package com.biz.crm.role.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.base.BusinessException;
import com.biz.crm.base.config.ThreadLocalUtil;
import com.biz.crm.button.service.MdmButtonRoleService;
import com.biz.crm.common.GlobalParam;
import com.biz.crm.common.PageResult;
import com.biz.crm.common.param.ParameterParam;
import com.biz.crm.crmlog.handle.util.CrmLogSendUtil;
import com.biz.crm.eunm.CodeRuleEnum;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.function.service.MdmFunctionRoleService;
import com.biz.crm.function.service.MdmFunctionService;
import com.biz.crm.nebular.mdm.constant.UserTypeEnum;
import com.biz.crm.nebular.mdm.role.req.*;
import com.biz.crm.nebular.mdm.role.resp.MdmRoleRespVo;
import com.biz.crm.nebular.mdm.role.resp.MdmRoleSelectRespVo;
import com.biz.crm.nebular.mdm.user.resp.MdmUserRespVo;
import com.biz.crm.permission.model.MdmDataPermissionEntity;
import com.biz.crm.permission.service.IMdmDataPermissionService;
import com.biz.crm.position.service.MdmPositionRoleService;
import com.biz.crm.position.service.MdmPositionService;
import com.biz.crm.position.service.MdmPositionUserService;
import com.biz.crm.positionlevel.service.MdmPositionLevelRoleService;
import com.biz.crm.role.mapper.MdmRoleMapper;
import com.biz.crm.role.mode.MdmRoleEntity;
import com.biz.crm.role.service.MdmRoleService;
import com.biz.crm.tableconfig.service.IMdmFunctionSubButtonService;
import com.biz.crm.user.service.MdmUserRelRoleService;
import com.biz.crm.user.service.MdmUserService;
import com.biz.crm.util.*;
import com.biz.crm.util.cache.TableConfigUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author liaohua
 * @date 2020/10/22 11:32 上午
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "MdmRoleServiceExpandImpl")
//TODO 角色模块的依赖问题特别严重，需要尽快进行解耦，RBAC模块应该处于 用户、岗位、职位、组织机构等模块的底层，所以不能直接调用这些模块的接口，否则会引循环依赖
public class MdmRoleServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<MdmRoleMapper, MdmRoleEntity> implements MdmRoleService {

    @Resource
    private MdmRoleMapper mdmRoleMapper;

    @Resource
    private MdmPositionRoleService mdmPositionRoleService;

    @Resource
    private MdmPositionLevelRoleService mdmPositionLevelRoleService;

    @Resource
    private MdmPositionService mdmPositionService;

    @Resource
    private MdmFunctionService mdmFunctionService;

    @Resource
    private IMdmFunctionSubButtonService mdmFunctionSubButtonService;

    @Resource
    private MdmFunctionRoleService mdmFunctionRoleService;

    @Resource
    private MdmButtonRoleService mdmButtonRoleService;

    @Resource
    private IMdmDataPermissionService mdmDataPermissionService;

    @Autowired
    private CrmLogSendUtil crmLogSendUtil;

    @Resource
    private MdmUserService mdmUserService;

    @Resource
    private MdmUserRelRoleService mdmUserRelRoleService;

    @Resource
    private MdmPositionUserService mdmPositionUserService;

    @Override
    public PageResult<MdmRoleRespVo> findList(MdmRoleReqVo mdmRoleReqVo) {
        Page<MdmRoleReqVo> page = PageUtil.buildPage(mdmRoleReqVo.getPageNum(), mdmRoleReqVo.getPageSize());
        List<MdmRoleRespVo> list = mdmRoleMapper.findList(page, mdmRoleReqVo);
        return PageResult.<MdmRoleRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    @Override
    public MdmRoleRespVo detail(String id, String roleCode) {
        if (StringUtils.isNotEmpty(id) || StringUtils.isNotEmpty(roleCode)) {
            MdmRoleEntity one = this.lambdaQuery()
                    .eq(!StringUtils.isEmpty(id), MdmRoleEntity::getId, id)
                    .eq(!StringUtils.isEmpty(roleCode), MdmRoleEntity::getRoleCode, roleCode)
                    .one();
            return CrmBeanUtil.copy(one, MdmRoleRespVo.class);
        }
        return null;
    }

    @Override
    public List<MdmRoleRespVo> queryBatchByRoleCodeList(List<String> roleCodeList) {
        if (CollectionUtil.listNotEmptyNotSizeZero(roleCodeList)) {
            List<MdmRoleEntity> list = this.lambdaQuery()
                    .in(MdmRoleEntity::getRoleCode, roleCodeList)
                    .list();
            if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
                return CrmBeanUtil.copyList(list, MdmRoleRespVo.class);
            }
        }
        return new ArrayList<>();
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void save(MdmRoleReqVo mdmRoleReqVo) {
        mdmRoleReqVo.setId(null);
        if (StringUtils.isBlank(mdmRoleReqVo.getRoleName())) {
            throw new BusinessException("角色名称不能为空。");
        }
//        Assert.hasText(mdmRoleReqVo.getEnableStatus(), "启用禁用状态必填");
        Integer count = this.lambdaQuery()
                .eq(MdmRoleEntity::getRoleName, mdmRoleReqVo.getRoleName())
                .count();
        Assert.isTrue(count == null || count < 1, "当前角色名称已存在，请重新输入");
        if (StringUtils.isBlank(mdmRoleReqVo.getRoleCode())) {
            mdmRoleReqVo.setRoleCode(CodeUtil.generateCode(CodeRuleEnum.MDM_ROLE_CODE.getCode()));
        } else {
//            Assert.isTrue(mdmRoleReqVo.getRoleCode().matches("^[A-Za-z0-9]+$"), "角色编码请输入数字和字母组合");
            MdmRoleEntity roleCodeEntity = this.lambdaQuery()
                    .eq(MdmRoleEntity::getRoleCode, mdmRoleReqVo.getRoleCode())
                    .select(MdmRoleEntity::getId)
                    .last(DatabaseTypeUtil.SEGMENT)
                    .one();
            Assert.isNull(roleCodeEntity, "当前角色编码已存在，请重新输入");
        }
        MdmRoleEntity entity = CrmBeanUtil.copy(mdmRoleReqVo, MdmRoleEntity.class);
        this.save(entity);
        Object menuObject = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        if (menuObject != null) {
            crmLogSendUtil.sendForAdd(menuObject.toString(), entity.getId(), null, mdmRoleReqVo);
        }

    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(MdmRoleReqVo mdmRoleReqVo) {
        String id = mdmRoleReqVo.getId();
        Assert.hasText(id, "数据主键不能为空");
        String roleCode = mdmRoleReqVo.getRoleCode();
        Assert.hasText(roleCode, "角色编码不能为空");
        String roleName = mdmRoleReqVo.getRoleName();
        Assert.hasText(roleName, "角色名称不能为空");
        MdmRoleEntity oldEntity = this.getById(id);
        Assert.notNull(oldEntity, "未找到角色");

        List<MdmRoleEntity> collect = this.lambdaQuery()
                .eq(MdmRoleEntity::getRoleCode, roleCode)
                .select(MdmRoleEntity::getId)
                .list()
                .stream().filter(x -> !id.equals(x.getId())).collect(Collectors.toList());
        Assert.isTrue(CollectionUtil.listEmpty(collect),"角色编码已经存在");

        List<MdmRoleEntity> collect2 = this.lambdaQuery()
                .eq(MdmRoleEntity::getRoleName, roleName)
                .select(MdmRoleEntity::getId)
                .list()
                .stream().filter(x -> !id.equals(x.getId())).collect(Collectors.toList());
        Assert.isTrue(CollectionUtil.listEmpty(collect2),"角色名称已经存在");
        MdmRoleEntity mdmRoleEntity = this.getById(mdmRoleReqVo.getId());
        MdmRoleReqVo oldObject = CrmBeanUtil.copy(mdmRoleEntity,MdmRoleReqVo.class);

        this.updateById(CrmBeanUtil.copy(mdmRoleReqVo, MdmRoleEntity.class));
        Object menuObject = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        if (menuObject != null) {
            MdmRoleEntity mdmRoleEntityNew = this.getById(mdmRoleReqVo.getId());
            MdmRoleReqVo newObject = CrmBeanUtil.copy(mdmRoleEntityNew, MdmRoleReqVo.class);
            crmLogSendUtil.sendForUpdate(menuObject.toString(), mdmRoleReqVo.getId(), newObject.getRoleName(), oldObject, newObject);
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteBatch(MdmRoleReqVo mdmRoleReqVo) {
        List<String> ids = mdmRoleReqVo.getIds();
        Assert.notEmpty(ids, "id集合必填");
        StringBuilder errorMsg = new StringBuilder();
        List<MdmRoleEntity> list = this.lambdaQuery()
                .in(MdmRoleEntity::getId, ids)
                .list();
        Assert.notEmpty(list, "未找到角色");
        for (MdmRoleEntity role : list) {

            List<String> positionCodeList = mdmPositionRoleService.findPositionCodeCodeByRole(Collections.singletonList(role.getRoleCode()));
            if(!CollectionUtils.isEmpty(positionCodeList)){
                errorMsg.append("角色:[").append(role.getRoleName()).append("]").append("关联职位,无法删除").append(System.lineSeparator());
            }

            List<String> functionCodeList = mdmFunctionRoleService.findFunctionCodeByRole(role.getRoleCode());
            if(!CollectionUtils.isEmpty(functionCodeList)){
                errorMsg.append("角色:[").append(role.getRoleName()).append("]").append("关联菜单,无法删除").append(System.lineSeparator());
            }

//            //移除角色与职位关联
//            mdmPositionRoleService.removeRoleCode(role.getRoleCode());
//
//            //移除角色关联的菜单
//            mdmFunctionRoleService.deleteByRoleCodeList(Collections.singletonList(role.getRoleCode()));

        }
        String msg = String.valueOf(errorMsg);
        if (!StringUtils.isEmpty(msg)) {
            throw new BusinessException(msg);
        }

        List<String> roleCodeList = list.stream().map(MdmRoleEntity::getRoleCode).collect(Collectors.toList());

        //移除角色与职位级别的关联关系
        mdmPositionLevelRoleService.removeRoleCodes(roleCodeList);
        //移除角色关联的按钮
        mdmButtonRoleService.deleteByRoleCodeList(roleCodeList);
        //移除数据权限
        mdmDataPermissionService.lambdaUpdate()
                .in(MdmDataPermissionEntity::getRoleCode, roleCodeList)
                .remove();
        Object menuObject = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        if (menuObject != null) {
            for (MdmRoleEntity role :
                    list) {
                crmLogSendUtil.sendForDel(menuObject.toString(),
                        role.getId(), role.getRoleCode(), CrmBeanUtil.copy(role, MdmRoleReqVo.class));
            }
        }
        this.removeByIds(ids);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void enableBatch(MdmRoleReqVo mdmRoleReqVo) {
        List<String> ids = mdmRoleReqVo.getIds();
        Assert.notEmpty(ids, "ids不能为空");
        this.lambdaUpdate()
                .in(MdmRoleEntity::getId, ids)
                .set(MdmRoleEntity::getEnableStatus, CrmEnableStatusEnum.ENABLE.getCode())
                .update();
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void disableBatch(MdmRoleReqVo mdmRoleReqVo) {
        List<String> ids = mdmRoleReqVo.getIds();
        Assert.notEmpty(ids, "ids不能为空");
        this.lambdaUpdate()
                .in(MdmRoleEntity::getId, ids)
                .set(MdmRoleEntity::getEnableStatus, CrmEnableStatusEnum.DISABLE.getCode())
                .update();
    }

    @Override
    public List<MdmRoleRespVo> roleList(MdmRoleReqVo reqVo) {
        List<MdmRoleEntity> list = this.lambdaQuery()
                .eq(StringUtils.isNotEmpty(reqVo.getEnableStatus()), MdmRoleEntity::getEnableStatus, reqVo.getEnableStatus())
                .like(StringUtils.isNotEmpty(reqVo.getRoleCode()), MdmRoleEntity::getRoleCode, reqVo.getRoleCode())
                .in(!CollectionUtils.isEmpty(reqVo.getRoleCodeList()), MdmRoleEntity::getRoleCode, reqVo.getRoleCodeList())
                .like(StringUtils.isNotEmpty(reqVo.getRoleName()), MdmRoleEntity::getRoleName, reqVo.getRoleName())
                .eq(MdmRoleEntity::getEnableStatus, CrmEnableStatusEnum.ENABLE.getCode())
                .list();
        return CrmBeanUtil.copyList(list, MdmRoleRespVo.class);
    }

    @Override
    public List<MdmRoleSelectRespVo> roleSelectList(MdmRoleSelectReqVo reqVo) {
        Integer pageSize = reqVo.getPageSize();
        if (pageSize == null || pageSize == 0) {
            pageSize = Integer.valueOf(ParamUtil.getParameterValue(ParameterParam.SELECT_BOX_OPTION_SIZE));
        }
        List<MdmRoleSelectRespVo> list = new ArrayList<>();
        Set<String> codeList = new HashSet<>(16);
        if (StringUtils.isNotEmpty(reqVo.getSelectedCode())) {
            codeList.add(reqVo.getSelectedCode());
        }
        if (CollectionUtil.listNotEmptyNotSizeZero(reqVo.getSelectedCodeList())) {
            codeList.addAll(reqVo.getSelectedCodeList());
        }
        if (!codeList.isEmpty()) {
            reqVo.setSelectedCodeList(new ArrayList<>(codeList));
        }
        if (pageSize > 0) {
            list.addAll(mdmRoleMapper.roleSelectList(new Page<>(1L, pageSize, false), reqVo));
        }
        return list;
    }

    @Override
    public MdmRoleFunctionRespVo roleFunctionTree(MdmRoleFunctionReqVo mdmRoleFunctionReqVo) {
        String roleCode = mdmRoleFunctionReqVo.getRoleCode();
        Assert.hasText(roleCode, "角色编码不能为空");
        MdmRoleEntity mdmRoleEntity = this.lambdaQuery()
                .eq(MdmRoleEntity::getRoleCode, roleCode)
                .select(MdmRoleEntity::getRoleCode, MdmRoleEntity::getRoleName, MdmRoleEntity::getRemarks)
                .one();
        Assert.notNull(mdmRoleEntity, "角色不存在");
        List<MdmRoleFunctionTreeVo> mdmRoleFunctionTreeVos = mdmFunctionService.roleFunctionTree(mdmRoleFunctionReqVo);
        MdmRoleFunctionRespVo mdmRoleFunctionRespVo = new MdmRoleFunctionRespVo();
        mdmRoleFunctionRespVo.setRoleCode(mdmRoleEntity.getRoleCode());
        mdmRoleFunctionRespVo.setRoleName(mdmRoleEntity.getRoleName());
        mdmRoleFunctionRespVo.setRemarks(mdmRoleEntity.getRemarks());
        mdmRoleFunctionRespVo.setMdmRoleFunctionTreeVos(mdmRoleFunctionTreeVos);
        return mdmRoleFunctionRespVo;
    }

    @Override
    public List<MdmRoleSubButtonVo> roleButtonList(MdmRoleSubButtonReqVo mdmRoleSubButtonReqVo) {
        return mdmFunctionSubButtonService.buttonList(mdmRoleSubButtonReqVo);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void saveFunctionButton(MdmRoleFunctionButtonStoreVo mdmRoleFunctionButtonStoreVo) {
        String roleCode = mdmRoleFunctionButtonStoreVo.getRoleCode();
        Assert.hasText(roleCode, "角色编码不能为空");
        String roleName = mdmRoleFunctionButtonStoreVo.getRoleName();
        Assert.hasText(roleName, "角色名称不能为空");
        Integer count = this.lambdaQuery()
                .eq(MdmRoleEntity::getRoleCode, roleCode)
                .count();
        Assert.isTrue(count != null && count > 0, "角色不存在");
        this.lambdaUpdate()
                .eq(MdmRoleEntity::getRoleCode, roleCode)
                .set(MdmRoleEntity::getRoleName, roleName)
                .set(!StringUtils.isEmpty(mdmRoleFunctionButtonStoreVo.getRemarks()),MdmRoleEntity::getRemarks,mdmRoleFunctionButtonStoreVo.getRemarks())
                .update();

        List<MdmRoleFunctionTreeVo> mdmRoleFunctionTreeVos = mdmRoleFunctionButtonStoreVo.getMdmRoleFunctionTreeVos();
        List<String> functionCodeList = new ArrayList<>();
        if(!CollectionUtils.isEmpty(mdmRoleFunctionTreeVos)){
            setUpFunction(functionCodeList, mdmRoleFunctionTreeVos);
        }
        mdmFunctionRoleService.update(roleCode, functionCodeList);
        List<MdmRoleSubButtonVo> mdmRoleSubButtonVos = mdmRoleFunctionButtonStoreVo.getMdmRoleSubButtonVos();
        if(!CollectionUtils.isEmpty(mdmRoleSubButtonVos)){
            mdmButtonRoleService.setUp(roleCode, mdmRoleSubButtonVos);
        }
        //清除页面配置缓存
        TableConfigUtil.removeAll();
    }

    @Override
    public void switchFunctionButton(MdmSwitchFunctionButtonReqVo mdmSwitchFunctionButtonReqVo) {
        mdmFunctionSubButtonService.switchFunctionButton(mdmSwitchFunctionButtonReqVo);
    }

    @Override
    public List<String> getRoleByUserName(String userName, String positionCode) {
        Assert.hasText(userName, "缺失账号");
        MdmUserRespVo user = mdmUserService.queryByUserName(userName);
        if (user != null) {
            if (UserTypeEnum.USER.getCode().equals(user.getUserType())&&!StringUtils.isEmpty(positionCode)) {
                return mdmPositionRoleService.findRoleCodeByPosition(Collections.singletonList(positionCode));
            } else {
                return mdmUserRelRoleService.findRoleCodeListByUserName(userName);
            }
        }
        return Collections.emptyList();
    }

    @Override
    public List<String> getRoleByUserName(String userName) {
        Assert.hasText(userName, "缺失账号");
        MdmUserRespVo user = mdmUserService.queryByUserName(userName);
        if (user != null) {
            if (UserTypeEnum.USER.getCode().equals(user.getUserType())) {
                //企业用户
                List<String> positionCodeList = mdmPositionUserService.getPositionCodeListByUserName(userName);
                if (CollectionUtil.listNotEmptyNotSizeZero(positionCodeList)) {
                   return mdmPositionRoleService.findRoleCodeByPosition(positionCodeList);
                }
            } else {
                return mdmUserRelRoleService.findRoleCodeListByUserName(userName);
            }
        }
        return new ArrayList<>();
    }

    @Override
    public List<MdmRoleRespVo> obtainUserRoleList(MdmUserRoleReqVo mdmUserRoleReqVo) {
        String userName = mdmUserRoleReqVo.getUserName();
        if(StringUtils.isEmpty(userName)){
            return Collections.emptyList();
        }
        MdmUserRespVo user = mdmUserService.queryByUserName(userName);
        if (user == null) {
            return Collections.emptyList();
        }
        String userType = user.getUserType();
        if(StringUtils.isEmpty(userType)){
            return Collections.emptyList();
        }
        mdmUserRoleReqVo.setUserType(userType);
        return mdmRoleMapper.obtainUserRoleList(mdmUserRoleReqVo);
    }

    public void setUpFunction(List<String> functionCodeList, List<MdmRoleFunctionTreeVo> mdmRoleFunctionTreeVos) {
        if(!CollectionUtils.isEmpty(mdmRoleFunctionTreeVos)){
            List<String> currentFunctionCodeList = mdmRoleFunctionTreeVos.stream()
                    .filter(x -> !StringUtils.isEmpty(x.getFunctionCode()))
                    .filter(x->!StringUtils.isEmpty(x.getFunctionSelect())&& YesNoEnum.yesNoEnum.ONE.getValue().equals(x.getFunctionSelect()))
                    .map(MdmRoleFunctionTreeVo::getFunctionCode)
                    .collect(Collectors.toList());
            if(!CollectionUtils.isEmpty(currentFunctionCodeList)){
                functionCodeList.addAll(currentFunctionCodeList);
            }
            List<MdmRoleFunctionTreeVo> collect = mdmRoleFunctionTreeVos.stream().flatMap(x -> {
                List<MdmRoleFunctionTreeVo> children = x.getChildren();
                if (children == null) {
                    children = new ArrayList<>();
                }
                return children.stream();
            }).collect(Collectors.toList());
            setUpFunction(functionCodeList,collect);
        }
    }

}
