package com.biz.crm.tableconfig.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.nebular.mdm.tableconfig.MdmColumnConfigPersonalBatchReqVo;
import com.biz.crm.nebular.mdm.tableconfig.MdmColumnConfigPersonalRedisVo;
import com.biz.crm.nebular.mdm.tableconfig.MdmColumnConfigPersonalReqVo;
import com.biz.crm.nebular.mdm.tableconfig.MdmColumnConfigPersonalRespVo;
import com.biz.crm.tableconfig.mapper.MdmColumnConfigPersonalMapper;
import com.biz.crm.tableconfig.model.MdmColumnConfigEntity;
import com.biz.crm.tableconfig.model.MdmColumnConfigPersonalEntity;
import com.biz.crm.tableconfig.service.IMdmColumnConfigService;
import com.biz.crm.tableconfig.service.MdmColumnConfigPersonalService;
import com.biz.crm.util.*;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import javax.annotation.Resource;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 字段配置个性设置接口实现
 *
 * @author Tao.Chen
 * @date 2021-02-20 15:30:42
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "MdmColumnConfigPersonalServiceExpandImpl")
public class MdmColumnConfigPersonalServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<MdmColumnConfigPersonalMapper, MdmColumnConfigPersonalEntity> implements MdmColumnConfigPersonalService {

    @Resource
    private MdmColumnConfigPersonalMapper mdmColumnConfigPersonalMapper;
    @Resource
    private IMdmColumnConfigService mdmColumnConfigService;

    @Override
    public List<MdmColumnConfigPersonalRespVo> findColumnConfigPersonalList(MdmColumnConfigPersonalBatchReqVo reqVo) {
        String positionCode = getCurPositionCode();
        Assert.hasText(reqVo.getParentCode(), "缺失菜单编码");
        Assert.hasText(reqVo.getFunctionCode(), "缺失功能编码");

        //查询列表配置
        List<MdmColumnConfigEntity> columnList = mdmColumnConfigService.lambdaQuery()
                .eq(MdmColumnConfigEntity::getParentCode, reqVo.getParentCode())
                .eq(MdmColumnConfigEntity::getFunctionCode, reqVo.getFunctionCode())
                .select(MdmColumnConfigEntity::getField, MdmColumnConfigEntity::getVisible, MdmColumnConfigEntity::getTitle, MdmColumnConfigEntity::getFormorder)
                .list();
        if (CollectionUtil.listNotEmptyNotSizeZero(columnList)) {

            //转换
            List<MdmColumnConfigPersonalRespVo> collect = columnList.stream().map(x -> {
                MdmColumnConfigPersonalRespVo personalRespVo = new MdmColumnConfigPersonalRespVo();
                personalRespVo.setTitle(x.getTitle());
                personalRespVo.setField(x.getField());
                personalRespVo.setVisible(x.getVisible());
                personalRespVo.setFormOrder(StringUtils.isNotEmpty(x.getFormorder()) ? x.getFormorder() : "100");
                return personalRespVo;
            }).collect(Collectors.toList());

            //查询个性配置
            List<MdmColumnConfigPersonalEntity> list = this.lambdaQuery()
                    .eq(MdmColumnConfigPersonalEntity::getPositionCode, positionCode)
                    .eq(MdmColumnConfigPersonalEntity::getParentCode, reqVo.getParentCode())
                    .eq(MdmColumnConfigPersonalEntity::getFunctionCode, reqVo.getFunctionCode())
                    .orderByAsc(MdmColumnConfigPersonalEntity::getFormOrder)
                    .list();
            if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
                Map<String, MdmColumnConfigPersonalEntity> filedMap = list.stream().collect(Collectors.toMap(MdmColumnConfigPersonalEntity::getField, v -> v));
                for (MdmColumnConfigPersonalRespVo item :
                        collect) {
                    if (!filedMap.containsKey(item.getField())) {
                        item.setFormOrder("100");
                    } else {
                        MdmColumnConfigPersonalEntity mdmColumnConfigPersonalEntity = filedMap.get(item.getField());
                        item.setVisible(mdmColumnConfigPersonalEntity.getVisible());
                        item.setFormOrder(mdmColumnConfigPersonalEntity.getFormOrder());
                    }
                }
            }
            collect.sort(Comparator.comparing(x -> Optional.ofNullable(x.getFormOrder()).map(Integer::valueOf).orElse(100)));
            return collect;
        }
        return new ArrayList<>();
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void saveColumnConfigPersonal(MdmColumnConfigPersonalBatchReqVo reqVo) {
        String positionCode = getCurPositionCode();
        Assert.hasText(reqVo.getParentCode(), "缺失菜单编码");
        Assert.hasText(reqVo.getFunctionCode(), "缺失功能编码");
        Assert.notEmpty(reqVo.getColumnList(), "缺失字段配置");
        for (MdmColumnConfigPersonalReqVo item :
                reqVo.getColumnList()) {
            Assert.hasText(item.getField(), "缺失属性");
            Assert.hasText(item.getVisible(), "缺失显示模式");
            Assert.hasText(item.getFormOrder(), "缺失显示顺序");
            Assert.isTrue(YesNoEnum.yesNoEnum.ONE.getValue().equals(item.getVisible()) || YesNoEnum.yesNoEnum.ZERO.getValue().equals(item.getVisible()), "显示模式参数不合法");
            item.setFunctionCode(reqVo.getFunctionCode());
            item.setParentCode(reqVo.getParentCode());
            item.setPositionCode(positionCode);
        }
        this.lambdaUpdate()
                .eq(MdmColumnConfigPersonalEntity::getPositionCode, positionCode)
                .eq(MdmColumnConfigPersonalEntity::getParentCode, reqVo.getParentCode())
                .eq(MdmColumnConfigPersonalEntity::getFunctionCode, reqVo.getFunctionCode())
                .remove();
        this.saveBatch(CrmBeanUtil.copyList(reqVo.getColumnList(), MdmColumnConfigPersonalEntity.class));
        //清除缓存
        ColumnConfigPersonalUtil.deleteColumnConfigPersonal(reqVo.getParentCode(), reqVo.getFunctionCode(), positionCode);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteColumnConfigPersonal(MdmColumnConfigPersonalBatchReqVo reqVo) {
        String positionCode = getCurPositionCode();
        Assert.hasText(reqVo.getParentCode(), "缺失菜单编码");
        Assert.hasText(reqVo.getFunctionCode(), "缺失功能编码");
        this.lambdaUpdate()
                .eq(MdmColumnConfigPersonalEntity::getPositionCode, positionCode)
                .eq(MdmColumnConfigPersonalEntity::getParentCode, reqVo.getParentCode())
                .eq(MdmColumnConfigPersonalEntity::getFunctionCode, reqVo.getFunctionCode())
                .remove();
        //清除缓存
        ColumnConfigPersonalUtil.deleteColumnConfigPersonal(reqVo.getParentCode(), reqVo.getFunctionCode(), positionCode);

    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteColumnConfig(String parentCode, String functionCode) {
        Assert.hasText(parentCode, "缺失菜单编码");
        Assert.hasText(functionCode, "缺失功能编码");
        this.lambdaUpdate()
                .eq(MdmColumnConfigPersonalEntity::getParentCode, parentCode)
                .eq(MdmColumnConfigPersonalEntity::getFunctionCode, functionCode)
                .remove();
        //清除缓存
        ColumnConfigPersonalUtil.deleteColumnConfig(parentCode, functionCode);

    }

    @Override
    public List<MdmColumnConfigPersonalRedisVo> findColumnConfigPersonalRedisList(String parentCode, String functionCode, String positionCode) {
        Assert.hasText(positionCode, "缺失职位编码");
        Assert.hasText(parentCode, "缺失菜单编码");
        Assert.hasText(functionCode, "缺失功能编码");

        List<MdmColumnConfigPersonalRedisVo> collect = new ArrayList<>();

        //查询个性配置
        List<MdmColumnConfigPersonalEntity> list = this.lambdaQuery()
                .eq(MdmColumnConfigPersonalEntity::getPositionCode, positionCode)
                .eq(MdmColumnConfigPersonalEntity::getParentCode, parentCode)
                .eq(MdmColumnConfigPersonalEntity::getFunctionCode, functionCode)
                .orderByAsc(MdmColumnConfigPersonalEntity::getFormOrder)
                .list();
        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            collect.addAll(list.stream().map(x -> {
                MdmColumnConfigPersonalRedisVo personalRespVo = new MdmColumnConfigPersonalRedisVo();
                personalRespVo.setField(x.getField());
                personalRespVo.setVisible(x.getVisible());
                personalRespVo.setFormOrder(x.getFormOrder());
                return personalRespVo;
            }).collect(Collectors.toList()));
        }
        collect.sort(Comparator.comparing(x -> Optional.ofNullable(x.getFormOrder()).map(Integer::valueOf).orElse(100)));
        return collect;
    }

    private String getCurPositionCode() {
        UserRedis user = UserUtils.getUser();
        Assert.notNull(user, "请先登录");
        Assert.hasText(user.getPoscode(), "请先登录");
        return user.getPoscode();
    }
}
