package com.biz.crm.tableconfig.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.common.PageResult;
import com.biz.crm.common.param.CacheManagerConstant;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.function.service.MdmFunctionService;
import com.biz.crm.nebular.mdm.constant.DictConstant;
import com.biz.crm.nebular.mdm.dict.resp.DictDataVo;
import com.biz.crm.nebular.mdm.function.MdmFunctionReqVo;
import com.biz.crm.nebular.mdm.function.MdmFunctionRespVo;
import com.biz.crm.nebular.mdm.pageconfig.*;
import com.biz.crm.nebular.mdm.permission.MdmDataPermissionReqVo;
import com.biz.crm.nebular.mdm.permission.MdmPermissionObjReqVo;
import com.biz.crm.nebular.mdm.permission.MdmPermissionObjRespVo;
import com.biz.crm.permission.service.IMdmDataPermissionService;
import com.biz.crm.tableconfig.mapper.MdmFunctionSubMapper;
import com.biz.crm.tableconfig.model.MdmColumnConfigEntity;
import com.biz.crm.tableconfig.model.MdmFunctionSubButtonEntity;
import com.biz.crm.tableconfig.model.MdmFunctionSubEntity;
import com.biz.crm.tableconfig.service.IMdmColumnConfigService;
import com.biz.crm.tableconfig.service.IMdmFunctionSubButtonService;
import com.biz.crm.tableconfig.service.IMdmFunctionSubService;
import com.biz.crm.util.CrmBeanUtil;
import com.biz.crm.util.DictUtil;
import com.biz.crm.util.cache.TableConfigUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.cache.annotation.CacheConfig;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 下级菜单表接口实现
 *
 * @author zxw
 * @date 2020-11-21 00:22:17
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "MdmFunctionSubServiceExpandImpl")
@CacheConfig(cacheManager = CacheManagerConstant.CACHE_MANAGER, cacheNames = CacheManagerConstant.MDM_FUNCTION_SUB)
public class MdmFunctionSubServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<MdmFunctionSubMapper, MdmFunctionSubEntity> implements IMdmFunctionSubService {

    @Autowired
    private MdmFunctionSubMapper mdmFunctionSubMapper;
    @Autowired
    private MdmFunctionService mdmFunctionService;
    @Autowired
    private IMdmFunctionSubButtonService mdmFunctionSubButtonService;
    @Autowired
    private IMdmColumnConfigService mdmColumnConfigService;
    @Autowired
    private IMdmDataPermissionService mdmDataPermissionService;

    @Override
    public PageResult<MdmFunctionSubRespVo> findList(MdmFunctionSubReqVo reqVo) {
        Page<MdmFunctionSubRespVo> page = new Page<>(reqVo.getPageNum(), reqVo.getPageSize());
        List<MdmFunctionSubRespVo> list = mdmFunctionSubMapper.findList(page, reqVo);
        return PageResult.<MdmFunctionSubRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    @Override
    public MdmFunctionSubRespVo query(String id) {
        Assert.hasText(id, "id不能为空");
        MdmFunctionSubEntity one = this.lambdaQuery()
                .eq(MdmFunctionSubEntity::getId, id)
                .one();
        return CrmBeanUtil.copy(one, MdmFunctionSubRespVo.class);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    @CacheEvict(allEntries = true)
    public void save(MdmFunctionSubReqVo reqVo) {
        String parentCode = reqVo.getParentCode();
        Assert.hasText(parentCode, "菜单编码不能为空");
        valid(parentCode);
        String functionCode = reqVo.getFunctionCode();
        Assert.hasText(functionCode, "功能编码不能为空");
        Integer count = this.lambdaQuery()
                .eq(MdmFunctionSubEntity::getParentCode, parentCode)
                .eq(MdmFunctionSubEntity::getFunctionCode, functionCode)
                .count();
        Assert.isTrue(count == null || count < 1, "功能编码已存在");
        MdmFunctionSubEntity entity = CrmBeanUtil.copy(reqVo, MdmFunctionSubEntity.class);
        this.save(entity);
        //清除页面配置缓存
        TableConfigUtil.removeListConfig(reqVo.getParentCode(), reqVo.getFunctionCode());
    }

    private void valid(String parentCode) {
        Assert.hasText(parentCode, "菜单编码不能为空");
        MdmFunctionReqVo mdmFunctionReqVo = new MdmFunctionReqVo();
        mdmFunctionReqVo.setFunctionCode(parentCode);
        MdmFunctionRespVo mdmFunctionRespVo = mdmFunctionService.query(mdmFunctionReqVo);
        Assert.notNull(mdmFunctionRespVo, "菜单不存在");
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    @CacheEvict(allEntries = true)
    public void update(MdmFunctionSubReqVo reqVo) {
        String functionCode = reqVo.getFunctionCode();
        Assert.hasText(functionCode, "功能编码不能为空");
        String functionName = reqVo.getFunctionName();
        Assert.hasText(functionName, "名称不能为空");
        String parentCode = reqVo.getParentCode();
        Assert.hasText(reqVo.getId(), "主键id不能为空");
        MdmFunctionSubEntity entity = this.lambdaQuery()
                .eq(MdmFunctionSubEntity::getId, reqVo.getId())
                .select(MdmFunctionSubEntity::getId, MdmFunctionSubEntity::getFunctionCode)
                .one();
        Assert.notNull(entity, "功能列表不存在");
        valid(parentCode);
        long count = this.lambdaQuery()
                .eq(MdmFunctionSubEntity::getParentCode, parentCode)
                .eq(MdmFunctionSubEntity::getFunctionCode, reqVo.getFunctionCode())
                .select(MdmFunctionSubEntity::getId)
                .list().stream().filter(x -> !entity.getId().equals(x.getId()))
                .count();
        Assert.isTrue(count < 1, "功能编码已存在");
        this.updateById(CrmBeanUtil.copy(reqVo, MdmFunctionSubEntity.class));
        if (!functionCode.equals(entity.getFunctionCode())) {
            mdmFunctionSubButtonService.lambdaUpdate()
                    .eq(MdmFunctionSubButtonEntity::getParentCode, parentCode)
                    .eq(MdmFunctionSubButtonEntity::getFunctionCode, entity.getFunctionCode())
                    .set(MdmFunctionSubButtonEntity::getFunctionCode, functionCode)
                    .update();
            mdmColumnConfigService.lambdaUpdate()
                    .eq(MdmColumnConfigEntity::getParentCode, parentCode)
                    .eq(MdmColumnConfigEntity::getFunctionCode, entity.getFunctionCode())
                    .set(MdmColumnConfigEntity::getFunctionCode, functionCode)
                    .update();
        }
        //清除页面配置缓存
        TableConfigUtil.removeListConfig(reqVo.getParentCode(), reqVo.getFunctionCode());
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    @CacheEvict(allEntries = true)
    public void deleteBatch(List<String> ids) {
        Assert.notEmpty(ids, "id集合不能为空");
        List<MdmFunctionSubEntity> list = this.lambdaQuery()
                .in(MdmFunctionSubEntity::getId, ids)
                .select(MdmFunctionSubEntity::getParentCode, MdmFunctionSubEntity::getFunctionCode)
                .list();
        if (!CollectionUtils.isEmpty(list)) {
            List<String> parentCodeList = list.stream()
                    .filter(x -> x != null && !StringUtils.isEmpty(x.getParentCode()))
                    .map(MdmFunctionSubEntity::getParentCode)
                    .collect(Collectors.toList());
            List<String> functionCodeList = list.stream()
                    .filter(x -> x != null && !StringUtils.isEmpty(x.getFunctionCode()))
                    .map(MdmFunctionSubEntity::getFunctionCode)
                    .collect(Collectors.toList());
            if (!CollectionUtils.isEmpty(parentCodeList) && !CollectionUtils.isEmpty(functionCodeList)) {
                //按钮删除
                mdmFunctionSubButtonService.lambdaUpdate()
                        .in(MdmFunctionSubButtonEntity::getParentCode, parentCodeList)
                        .in(MdmFunctionSubButtonEntity::getFunctionCode, functionCodeList)
                        .remove();
                //字段删除
                mdmColumnConfigService.lambdaUpdate()
                        .in(MdmColumnConfigEntity::getParentCode, parentCodeList)
                        .in(MdmColumnConfigEntity::getFunctionCode, functionCodeList)
                        .remove();
            }
            MdmFunctionSubEntity mdmFunctionSubEntity = list.get(0);
            String parentCode = mdmFunctionSubEntity.getParentCode();
            if (!StringUtils.isEmpty(parentCode)) {
                //清除页面配置缓存
                TableConfigUtil.removeFunction(parentCode);
            }
        }
        this.removeByIds(ids);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    @CacheEvict(allEntries = true)
    public void enableBatch(MdmFunctionSubReqVo reqVo) {
        List<String> ids = reqVo.getIds();
        Assert.notEmpty(ids, "id集合不能为空");
        this.lambdaUpdate()
                .in(MdmFunctionSubEntity::getId, ids)
                .set(MdmFunctionSubEntity::getEnableStatus, CrmEnableStatusEnum.ENABLE.getCode())
                .update();
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    @CacheEvict(allEntries = true)
    public void disableBatch(MdmFunctionSubReqVo reqVo) {
        List<String> ids = reqVo.getIds();
        Assert.notEmpty(ids, "id集合不能为空");
        this.lambdaUpdate()
                .in(MdmFunctionSubEntity::getId, ids)
                .set(MdmFunctionSubEntity::getEnableStatus, CrmEnableStatusEnum.DISABLE.getCode())
                .update();
    }

    @Override
    public PageResult<MdmFunctionRespVo> functionPage(MdmFunctionReqVo mdmFunctionReqVo) {
        mdmFunctionReqVo.setPageConfig(YesNoEnum.yesNoEnum.YES.getValue());
        return mdmFunctionService.functionPage(mdmFunctionReqVo);
    }

    @Override
    @Cacheable(sync = true)
    public List<MdmFunctionSubRespVo> findSubList(MdmFunctionSubReqVo mdmFunctionSubReqVo) {
        return mdmFunctionSubMapper.findSubList(mdmFunctionSubReqVo);
    }

    @Override
    public List<MdmFunctionSubRespVo> findPermissionList(MdmFunctionSubReqVo mdmFunctionSubReqVo) {
        List<MdmFunctionSubEntity> list = this.lambdaQuery()
                .eq(!StringUtils.isEmpty(mdmFunctionSubReqVo.getFunctionType()), MdmFunctionSubEntity::getFunctionType, mdmFunctionSubReqVo.getFunctionType())
                .in(!CollectionUtils.isEmpty(mdmFunctionSubReqVo.getParentCodeList()), MdmFunctionSubEntity::getParentCode, mdmFunctionSubReqVo.getParentCodeList())
                .isNotNull(MdmFunctionSubEntity::getPermissionObj)
                .select(MdmFunctionSubEntity::getFunctionCode, MdmFunctionSubEntity::getParentCode)
                .list();
        return CrmBeanUtil.copyList(list, MdmFunctionSubRespVo.class);
    }

    @Override
    public List<MdmPermissionObjRespVo> objList(MdmFunctionSubReqVo mdmFunctionSubReqVo) {
        String functionCode = mdmFunctionSubReqVo.getParentCode();
        String listConfigCode = mdmFunctionSubReqVo.getFunctionCode();
        List<DictDataVo> permissionTree = DictUtil.tree(DictConstant.PERMISSION_OBJ);
        if (CollectionUtils.isEmpty(permissionTree)) {
            return new ArrayList<>();
        }
        MdmFunctionSubEntity mdmFunctionSubEntity = this.lambdaQuery()
                .eq(MdmFunctionSubEntity::getParentCode, functionCode)
                .eq(MdmFunctionSubEntity::getFunctionCode, listConfigCode)
                .select(MdmFunctionSubEntity::getPermissionObj)
                .one();
        log.info("[数据权限][已配置的权限对象]:{}", mdmFunctionSubEntity);
        if (mdmFunctionSubEntity == null || StringUtils.isEmpty(mdmFunctionSubEntity.getPermissionObj())) {
            return permissionTree.stream().map(x -> {
                MdmPermissionObjRespVo mdmPermissionObjRespVo = new MdmPermissionObjRespVo();
                mdmPermissionObjRespVo.setSelect(YesNoEnum.yesNoEnum.ZERO.getValue());
                mdmPermissionObjRespVo.setPermissionObjCode(x.getDictCode());
                mdmPermissionObjRespVo.setPermissionObjName(x.getDictValue());
                mdmPermissionObjRespVo.setRemarks(x.getDictValue());
                return mdmPermissionObjRespVo;
            }).collect(Collectors.toList());
        }

        List<String> permissionObjList = Arrays.asList(mdmFunctionSubEntity.getPermissionObj().split(","));
        return permissionTree
                .stream()
                .map(x -> {
                    MdmPermissionObjRespVo mdmPermissionObjRespVo = new MdmPermissionObjRespVo();
                    if (permissionObjList.contains(x.getDictCode())) {
                        mdmPermissionObjRespVo.setSelect(YesNoEnum.yesNoEnum.ONE.getValue());
                    } else {
                        mdmPermissionObjRespVo.setSelect(YesNoEnum.yesNoEnum.ZERO.getValue());
                    }
                    mdmPermissionObjRespVo.setPermissionObjCode(x.getDictCode());
                    mdmPermissionObjRespVo.setPermissionObjName(x.getDictValue());
                    mdmPermissionObjRespVo.setRemarks(x.getDictValue());
                    return mdmPermissionObjRespVo;
                }).collect(Collectors.toList());
    }

    @Override
    @CacheEvict(allEntries = true)
    public void saveObj(MdmPermissionObjReqVo mdmPermissionObjReqVo) {
        String functionCode = mdmPermissionObjReqVo.getFunctionCode();
        Assert.hasText(functionCode, "功能列表编码不能为空");
        String parentCode = mdmPermissionObjReqVo.getParentCode();
        Assert.hasText(parentCode, "菜单编码不能为空");
        Integer count = this.lambdaQuery()
                .eq(MdmFunctionSubEntity::getParentCode, parentCode)
                .eq(MdmFunctionSubEntity::getFunctionCode, functionCode)
                .count();
        Assert.isTrue(count != null && count > 0, "功能列表不存在");

        List<MdmPermissionObjRespVo> mdmPermissionObjRespVos = mdmPermissionObjReqVo.getMdmPermissionObjRespVos();
        if (CollectionUtils.isEmpty(mdmPermissionObjRespVos)) {
            this.lambdaUpdate()
                    .eq(MdmFunctionSubEntity::getParentCode, parentCode)
                    .eq(MdmFunctionSubEntity::getFunctionCode, functionCode)
                    .set(MdmFunctionSubEntity::getPermissionObj, null)
                    .update();
            return;
        }
        String collect = mdmPermissionObjRespVos.stream()
                .filter(x -> x != null && YesNoEnum.yesNoEnum.ONE.getValue().equals(x.getSelect()))
                .map(MdmPermissionObjRespVo::getPermissionObjCode)
                .collect(Collectors.joining(","));
        this.lambdaUpdate()
                .eq(MdmFunctionSubEntity::getParentCode, parentCode)
                .eq(MdmFunctionSubEntity::getFunctionCode, functionCode)
                .set(MdmFunctionSubEntity::getPermissionObj, collect)
                .update();
    }

    @Override
    public void buttonSetUp(MdmFunctionSubButtonReqVo mdmFunctionSubButtonReqVo) {
        mdmFunctionSubButtonService.setUp(mdmFunctionSubButtonReqVo);
    }

    @Override
    public List<MdmButtonConfigRespVo> buttonList(MdmFunctionSubReqVo mdmFunctionSubReqVo) {
        MdmFunctionSubButtonReqVo mdmFunctionSubButtonReqVo = new MdmFunctionSubButtonReqVo();
        mdmFunctionSubButtonReqVo.setParentCode(mdmFunctionSubReqVo.getParentCode());
        mdmFunctionSubButtonReqVo.setFunctionCode(mdmFunctionSubReqVo.getFunctionCode());
        return mdmFunctionSubButtonService.buttonList(mdmFunctionSubButtonReqVo);
    }

    @Override
    public void buttonRemove(List<String> ids) {
        mdmFunctionSubButtonService.buttonRemove(ids);
    }

    @Override
    public MdmButtonConfigRespVo buttonQuery(String id) {
        return mdmFunctionSubButtonService.buttonQuery(id);
    }

    @Override
    public MdmFunctionSubRespVo functionSubSearch(MdmFunctionSubReqVo mdmFunctionSubReqVo) {
        String parentCode = mdmFunctionSubReqVo.getParentCode();
        Assert.hasText(parentCode, "菜单编码不能为空");
        String functionCode = mdmFunctionSubReqVo.getFunctionCode();
        Assert.hasText(functionCode, "功能编码不能为空");
        MdmFunctionSubEntity one = this.lambdaQuery()
                .eq(MdmFunctionSubEntity::getParentCode, parentCode)
                .eq(MdmFunctionSubEntity::getFunctionCode, functionCode)
                .one();
        return CrmBeanUtil.copy(one, MdmFunctionSubRespVo.class);
    }

    @Override
    public MdmFunctionSubRespVo queryCondition(MdmFunctionSubReqVo mdmFunctionSubReqVo) {
        MdmFunctionSubEntity one = this.lambdaQuery()
                .eq(MdmFunctionSubEntity::getId, mdmFunctionSubReqVo.getId())
                .eq(MdmFunctionSubEntity::getFunctionCode, mdmFunctionSubReqVo.getFunctionCode())
                .one();
        return CrmBeanUtil.copy(one, MdmFunctionSubRespVo.class);
    }

    @Override
    public List<MdmFunctionRespVo> configTree(MdmFunctionReqVo mdmFunctionReqVo) {
        return mdmFunctionService.functionTree(mdmFunctionReqVo);

//        mdmFunctionReqVo.setPageConfig(YesNoEnum.yesNoEnum.YES.getValue());
//        List<String> configFunctionCodeList = mdmFunctionService.lambdaQuery()
//                .eq(MdmFunctionEntity::getPageConfig, YesNoEnum.yesNoEnum.YES.getValue())
//                .select(MdmFunctionEntity::getFunctionCode)
//                .list()
//                .stream()
//                .filter(Objects::nonNull)
//                .map(MdmFunctionEntity::getFunctionCode)
//                .collect(Collectors.toList());
//        List<MdmFunctionEntity> list = mdmFunctionService.lambdaQuery()
//                .list();
//        if (CollectionUtils.isEmpty(list)) {
//            return new ArrayList<>();
//        }
//        //菜单编码模糊区配
//        String functionCode = mdmFunctionReqVo.getFunctionCode();
//        if (!CollectionUtils.isEmpty(configFunctionCodeList) && !StringUtils.isEmpty(functionCode)) {
//            configFunctionCodeList = configFunctionCodeList.stream()
//                    .filter(x -> x.contains(functionCode))
//                    .collect(Collectors.toList());
//        }
//        //置灰匹配
//        List<MdmFunctionRespVo> mdmFunctionRespVos = CrmBeanUtil.copyList(list, MdmFunctionRespVo.class);
//        for (MdmFunctionRespVo mdmFunctionRespVo : mdmFunctionRespVos) {
//            if (!CollectionUtils.isEmpty(configFunctionCodeList) && configFunctionCodeList.contains(mdmFunctionRespVo.getFunctionCode())) {
//                mdmFunctionRespVo.setFunctionPermission(YesNoEnum.yesNoEnum.ONE.getValue());
//            } else {
//                mdmFunctionRespVo.setFunctionPermission(YesNoEnum.yesNoEnum.ZERO.getValue());
//            }
//        }
//        return mdmFunctionService.constructFunctionTree(mdmFunctionReqVo.getFunctionName(), mdmFunctionRespVos, configFunctionCodeList);
    }

    @Override
    public List<MdmFunctionSubPermissionVo> findAuthorizedSubList(MdmFunctionSubReqVo mdmFunctionSubReqVo) {
        Assert.hasText(mdmFunctionSubReqVo.getParentCode(), "上级菜单编码不能为空");
        List<MdmFunctionSubPermissionVo> list = findMdmFunctionSubPermissionVos(mdmFunctionSubReqVo);
        if (CollectionUtils.isEmpty(list)) {
            return new ArrayList<>();
        }
        for (MdmFunctionSubPermissionVo mdmFunctionSubPermissionVo : list) {
            mdmFunctionSubPermissionVo.setFunctionAuthorized(YesNoEnum.yesNoEnum.ZERO.getValue());
        }
        //设置已配置数据权限作为列表回显
        List<String> functionCodeList = list.stream().filter(x -> !StringUtils.isEmpty(x.getFunctionCode()))
                .map(MdmFunctionSubPermissionVo::getFunctionCode)
                .distinct()
                .collect(Collectors.toList());
        if (!CollectionUtils.isEmpty(functionCodeList)) {
            MdmDataPermissionReqVo mdmDataPermissionReqVo = new MdmDataPermissionReqVo();
            mdmDataPermissionReqVo.setFunctionCode(mdmFunctionSubReqVo.getParentCode());
            mdmDataPermissionReqVo.setListConfigCodeList(functionCodeList);
            Map<String, List<String>> dataPermissionMap = mdmDataPermissionService.findDataPermissionMap(mdmDataPermissionReqVo);
            if (dataPermissionMap != null && dataPermissionMap.size() > 0) {
                for (MdmFunctionSubPermissionVo mdmFunctionSubPermissionVo : list) {
                    List<String> currentFunctionCodeList = dataPermissionMap.get(mdmFunctionSubPermissionVo.getParentCode());
                    if(!CollectionUtils.isEmpty(currentFunctionCodeList)&&currentFunctionCodeList.contains(mdmFunctionSubPermissionVo.getFunctionCode())){
                        mdmFunctionSubPermissionVo.setFunctionAuthorized(YesNoEnum.yesNoEnum.ONE.getValue());
                    }
                }
            }
        }
        return list;
    }

    @Override
    public List<MdmFunctionSubPermissionVo> findMdmFunctionSubPermissionVos(MdmFunctionSubReqVo mdmFunctionSubReqVo) {
        return mdmFunctionSubMapper.findAuthorizedSubList(mdmFunctionSubReqVo);
    }

}
