package com.biz.crm.terminal.controller;

import com.biz.crm.aop.CrmGlobalLog;
import com.biz.crm.common.PageResult;
import com.biz.crm.config.CrmDictMethod;
import com.biz.crm.eunm.mdm.MdmBpmStatusEnum;
import com.biz.crm.nebular.activiti.act.ActivitiCallBackVo;
import com.biz.crm.nebular.mdm.org.resp.MdmOrgRespVo;
import com.biz.crm.nebular.mdm.terminal.*;
import com.biz.crm.nebular.mdm.terminal.req.MdmOrgTerminalReqVo;
import com.biz.crm.nebular.mdm.terminal.req.MdmTerminalSelectReqVo;
import com.biz.crm.nebular.mdm.terminal.req.UpdateCoordinateReqVo;
import com.biz.crm.nebular.mdm.terminal.resp.MdmTerminalSelectRespVo;
import com.biz.crm.nebular.sfa.audit.req.SfaTPMTerminalReqVo;
import com.biz.crm.nebular.sfa.tpmact.SfaTpmOrgRCusVo;
import com.biz.crm.terminal.service.MdmTerminalService;
import com.biz.crm.nebular.mdm.terminal.MdmTencentMapData;
import com.biz.crm.terminal.vo.UpdateTerminalCusOrgCodeVo;
import com.biz.crm.util.Result;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * @author Liao Hua
 * @date 2020/9/3 3:30 下午
 */
@Slf4j
@RestController
@RequestMapping("/mdmTerminalController")
@Api(tags = "终端信息")
@CrmGlobalLog
public class MdmTerminalController {

    @Autowired
    private MdmTerminalService mdmTerminalService;

    @ApiOperation(value = "终端信息页面查询主列表（其它功能请勿使用）")
    @PostMapping("/pageList")
    @CrmDictMethod
    public Result<PageResult<MdmTerminalVo>> pageList(@RequestBody MdmTerminalVo mdmTerminalReqVo) {
        PageResult<MdmTerminalVo> result = mdmTerminalService.findList(mdmTerminalReqVo);
        return Result.ok(result);
    }

    @ApiOperation(value = "查询列表")
    @PostMapping("/page")
    @CrmDictMethod
    public Result<PageResult<MdmTerminalVo>> page(@RequestBody MdmTerminalVo mdmTerminalReqVo) {
        mdmTerminalReqVo.setActApproveStatus(MdmBpmStatusEnum.PASS.getValue());
        //TODO 这里是提供出去的分页选择列表，可以重写一个简单的查询，提高查询效率
        //TODO暂时不动
        PageResult<MdmTerminalVo> result = mdmTerminalService.findList(mdmTerminalReqVo);
        return Result.ok(result);
    }

    @ApiOperation(value = "查询", httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "ID", required = false, dataType = "String", paramType = "query"),
            @ApiImplicitParam(name = "terminalCode", value = "终端编码", required = false, dataType = "String", paramType = "query")
    })
    @GetMapping("/query")
    @CrmDictMethod
    public Result<MdmTerminalVo> query(@RequestParam(value = "id", required = false) String id,
                                       @RequestParam(value = "terminalCode", required = false) String terminalCode) {
        MdmTerminalVo model = mdmTerminalService.query(id, terminalCode);
        return Result.ok(model);
    }

    @ApiOperation(value = "新增")
    @PostMapping("/save")
    public Result<MdmTerminalVo> save(@RequestBody MdmTerminalVo mdmTerminalVo) {
        mdmTerminalService.save(mdmTerminalVo);
        return Result.ok(mdmTerminalService.query(null, mdmTerminalVo.getTerminalCode()));
    }

    @ApiOperation(value = "批量新增(只适用于腾讯地图数据同步")
    @PostMapping("/saveTxBatch")
    public Result<Object> saveTxBatch(@RequestBody MdmTencentMapData mdmTencentMapData){
        mdmTerminalService.saveTxBatch(mdmTencentMapData);
        return Result.ok();
    }


    @ApiOperation(value = "新增并提交流程")
    @PostMapping("/saveAndSubmit")
    public Result<Object> saveAndSubmit(@RequestBody MdmTerminalVo mdmTerminalVo) {
        return Result.ok(mdmTerminalService.saveAndSubmit(mdmTerminalVo));
    }

    @ApiOperation(value = "sfa校验新增终端参数重复")
    @PostMapping("/validateSaveParamFromSfa")
    public Result<Object> validateSaveParamFromSfa(@RequestBody MdmTerminalVo mdmTerminalVo) {
        Result<Object> result = Result.ok();
        result.setResult(mdmTerminalService.validateSaveParamFromSfa(mdmTerminalVo));
        return result;
    }

    @ApiOperation(value = "更新")
    @PostMapping("/update")
    public Result<Object> update(@RequestBody MdmTerminalVo mdmTerminalVo) {
        mdmTerminalService.update(mdmTerminalVo);
        return Result.ok();
    }

    @ApiOperation(value = "根据客户编码集合修改经纬度")
    @PostMapping("/updateCoordinate")
    public Result<Object> updateCoordinate(@RequestBody List<UpdateCoordinateReqVo> updateCoordinateReqVos) {
        mdmTerminalService.updateCoordinate(updateCoordinateReqVos);
        return Result.ok();
    }

    @ApiOperation(value = "更新并提交")
    @PostMapping("/updateAndSubmit")
    public Result<Object> updateAndSubmit(@RequestBody MdmTerminalVo mdmTerminalVo) {
        mdmTerminalService.updateAndSubmit(mdmTerminalVo);
        return Result.ok();
    }

    @ApiOperation(value = "删除")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "ids", value = "id集合", required = true, paramType = "body")
    })
    @PostMapping("/delete")
    public Result<Object> delete(@RequestBody List<String> ids) {
        mdmTerminalService.deleteBatch(ids);
        return Result.ok();
    }

    @ApiOperation(value = "启用")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "ids", value = "id集合", required = true, paramType = "body")
    })
    @PostMapping("/enable")
    public Result<Object> enable(@RequestBody List<String> ids) {
        mdmTerminalService.enableBatch(ids);
        return Result.ok();
    }

    @ApiOperation(value = "禁用")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "ids", value = "id集合", required = true, paramType = "body")
    })
    @PostMapping("/disable")
    public Result<Object> disable(@RequestBody List<String> ids) {
        mdmTerminalService.disableBatch(ids);
        return Result.ok();
    }

    @ApiOperation("批量修改终端客户组织编码")
    @PostMapping("/updateCusOrgCode")
    public Result<Object> updateCusOrgCode(@RequestBody UpdateTerminalCusOrgCodeVo vo) {
        mdmTerminalService.updateCusOrgCode(vo);
        return Result.ok();
    }

    @ApiImplicitParams({
            @ApiImplicitParam(name = "orgCodes", value = "组织编码集合", required = true, paramType = "body")
    })
    @ApiOperation(value = "根据组织编码查询终端编码集合", httpMethod = "POST")
    @PostMapping("/findByOrgCodeList")
    public Result<List<String>> findByOrgCodeList(@RequestBody List<String> orgCodes) {
        return Result.ok(mdmTerminalService.findByOrgCodeList(orgCodes));
    }

    @ApiOperation("条件查询终端及主联系人列表")
    @PostMapping("/findTerminalAndContactList")
    public Result<PageResult<MdmTerminalContactPageVo>> findTerminalAndContactList(@RequestBody MdmTerminalCurrentAndSubVo mdmTerminalCurrentAndSubVo) {
        return Result.ok(mdmTerminalService.findTerminalAndContactList(mdmTerminalCurrentAndSubVo));
    }

    @ApiOperation("根据组织编码查询当前及下级终端")
    @PostMapping("/findCurrentAndSubTerminalList")
    public Result<List<MdmTerminalVo>> findCurrentAndSubTerminalList(@RequestBody List<String> orgCodes) {
        return Result.ok(mdmTerminalService.findCurrentAndSubTerminalList(orgCodes));
    }

    @ApiOperation("条件列表查询")
    @PostMapping("/listCondition")
    public Result<List<MdmTerminalVo>> listCondition(@RequestBody MdmTerminalVo vo) {
        return Result.ok(mdmTerminalService.listCondition(vo));
    }

    @ApiImplicitParams({
            @ApiImplicitParam(name = "posId", value = "职位ID", required = false, dataType = "String", paramType = "query"),
            @ApiImplicitParam(name = "posCode", value = "职位编码", required = false, dataType = "String", paramType = "query")
    })
    @ApiOperation(value = "根据职位id或职位编码查询终端列表", httpMethod = "GET")
    @GetMapping("/findPositionTerminalList")
    public Result<List<MdmTerminalVo>> findPositionTerminalList(@RequestParam(value = "posId", required = false) String posId, @RequestParam(value = "posCode", required = false) String posCode) {
        return Result.ok(mdmTerminalService.findPositionTerminalList(posId, posCode));
    }

    @ApiImplicitParams({
            @ApiImplicitParam(name = "posId", value = "职位ID", required = false, dataType = "String", paramType = "query"),
            @ApiImplicitParam(name = "posCode", value = "职位编码", required = false, dataType = "String", paramType = "query")
    })
    @ApiOperation(value = "根据职位id或职位编码查询当前及所有下级职位对应终端列表", httpMethod = "GET")
    @GetMapping("/findCurrentAndSybPositionTerminalList")
    public Result<List<MdmTerminalVo>> findCurrentAndSybPositionTerminalList(@RequestParam(value = "posId", required = false) String posId, @RequestParam(value = "posCode", required = false) String posCode) {
        return Result.ok(mdmTerminalService.findCurrentAndSybPositionTerminalList(posId, posCode));
    }


    @ApiOperation("批量修改与终端关联的企业组织编码")
    @PostMapping("/batchUpdateTerminalOrg")
    public Result batchUpdateTerminalOrg(@RequestBody MdmOrgTerminalReqVo reqVo) {
        mdmTerminalService.batchUpdateTerminalOrg(reqVo);
        return Result.ok();
    }

    @ApiOperation("获取指定职位终端及主联系人列表")
    @PostMapping("/userTerminalContactPage")
    @CrmDictMethod
    public Result<PageResult<MdmTerminalContactPageVo>> userTerminalContactPage(@RequestBody MdmTerminalContactPageReqVo mdmTerminalContactPageReqVo) {
        PageResult<MdmTerminalContactPageVo> result = mdmTerminalService.userTerminalContactPage(mdmTerminalContactPageReqVo);
        return Result.ok(result);
    }

    @ApiOperation("根据终端编码集合查询终端及主联系人列表")
    @PostMapping("/terminalContactPage")
    @CrmDictMethod
    public Result<PageResult<MdmTerminalContactPageVo>> terminalContactPage(@RequestBody MdmTerminalContactPageReqVo mdmTerminalContactPageReqVo) {
        PageResult<MdmTerminalContactPageVo> result = mdmTerminalService.terminalContactPage(mdmTerminalContactPageReqVo);
        return Result.ok(result);
    }

    @ApiOperation(value = "sfa-根据终端编码集合查询终端及主联系人列表")
    @PostMapping("sfaFindTerminalContactPage")
    @CrmDictMethod
    public Result<List<MdmTerminalContactPageVo>> sfaFindTerminalContactPage(@RequestBody List<String> terminalCodes) {
        return Result.ok(mdmTerminalService.sfaFindTerminalContactPage(terminalCodes));
    }

    @ApiOperation("条件查询终端主联系人及距离分页列表（只查询审批通过的）")
    @PostMapping("/distanceContactPage")
    @CrmDictMethod
    public Result<PageResult<MdmTerminalContactPageVo>> distanceContactPage(@RequestBody MdmTerminalDistanceContactPageReqVo mdmTerminalDistanceContactPageReqVo) {
        mdmTerminalDistanceContactPageReqVo.setActApproveStatus(MdmBpmStatusEnum.PASS.getValue());
        PageResult<MdmTerminalContactPageVo> result = mdmTerminalService.distanceContactPage(mdmTerminalDistanceContactPageReqVo);
        return Result.ok(result);
    }

    @ApiOperation("条件查询终端主联系人及距离分页列表")
    @PostMapping("/distanceContactNoBpmFilterPage")
    @CrmDictMethod
    public Result<PageResult<MdmTerminalContactPageVo>> distanceContactNoBpmFilterPage(@RequestBody MdmTerminalDistanceContactPageReqVo mdmTerminalDistanceContactPageReqVo) {
        PageResult<MdmTerminalContactPageVo> result = mdmTerminalService.distanceContactPage(mdmTerminalDistanceContactPageReqVo);
        return Result.ok(result);
    }

    @ApiOperation("获取终端组织集合")
    @PostMapping("/detailBatch")
    public Result<List<MdmOrgRespVo>> detailBatch(@RequestBody MdmTerminalVo vo) {
        List<MdmOrgRespVo> list = mdmTerminalService.detailBatch(vo);
        return Result.ok(list);
    }

    @ApiOperation("根据客户编码或组织编码查询终端信息")
    @PostMapping("/customerTerminalList")
    public Result<List<MdmTerminalVo>> customerTerminalList(@RequestBody MdmCustomerTerminalVo mdmCustomerTerminalVo) {
        List<MdmTerminalVo> list = mdmTerminalService.customerTerminalList(mdmCustomerTerminalVo);
        return Result.ok(list);
    }

    @ApiOperation("查询职位关联的终端数量")
    @GetMapping("/findPositionRelationTerminalCount")
    public Result<Long> findPositionRelationTerminalCount(@RequestParam("positionCode") String positionCode) {
        Long count = mdmTerminalService.findPositionRelationTerminalCount(positionCode);
        return Result.ok(count);
    }

    @ApiOperation("查询职位关联的二批商数量")
    @GetMapping("/findPositionRelationTerminalCustomerCount")
    public Result<Long> findPositionRelationTerminalCustomerCount(@RequestParam("positionCode") String positionCode) {
        Long count = mdmTerminalService.findPositionRelationTerminalCustomerCount(positionCode);
        return Result.ok(count);
    }

    @ApiOperation("根据营业注册号或终端名称查询详情")
    @GetMapping("/findDetailByRegisterOrName")
    public Result<MdmTerminalVo> findDetailByRegisterOrName(@RequestParam(value = "licenseRegisterNumber", required = false) String licenseRegisterNumber,
                                                            @RequestParam(value = "terminalName", required = false) String terminalName) {
        MdmTerminalVo mdmTerminalVo = mdmTerminalService.findDetailByRegisterOrName(licenseRegisterNumber, terminalName);
        return Result.ok(mdmTerminalVo);
    }

    @ApiOperation(value = "终端下拉框（限制数量）")
    @PostMapping("/select")
    public Result<List<MdmTerminalSelectRespVo>> select(@RequestBody MdmTerminalSelectReqVo reqVo) {
        return Result.ok(mdmTerminalService.findTerminalSelectList(reqVo));
    }

    @ApiOperation(value = "提交工作流审批")
    @PostMapping("/submit")
    public Result<Object> submit(@RequestBody MdmTerminalVo reqVo) {
        mdmTerminalService.submit(reqVo);
        return Result.ok("提交审批成功");
    }

    @ApiOperation(value = "工作流回调接口")
    @PostMapping("/activityCallBack")
    public Result<Object> callback(@RequestBody ActivitiCallBackVo reqVo) {
        mdmTerminalService.callback(reqVo);
        return Result.ok("活动审批回调成功");
    }

    @ApiOperation(value = "状态审批，提交")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "ids", value = "id集合", required = true, paramType = "body")
    })
    @PostMapping("/statusApprovalSubmit")
    public Result<Object> statusApprovalSubmit(@RequestBody List<String> ids) {
        mdmTerminalService.statusApprovalSubmit(ids);
        return Result.ok();
    }

    @ApiOperation(value = "状态审批，驳回")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "ids", value = "id集合", required = true, paramType = "body")
    })
    @PostMapping("/statusApprovalReject")
    public Result<Object> statusApprovalReject(@RequestBody List<String> ids) {
        mdmTerminalService.statusApprovalReject(ids);
        return Result.ok();
    }

    @ApiOperation(value = "状态审批，审批通过")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "ids", value = "id集合", required = true, paramType = "body")
    })
    @PostMapping("/statusApprovalAgree")
    public Result<Object> statusApprovalAgree(@RequestBody List<String> ids) {
        mdmTerminalService.statusApprovalAgree(ids);
        return Result.ok();
    }

    @ApiOperation("根据终端编码集合查询供货关系")
    @PostMapping("/findCustomerByTerminalCode")
    public Result<List<MdmTerminalSupplyVo>> findTerminalSupply(@RequestBody MdmTerminalVo mdmTerminalVo) {
        return Result.ok(mdmTerminalService.findTerminalSupply(mdmTerminalVo));
    }

    @ApiOperation("校验终端编码有没有被组织编码关联")
    @PostMapping("/checkTerminalCodeRelByOrgCodeList")
    public Result<Boolean> checkTerminalCodeRelByOrgCodeList(@RequestBody List<String> orgCodeList, @RequestParam("terminalCode") String terminalCode) {
        return Result.ok(mdmTerminalService.checkTerminalCodeRelByOrgCodeList(orgCodeList, terminalCode));
    }

    @ApiOperation("校验终端编码有没有被客户编码关联")
    @PostMapping("/checkTerminalCodeRelByCustomerCodeList")
    public Result<Boolean> checkTerminalCodeRelByCustomerCodeList(@RequestBody List<String> customerCodeList, @RequestParam("terminalCode") String terminalCode) {
        return Result.ok(mdmTerminalService.checkTerminalCodeRelByCustomerCodeList(customerCodeList, terminalCode));
    }


    @ApiOperation(value = "校验终端是否被组织/客户关联-sfa专用")
    @PostMapping("checkTerminalCodeRelByOrgOrCusCode")
    public Result<List<String>> checkTerminalCodeRelByOrgOrCusCode(@RequestBody SfaTpmOrgRCusVo orgRCusVo) {
        return Result.ok(mdmTerminalService.checkTerminalCodeRelByOrgOrCusCode(orgRCusVo));
    }

    @ApiOperation(value = "根据客户组织编码集合查询客户组织下的终端")
    @PostMapping("findTerminalByCustomerOrgCodeList")
    public Result<List<MdmTerminalVo>> findTerminalByCustomerOrgCodeList(@RequestBody List<String> customerOrgCodeList) {
        return Result.ok(mdmTerminalService.findTerminalByCustomerOrgCodeList(customerOrgCodeList));
    }

    @ApiOperation(value = "根据名称和地址查询终端详情", httpMethod = "GET")
    @GetMapping("/findTerminalByNameOrAddress")
    public Result<MdmTerminalVo> findTerminalByNameOrAddress(@RequestParam(value = "terminalName", required = false) String terminalName,
                                                             @RequestParam(value = "terminalAddress", required = false) String terminalAddress){
        return Result.ok(mdmTerminalService.findTerminalByNameOrAddress(terminalName, terminalAddress));
    }


    /**
     * 根据终端编码集合批量查询终端主表(以及附属信息)
     *
     * @param terminalCodeList list
     * @return list
     */
    @ApiOperation(value = "根据终端编码集合批量查询终端主表(以及附属信息)", httpMethod = "POST")
    @PostMapping("/queryTerminalAndAttachByCode")
    public Result<List<MdmTerminalVo>> queryTerminalAndAttachByCode(@RequestBody List<String> terminalCodeList) {
        return Result.ok(mdmTerminalService.queryTerminalAndAttachByCode(terminalCodeList));
    }

    @ApiOperation(value = "根据组织编码集合查询终端信息")
    @PostMapping("/queryTerminalByOrgCodeList")
    public Result<List<MdmTerminalVo>> queryTerminalByOrgCodeList(@RequestBody SfaTPMTerminalReqVo reqVo){
        return Result.ok(mdmTerminalService.queryTerminalByOrgCodeList(reqVo));
    }

}