package com.biz.crm.terminal.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.base.BaseIdEntity;
import com.biz.crm.common.PageResult;
import com.biz.crm.eunm.CrmDelFlagEnum;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.eunm.mdm.TerminalSupplyTypeEnum;
import com.biz.crm.material.service.MdmMaterialService;
import com.biz.crm.nebular.mdm.customer.*;
import com.biz.crm.nebular.mdm.material.MdmMaterialRespVo;
import com.biz.crm.nebular.mdm.poi.resp.MdmTerminalSupplyDetailVo;
import com.biz.crm.nebular.mdm.position.resp.MdmPositionRespVo;
import com.biz.crm.nebular.mdm.terminal.MdmTerminalSupplyVo;
import com.biz.crm.nebular.mdm.terminal.MdmTerminalVo;
import com.biz.crm.position.service.MdmPositionService;
import com.biz.crm.terminal.mapper.MdmTerminalSupplyMapper;
import com.biz.crm.terminal.model.MdmTerminalSupplyDetailEntity;
import com.biz.crm.terminal.model.MdmTerminalSupplyEntity;
import com.biz.crm.terminal.service.MdmTerminalSupplyDetailService;
import com.biz.crm.terminal.service.MdmTerminalSupplyService;
import com.biz.crm.util.AssertUtils;
import com.biz.crm.util.CollectionUtil;
import com.biz.crm.util.CrmBeanUtil;
import com.biz.crm.util.PageUtil;
import com.biz.crm.util.StringUtils;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.time.DateUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;

import javax.validation.Validator;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author Liao Hua
 * @date 2020/9/7 4:30 下午
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "MdmTerminalSupplyServiceExpandImpl")
public class MdmTerminalSupplyServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<MdmTerminalSupplyMapper, MdmTerminalSupplyEntity> implements MdmTerminalSupplyService {

  @Autowired
  private MdmTerminalSupplyDetailService mdmTerminalSupplyDetailService;

  @Autowired
  private MdmTerminalSupplyMapper mdmTerminalSupplyMapper;

  @Autowired
  private MdmPositionService mdmPositionService;

  @Autowired
  private MdmMaterialService mdmMaterialService;


  @Override
  @Transactional(rollbackFor = Exception.class)
  public void deleteByCodes(List<String> terminalCodes) {
    if (CollectionUtils.isEmpty(terminalCodes)) {
      return;
    }
    List<MdmTerminalSupplyEntity> list = this.lambdaQuery()
            .in(MdmTerminalSupplyEntity::getTerminalCode, terminalCodes)
            .select(MdmTerminalSupplyEntity::getId)
            .list();
    if (CollectionUtils.isEmpty(list)) {
      return;
    }
    List<String> supplyIds = list.stream().map(BaseIdEntity::getId).collect(Collectors.toList());
    mdmTerminalSupplyDetailService.lambdaUpdate()
            .in(MdmTerminalSupplyDetailEntity::getSupplyId, supplyIds)
            .remove();
    this.removeByIds(supplyIds);
  }

  @Override
  @Transactional(rollbackFor = Exception.class)
  public void setUpSupply(MdmTerminalVo mdmTerminalVo) {
    String terminalCode = mdmTerminalVo.getTerminalCode();
    Assert.hasText(terminalCode, "终端编码不能为空");
    List<MdmTerminalSupplyEntity> list = this.lambdaQuery()
            .eq(MdmTerminalSupplyEntity::getTerminalCode, terminalCode)
            .select(MdmTerminalSupplyEntity::getId)
            .list();
    if (!CollectionUtils.isEmpty(list)) {
      List<String> supplyIds = list.stream().map(BaseIdEntity::getId).collect(Collectors.toList());
      mdmTerminalSupplyDetailService.lambdaUpdate()
              .in(MdmTerminalSupplyDetailEntity::getSupplyId, supplyIds)
              .remove();
    }
    this.lambdaUpdate()
            .eq(MdmTerminalSupplyEntity::getTerminalCode, terminalCode)
            .remove();
    if (CollectionUtil.listEmpty(mdmTerminalVo.getSupplys())) {
      return;
    }
    List<MdmTerminalSupplyVo> supplys = mdmTerminalVo.getSupplys();
    if (!CollectionUtils.isEmpty(supplys)) {
      List<String> supplyProductLists = Lists.newArrayList();
      List<String> supplyProductLevels = Lists.newArrayList();
      for (MdmTerminalSupplyVo supply : supplys) {
        if (StringUtils.isEmpty(supply.getSupplyType())) {
          supply.setSupplyType(TerminalSupplyTypeEnum.DEFAULT.getValue());
        }
        supply.setId(UUID.randomUUID().toString());
        supply.setTerminalCode(terminalCode);
        Assert.hasText(supply.getSupplyType(), "缺失供货关系类型");
        if (TerminalSupplyTypeEnum.CUSTOMER_USER.getValue().equals(supply.getSupplyType())) {
          Assert.hasText(supply.getCustomerCode(), "缺失客户");
        }
        List<MdmTerminalSupplyDetailVo> details = supply.getDetails();
        if (!CollectionUtils.isEmpty(details)) {
          for (MdmTerminalSupplyDetailVo detail : details) {
            Assert.hasText(detail.getCode(), "[终端:{" + terminalCode + "}]供货关系编码不能为空");
            Assert.hasText(detail.getName(), "[终端:{" + terminalCode + "}]供货关系名称不能为空");
            Assert.hasText(detail.getDataType(), "[终端:{" + terminalCode + "}]供货关系产品类型不能为空");
            if (detail.getDataType().equals("1")) {
              //如果是商品，则校验商品是否重复，以及商品的商品层级是否已经设置过供货关系
              if (!CollectionUtils.isEmpty(supplyProductLists)) {
                Validate.isTrue(!supplyProductLists.contains(detail.getSupplyId()), "供货关系商品重复！");
              }
              if (!CollectionUtils.isEmpty(supplyProductLevels)) {
                MdmMaterialRespVo mdmMaterialRespVo = mdmMaterialService.detail(null, detail.getSupplyId());
                Validate.notNull(mdmMaterialRespVo, "所选物料编码对应物料不存在");
                Validate.isTrue(!supplyProductLevels.contains(mdmMaterialRespVo.getProductLevelCode()), "供货关系商品所在产品层级重复！");
              }
              supplyProductLists.add(detail.getCode());
            } else {
              //如果是商品层级，则校验产品层级是否重复
              if (!CollectionUtils.isEmpty(supplyProductLevels)) {
                Validate.isTrue(!supplyProductLevels.contains(detail.getSupplyId()), "供货关系产品层级重复！");
              }
              supplyProductLevels.add(detail.getCode());
            }
            detail.setId(UUID.randomUUID().toString());
            detail.setSupplyId(supply.getId());
          }
          mdmTerminalSupplyDetailService.saveBatch(CrmBeanUtil.copyList(details, MdmTerminalSupplyDetailEntity.class));
        }
        this.save(CrmBeanUtil.copy(supply, MdmTerminalSupplyEntity.class));
      }
    }
  }

  @Override
  public List<MdmTerminalSupplyVo> findByTerminalCode(String terminalCode) {
    Assert.hasText(terminalCode, "终端编码不能为空");
    List<MdmTerminalSupplyVo> list = mdmTerminalSupplyMapper.queryDetails(terminalCode);
    if (CollectionUtils.isEmpty(list)) {
      return new ArrayList<>();
    }
    List<String> supplyIds = list.stream().map(MdmTerminalSupplyVo::getId).collect(Collectors.toList());
    //组装供货商品
    List<MdmTerminalSupplyDetailEntity> supplyDetailEntities = mdmTerminalSupplyDetailService.lambdaQuery()
            .in(MdmTerminalSupplyDetailEntity::getSupplyId, supplyIds)
            .list();
    if (!CollectionUtils.isEmpty(supplyDetailEntities)) {
      Map<String, List<MdmTerminalSupplyDetailEntity>> detailsMap = supplyDetailEntities.stream()
              .collect(Collectors.groupingBy(MdmTerminalSupplyDetailEntity::getSupplyId));
      for (MdmTerminalSupplyVo mdmTerminalSupplyVo : list) {
        List<MdmTerminalSupplyDetailEntity> source = detailsMap.get(mdmTerminalSupplyVo.getId());
        if (source == null) {
          mdmTerminalSupplyVo.setDetails(new ArrayList<>());
          continue;
        }
        List<MdmTerminalSupplyDetailVo> detailVos = CrmBeanUtil.copyList(source, MdmTerminalSupplyDetailVo.class);
        mdmTerminalSupplyVo.setDetails(detailVos);
      }
    } else {
      list = list.stream().peek(x -> x.setDetails(new ArrayList<>())).collect(Collectors.toList());
    }
    return list;
  }

  @Override
  public void customerRelationTerminal(MdmCustomerCorrelatedTerminalReqVo mdmCustomerCorrelatedTerminalReqVo) {
    String customerCode = mdmCustomerCorrelatedTerminalReqVo.getCustomerCode();
    Assert.hasText(customerCode, "客户编码不能为空");
    Set<String> terminalCodeList = mdmCustomerCorrelatedTerminalReqVo.getTerminalCodeList();
    Assert.notEmpty(terminalCodeList, "终端编码集合不能为空");
    List<MdmTerminalSupplyEntity> list = this.lambdaQuery()
            .eq(MdmTerminalSupplyEntity::getCustomerCode, customerCode)
            .in(MdmTerminalSupplyEntity::getTerminalCode, terminalCodeList)
            .select(MdmTerminalSupplyEntity::getTerminalCode)
            .list();
    if (!CollectionUtils.isEmpty(list)) {
      List<String> oldTerminalCodeList = list.stream()
              .filter(x -> x != null && !StringUtils.isEmpty(x.getTerminalCode()))
              .map(MdmTerminalSupplyEntity::getTerminalCode)
              .collect(Collectors.toList());
      if (!CollectionUtils.isEmpty(oldTerminalCodeList)) {
        terminalCodeList.removeIf(oldTerminalCodeList::contains);
      }
    }
    if (CollectionUtils.isEmpty(terminalCodeList)) {
      return;
    }
    List<MdmTerminalSupplyEntity> collect = terminalCodeList.stream().map(x -> {
      MdmTerminalSupplyEntity mdmTerminalSupplyEntity = new MdmTerminalSupplyEntity();
      mdmTerminalSupplyEntity.setId(UUID.randomUUID().toString());
      mdmTerminalSupplyEntity.setTerminalCode(x);
      mdmTerminalSupplyEntity.setCustomerCode(customerCode);
      mdmTerminalSupplyEntity.setSupplyType(TerminalSupplyTypeEnum.DEFAULT.getValue());
      return mdmTerminalSupplyEntity;
    }).collect(Collectors.toList());
    this.saveBatch(collect);
  }

  @Override
  public PageResult<MdmCustomerTerminalRespVo> correlatedCustomerPage(MdmCustomerTerminalReqVo mdmCustomerTerminalReqVo) {
    Assert.hasText(mdmCustomerTerminalReqVo.getCustomerCode(), "客户编码不能为空");
    Page<MdmCustomerTerminalRespVo> page = PageUtil.buildPage(mdmCustomerTerminalReqVo.getPageNum(), mdmCustomerTerminalReqVo.getPageSize());
    List<MdmCustomerTerminalRespVo> list = mdmTerminalSupplyMapper.correlatedCustomerPage(page, mdmCustomerTerminalReqVo);
    return PageResult.<MdmCustomerTerminalRespVo>builder()
            .data(list)
            .count(page.getTotal())
            .build();
  }

  @Override
  public PageResult<MdmCustomerTerminalRespVo> unCorrelatedAnyCustomerPage(MdmCustomerTerminalReqVo mdmCustomerTerminalReqVo) {
//        List<MdmTerminalSupplyEntity> mdmTerminalSupplyEntities = this.lambdaQuery()
//                .isNotNull(MdmTerminalSupplyEntity::getCustomerCode)
//                .select(MdmTerminalSupplyEntity::getTerminalCode)
//                .list();
//        if (!CollectionUtils.isEmpty(mdmTerminalSupplyEntities)) {
//            List<String> terminalCodeList = mdmTerminalSupplyEntities.stream()
//                    .filter(x -> x != null && !StringUtils.isEmpty(x.getTerminalCode()))
//                    .map(MdmTerminalSupplyEntity::getTerminalCode)
//                    .distinct()
//                    .collect(Collectors.toList());
//            mdmCustomerTerminalReqVo.setTerminalCodeList(terminalCodeList);
//        }
    Page<MdmCustomerTerminalRespVo> page = new Page<>(mdmCustomerTerminalReqVo.getPageNum(), mdmCustomerTerminalReqVo.getPageSize());
    List<MdmCustomerTerminalRespVo> list = mdmTerminalSupplyMapper.uncorrelatedAnyCustomer(page, mdmCustomerTerminalReqVo);
    return PageResult.<MdmCustomerTerminalRespVo>builder()
            .data(list)
            .count(page.getTotal())
            .build();
  }

  @Override
  public PageResult<MdmCustomerTerminalRespVo> unCorrelatedCurrentCustomerPage(MdmCustomerTerminalReqVo mdmCustomerTerminalReqVo) {
    String customerCode = mdmCustomerTerminalReqVo.getCustomerCode();
    Assert.hasText(customerCode, "客户编码不能为空");
    List<MdmTerminalSupplyEntity> mdmTerminalSupplyEntities = this.lambdaQuery()
            .eq(MdmTerminalSupplyEntity::getCustomerCode, customerCode)
            .select(MdmTerminalSupplyEntity::getTerminalCode)
            .list();
    if (!CollectionUtils.isEmpty(mdmTerminalSupplyEntities)) {
      List<String> terminalCodeList = mdmTerminalSupplyEntities.stream()
              .filter(x -> x != null && !StringUtils.isEmpty(x.getTerminalCode()))
              .map(MdmTerminalSupplyEntity::getTerminalCode)
              .distinct()
              .collect(Collectors.toList());
      mdmCustomerTerminalReqVo.setTerminalCodeList(terminalCodeList);
    }
    Page<MdmCustomerTerminalRespVo> page = new Page<>(mdmCustomerTerminalReqVo.getPageNum(), mdmCustomerTerminalReqVo.getPageSize());
    List<MdmCustomerTerminalRespVo> list = mdmTerminalSupplyMapper.unCorrelatedCurrentCustomer(page, mdmCustomerTerminalReqVo);
    return PageResult.<MdmCustomerTerminalRespVo>builder()
            .data(list)
            .count(page.getTotal())
            .build();
  }

  @Override
  public void replaceCustomer(MdmCustomerCorrelatedTerminalReplaceReqVo mdmCustomerCorrelatedTerminalReplaceReqVo) {
    io.jsonwebtoken.lang.Assert.hasText(mdmCustomerCorrelatedTerminalReplaceReqVo.getOldCustomerCode(), "原客户编码不能为空");
    io.jsonwebtoken.lang.Assert.hasText(mdmCustomerCorrelatedTerminalReplaceReqVo.getNewCustomerCode(), "新客户编码不能为空");
    io.jsonwebtoken.lang.Assert.notEmpty(mdmCustomerCorrelatedTerminalReplaceReqVo.getTerminalCodeList(), "终端编码集合不能为空");
    this.lambdaUpdate()
            .eq(MdmTerminalSupplyEntity::getCustomerCode, mdmCustomerCorrelatedTerminalReplaceReqVo.getOldCustomerCode())
            .in(MdmTerminalSupplyEntity::getTerminalCode, mdmCustomerCorrelatedTerminalReplaceReqVo.getTerminalCodeList())
            .set(MdmTerminalSupplyEntity::getCustomerCode, mdmCustomerCorrelatedTerminalReplaceReqVo.getNewCustomerCode())
            .update();
  }

  @Override
  public void unbind(MdmCustomerCorrelatedTerminalReqVo mdmCustomerCorrelatedTerminalReqVo) {
    Assert.hasText(mdmCustomerCorrelatedTerminalReqVo.getCustomerCode(), "客户编码不能为空");
    Assert.notEmpty(mdmCustomerCorrelatedTerminalReqVo.getTerminalCodeList(), "终端编码集合不能为空");
    this.lambdaUpdate()
            .eq(MdmTerminalSupplyEntity::getCustomerCode, mdmCustomerCorrelatedTerminalReqVo.getCustomerCode())
            .in(MdmTerminalSupplyEntity::getTerminalCode, mdmCustomerCorrelatedTerminalReqVo.getTerminalCodeList())
            .set(MdmTerminalSupplyEntity::getCustomerCode, null)
            .update();
  }

  @Override
  @Transactional(rollbackFor = Exception.class)
  public void addTerminalPosition(List<String> terminalCodeList, String positionCode) {
    if (CollectionUtil.listNotEmptyNotSizeZero(terminalCodeList) && StringUtils.isNotEmpty(positionCode)) {
      MdmPositionRespVo one = mdmPositionService.queryByPositionCode(positionCode);
      if (one != null) {
        this.addTerminalPosition(terminalCodeList, positionCode, one.getOrgCode());
      }
    }
  }

  @Override
  @Transactional(rollbackFor = Exception.class)
  public void addTerminalPosition(List<String> terminalCodeList, String positionCode, String orgCode) {
    if (CollectionUtil.listNotEmptyNotSizeZero(terminalCodeList) && StringUtils.isNotEmpty(positionCode)) {
      terminalCodeList.forEach(terminalCode -> {
        MdmTerminalSupplyEntity entity = new MdmTerminalSupplyEntity();
        entity.setTerminalCode(terminalCode);
        entity.setPositionCode(positionCode);
        entity.setOrgCode(orgCode);
        entity.setSupplyType(TerminalSupplyTypeEnum.DEFAULT.getValue());
        this.save(entity);
      });
    }
  }

  @Override
  @Transactional(rollbackFor = Exception.class)
  public void unbindByIds(List<String> ids) {
    if (CollectionUtil.listNotEmptyNotSizeZero(ids)) {
      this.removeByIds(ids);
      mdmTerminalSupplyDetailService.lambdaUpdate()
              .in(MdmTerminalSupplyDetailEntity::getSupplyId, ids)
              .remove();
    }
  }

  @Override
  @Transactional(rollbackFor = Exception.class)
  public void replaceRelationPosition(List<String> ids, String positionCode) {
    if (CollectionUtil.listNotEmptyNotSizeZero(ids) && StringUtils.isNotEmpty(positionCode)) {
      MdmPositionRespVo one = mdmPositionService.queryByPositionCode(positionCode);
      if (one != null) {
        this.replaceRelationPosition(ids, positionCode, one.getOrgCode());
      }
    }
  }

  @Override
  @Transactional(rollbackFor = Exception.class)
  public void replaceRelationPosition(List<String> ids, String positionCode, String orgCode) {
    if (CollectionUtil.listNotEmptyNotSizeZero(ids) && StringUtils.isNotEmpty(positionCode)) {
      this.lambdaUpdate()
              .in(MdmTerminalSupplyEntity::getId, ids)
              .set(MdmTerminalSupplyEntity::getPositionCode, positionCode)
              .set(MdmTerminalSupplyEntity::getOrgCode, orgCode)
              .update();
    }
  }

  @Override
  public List<String> findTerminalByCustomer(List<String> customerCodeList) {
    if (CollectionUtils.isEmpty(customerCodeList)) {
      return new ArrayList<>();
    }
    List<String> list = new ArrayList<>();
    List<List<String>> partition = Lists.partition(customerCodeList, 500);
    for (List<String> part :
            partition) {
      list.addAll(this.lambdaQuery()
              .in(MdmTerminalSupplyEntity::getCustomerCode, part)
              .select(MdmTerminalSupplyEntity::getTerminalCode)
              .list()
              .stream()
              .filter(x -> x != null && !StringUtils.isEmpty(x.getTerminalCode()))
              .map(MdmTerminalSupplyEntity::getTerminalCode)
              .collect(Collectors.toList()));
    }
    return list;
  }

  @Override
  public List<String> findCustomerCodeListByTerminalCode(String terminalCode) {
    if (StringUtils.isNotEmpty(terminalCode)) {
      List<MdmTerminalSupplyEntity> list = this.lambdaQuery()
              .eq(MdmTerminalSupplyEntity::getTerminalCode, terminalCode)
              .list();
      if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
        List<MdmTerminalSupplyEntity> collect = list.stream().filter(x -> StringUtils.isNotEmpty(x.getCustomerCode())).collect(Collectors.toList());
        if (CollectionUtil.listNotEmptyNotSizeZero(collect)) {
          Set<String> set = collect.stream().map(MdmTerminalSupplyEntity::getCustomerCode).collect(Collectors.toSet());
          if (!set.isEmpty()) {
            return new ArrayList<>(set);
          }
        }
      }
    }
    return new ArrayList<>();
  }

  @Override
  public List<String> findTerminalByPositionCodeList(List<String> positionCodeList) {
    if (CollectionUtils.isEmpty(positionCodeList)) {
      return new ArrayList<>();
    }
    return this.lambdaQuery()
            .in(MdmTerminalSupplyEntity::getPositionCode, positionCodeList)
            .select(MdmTerminalSupplyEntity::getTerminalCode)
            .list()
            .stream()
            .filter(x -> x != null && !StringUtils.isEmpty(x.getTerminalCode()))
            .map(MdmTerminalSupplyEntity::getTerminalCode)
            .distinct()
            .collect(Collectors.toList());
  }

  @Override
  public List<MdmTerminalSupplyVo> findTerminalSupply(MdmTerminalVo mdmTerminalVo) {
    if (CollectionUtils.isEmpty(mdmTerminalVo.getTerminalCodeList())) {
      return new ArrayList<>();
    }
    return mdmTerminalSupplyMapper.findTerminalSupply(mdmTerminalVo);
  }

  @Override
  @Transactional(rollbackFor = Exception.class)
  public void addCustomerUserTerminalSupply(List<String> terminalCodeList, String userName) {
    if (StringUtils.isNotEmpty(userName) && CollectionUtil.listNotEmptyNotSizeZero(terminalCodeList)) {
      List<MdmTerminalSupplyEntity> list = this.lambdaQuery()
              .eq(MdmTerminalSupplyEntity::getSupplyType, TerminalSupplyTypeEnum.CUSTOMER_USER.getValue())
              .eq(MdmTerminalSupplyEntity::getUserName, userName)
              .list();
      final Set<String> terminalCodeSet = list.stream().map(MdmTerminalSupplyEntity::getTerminalCode).collect(Collectors.toSet());
      for (String terminalCode :
              terminalCodeList) {
        if (terminalCodeSet.contains(terminalCode)) {
          continue;
        }
        MdmTerminalSupplyEntity supply = new MdmTerminalSupplyEntity();
        supply.setSupplyType(TerminalSupplyTypeEnum.CUSTOMER_USER.getValue());
        supply.setTerminalCode(terminalCode);
        supply.setUserName(userName);
        this.save(supply);
      }
    }
  }

  @Override
  @Transactional(rollbackFor = Exception.class)
  public void delCustomerUserTerminalSupply(List<String> terminalCodeList, String userName) {
    if (StringUtils.isNotEmpty(userName) && CollectionUtil.listNotEmptyNotSizeZero(terminalCodeList)) {
      List<MdmTerminalSupplyEntity> list = this.lambdaQuery()
              .eq(MdmTerminalSupplyEntity::getSupplyType, TerminalSupplyTypeEnum.CUSTOMER_USER.getValue())
              .eq(MdmTerminalSupplyEntity::getUserName, userName)
              .in(MdmTerminalSupplyEntity::getTerminalCode, terminalCodeList)
              .list();
      if (list.size() > 0) {
        List<String> supplyIds = list.stream().map(MdmTerminalSupplyEntity::getId).collect(Collectors.toList());
        mdmTerminalSupplyDetailService.lambdaUpdate()
                .in(MdmTerminalSupplyDetailEntity::getSupplyId, supplyIds)
                .remove();
        this.removeByIds(supplyIds);
      }
    }
  }

  @Override
  @Transactional(rollbackFor = Exception.class)
  public void replaceCustomerUserTerminalSupply(List<String> terminalCodeList, String userName, String newUserName) {
    if (StringUtils.isNotEmpty(userName) && CollectionUtil.listNotEmptyNotSizeZero(terminalCodeList) && StringUtils.isNotEmpty(newUserName)) {
      List<MdmTerminalSupplyEntity> list = this.lambdaQuery()
              .eq(MdmTerminalSupplyEntity::getSupplyType, TerminalSupplyTypeEnum.CUSTOMER_USER.getValue())
              .eq(MdmTerminalSupplyEntity::getUserName, userName)
              .in(MdmTerminalSupplyEntity::getTerminalCode, terminalCodeList)
              .list();
      if (list.size() > 0) {
        final Set<String> terminalCodeSet = this.lambdaQuery()
                .eq(MdmTerminalSupplyEntity::getSupplyType, TerminalSupplyTypeEnum.CUSTOMER_USER.getValue())
                .eq(MdmTerminalSupplyEntity::getUserName, newUserName)
                .in(MdmTerminalSupplyEntity::getTerminalCode, terminalCodeList)
                .select(MdmTerminalSupplyEntity::getTerminalCode)
                .list()
                .stream().map(MdmTerminalSupplyEntity::getTerminalCode).collect(Collectors.toSet());
        list.forEach(item -> {
          if (!terminalCodeSet.contains(item.getTerminalCode())) {
            item.setUserName(newUserName);
            this.updateById(item);
          } else {
            mdmTerminalSupplyDetailService.lambdaUpdate()
                    .eq(MdmTerminalSupplyDetailEntity::getSupplyId, item.getId())
                    .remove();
            this.removeById(item.getId());
          }
        });
      }
    }
  }

  /**
   * 通过终端门店编码查询挂靠的客户信息
   *
   * @param mdmTerminalSupplyVo
   * @return
   */
  @Override
  public List<MdmCustomerRespVo> findBySupply(MdmTerminalSupplyVo mdmTerminalSupplyVo) {
    mdmTerminalSupplyVo.setDelFlag(CrmDelFlagEnum.NORMAL.getCode());
    mdmTerminalSupplyVo.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
    return mdmTerminalSupplyMapper.findBySupply(mdmTerminalSupplyVo);
  }

  @Override
  public List<MdmTerminalSupplyDetailEntity> findGoodsLevelBySupplyId(String supplyId, String dataType) {
    AssertUtils.isNotEmpty(supplyId, "供货关系id不能为空");
    List<String> supplyIds = Lists.newArrayList();
    supplyIds.add(supplyId);
    List<MdmTerminalSupplyDetailEntity> supplyDetailEntities = mdmTerminalSupplyDetailService.getListBySupplyIdOrDataType(supplyIds, dataType);
    return supplyDetailEntities;
  }
}
