package com.biz.crm.user.controller;

import com.biz.crm.aop.CrmGlobalLog;
import com.biz.crm.common.PageResult;
import com.biz.crm.common.param.ParameterParam;
import com.biz.crm.config.CrmDictMethod;
import com.biz.crm.customer.service.MdmCustomerMsgService;
import com.biz.crm.nebular.mdm.customer.MdmCustomerMsgRespVo;
import com.biz.crm.nebular.mdm.user.req.*;
import com.biz.crm.nebular.mdm.user.resp.*;
import com.biz.crm.user.service.MdmCustomerUserService;
import com.biz.crm.user.service.MdmUserService;
import com.biz.crm.util.ParamUtil;
import com.biz.crm.util.Result;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import java.util.List;

/**
 * 用户信息表
 *
 * @author Tao.Chen
 * @date 2020-11-17 19:04:29
 */
@Slf4j
@RestController
@RequestMapping("/mdmCustomerUserController")
@CrmGlobalLog
@Api(tags = "MDM-用户管理（客户用户）")
public class MdmCustomerUserController {

    @Autowired
    private MdmCustomerUserService mdmCustomerUserService;
    @Autowired
    private MdmUserService mdmUserService;
    @Resource
    private MdmCustomerMsgService mdmCustomerMsgService;

    @ApiOperation(value = "查询分页列表")
    @PostMapping("/pageList")
    @CrmDictMethod
    public Result<PageResult<MdmCustomerUserPageRespVo>> pageList(@RequestBody MdmCustomerUserPageReqVo reqVo) {
        PageResult<MdmCustomerUserPageRespVo> result = mdmCustomerUserService.findList(reqVo);
        return Result.ok(result);
    }

    @ApiOperation(value = "详情（客户用户）", httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "ID", required = false, dataType = "String", paramType = "query"),
            @ApiImplicitParam(name = "userName", value = "用户登录名", required = false, dataType = "String", paramType = "query")
    })
    @GetMapping("/detail")
    @CrmDictMethod
    public Result<MdmUserRespVo> detail(@RequestParam(value = "id", required = false) String id, @RequestParam(value = "userName", required = false) String userName) {
        return Result.ok(mdmCustomerUserService.query(id, userName));
    }

    @ApiOperation(value = "新增")
    @PostMapping("/save")
    public Result save(@RequestBody MdmUserEditReqVo reqVo) {
        mdmCustomerUserService.save(reqVo);
        return Result.ok();
    }

    @ApiOperation(value = "更新")
    @PostMapping("/update")
    public Result update(@RequestBody MdmUserEditReqVo reqVo) {
        mdmCustomerUserService.update(reqVo);
        return Result.ok("修改成功");
    }

    @ApiOperation(value = "删除", httpMethod = "POST")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "ids", value = "id集合", required = true, paramType = "body")
    })
    @PostMapping("/delete")
    public Result delete(@RequestBody List<String> ids) {
        mdmCustomerUserService.deleteBatch(ids);
        return Result.ok("删除成功");
    }

    @ApiOperation(value = "启用", httpMethod = "POST")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "ids", value = "id集合", required = true, paramType = "body")
    })
    @PostMapping("/enable")
    public Result enable(@RequestBody List<String> ids) {
        mdmUserService.enableBatch(ids);
        return Result.ok("启用成功");
    }

    @ApiOperation(value = "禁用", httpMethod = "POST")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "ids", value = "id集合", required = true, paramType = "body")
    })
    @PostMapping("/disable")
    public Result disable(@RequestBody List<String> ids) {
        mdmUserService.disableBatch(ids);
        return Result.ok("禁用成功");
    }

    @ApiOperation(value = "用户解锁", httpMethod = "POST")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "ids", value = "id集合", required = true, paramType = "body")
    })
    @PostMapping("/unlock")
    public Result unlock(@RequestBody List<String> ids) {
        mdmUserService.unlock(ids);
        return Result.ok("禁用成功");
    }

    @ApiOperation(value = "解绑用户全部关联微信", httpMethod = "POST")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "ids", value = "用户id集合", required = true, paramType = "body")
    })
    @PostMapping("/unbindWeChat")
    public Result unbindWeChat(@RequestBody List<String> ids) {
        mdmUserService.unbindWeChat(ids);
        return Result.ok("解绑成功");
    }

    @ApiOperation(value = "强制批量修改密码", httpMethod = "POST")
    @PostMapping("/forceChangePassword")
    public Result forceChangePassword(@RequestBody MdmUserForceChangePasswordReqVo reqVo) {
        mdmUserService.forceChangePassword(reqVo);
        return Result.ok("修改成功");
    }

    @ApiOperation(value = "获取客户用户新增编辑页面控制变量")
    @GetMapping("/getCustomerUserControl")
    public Result<MdmCustomerUserControlRespVo> getCustomerUserControl() {
        MdmCustomerUserControlRespVo respVo = new MdmCustomerUserControlRespVo();
        respVo.setUserCanRelationManyCustomer(ParamUtil.getParameterValue(ParameterParam.USER_CAN_RELATION_MANY_CUSTOMER));
        return Result.ok(respVo);
    }

    @ApiOperation(value = "根据userName获取客户用户关联的全部客户", httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "userName", value = "用户登录名", dataType = "String", paramType = "query")
    })
    @GetMapping("/getUserAllCustomerList")
    @CrmDictMethod
    public Result<List<MdmCustomerMsgRespVo>> getUserAllCustomerList(@RequestParam(value = "userName") String userName) {
        return Result.ok(mdmCustomerMsgService.getCustomerListByUserName(userName));
    }

    @ApiOperation(value = "切换客户用户当前客户")
    @PostMapping("/changeUserCurrentCustomer")
    public Result changeUserCurrentCustomer(@RequestBody MdmUserChangeCurrentCustomerReqVo reqVo) {
        mdmCustomerUserService.changeUserCurrentCustomer(reqVo);
        return Result.ok();
    }

    @ApiOperation(value = "客户用户分页弹框选择列表")
    @PostMapping("/findCustomerUserSelectPageList")
    @CrmDictMethod
    public Result<PageResult<MdmUserSelectRespVo>> findCustomerUserSelectPageList(@RequestBody MdmUserSelectReqVo reqVo) {
        return Result.ok(mdmCustomerUserService.findCustomerUserSelectPageList(reqVo));
    }

    @ApiOperation(value = "查询经销商关联的客户用户列表")
    @PostMapping("/findCustomerRelCustomerUserList")
    @CrmDictMethod
    public Result<List<MdmUserRespVo>> findCustomerRelCustomerUserList(@RequestBody MdmCustomerRelCustomerUserReqVo reqVo) {
        return Result.ok(mdmCustomerUserService.findCustomerRelCustomerUserList(reqVo));
    }

    @ApiOperation(value = "客户用户下拉框")
    @PostMapping("/findCustomerUserSelectList")
    public Result<List<MdmUserSelectRespVo>> findCustomerUserSelectList(@RequestBody MdmUserSelectReqVo reqVo) {
        return Result.ok(mdmCustomerUserService.findCustomerUserSelectList(reqVo));
    }

    @ApiOperation(value = "查询未关联任何角色的客户用户")
    @PostMapping("/findCustomerUserNotRelAnyRole")
    @CrmDictMethod
    public Result<PageResult<MdmCustomerUserRelRolePageRespVo>> findCustomerUserNotRelAnyRole(@RequestBody MdmCustomerUserRelRolePageReqVo reqVo) {
        return Result.ok(mdmCustomerUserService.findCustomerUserNotRelAnyRole(reqVo));
    }

    @ApiOperation(value = "查询未关联当前角色的客户用户", notes = "角色编码必传")
    @PostMapping("/findCustomerUserNotRelCurRole")
    @CrmDictMethod
    public Result<PageResult<MdmCustomerUserRelRolePageRespVo>> findCustomerUserNotRelCurRole(@RequestBody MdmCustomerUserRelRolePageReqVo reqVo) {
        return Result.ok(mdmCustomerUserService.findCustomerUserNotRelCurRole(reqVo));
    }

    @ApiOperation(value = "查询已关联当前角色的客户用户", notes = "角色编码必传")
    @PostMapping("/findCustomerUserHasRelCurRole")
    @CrmDictMethod
    public Result<PageResult<MdmCustomerUserRelRolePageRespVo>> findCustomerUserHasRelCurRole(@RequestBody MdmCustomerUserRelRolePageReqVo reqVo) {
        return Result.ok(mdmCustomerUserService.findCustomerUserHasRelCurRole(reqVo));
    }

    @ApiOperation(value = "添加客户用户与角色关联")
    @PostMapping("/bindUserRelRole")
    public Result bindUserRelRole(@RequestBody MdmCustomerUserRelRoleReqVo reqVo) {
        mdmCustomerUserService.bindUserRelRole(reqVo);
        return Result.ok();
    }

    @ApiOperation(value = "替换客户用户关联的角色")
    @PostMapping("/replaceUserRelRole")
    public Result replaceUserRelRole(@RequestBody MdmCustomerUserRelRoleReqVo reqVo) {
        mdmCustomerUserService.replaceUserRelRole(reqVo);
        return Result.ok();
    }

    @ApiOperation(value = "解除客户用户与角色关联", httpMethod = "POST")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "ids", value = "关联关系id集合", required = true, paramType = "body")
    })
    @PostMapping("/unbindUserRelRole")
    public Result unbindUserRelRole(@RequestBody List<String> ids) {
        mdmCustomerUserService.unbindUserRelRole(ids);
        return Result.ok();
    }

    @ApiOperation(value = "查询未关联任何客户用户的终端")
    @PostMapping("/findTerminalNotRelAnyCustomerUser")
    @CrmDictMethod
    public Result<PageResult<MdmCustomerUserRelTerminalPageRespVo>> findTerminalNotRelAnyCustomerUser(@RequestBody MdmCustomerUserRelTerminalPageReqVo reqVo) {
        return Result.ok(mdmCustomerUserService.findTerminalNotRelAnyCustomerUser(reqVo));
    }

    @ApiOperation(value = "查询未关联当前客户用户的终端", notes = "客户用户账号必填")
    @PostMapping("/findTerminalNotRelCurCustomerUser")
    @CrmDictMethod
    public Result<PageResult<MdmCustomerUserRelTerminalPageRespVo>> findTerminalNotRelCurCustomerUser(@RequestBody MdmCustomerUserRelTerminalPageReqVo reqVo) {
        return Result.ok(mdmCustomerUserService.findTerminalNotRelCurCustomerUser(reqVo));
    }

    @ApiOperation(value = "查询已关联当前客户用户的终端", notes = "客户用户账号必填")
    @PostMapping("/findTerminalHasRelCurCustomerUser")
    @CrmDictMethod
    public Result<PageResult<MdmCustomerUserRelTerminalPageRespVo>> findTerminalHasRelCurCustomerUser(@RequestBody MdmCustomerUserRelTerminalPageReqVo reqVo) {
        return Result.ok(mdmCustomerUserService.findTerminalHasRelCurCustomerUser(reqVo));
    }

    @ApiOperation(value = "添加客户用户与终端的关联")
    @PostMapping("/bindCustomerUserRelTerminal")
    public Result bindCustomerUserRelTerminal(@RequestBody MdmCustomerUserRelTerminalReqVo reqVo) {
        mdmCustomerUserService.bindCustomerUserRelTerminal(reqVo);
        return Result.ok();
    }

    @ApiOperation(value = "替换终端关联的客户用户")
    @PostMapping("/replaceCustomerUserRelTerminal")
    public Result replaceCustomerUserRelTerminal(@RequestBody MdmCustomerUserRelTerminalReqVo reqVo) {
        mdmCustomerUserService.replaceCustomerUserRelTerminal(reqVo);
        return Result.ok();
    }

    @ApiOperation(value = "解除客户用户与终端关联")
    @PostMapping("/unbindCustomerUserRelTerminal")
    public Result unbindCustomerUserRelTerminal(@RequestBody MdmCustomerUserRelTerminalReqVo reqVo) {
        mdmCustomerUserService.unbindCustomerUserRelTerminal(reqVo);
        return Result.ok();
    }

}
