package com.biz.crm.user.controller;

import com.biz.crm.aop.CrmLog;
import com.biz.crm.common.PageResult;
import com.biz.crm.config.CrmDictMethod;
import com.biz.crm.nebular.mdm.position.resp.MdmPositionSelectRespVo;
import com.biz.crm.nebular.mdm.position.resp.MdmPositionUserOrgRespVo;
import com.biz.crm.nebular.mdm.user.req.*;
import com.biz.crm.nebular.mdm.user.resp.MdmUserPageRespVo;
import com.biz.crm.nebular.mdm.user.resp.MdmUserRelWeChatRespVo;
import com.biz.crm.nebular.mdm.user.resp.MdmUserRespVo;
import com.biz.crm.nebular.mdm.user.resp.MdmUserSelectRespVo;
import com.biz.crm.user.service.MdmUserRelWeChatService;
import com.biz.crm.user.service.MdmUserService;
import com.biz.crm.util.Result;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * 用户信息表
 *
 * @author Tao.Chen
 * @date 2020-11-17 19:04:29
 */
@Slf4j
@RestController
@RequestMapping("/mdmUserController")
@Api(tags = "MDM-用户管理（企业用户）")
public class MdmUserController {

    @Autowired
    private MdmUserService mdmUserService;
    @Autowired
    private MdmUserRelWeChatService mdmUserRelWeChatService;

    @ApiOperation(value = "查询分页列表")
    @PostMapping("/pageList")
    @CrmDictMethod
    @CrmLog
    public Result<PageResult<MdmUserPageRespVo>> pageList(@RequestBody MdmUserPageReqVo reqVo) {
        PageResult<MdmUserPageRespVo> result = mdmUserService.findList(reqVo);
        return Result.ok(result);
    }

    @ApiOperation(value = "详情", httpMethod = "POST")
    @PostMapping("/query")
    @CrmLog
    @CrmDictMethod
    public Result<MdmUserRespVo> query(@RequestBody MdmUserReqVo mdmUserReqVo) {
        return Result.ok(mdmUserService.detail(mdmUserReqVo.getId(), mdmUserReqVo.getUserName()));
    }

    @ApiOperation(value = "详情", httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "ID", required = false, dataType = "String", paramType = "query"),
            @ApiImplicitParam(name = "userName", value = "用户登录名", required = false, dataType = "String", paramType = "query")
    })
    @GetMapping("/detail")
    @CrmLog
    @CrmDictMethod
    public Result<MdmUserRespVo> detail(@RequestParam(value = "id", required = false) String id, @RequestParam(value = "userName", required = false) String userName) {
        return Result.ok(mdmUserService.detail(id, userName));
    }

    @ApiOperation(value = "新增")
    @PostMapping("/save")
    @CrmLog
    public Result save(@RequestBody MdmUserEditReqVo reqVo) {
        mdmUserService.save(reqVo);
        return Result.ok();
    }

    @ApiOperation(value = "更新")
    @PostMapping("/update")
    @CrmLog
    public Result update(@RequestBody MdmUserEditReqVo reqVo) {
        mdmUserService.update(reqVo);
        return Result.ok("修改成功");
    }

    @ApiOperation(value = "删除", httpMethod = "POST")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "ids", value = "id集合", required = true, paramType = "body")
    })
    @PostMapping("/delete")
    @CrmLog
    public Result delete(@RequestBody List<String> ids) {
        mdmUserService.deleteBatch(ids);
        return Result.ok("删除成功");
    }

    @ApiOperation(value = "启用", httpMethod = "POST")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "ids", value = "id集合", required = true, paramType = "body")
    })
    @PostMapping("/enable")
    @CrmLog
    public Result enable(@RequestBody List<String> ids) {
        mdmUserService.enableBatch(ids);
        return Result.ok("启用成功");
    }

    @ApiOperation(value = "禁用", httpMethod = "POST")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "ids", value = "id集合", required = true, paramType = "body")
    })
    @PostMapping("/disable")
    @CrmLog
    public Result disable(@RequestBody List<String> ids) {
        mdmUserService.disableBatch(ids);
        return Result.ok("禁用成功");
    }

    @ApiOperation("修改用户密码")
    @PostMapping("/changePassword")
    public Result changePassword(@RequestBody MdmUserChangePasswordReqVo reqVo) {
        mdmUserService.changePassword(reqVo);
        return Result.ok();
    }

    @ApiOperation("重新设置自己的密码")
    @PostMapping("/resetMyselfPassword")
    public Result resetMyselfPassword(@RequestBody MdmUserChangePasswordReqVo reqVo) {
        mdmUserService.resetMyselfPassword(reqVo);
        return Result.ok();
    }

    @ApiOperation(value = "用户解锁", httpMethod = "POST")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "ids", value = "id集合", required = true, paramType = "body")
    })
    @PostMapping("/unlock")
    @CrmLog
    public Result unlock(@RequestBody List<String> ids) {
        mdmUserService.unlock(ids);
        return Result.ok("解锁成功");
    }

    @ApiOperation(value = "（过期）查询用户关联的微信列表", httpMethod = "GET")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "userName", value = "用户登录名", required = true, dataType = "String", paramType = "query")
    })
    @GetMapping("/findUserRelWeChatList")
    public Result<List<MdmUserRelWeChatRespVo>> findUserRelWeChatList(@RequestParam(value = "userName", required = true) String userName) {
        return Result.ok(mdmUserRelWeChatService.findUserRelWeChatList(new MdmUserRelWeChatReqVo().setUserName(userName)));
    }

    @ApiOperation(value = "（过期）解绑指定关联的微信", httpMethod = "POST")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "ids", value = "用户关联微信表的id集合", required = true, paramType = "body")
    })
    @PostMapping("/unbindUserWeChat")
    public Result unbindUserWeChat(@RequestBody List<String> ids) {
        mdmUserRelWeChatService.unbindByIds(ids);
        return Result.ok("解绑成功");
    }

    @ApiOperation(value = "解绑用户全部关联微信", httpMethod = "POST")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "ids", value = "用户id集合", required = true, paramType = "body")
    })
    @PostMapping("/unbindWeChat")
    public Result unbindWeChat(@RequestBody List<String> ids) {
        mdmUserService.unbindWeChat(ids);
        return Result.ok("解绑成功");
    }

    @ApiOperation(value = "强制批量修改密码", httpMethod = "POST")
    @PostMapping("/forceChangePassword")
    public Result forceChangePassword(@RequestBody MdmUserForceChangePasswordReqVo reqVo) {
        mdmUserService.forceChangePassword(reqVo);
        return Result.ok("修改成功");
    }

    @ApiOperation("用户基本信息列表查询")
    @PostMapping("/list_detail")
    public Result<List<MdmUserRespVo>> listDetail(@RequestBody MdmUserReqVo mdmUserReqVo) {
        return Result.ok(mdmUserService.listDetail(mdmUserReqVo));
    }

    @ApiOperation("获取当前登录企业用户的当前职位关联的个人信息详情")
    @GetMapping("/getCurUserPositionDetail")
    public Result<MdmPositionUserOrgRespVo> getCurUserPositionDetail() {
        return Result.ok(mdmUserService.getCurUserPositionDetail());
    }

    @ApiOperation(value = "修改用户头像")
    @PostMapping("/changeUserHeadUrl")
    public Result changeUserHeadUrl(@RequestBody MdmUserChangeHeadReqVo reqVo) {
        mdmUserService.changeUserHeadUrl(reqVo);
        return Result.ok();
    }

    @ApiOperation(value = "（用户新增编辑专用）查询已有职位下拉框列表", notes = "查询非主职位或者未关联用户的主职位，只查启用的，组织直接关联的")
    @PostMapping("/findUserPositionSelectList")
    public Result<List<MdmPositionSelectRespVo>> findUserPositionSelectList(@RequestBody MdmUserPositionSelectReqVo reqVo) {
        return Result.ok(mdmUserService.findUserPositionSelectList(reqVo));
    }

    @ApiOperation(value = "（用户新增编辑专用）查询上级职位下拉框列表", notes = "查询启用、排除当前用户、排除组织下级、排除职位下级职位对应组织及下级组织")
    @PostMapping("/findUserParentPositionSelectList")
    public Result<List<MdmPositionSelectRespVo>> findUserParentPositionSelectList(@RequestBody MdmUserParentPositionSelectReqVo reqVo) {
        return Result.ok(mdmUserService.findUserParentPositionSelectList(reqVo));
    }

    @ApiOperation(value = "企业用户下拉框", notes = "")
    @PostMapping("/findUserSelectList")
    public Result<List<MdmUserSelectRespVo>> findUserSelectList(@RequestBody MdmUserSelectReqVo reqVo) {
        return Result.ok(mdmUserService.findUserSelectList(reqVo));
    }


}
