package com.biz.crm.user.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.base.BaseIdEntity;
import com.biz.crm.base.BusinessException;
import com.biz.crm.base.config.ThreadLocalUtil;
import com.biz.crm.common.GlobalParam;
import com.biz.crm.common.PageResult;
import com.biz.crm.common.param.ParameterParam;
import com.biz.crm.crmlog.handle.util.CrmLogSendUtil;
import com.biz.crm.eunm.CodeRuleEnum;
import com.biz.crm.eunm.CrmDelFlagEnum;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.eunm.mdm.MdmBpmStatusEnum;
import com.biz.crm.nebular.mdm.constant.UserTypeEnum;
import com.biz.crm.nebular.mdm.terminal.MdmTerminalVo;
import com.biz.crm.nebular.mdm.terminal.resp.MdmTerminalDetailRespVo;
import com.biz.crm.nebular.mdm.user.req.MdmTerminalUserEditReqVo;
import com.biz.crm.nebular.mdm.user.req.MdmTerminalUserPageReqVo;
import com.biz.crm.nebular.mdm.user.req.MdmTerminalUserTerminalSelectReqVo;
import com.biz.crm.nebular.mdm.user.req.MdmUserTerminalEditReqVo;
import com.biz.crm.nebular.mdm.user.resp.MdmTerminalUserDetailRespVo;
import com.biz.crm.nebular.mdm.user.resp.MdmTerminalUserPageRespVo;
import com.biz.crm.role.service.MdmRoleService;
import com.biz.crm.terminal.service.MdmTerminalService;
import com.biz.crm.user.mapper.MdmTerminalUserMapper;
import com.biz.crm.user.mapper.MdmUserMapper;
import com.biz.crm.user.model.MdmTerminalRelRoleEntity;
import com.biz.crm.user.model.MdmUserEntity;
import com.biz.crm.user.model.MdmUserRelTerminalEntity;
import com.biz.crm.user.service.MdmTerminalRelRoleService;
import com.biz.crm.user.service.MdmTerminalUserService;
import com.biz.crm.user.service.MdmUserRelRoleService;
import com.biz.crm.user.service.MdmUserRelTerminalService;
import com.biz.crm.util.*;
import com.biz.crm.utils.LoginHelpUtil;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import javax.annotation.Resource;
import java.util.*;
import java.util.stream.Collectors;
import org.springframework.util.CollectionUtils;


/**
 * 终端用户信息表接口实现
 *
 * @author ql
 * @date 2021/2/3
 * @time 20:31
 **/
@Slf4j
@Service
@ConditionalOnMissingBean(name = "MdmTerminalUserServiceExpandImpl")
public class MdmTerminalUserServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<MdmUserMapper, MdmUserEntity>
    implements MdmTerminalUserService {

  @Resource
  private MdmTerminalUserMapper mdmTerminalUserMapper;

  @Resource
  private MdmTerminalService mdmTerminalService;

  @Resource
  private MdmUserRelTerminalService mdmUserRelTerminalService;

  @Resource
  private MdmUserRelRoleService mdmUserRelRoleService;

  @Autowired
  private CrmLogSendUtil crmLogSendUtil;

  @Resource
  private MdmTerminalRelRoleService mdmTerminalRelRoleService;



  /**
   * 分页查询
   *
   * @param reqVo 请求参数
   * @return 记录
   */
  @Override

  public PageResult<MdmTerminalUserPageRespVo> findList(MdmTerminalUserPageReqVo reqVo) {
    // 将用户类型设置进查询条件
    reqVo.setUserType(UserTypeEnum.TERMINAL.getCode());
    // 构建分页
    Page<MdmTerminalUserPageRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
    // 根据分页查询所有的用户
    List<MdmTerminalUserPageRespVo> list = mdmTerminalUserMapper.findList(page, reqVo);
    // 给查询出来的终端用户添加对应的终端
    this.setTerminalCodeName(list);
    this.setLockState(list);
    // 返回结果
    return PageResult.<MdmTerminalUserPageRespVo>builder()
        .data(list)
        .count(page.getTotal())
        .build();

  }


  /**
   * 保存方法，包括新增和更新
   *
   * @param reqVo 携带的信息
   */
  @Override
  @Transactional(rollbackFor = Exception.class)
  public void save(MdmTerminalUserEditReqVo reqVo) {
    // 处理默认信息
    reqVo.setId(null);
    reqVo.setUserType(UserTypeEnum.TERMINAL.getCode());
    // 参数校验
    this.verifyData(reqVo);
    // 将请求vo转成user实体
    MdmUserEntity entity = CrmBeanUtil.copy(reqVo, MdmUserEntity.class);
    /// 设置用户编码，不过因为用户编码这个字段已经过时了，所以不用再设置，但是不设置会出问题、、、、、SQL问题
    if (StringUtils.isEmpty(entity.getUserCode())) {
      entity.setUserCode(CodeUtil.generateCode(CodeRuleEnum.MDM_USER_CODE.getCode()));
    }
    entity.setLockState(CrmEnableStatusEnum.ENABLE.getCode());
    entity.setForceChangePassword(YesNoEnum.yesNoEnum.ONE.getValue());
    entity.setDelFlag(CrmDelFlagEnum.NORMAL.getCode());
    // 调用保存方法，保存用户这个实体
    this.save(entity);
    //更新终端用户和终端关系
    saveOrUpdateTerminalUserRelation(reqVo, entity);
    //增加日志
    Object objectMenu = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
    if(objectMenu!=null){
      crmLogSendUtil.sendForAdd(ThreadLocalUtil.getObj(GlobalParam.MENU_CODE).toString(), entity.getId(), entity.getId(), entity);
    }
  }


  /**
   * @param id id
   * @return 终端用户详细信息，包括终端列表、角色 根据id或者username查询终端用户的详细信息：包括基本信息、关联的终端、终端关联的角色 思路： 1. 根据ID或者username查询出终端的基本信息 2. 去终端用户-终端中间表中查询出该终端用户关联的所有终端的终端用户-终端中间记录 3. 遍历终端用户-终端中间记录，去终端表中，根据终端用户-终端中间记录查询出终端的详细信息
   * 4. 根据终端的详细信息去终端-角色中间表中查询终端关联的角色中间记录 5. 根据 终端关联的角色中间记录 去 角色表中查询角色的详细信息 6. 将角色信息设置到终端中，将所有的终端组成一个集合设置到终端详细信息中
   */
  @Override
  public MdmTerminalUserDetailRespVo queryByIdOrUsername(String id, String username) {
    // 如果ID或者username不为empty，
    if (StringUtils.isNotEmpty(id) || StringUtils.isNotEmpty(username)) {
      // 哪个不为empty，就根据哪个进行查询
      MdmUserEntity mdmUserEntity = this.lambdaQuery()
          .eq(!StringUtils.isEmpty(id), MdmUserEntity::getId, id)
          .eq(!StringUtils.isEmpty(username), MdmUserEntity::getUserName, username)
          .eq(MdmUserEntity::getUserType, UserTypeEnum.TERMINAL.getCode())
          .one();
      if (mdmUserEntity != null) {
        // 将查询出来的终端用户信息复制到对应的响应vo中
        MdmTerminalUserDetailRespVo mdmTerminalUserDetailRespVo = CrmBeanUtil.copy(mdmUserEntity, MdmTerminalUserDetailRespVo.class);
        // 去 终端用户-终端 中间表中根据终端用户查询该终端用户所有的终端信息
        List<MdmUserRelTerminalEntity> mdmUserRelTerminalEntities = mdmUserRelTerminalService.findTerminalCodeListByUserName(mdmUserEntity.getUserName());

        Map<String, MdmUserRelTerminalEntity> relTerminalEntityMap = mdmUserRelTerminalEntities.stream().collect(Collectors.toMap(MdmUserRelTerminalEntity::getTerminalCode, v -> v));

        List<String> terminalCodeList = mdmUserRelTerminalEntities.stream().map(MdmUserRelTerminalEntity::getTerminalCode).collect(Collectors.toList());
        if (terminalCodeList.size() > 0) {
          List<MdmTerminalVo> mdmTerminalVoList = mdmTerminalService.queryBatchByTerminalCodeList(terminalCodeList);
          if (CollectionUtil.listNotEmptyNotSizeZero(mdmTerminalVoList)) {
            List<MdmTerminalDetailRespVo> mdmTerminalDetailList = CrmBeanUtil.copyList(mdmTerminalVoList, MdmTerminalDetailRespVo.class);
            mdmTerminalDetailList.forEach(item->{
              // 插入角色信息
              MdmUserRelTerminalEntity mdmUserRelTerminalEntity = relTerminalEntityMap.get(item.getTerminalCode());
              if(mdmUserRelTerminalEntity!=null){
                List<MdmTerminalRelRoleEntity> relRoleList = mdmTerminalRelRoleService.list(
                    Wrappers.lambdaQuery(MdmTerminalRelRoleEntity.class).eq(MdmTerminalRelRoleEntity::getUserRelId, mdmUserRelTerminalEntity.getId())
                        .eq(MdmTerminalRelRoleEntity::getTerminalCode, item.getTerminalCode()));
                List<String> roleCodeList = relRoleList.stream().map(MdmTerminalRelRoleEntity::getRoleCode).collect(Collectors.toList());
                if(CollectionUtil.listNotEmpty(roleCodeList)){
                  item.setRoleCodeList(roleCodeList);
                }
              }
            });
            mdmTerminalUserDetailRespVo.setTerminalList(mdmTerminalDetailList);
          }
        }
        // 如果该终端用户只有一个终端，不管是否为一对一，都就这个单独的终端和角色设置进前端展示的地方
     /*   if (CollectionUtil.listNotEmptyNotSizeZero(mdmTerminalUserDetailRespVo.getTerminalList()) && YesNoEnum.yesNoEnum.N.getValue()
            .equals(ParamUtil.getParameterValue(ParameterParam.USER_CAN_RELATION_MANY_TERMINAL))) {
          // 设置终端用户的 终端名称、终端代码、角色名称、角色代码
          mdmTerminalUserDetailRespVo.setTerminalCode(mdmTerminalUserDetailRespVo.getTerminalList().get(0).getTerminalCode());
          mdmTerminalUserDetailRespVo.setTerminalName(mdmTerminalUserDetailRespVo.getTerminalList().get(0).getTerminalName());
        }*/
        mdmTerminalUserDetailRespVo.setRoleCodeList(mdmUserRelRoleService.findRoleCodeListByUserName(mdmUserEntity.getUserName()));
        return mdmTerminalUserDetailRespVo;
      }
      return null;
    }
    return null;
  }


  /**
   * 更新方法
   *
   * @param mdmTerminalUserEditReqVo 携带的修改信息的请求vo
   *                                 <p>
   *                                 先更用户表中的基本用户信息，再更新终端用户与终端、终端与角色之间的关联信息
   */
  @Override
  @Transactional(rollbackFor = Exception.class)
  public void update(MdmTerminalUserEditReqVo mdmTerminalUserEditReqVo) {
    // 其实这里这样做，我不知道好不好。按道理在修改的时候，这个用户类型是应该被带过来的，一起带过来就能正常更新。但是如果在接口调试的时候，忘了穿这个字段，就会出错
    mdmTerminalUserEditReqVo.setUserType(UserTypeEnum.TERMINAL.getCode());
    // 校验参数
    this.verifyData(mdmTerminalUserEditReqVo);
    // 根据ID查询查询出终端用户实体
    MdmUserEntity entity = this.getById(mdmTerminalUserEditReqVo.getId());
    //复制旧数据到VO
    MdmTerminalUserEditReqVo oldVo = CrmBeanUtil.copy(entity, MdmTerminalUserEditReqVo.class);
    // 将查询出的实体复制到一个MdmTerminalUserEditReqVo中，用于更新
    CrmBeanUtil.copyProperties(mdmTerminalUserEditReqVo, entity);
    // 更新终端用户之后
    this.updateById(entity);
    // 再更新终端用户与终端的中间表
    saveOrUpdateTerminalUserRelation(mdmTerminalUserEditReqVo, entity);
    //更新日志
    crmLogSendUtil.sendForUpdate(ThreadLocalUtil.getObj(GlobalParam.MENU_CODE).toString(), entity.getId(), entity.getId(), oldVo, mdmTerminalUserEditReqVo);
  }

  /**
   * @param ids 待删除的ID集合
   *            <p>
   *            根据ID集合删除用户 思路： 1. 进行一些参数校验 2. 遍历ID集合，根据ID去查询用户详情，判断用户是否真实存在 3. 根据用户详情中的username去 终端用户-终端中间表中 删除该终端用户所关联的所有终端，不是删除终端，是删除中间表中的记录。终端与角色的关系不处理 4. 关联关系处理完毕了之后，再删除这个终端用户
   */
  @Override
  @Transactional(rollbackFor = Exception.class)
  public void deleteBatch(List<String> ids) {
    Assert.notEmpty(ids, "id集合不能为空");
    // 遍历这个ID集合，校验ID有效性，处理关联关系
    ids.forEach(id -> {
      Assert.hasText(id, "不能传空值");
      MdmUserEntity mdmUser = this.getById(id);
      Assert.notNull(mdmUser, "无效的用户");
      Assert.isTrue(UserTypeEnum.TERMINAL.getCode().equals(mdmUser.getUserType()),
          "用户[" + mdmUser.getFullName() + "]不是终端用户");
      // 清空终端用户尚关联的所有终端，根据终端用户用户名删除，删除终端上关联的角色

      List<MdmUserRelTerminalEntity> list = mdmUserRelTerminalService.lambdaQuery()
          .eq(MdmUserRelTerminalEntity::getUserName, mdmUser.getUserName())
          .select(MdmUserRelTerminalEntity::getId)
          .list();

      // 清除此终端用户终端关联角色信息
      if (!CollectionUtils.isEmpty(list)) {
        List<String> userRelIds = list.stream().map(BaseIdEntity::getId).collect(Collectors.toList());
        mdmTerminalRelRoleService.lambdaUpdate()
            .in(MdmTerminalRelRoleEntity::getUserRelId, userRelIds)
            .remove();
      }
      mdmUserRelTerminalService.lambdaUpdate()
          .eq(MdmUserRelTerminalEntity::getUserName, mdmUser.getUserName())
          .remove();
    });
    //删除用户
    this.removeByIds(ids);
  }




  /**
   * 获取终端下拉列表，如果是新增，那么需要返回没有被用户关联的终端 如果是编辑，前端需要将当前编辑的用户名传过来，然后将当前用户已经关联的终端放在前面
   *
   * @return 满足条件的终端列表
   */
  @Override
  public List<MdmTerminalDetailRespVo> getTerminalListByCondition(MdmTerminalUserTerminalSelectReqVo reqVo) {
    // 处理启用状态
    reqVo.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
    // 处理审批状态
    reqVo.setActApproveStatus(MdmBpmStatusEnum.PASS.getValue());
    // 处理每页显示条数，没有就为全局参数
    Integer pageSize = reqVo.getPageSize();
    if (pageSize == null || pageSize == 0) {
      pageSize = Integer.valueOf(ParamUtil.getParameterValue(ParameterParam.SELECT_BOX_OPTION_SIZE));
    }
    Set<String> selectedCodeList = new HashSet<>();
    if (StringUtils.isNotEmpty(reqVo.getSelectedCode())) {
      selectedCodeList.add(reqVo.getSelectedCode());
    }
    if (CollectionUtil.listNotEmptyNotSizeZero(reqVo.getSelectedCodeList())) {
      selectedCodeList.addAll(reqVo.getSelectedCodeList());
    }
    reqVo.setSelectedCodeList(new ArrayList<>(selectedCodeList));
    return mdmTerminalUserMapper.getTerminalListByCondition(new Page<>(1L, pageSize, false), reqVo);
  }


  /**
   * 保存终端用户与终端的关联关系
   *
   * @param reqVo      页面参数
   * @param userEntity 实体 更新终端用户和终端的关系逻辑 1. 对终端用户与终端关联表mdm_user_rel_terminal中该该用户的所有关联信息删除 2. 更新终端用户与终端的关系 1. 判断新添加的终端是否存在 2. 判断终端是否被其他终端用户所关联 3. 对终端与角色关联信息进行更新 1. 判断角色是否存在 2. 判断角色是否被其他终端关联 3.
   *                   判断终端是否关联了其它角色
   */
  private void saveOrUpdateTerminalUserRelation(MdmTerminalUserEditReqVo reqVo, MdmUserEntity userEntity) {
    // 读取全局配置，终端用户是否能与终端一对多,2021-08-20 确认做成一对多的关系 不用读取全局配置了
       /* if (YesNoEnum.yesNoEnum.N.getValue().equals(ParamUtil.getParameterValue(ParameterParam.USER_CAN_RELATION_MANY_TERMINAL))) {
            // 如果不能一对多，那么将单独的terminalCode和roleCode设置进terminalList，并且设置为主终端
            reqVo.setTerminalList(Collections.singletonList(new MdmUserTerminalEditReqVo()
                    .setTerminalCode(reqVo.getTerminalCode())));
        }*/
    // todo 后面直接往terminalList里面放值 这里要去掉
    if (StringUtils.isNotEmpty(reqVo.getTerminalCode())) {
      reqVo.setTerminalList(Collections.singletonList(new MdmUserTerminalEditReqVo()
          .setTerminalCode(reqVo.getTerminalCode())));
    }
    if (CollectionUtil.listNotEmptyNotSizeZero(reqVo.getRoleCodeList())) {
      mdmUserRelRoleService.add(userEntity.getUserName(), reqVo.getRoleCodeList());
    }

    // mdmUserRelTerminalService.add(userEntity.getUserName(), reqVo.getTerminalList().stream().map(MdmUserTerminalEditReqVo::getTerminalCode).collect(Collectors.toList()));

    mdmUserRelTerminalService.add(userEntity.getUserName(),reqVo.getTerminalList());
  }


  /**
   * 对增加/编辑 参数进行校验
   *
   * @param reqVo 参数
   */
  private void verifyData(MdmTerminalUserEditReqVo reqVo) {
    // 校验参数
    String userName = reqVo.getUserName();
    Assert.hasText(userName, "用户帐号不能为空");
    Assert.hasText(reqVo.getFullName(), "用户姓名不能为空");
    // 校验用户名是否重复
    MdmUserEntity one = this.lambdaQuery()
        .eq(MdmUserEntity::getUserName, userName)
        .one();
    if (StringUtils.isNotEmpty(reqVo.getUserPhone())) {
      List<MdmUserEntity> list = this.lambdaQuery()
          .eq(MdmUserEntity::getUserPhone, reqVo.getUserPhone())
          .list()
          .stream().filter(x -> StringUtils.isEmpty(reqVo.getId()) || !reqVo.getId().equals(x.getId()))
          .collect(Collectors.toList());
      Assert.isTrue(CollectionUtil.listEmpty(list), "手机号码已经被占用");
    }
    if (StringUtils.isNotEmpty(reqVo.getEmail())) {
      List<MdmUserEntity> list = this.lambdaQuery()
          .eq(MdmUserEntity::getEmail, reqVo.getEmail())
          .list()
          .stream().filter(x -> StringUtils.isEmpty(reqVo.getId()) || !reqVo.getId().equals(x.getId()))
          .collect(Collectors.toList());
      Assert.isTrue(CollectionUtil.listEmpty(list), "邮箱已经被占用");
    }
    if (StringUtils.isNotEmpty(reqVo.getIdentityCardNumber())) {
      List<MdmUserEntity> list = this.lambdaQuery()
          .eq(MdmUserEntity::getIdentityCardNumber, reqVo.getIdentityCardNumber())
          .list()
          .stream().filter(x -> StringUtils.isEmpty(reqVo.getId()) || !reqVo.getId().equals(x.getId()))
          .collect(Collectors.toList());
      Assert.isTrue(CollectionUtil.listEmpty(list), "身份证号码已经存在");
    }
    // 如果id为空，说明是新增。这里要注意：userName是不能编辑的，所以只有在增加的时候才进行重名验证
    if (StringUtils.isEmpty(reqVo.getId())) {
      // 校验
      if (one != null) {
        String delMark = "";
        if (CrmDelFlagEnum.DELETE.getCode().equals(one.getDelFlag())) {
          delMark = "（已逻辑删除）";
        }
        throw new BusinessException("帐号[" + userName + "]已存在" + delMark);
      }
      Assert.hasText(reqVo.getUserPassword(), "密码不能为空");
    }
    // 密码为空，那么肯定是修改，因为如果是新增，密码非空已经校验过了
    if (StringUtils.isEmpty(reqVo.getUserPassword())) {
      // 修改密码为空，那么将密码设置原来的
      reqVo.setUserPassword(one.getUserPassword());
    } else {
      //密码加密
      reqVo.setUserPassword(Md5EncryptionAndDecryption.encryPwd(reqVo.getUserPassword()));
    }

    /**
     *  String parameterValue = ParamUtil.getParameterValue(ParameterParam.USER_CAN_RELATION_MANY_TERMINAL);
     * 终端用户 2021-08-27 需求：终端用户与终端为一对多的关系
     * 获取从前端拿到的终端列表
     */
    // todo 后面直接往terminalList里面放值 这里要去掉
    if (StringUtils.isNotEmpty(reqVo.getTerminalCode())) {
      reqVo.setTerminalList(Collections.singletonList(new MdmUserTerminalEditReqVo()
          .setTerminalCode(reqVo.getTerminalCode())));
    }
    List<MdmUserTerminalEditReqVo> terminalList = reqVo.getTerminalList();
    // 终端列表为空
    Assert.notEmpty(terminalList, "终端用户必须关联终端");
    // 对终端用户关联的终端进行去重和主终端唯一验证
    Map<String, MdmUserTerminalEditReqVo> terminalUniqueMap = new HashMap<>(16);
    for (MdmUserTerminalEditReqVo item : terminalList) {
      // 终端数据安全性校验
      Assert.hasText(item.getTerminalCode(), "终端不能为空");
      if (terminalUniqueMap.containsKey(item.getTerminalCode())) {
        throw new BusinessException("不能重复关联相同终端");
      }
      terminalUniqueMap.put(item.getTerminalCode(), item);
    }

  }


  /**
   * 将查询出来的终端用户的终端设置到返回vo中，同时也需要设置角色 逻辑： 如果是一对一，直接将终端编码和终端名称设置进对应的属性中 如果是一对多，就用,分隔
   */
  private void setTerminalCodeName(List<MdmTerminalUserPageRespVo> allList) {
    if (CollectionUtil.listEmpty(allList)) {
      return;
    }
    List<List<MdmTerminalUserPageRespVo>> partition = Lists.partition(allList, 500);
    for (List<MdmTerminalUserPageRespVo> list :
        partition) {
      final Map<String, List<String>> map = mdmUserRelTerminalService.getTerminalCodeListGroupByUserNameList(list.stream().map(MdmTerminalUserPageRespVo::getUserName).collect(Collectors.toList()));
      if (!map.isEmpty()) {
        Set<String> collect = map.values().stream().flatMap(Collection::stream).collect(Collectors.toSet());
        final Map<String, String> terminalCodeNameMap = mdmTerminalService.getTerminalCodeNameMap(new ArrayList<>(collect));
        list.forEach(item -> {
          if (map.containsKey(item.getUserName())) {
            List<String> customerCodeList = map.get(item.getUserName()).stream().filter(terminalCodeNameMap::containsKey).collect(Collectors.toList());
            if (CollectionUtil.listNotEmptyNotSizeZero(customerCodeList)) {
              String names = customerCodeList.stream().map(terminalCodeNameMap::get).collect(Collectors.joining(","));
              item.setTerminalCode(String.join(",", customerCodeList));
              item.setTerminalName(names);
            }
          }
        });
      }
    }

  }

  private void setLockState(List<MdmTerminalUserPageRespVo> list) {
    if (CollectionUtil.listEmpty(list)) {
      return;
    }
    final Set<String> lockUserNameSet = LoginHelpUtil.checkAndGetLockUserName(list.stream().map(MdmTerminalUserPageRespVo::getUserName).collect(Collectors.toSet()));
    list.forEach(item -> {
      item.setLockState(!lockUserNameSet.contains(item.getUserName()) ? CrmEnableStatusEnum.ENABLE.getCode() : CrmEnableStatusEnum.DISABLE.getCode());
    });
  }

}
