package com.biz.crm.user.service.impl;

import com.biz.crm.common.param.ParameterParam;
import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.user.model.MdmUserLoginLogEntity;
import com.biz.crm.user.service.MdmUserLoginLogAsyncService;
import com.biz.crm.user.service.MdmUserLoginLogService;
import com.biz.crm.user.service.MdmUserService;
import com.biz.crm.util.IpUtil;
import com.biz.crm.util.ParamUtil;
import com.biz.crm.util.StringUtils;
import com.biz.crm.utils.ApplicationProperties;
import com.biz.crm.utils.GeoLiteReader;
import com.maxmind.geoip2.DatabaseReader;
import com.maxmind.geoip2.model.CityResponse;
import com.maxmind.geoip2.record.*;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import java.net.InetAddress;

/**
 * 用户登录日志表接口实现
 *
 * @author Tao.Chen
 * @date 2021-03-18 21:38:20
 */
@Slf4j
@Component
@ConditionalOnMissingBean(name = "MdmUserLoginLogAsyncServiceExpandImpl")
public class MdmUserLoginLogAsyncServiceImpl implements MdmUserLoginLogAsyncService {

    @Resource
    private MdmUserLoginLogService mdmUserLoginLogService;
    @Resource
    private MdmUserService mdmUserService;
    @Resource
    private ApplicationProperties applicationProperties;

    @Override
    @Async
    public void saveLoginLog(HttpServletRequest request, String userName, String fullName, String fromType, String loginDateTime) {
        if (request == null) {
            return;
        }
        if (StringUtils.isEmpty(userName)) {
            return;
        }
        if (StringUtils.isEmpty(fullName)) {
            fullName = mdmUserService.getFullNameByUserName(userName);
        }
        MdmUserLoginLogEntity logEntity = new MdmUserLoginLogEntity();
        logEntity.setUserName(userName);
        logEntity.setFullName(fullName);
        logEntity.setFromType(fromType);
        logEntity.setLoginDateTime(loginDateTime);
        this.setClient(logEntity, request);
        this.setServer(logEntity, request);
        mdmUserService.updateUserLastLoginDateTime(userName, loginDateTime);
        mdmUserLoginLogService.save(logEntity);
    }

    /**
     * 获取客户端信息
     *
     * @param logEntity
     * @param request
     */
    private void setClient(MdmUserLoginLogEntity logEntity, HttpServletRequest request) {
        try {
            String clientIp = IpUtil.getIpAddrByRequest(request);
            logEntity.setClientIp(clientIp);
            if (StringUtils.isEmpty(clientIp) || clientIp.contains("0:0:0:0:0:0:0:1") || clientIp.contains("127.0.0.1")) {
                return;
            }
            if (!YesNoEnum.yesNoEnum.Y.getValue().equalsIgnoreCase(ParamUtil.getParameterValueNoException(ParameterParam.IP_RESOURCE))) {
                GeoLiteReader.clear();
                return;
            }
            DatabaseReader reader = GeoLiteReader.getInstance();
            InetAddress inetAddress = InetAddress.getByName(clientIp);
            CityResponse cityResponse = reader.city(inetAddress);

            // 获取大洲
            Continent continent = cityResponse.getContinent();
            logEntity.setClientContinentZh(continent.getNames().get("zh-CN"));
            logEntity.setClientContinentEn(continent.getNames().get("en"));

            // 获取国家信息
            Country country = cityResponse.getCountry();
            logEntity.setClientCountryZh(country.getNames().get("zh-CN"));
            logEntity.setClientCountryEn(country.getNames().get("en"));

            // 获取省份
            Subdivision subdivision = cityResponse.getMostSpecificSubdivision();
            logEntity.setClientSubdivisionZh(subdivision.getNames().get("zh-CN"));
            logEntity.setClientSubdivisionEn(subdivision.getNames().get("en"));

            // 获取城市
            City city = cityResponse.getCity();
            logEntity.setClientCityZh(city.getNames().get("zh-CN"));
            logEntity.setClientCityEn(city.getNames().get("en"));

            // 经纬度
            Location location = cityResponse.getLocation();
            logEntity.setClientLatitude(location.getLatitude() == null ? null : location.getLatitude().toString());
            logEntity.setClientLongitude(location.getLongitude() == null ? null : location.getLongitude().toString());

        } catch (Exception e) {
            log.error("获取登录客户端信息失败");
        }
    }


    /**
     * 获取服务端信息
     *
     * @param logEntity
     * @param request
     */
    private void setServer(MdmUserLoginLogEntity logEntity, HttpServletRequest request) {
        try {
            logEntity.setServerPort(applicationProperties.getPort());
            String serverName = request.getServerName();
//            logEntity.setServerName(serverName);
//            logEntity.setServerPort(String.valueOf(request.getLocalPort()));
            InetAddress address = InetAddress.getByName(serverName);
            logEntity.setServerIp(address.getHostAddress());
            logEntity.setServerName(address.getHostName());
        } catch (Exception e) {
            log.error("获取登录服务器信息失败");
        }
    }

}
