package com.biz.crm.user.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.user.mapper.MdmUserRelRoleMapper;
import com.biz.crm.user.model.MdmUserRelRoleEntity;
import com.biz.crm.user.service.MdmUserRelRoleService;
import com.biz.crm.util.CollectionUtil;
import com.biz.crm.util.StringUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 用户与权限角色关联表接口实现
 *
 * @author Tao.Chen
 * @date 2021-04-19 14:55:59
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "MdmUserRelRoleServiceExpandImpl")
public class MdmUserRelRoleServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<MdmUserRelRoleMapper, MdmUserRelRoleEntity> implements MdmUserRelRoleService {

    @Resource
    private MdmUserRelRoleMapper mdmUserRelRoleMapper;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void add(String userName, String roleCode) {
        if (StringUtils.isNotEmpty(userName) && StringUtils.isNotEmpty(roleCode)) {
            this.add(userName, Collections.singletonList(roleCode));
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void add(String userName, List<String> roleCodeList) {
        if (StringUtils.isNotEmpty(userName) && CollectionUtil.listNotEmptyNotSizeZero(roleCodeList)) {
            Set<String> existCodeSet = this.lambdaQuery()
                    .eq(MdmUserRelRoleEntity::getUserName, userName)
                    .in(MdmUserRelRoleEntity::getRoleCode, roleCodeList)
                    .select(MdmUserRelRoleEntity::getRoleCode)
                    .list()
                    .stream().map(MdmUserRelRoleEntity::getRoleCode).collect(Collectors.toSet());
            Set<String> roleCodeSet = new HashSet<>(roleCodeList);
            List<MdmUserRelRoleEntity> entityList = new ArrayList<>();
            for (String roleCode :
                    roleCodeSet) {
                if (existCodeSet.contains(roleCode)) {
                    continue;
                }
                MdmUserRelRoleEntity entity = new MdmUserRelRoleEntity();
                entity.setUserName(userName);
                entity.setRoleCode(roleCode);
                entityList.add(entity);
            }
            if (entityList.size() > 0) {
                this.saveBatch(entityList);
            }
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void add(List<String> userNameList, String roleCode) {
        if (CollectionUtil.listNotEmptyNotSizeZero(userNameList) && StringUtils.isNotEmpty(roleCode)) {
            Set<String> existCodeSet = this.lambdaQuery()
                    .eq(MdmUserRelRoleEntity::getRoleCode, roleCode)
                    .in(MdmUserRelRoleEntity::getUserName, userNameList)
                    .select(MdmUserRelRoleEntity::getUserName)
                    .list()
                    .stream().map(MdmUserRelRoleEntity::getUserName).collect(Collectors.toSet());
            Set<String> userNameSet = new HashSet<>(userNameList);
            List<MdmUserRelRoleEntity> entityList = new ArrayList<>();
            for (String userName :
                    userNameSet) {
                if (existCodeSet.contains(userName)) {
                    continue;
                }
                MdmUserRelRoleEntity entity = new MdmUserRelRoleEntity();
                entity.setUserName(userName);
                entity.setRoleCode(roleCode);
                entityList.add(entity);
            }
            if (entityList.size() > 0) {
                this.saveBatch(entityList);
            }
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void del(String userName, String roleCode) {
        if (StringUtils.isNotEmpty(userName) && StringUtils.isNotEmpty(roleCode)) {
            this.del(userName, Collections.singletonList(roleCode));
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void del(String userName, List<String> roleCodeList) {
        if (StringUtils.isNotEmpty(userName) && CollectionUtil.listNotEmptyNotSizeZero(roleCodeList)) {
            this.lambdaUpdate()
                    .eq(MdmUserRelRoleEntity::getUserName, userName)
                    .in(MdmUserRelRoleEntity::getRoleCode, roleCodeList)
                    .remove();
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void del(List<String> userNameList, String roleCode) {
        if (CollectionUtil.listNotEmptyNotSizeZero(userNameList) && StringUtils.isNotEmpty(roleCode)) {
            this.lambdaUpdate()
                    .eq(MdmUserRelRoleEntity::getRoleCode, roleCode)
                    .in(MdmUserRelRoleEntity::getUserName, userNameList)
                    .remove();
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void delByUserName(String userName) {
        if (StringUtils.isNotEmpty(userName)) {
            this.delByUserName(Collections.singletonList(userName));
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void delByUserName(List<String> userNameList) {
        if (CollectionUtil.listNotEmptyNotSizeZero(userNameList)) {
            this.lambdaUpdate()
                    .in(MdmUserRelRoleEntity::getUserName, userNameList)
                    .remove();
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void delByRoleCode(String roleCode) {
        if (StringUtils.isNotEmpty(roleCode)) {
            this.delByRoleCode(Collections.singletonList(roleCode));
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void delByRoleCode(List<String> roleCodeList) {
        if (CollectionUtil.listNotEmptyNotSizeZero(roleCodeList)) {
            this.lambdaUpdate()
                    .in(MdmUserRelRoleEntity::getRoleCode, roleCodeList)
                    .remove();
        }
    }

    @Override
    public List<String> findRoleCodeListByUserName(String userName) {
        if (StringUtils.isNotEmpty(userName)) {
            return this.lambdaQuery()
                    .eq(MdmUserRelRoleEntity::getUserName, userName)
                    .select(MdmUserRelRoleEntity::getRoleCode)
                    .list()
                    .stream().map(MdmUserRelRoleEntity::getRoleCode).collect(Collectors.toList());
        }
        return new ArrayList<>();
    }

    @Override
    public List<String> findUserNameListByRoleCodeList(List<String> roleCodeList) {
        if (CollectionUtil.listNotEmptyNotSizeZero(roleCodeList)) {
           return this.lambdaQuery()
                    .in(MdmUserRelRoleEntity::getRoleCode, roleCodeList)
                    .select(MdmUserRelRoleEntity::getUserName)
                    .list()
                    .stream()
                    .map(MdmUserRelRoleEntity::getUserName)
                    .distinct()
                    .collect(Collectors.toList());
        }
        return new ArrayList<>();
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void replaceUserRelRole(List<String> ids, String roleCode) {
        if (CollectionUtil.listNotEmptyNotSizeZero(ids) && StringUtils.isNotEmpty(roleCode)) {
            List<MdmUserRelRoleEntity> list = this.lambdaQuery()
                    .in(MdmUserRelRoleEntity::getId, ids)
                    .list();
            if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
                List<String> userNameList = list.stream().map(MdmUserRelRoleEntity::getUserName).collect(Collectors.toList());
                this.removeByIds(ids);
                this.add(userNameList, roleCode);
            }
        }
    }

}
