package com.biz.crm.user.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.base.BusinessException;
import com.biz.crm.base.config.ThreadLocalUtil;
import com.biz.crm.common.GlobalParam;
import com.biz.crm.common.PageResult;
import com.biz.crm.common.param.ParameterParam;
import com.biz.crm.crmlog.handle.util.CrmLogSendUtil;
import com.biz.crm.customer.service.MdmCustomerMsgService;
import com.biz.crm.eunm.CodeRuleEnum;
import com.biz.crm.eunm.CrmDelFlagEnum;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.nebular.mdm.constant.PositionOperationEnum;
import com.biz.crm.nebular.mdm.constant.UserTypeEnum;
import com.biz.crm.nebular.mdm.customer.MdmCustomerMsgRespVo;
import com.biz.crm.nebular.mdm.org.resp.MdmOrgRespVo;
import com.biz.crm.nebular.mdm.position.req.MdmPositionCurrentUserReqVo;
import com.biz.crm.nebular.mdm.position.req.MdmPositionReqVo;
import com.biz.crm.nebular.mdm.position.req.MdmPositionUserOrgReqVo;
import com.biz.crm.nebular.mdm.position.resp.MdmPositionPageRespVo;
import com.biz.crm.nebular.mdm.position.resp.MdmPositionRespVo;
import com.biz.crm.nebular.mdm.position.resp.MdmPositionSelectRespVo;
import com.biz.crm.nebular.mdm.position.resp.MdmPositionUserOrgRespVo;
import com.biz.crm.nebular.mdm.role.resp.MdmRoleRespVo;
import com.biz.crm.nebular.mdm.user.req.*;
import com.biz.crm.nebular.mdm.user.resp.MdmUserPageRespVo;
import com.biz.crm.nebular.mdm.user.resp.MdmUserRespVo;
import com.biz.crm.nebular.mdm.user.resp.MdmUserSelectRespVo;
import com.biz.crm.org.service.MdmOrgService;
import com.biz.crm.position.service.MdmPositionRoleService;
import com.biz.crm.position.service.MdmPositionService;
import com.biz.crm.position.service.MdmPositionUserService;
import com.biz.crm.user.mapper.MdmUserMapper;
import com.biz.crm.user.model.MdmUserEntity;
import com.biz.crm.user.service.MdmUserRelRoleService;
import com.biz.crm.user.service.MdmUserRelWeChatService;
import com.biz.crm.user.service.MdmUserService;
import com.biz.crm.util.*;
import com.biz.crm.utils.LoginAesUtil;
import com.biz.crm.utils.LoginEncryptUtil;
import com.biz.crm.utils.LoginHelpUtil;
import com.biz.crm.utils.VerificationUtil;
import jodd.util.StringUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import javax.annotation.Resource;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 用户信息表接口实现
 *
 * @author Tao.Chen
 * @date 2020-11-17 19:04:29
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "MdmUserServiceExpandImpl")
public class MdmUserServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<MdmUserMapper, MdmUserEntity> implements MdmUserService {

    @Resource
    private MdmUserMapper mdmUserMapper;
    @Resource
    private MdmPositionUserService mdmPositionUserService;
    @Resource
    // TODO 和岗位模块存在循环依赖，岗位若处于用户的上层，则用户不能直接调用岗位接口
    private MdmPositionService mdmPositionService;
    @Resource
    private MdmOrgService mdmOrgService;
    @Resource
    // TODO 用户模块和客户模块的关系应该拆解
    private MdmCustomerMsgService mdmCustomerMsgService;
    @Resource
    private MdmPositionRoleService mdmPositionRoleService;
    @Resource
    private MdmUserRelWeChatService mdmUserRelWeChatService;
    @Resource
    private MdmUserRelRoleService mdmUserRelRoleService;
    @Autowired
    private CrmLogSendUtil crmLogSendUtil;

    @Override
    public PageResult<MdmUserPageRespVo> findList(MdmUserPageReqVo reqVo) {
        reqVo.setUserType(UserTypeEnum.USER.getCode());
        Page<MdmUserPageRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
        if (StringUtils.isNotEmpty(reqVo.getCode())) {
            MdmOrgRespVo org = mdmOrgService.queryByOrgCode(reqVo.getCode());
            if (org != null) {
                reqVo.setOrgRuleCode(org.getRuleCode());
            }
        }
        List<MdmUserPageRespVo> list = mdmUserMapper.findList(page, reqVo);
        this.setRoleCodeName(list);
        this.setLockState(list);
        return PageResult.<MdmUserPageRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    @Override
    public MdmUserRespVo query(MdmUserReqVo reqVo) {
        return detail(reqVo.getId(), reqVo.getUserName());
    }

    @Override
    public MdmUserRespVo detail(String id, String userName) {
        if (StringUtils.isNotEmpty(id) || StringUtils.isNotEmpty(userName)) {
            MdmUserEntity one = this.lambdaQuery()
                    .eq(MdmUserEntity::getDelFlag, CrmDelFlagEnum.NORMAL.getCode())
                    .eq(!StringUtils.isEmpty(id), MdmUserEntity::getId, id)
                    .eq(!StringUtils.isEmpty(userName), MdmUserEntity::getUserName, userName)
                    .one();
            if (one != null) {
                MdmUserRespVo respVo = CrmBeanUtil.copy(one, MdmUserRespVo.class);
                respVo.setUserPassword("");
                respVo.setRelWeChatList(mdmUserRelWeChatService.findUserRelWeChatList(new MdmUserRelWeChatReqVo().setUserName(one.getUserName())));
                if (UserTypeEnum.USER.getCode().equals(respVo.getUserType())) {
                    respVo.setPositionList(mdmPositionService.getUserAllPositionList(one.getUserName()));
                } else if (UserTypeEnum.CUSTOMER.getCode().equals(respVo.getUserType())) {
                    List<MdmCustomerMsgRespVo> customerList = mdmCustomerMsgService.getCustomerListByUserName(one.getUserName());
                    if (CollectionUtil.listNotEmptyNotSizeZero(customerList)) {
                        if (YesNoEnum.yesNoEnum.N.getValue().equals(ParamUtil.getParameterValue(ParameterParam.USER_CAN_RELATION_MANY_CUSTOMER))) {
                            respVo.setCustomerCode(customerList.get(0).getCustomerCode());
                            respVo.setCustomerName(customerList.get(0).getCustomerName());
                        }
                    }
                    respVo.setCustomerList(customerList);
                    respVo.setRoleCodeList(mdmUserRelRoleService.findRoleCodeListByUserName(one.getUserName()));
                } else {

                }
                return respVo;
            }

        }
        return null;
    }

    @Override
    public MdmUserRespVo queryByUserName(String userName) {
        if (StringUtils.isNotEmpty(userName)) {
            MdmUserEntity one = this.lambdaQuery()
                    .eq(MdmUserEntity::getDelFlag, CrmDelFlagEnum.NORMAL.getCode())
                    .eq(MdmUserEntity::getUserName, userName)
                    .one();
            if (one != null) {
                MdmUserRespVo copy = CrmBeanUtil.copy(one, MdmUserRespVo.class);
                copy.setUserPassword(null);
                return copy;
            }
        }
        return null;
    }

    @Override
    public List<MdmUserRespVo> queryBatchByUserNameList(List<String> userNameList) {
        if (CollectionUtil.listNotEmptyNotSizeZero(userNameList)) {
            return this.lambdaQuery()
                    .eq(MdmUserEntity::getDelFlag, CrmDelFlagEnum.NORMAL.getCode())
                    .in(MdmUserEntity::getUserName, userNameList)
                    .list()
                    .stream().map(x -> CrmBeanUtil.copy(x, MdmUserRespVo.class)).collect(Collectors.toList());
        }
        return new ArrayList<>();
    }

    @Override
    public MdmUserRespVo queryByUserCode(String userCode) {
        if (StringUtils.isNotEmpty(userCode)) {
            MdmUserEntity one = this.lambdaQuery()
                    .eq(MdmUserEntity::getDelFlag, CrmDelFlagEnum.NORMAL.getCode())
                    .eq(MdmUserEntity::getUserCode, userCode)
                    .one();
            if (one != null) {
                MdmUserRespVo copy = CrmBeanUtil.copy(one, MdmUserRespVo.class);
                copy.setUserPassword(null);
                return copy;
            }
        }
        return null;
    }

    @Override
    public String getFullNameByUserName(String userName) {
        if (StringUtils.isNotEmpty(userName)) {
            MdmUserEntity one = this.lambdaQuery()
                    .eq(MdmUserEntity::getDelFlag, CrmDelFlagEnum.NORMAL.getCode())
                    .eq(MdmUserEntity::getUserName, userName)
                    .select(MdmUserEntity::getFullName)
                    .one();
            if (one != null) {
                return one.getFullName();
            }
        }
        return null;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void save(MdmUserEditReqVo reqVo) {
        reqVo.setId(null);
        reqVo.setUserType(UserTypeEnum.USER.getCode());
        this.verifyData(reqVo);
        MdmUserEntity entity = CrmBeanUtil.copy(reqVo, MdmUserEntity.class);
        if (StringUtils.isEmpty(entity.getUserCode())) {
            entity.setUserCode(CodeUtil.generateCode(CodeRuleEnum.MDM_USER_CODE.getCode()));
        }
        entity.setLockState(CrmEnableStatusEnum.ENABLE.getCode());
        entity.setForceChangePassword(YesNoEnum.yesNoEnum.ONE.getValue());
        entity.setDelFlag(CrmDelFlagEnum.NORMAL.getCode());
        entity.setUpdatePasswordDate(DateUtil.getYYYYMMDD());
        this.save(entity);
        updateUserPositionRelation(reqVo, entity);

        Object menuCodeObj = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        if (menuCodeObj != null) {
            MdmUserRespVo newObject = detail(entity.getId(), null);
            crmLogSendUtil.sendForAdd(menuCodeObj.toString(), entity.getId(), entity.getUserName(), newObject);
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(MdmUserEditReqVo reqVo) {
        Assert.hasText(reqVo.getId(), "缺失id");
        this.verifyData(reqVo);
        MdmUserRespVo oldObject = detail(reqVo.getId(), null);
        MdmUserEntity entity = this.getById(reqVo.getId());

        // 如果密码发生变化 也要更新字段 updatePasswordDate
        if(!StringUtil.equals(reqVo.getUserPassword(),entity.getUserPassword())){
            entity.setUpdatePasswordDate(DateUtil.getYYYYMMDD());
        }
        CrmBeanUtil.copyProperties(reqVo, entity);
        this.updateById(entity);
        updateUserPositionRelation(reqVo, entity);
        Object menuCodeObj = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        if (menuCodeObj != null) {
            MdmUserRespVo newObject = detail(reqVo.getId(), null);
            crmLogSendUtil.sendForUpdate(menuCodeObj.toString(),
                    entity.getId(), entity.getUserName(), oldObject, newObject);
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteBatch(MdmUserReqVo reqVo) {
        this.deleteBatch(reqVo.getIds());
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteBatch(List<String> ids) {
        Assert.notEmpty(ids, "空参数");
        List<MdmUserEntity> list = this.lambdaQuery()
                .eq(MdmUserEntity::getDelFlag, CrmDelFlagEnum.NORMAL.getCode())
                .in(MdmUserEntity::getId, ids)
                .list();
        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            list.forEach(item -> {
                mdmPositionUserService.removeByUserName(item.getUserName());
                Object menuCodeObj = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
                if (menuCodeObj != null) {
                    crmLogSendUtil.sendForDel(menuCodeObj.toString(),
                            item.getId(), item.getUserName(), item);
                }
            });
            this.lambdaUpdate()
                    .in(MdmUserEntity::getId, ids)
                    .set(MdmUserEntity::getDelFlag, CrmDelFlagEnum.DELETE.getCode())
                    .update();
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void enableBatch(MdmUserReqVo reqVo) {
        this.enableBatch(reqVo.getIds());
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void enableBatch(List<String> ids) {
        Assert.notEmpty(ids, "ids不能为空");
//        this.lambdaUpdate()
//                .in(MdmUserEntity::getId, ids)
//                .set(MdmUserEntity::getEnableStatus, CrmEnableStatusEnum.ENABLE.getCode())
//                .update();

        List<MdmUserEntity> list = this.lambdaQuery()
                .eq(MdmUserEntity::getDelFlag, CrmDelFlagEnum.NORMAL.getCode())
                .in(MdmUserEntity::getId, ids)
                .select(MdmUserEntity::getId, MdmUserEntity::getUserName, MdmUserEntity::getEnableStatus)
                .list()
                .stream().filter(x -> !CrmEnableStatusEnum.ENABLE.getCode().equals(x.getEnableStatus()))
                .collect(Collectors.toList());
        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            Object menuCodeObj = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
            if (menuCodeObj != null) {
                list.forEach(item -> {
                    MdmUserRespVo oldObject = CrmBeanUtil.copy(item, MdmUserRespVo.class);
                    MdmUserRespVo newObject = CrmBeanUtil.copy(item, MdmUserRespVo.class);
                    newObject.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
                    crmLogSendUtil.sendForUpdate(menuCodeObj.toString(),
                            item.getId(), item.getUserName(), oldObject, newObject);
                });
            }

            this.lambdaUpdate()
                    .in(MdmUserEntity::getId, list.stream().map(MdmUserEntity::getId).collect(Collectors.toSet()))
                    .set(MdmUserEntity::getEnableStatus, CrmEnableStatusEnum.ENABLE.getCode())
                    .update();
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void disableBatch(MdmUserReqVo reqVo) {
        this.disableBatch(reqVo.getIds());
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void disableBatch(List<String> ids) {
        Assert.notEmpty(ids, "ids不能为空");
//        List<String> usernameList = this.lambdaQuery()
//                .in(MdmUserEntity::getId, ids)
//                .select(MdmUserEntity::getUserName)
//                .list()
//                .stream()
//                .filter(x -> x != null && !StringUtils.isEmpty(x.getUserName()))
//                .map(MdmUserEntity::getUserName)
//                .collect(Collectors.toList());
//        this.lambdaUpdate()
//                .in(MdmUserEntity::getId, ids)
//                .set(MdmUserEntity::getEnableStatus, CrmEnableStatusEnum.DISABLE.getCode())
//                .update();
//        UserUtils.deleteUser(usernameList);

        List<MdmUserEntity> list = this.lambdaQuery()
                .eq(MdmUserEntity::getDelFlag, CrmDelFlagEnum.NORMAL.getCode())
                .in(MdmUserEntity::getId, ids)
                .select(MdmUserEntity::getId, MdmUserEntity::getUserName, MdmUserEntity::getEnableStatus)
                .list()
                .stream().filter(x -> !CrmEnableStatusEnum.DISABLE.getCode().equals(x.getEnableStatus()))
                .collect(Collectors.toList());
        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            Object menuCodeObj = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
            if (menuCodeObj != null) {
                list.forEach(item -> {
                    MdmUserRespVo oldObject = CrmBeanUtil.copy(item, MdmUserRespVo.class);
                    MdmUserRespVo newObject = CrmBeanUtil.copy(item, MdmUserRespVo.class);
                    newObject.setEnableStatus(CrmEnableStatusEnum.DISABLE.getCode());
                    crmLogSendUtil.sendForUpdate(menuCodeObj.toString(),
                            item.getId(), item.getUserName(), oldObject, newObject);
                });
            }
            this.lambdaUpdate()
                    .in(MdmUserEntity::getId, list.stream().map(MdmUserEntity::getId).collect(Collectors.toSet()))
                    .set(MdmUserEntity::getEnableStatus, CrmEnableStatusEnum.DISABLE.getCode())
                    .update();
            UserUtils.deleteUser(list.stream().map(MdmUserEntity::getUserName).collect(Collectors.toList()));
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void changePassword(MdmUserChangePasswordReqVo reqVo) {
        Assert.hasText(reqVo.getUserName(), "缺失用户名");

        if (StringUtils.isNotEmpty(reqVo.getOofsksvclajyjcsp())) {
            reqVo.setOldPassword(LoginAesUtil.decrypt(reqVo.getOofsksvclajyjcsp()));
        } else if (StringUtils.isNotEmpty(reqVo.getOljxtwkylcygecop())) {
            reqVo.setOldPassword(LoginEncryptUtil.decrypt(reqVo.getOljxtwkylcygecop()));
        }
        if (StringUtils.isNotEmpty(reqVo.getPthxivgijazekrss())) {
            reqVo.setNewPassword(LoginAesUtil.decrypt(reqVo.getPthxivgijazekrss()));
        } else if (StringUtils.isNotEmpty(reqVo.getPazjofxkyvyoosot())) {
            reqVo.setNewPassword(LoginEncryptUtil.decrypt(reqVo.getPazjofxkyvyoosot()));
        }

        // 校验修改密码强度
        if(StringUtils.isNotEmpty(reqVo.getNewPassword())){
            VerificationUtil.isStrongPwd(reqVo.getNewPassword());
        }
        Assert.hasText(reqVo.getOldPassword(), "缺失原密码");
        Assert.hasText(reqVo.getNewPassword(), "缺失新密码");
        MdmUserEntity user = this.lambdaQuery()
                .eq(MdmUserEntity::getDelFlag, CrmDelFlagEnum.NORMAL.getCode())
                .eq(MdmUserEntity::getUserName, reqVo.getUserName())
                .one();
        Assert.notNull(user, "用户不存在");
        String oldPasswordEncryption = Md5EncryptionAndDecryption.encryPwd(reqVo.getOldPassword());
        String newPasswordEncryption = Md5EncryptionAndDecryption.encryPwd(reqVo.getNewPassword());
        Assert.isTrue(user.getUserPassword().equals(oldPasswordEncryption), "原密码输入错误");
        Assert.isTrue(!oldPasswordEncryption.equals(newPasswordEncryption), "新密码不能与原密码相同");
        user.setUserPassword(newPasswordEncryption);
        user.setUpdatePasswordDate(DateUtil.getYYYYMMDD());
        user.setForceChangePassword(YesNoEnum.yesNoEnum.ZERO.getValue());
        this.updateById(user);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void resetMyselfPassword(MdmUserChangePasswordReqVo reqVo) {
        if (StringUtils.isNotEmpty(reqVo.getPthxivgijazekrss())) {
            reqVo.setNewPassword(LoginAesUtil.decrypt(reqVo.getPthxivgijazekrss()));
        } else if (StringUtils.isNotEmpty(reqVo.getPazjofxkyvyoosot())) {
            reqVo.setNewPassword(LoginEncryptUtil.decrypt(reqVo.getPazjofxkyvyoosot()));
        } else {

        }
        Assert.hasText(reqVo.getNewPassword(), "缺失密码");

        // 校验修改密码强度
        if(StringUtils.isNotEmpty(reqVo.getNewPassword())){
            VerificationUtil.isStrongPwd(reqVo.getNewPassword());
        }

        UserRedis userRedis = UserUtils.getUser();
        Assert.isTrue(userRedis != null && StringUtils.isNotEmpty(userRedis.getUsername()), "请先登录");
        MdmUserEntity user = this.lambdaQuery()
                .eq(MdmUserEntity::getDelFlag, CrmDelFlagEnum.NORMAL.getCode())
                .eq(MdmUserEntity::getUserName, userRedis.getUsername())
                .one();
        Assert.notNull(user, "未找到当前用户");
//        Assert.isTrue(YesNoEnum.yesNoEnum.ONE.getValue().equals(user.getForceChangePassword()), "当前用户不可重设密码");
        user.setUserPassword(Md5EncryptionAndDecryption.encryPwd(reqVo.getNewPassword()));
        user.setForceChangePassword(YesNoEnum.yesNoEnum.ZERO.getValue());
        user.setUpdatePasswordDate(DateUtil.getYYYYMMDD());
        this.updateById(user);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void unlock(List<String> ids) {
        Assert.notNull(ids, "空参数");
        List<MdmUserEntity> list = this.lambdaQuery()
                .eq(MdmUserEntity::getDelFlag, CrmDelFlagEnum.NORMAL.getCode())
                .in(MdmUserEntity::getId, ids)
                .select(MdmUserEntity::getId, MdmUserEntity::getUserName, MdmUserEntity::getUserPhone, MdmUserEntity::getEmail, MdmUserEntity::getLockState)
                .list();
        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            list.forEach(item -> {
                LoginHelpUtil.unlock(item.getUserName(), item.getUserPhone(), item.getEmail());
            });
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void unlock(MdmUserReqVo reqVo) {
        this.unlock(reqVo.getIds());
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void unbindWeChat(List<String> ids) {
        Assert.notEmpty(ids, "空参数");
        List<MdmUserEntity> list = this.lambdaQuery()
                .eq(MdmUserEntity::getDelFlag, CrmDelFlagEnum.NORMAL.getCode())
                .in(MdmUserEntity::getId, ids)
                .select(MdmUserEntity::getUserName)
                .list();
        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            mdmUserRelWeChatService.unbindByUserName(list.stream().map(MdmUserEntity::getUserName).collect(Collectors.toList()));
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void forceChangePassword(MdmUserForceChangePasswordReqVo reqVo) {
        Assert.notNull(reqVo.getIds(), "缺失id");
        if (StringUtils.isNotEmpty(reqVo.getPthxivgijazekrss())) {
            reqVo.setPassword(LoginAesUtil.decrypt(reqVo.getPthxivgijazekrss()));
        } else if (StringUtils.isNotEmpty(reqVo.getPazjofxkyvyoosot())) {
            reqVo.setPassword(LoginEncryptUtil.decrypt(reqVo.getPazjofxkyvyoosot()));
        } else {

        }
        Assert.hasText(reqVo.getPassword(), "缺失密码");
        List<MdmUserEntity> list = this.lambdaQuery()
                .eq(MdmUserEntity::getDelFlag, CrmDelFlagEnum.NORMAL.getCode())
                .in(MdmUserEntity::getId, reqVo.getIds())
                .select(MdmUserEntity::getId)
                .list();
        Assert.notEmpty(list, "无效的id");
        Set<String> idSet = list.stream().map(MdmUserEntity::getId).collect(Collectors.toSet());
        reqVo.getIds().forEach(id -> {
            Assert.isTrue(idSet.contains(id), "无效的id");
        });
        String passwordEncryption = Md5EncryptionAndDecryption.encryPwd(reqVo.getPassword());
        this.lambdaUpdate()
                .in(MdmUserEntity::getId, reqVo.getIds())
                .set(MdmUserEntity::getUserPassword, passwordEncryption)
                .set(MdmUserEntity::getForceChangePassword, YesNoEnum.yesNoEnum.ONE.getValue())
                .update();
        List<String> usernameList = this.lambdaQuery()
                .eq(MdmUserEntity::getDelFlag, CrmDelFlagEnum.NORMAL.getCode())
                .in(MdmUserEntity::getId, reqVo.getIds())
                .select(MdmUserEntity::getUserName)
                .list()
                .stream()
                .filter(x -> x != null && !StringUtils.isEmpty(x.getUserName()))
                .map(MdmUserEntity::getUserName)
                .collect(Collectors.toList());
        UserUtils.deleteUser(usernameList);
    }

    /**
     * 校验新增编辑参数
     *
     * @param reqVo
     */
    public void verifyData(MdmUserEditReqVo reqVo) {
        String userName = reqVo.getUserName();
        Assert.hasText(userName, "用户帐号不能为空");
        Assert.hasText(reqVo.getFullName(), "用户姓名不能为空");
        MdmUserEntity one = this.lambdaQuery()
                .eq(MdmUserEntity::getUserName, userName)
                .one();
        if (StringUtils.isNotEmpty(reqVo.getUserPhone())) {
            List<MdmUserEntity> list = this.lambdaQuery()
                    .eq(MdmUserEntity::getUserPhone, reqVo.getUserPhone())
                    .list()
                    .stream().filter(x -> StringUtils.isEmpty(reqVo.getId()) || !reqVo.getId().equals(x.getId()))
                    .collect(Collectors.toList());
            Assert.isTrue(CollectionUtil.listEmpty(list), "手机号码已经被占用");
        }
        if (StringUtils.isNotEmpty(reqVo.getEmail())) {
            List<MdmUserEntity> list = this.lambdaQuery()
                    .eq(MdmUserEntity::getEmail, reqVo.getEmail())
                    .list()
                    .stream().filter(x -> StringUtils.isEmpty(reqVo.getId()) || !reqVo.getId().equals(x.getId()))
                    .collect(Collectors.toList());
            Assert.isTrue(CollectionUtil.listEmpty(list), "邮箱已经被占用");
        }
        if (StringUtils.isNotEmpty(reqVo.getIdentityCardNumber())) {
            List<MdmUserEntity> list = this.lambdaQuery()
                    .eq(MdmUserEntity::getIdentityCardNumber, reqVo.getIdentityCardNumber())
                    .list()
                    .stream().filter(x -> StringUtils.isEmpty(reqVo.getId()) || !reqVo.getId().equals(x.getId()))
                    .collect(Collectors.toList());
            Assert.isTrue(CollectionUtil.listEmpty(list), "身份证号码已经存在");
        }
        if (StringUtils.isEmpty(reqVo.getId())) {
            //新增
            if (one != null) {
                String delMark = "";
                if (CrmDelFlagEnum.DELETE.getCode().equals(one.getDelFlag())) {
                    delMark = "（已逻辑删除）";
                }
                throw new BusinessException("帐号[" + userName + "]已存在" + delMark);
            }
            reqVo.setUserPassword(Md5EncryptionAndDecryption.encryPwd(reqVo.getUserPassword()));
        } else {
            //编辑
            MdmUserEntity byId = this.getById(reqVo.getId());
            Assert.notNull(byId, "无效的id");
            Assert.isTrue(byId.getUserName().equals(userName), "用户账号不能修改");
            if (StringUtils.isNotEmpty(reqVo.getUserPassword())) {
                reqVo.setUserPassword(Md5EncryptionAndDecryption.encryPwd(reqVo.getUserPassword()));
            } else {
                reqVo.setUserPassword(byId.getUserPassword());
            }
        }

        UserTypeEnum userTypeEnum = UserTypeEnum.getEnumByCode(reqVo.getUserType());
        Assert.notNull(userTypeEnum, "用户类型无效");
        switch (userTypeEnum) {
            case USER:
                //企业用户
                List<MdmUserPositionEditReqVo> positionReqList = reqVo.getPositionList();
//                Assert.notEmpty(positionReqList, "职位不能为空");

                if (CollectionUtil.listNotEmptyNotSizeZero(positionReqList)) {
                    int count = 0;
                    for (MdmUserPositionEditReqVo item :
                            positionReqList) {
                        item.setId(null);
                        Assert.hasText(item.getOrgCode(), "组织不能为空");
                        Assert.hasText(item.getOperationType(), "操作类型不能为空");
                        Assert.notNull(item.getPrimaryFlag(), "是否主职位不能为空");
                        PositionOperationEnum positionOperationEnum = PositionOperationEnum.getEnumByCode(item.getOperationType());
                        Assert.notNull(positionOperationEnum, "无效的操作类型");
                        if (YesNoEnum.yesNoEnum.ONE.getValue().equals(item.getPrimaryFlag())) {
                            count++;
                            if (count > 1) {
                                throw new BusinessException("只能有一个主职位");
                            }
                        }
                        switch (positionOperationEnum) {
                            case NEW:
                                Assert.hasText(item.getPositionLevelCode(), "职位级别不能为空");
                                item.setPositionCode("");
                                ;
                                break;
                            case HISTORY:
                                Assert.hasText(item.getPositionCode(), "职位不能为空");
                                item.setPositionLevelCode("");
                                if (StringUtils.isNotEmpty(item.getParentCode()) && item.getPositionCode().equals(item.getParentCode())) {
                                    throw new BusinessException("上级职位不能是自己");
                                }
                                ;
                                break;
                            default:
                                ;
                        }
                    }
                }
                break;
            case CUSTOMER:
                if (YesNoEnum.yesNoEnum.N.getValue().equals(ParamUtil.getParameterValue(ParameterParam.USER_CAN_RELATION_MANY_CUSTOMER))) {
                    //客户用户（一对一）
                    Assert.hasText(reqVo.getCustomerCode(), "客户编码不能为空");
                } else if (YesNoEnum.yesNoEnum.Y.getValue().equals(ParamUtil.getParameterValue(ParameterParam.USER_CAN_RELATION_MANY_CUSTOMER))) {
                    //客户用户（一对多）
                    List<MdmUserCustomerEditReqVo> customerList = reqVo.getCustomerList();
                    Assert.notEmpty(customerList, "至少关联一个客户");
                    Map<String, MdmUserCustomerEditReqVo> customerUniqueMap = new HashMap<>(16);
                    for (MdmUserCustomerEditReqVo item :
                            customerList) {
                        Assert.hasText(item.getCustomerCode(), "客户不能为空");
                        if (customerUniqueMap.containsKey(item.getCustomerCode())) {
                            throw new BusinessException("不能重复关联客户相同客户");
                        }
                        customerUniqueMap.put(item.getCustomerCode(), item);
                    }
                } else {
                    throw new BusinessException("全局参数异常，请联系管理员");
                }

                break;
            default:
                ;
        }
    }

    /**
     * 保存/更新用户关联关系
     *
     * @param reqVo
     * @param userEntity
     */
    @Deprecated
    public void saveOrUpdateUserRelation(MdmUserEditReqVo reqVo, MdmUserEntity userEntity) {
        updateUserPositionRelation(reqVo, userEntity);
    }

    /**
     * 更新用户-职位关联关系
     *
     * @param reqVo privilege
     */
    public void updateUserPositionRelation(MdmUserEditReqVo reqVo, MdmUserEntity userEntity) {

        Map<String, String> positionCodeMap = new HashMap<>(16);
        if (CollectionUtil.listNotEmptyNotSizeZero(reqVo.getPositionList())) {
            for (MdmUserPositionEditReqVo item :
                    reqVo.getPositionList()) {
                //获取组织
                MdmOrgRespVo org = mdmOrgService.queryByOrgCode(item.getOrgCode());
                Assert.notNull(org, "组织不存在");
                if (PositionOperationEnum.NEW.getCode().equals(item.getOperationType())) {
                    //新增
                    MdmPositionReqVo copy = CrmBeanUtil.copy(item, MdmPositionReqVo.class);
                    copy.setRoleCodeList(item.getRoleList());
                    copy.setBpmRoleCodeList(item.getBpmRoleList());
                    copy.setPositionName(null);
                    String positionCode = mdmPositionService.save(copy);
                    positionCodeMap.put(positionCode, item.getPrimaryFlag());
                } else if (PositionOperationEnum.HISTORY.getCode().equals(item.getOperationType())) {
                    //已有职位
                    MdmPositionRespVo mdmPositionRespVo = mdmPositionService.queryByPositionCode(item.getPositionCode());
                    Assert.notNull(mdmPositionRespVo, "职位不存在");
                    MdmPositionReqVo copy = CrmBeanUtil.copy(mdmPositionRespVo, MdmPositionReqVo.class);
                    copy.setParentCode(item.getParentCode());
                    copy.setRoleCodeList(item.getRoleList());
                    copy.setBpmRoleCodeList(item.getBpmRoleList());
                    mdmPositionService.update(copy);
                    positionCodeMap.put(mdmPositionRespVo.getPositionCode(), item.getPrimaryFlag());
                } else {
                    throw new BusinessException("操作类型不存在");
                }
            }
            List<String> primaryPositionCodeList = positionCodeMap.entrySet().stream().filter(entry -> YesNoEnum.yesNoEnum.ONE.getValue().equals(entry.getValue())).map(Map.Entry::getKey).collect(Collectors.toList());
            Assert.isTrue(primaryPositionCodeList.size() <= 1, "一个用户有且只有一个主职位");
            mdmPositionUserService.resetUserPosition(userEntity.getUserName(), CollectionUtil.listNotEmptyNotSizeZero(primaryPositionCodeList) ? primaryPositionCodeList.get(0) : null, positionCodeMap.isEmpty() ? new ArrayList<>() : new ArrayList<>(positionCodeMap.keySet()));
        } else {
            mdmPositionUserService.removeByUserName(userEntity.getUserName());
        }
    }

    @Override
    public MdmUserRespVo queryBasic(MdmUserReqVo mdmUserReqVo) {
        MdmUserEntity one = this.lambdaQuery()
                .eq(MdmUserEntity::getDelFlag, CrmDelFlagEnum.NORMAL.getCode())
                .eq(!StringUtils.isEmpty(mdmUserReqVo.getUserCode()), MdmUserEntity::getUserCode, mdmUserReqVo.getUserCode())
                .eq(!StringUtils.isEmpty(mdmUserReqVo.getUserName()), MdmUserEntity::getUserName, mdmUserReqVo.getUserName())
                .eq(!StringUtils.isEmpty(mdmUserReqVo.getId()), MdmUserEntity::getId, mdmUserReqVo.getId())
                .eq(!StringUtils.isEmpty(mdmUserReqVo.getUserType()), MdmUserEntity::getUserType, mdmUserReqVo.getUserType())
                .one();
        return CrmBeanUtil.copy(one, MdmUserRespVo.class);
    }

    @Override
    public PageResult<MdmUserRespVo> pageBasic(MdmUserReqVo reqVo) {
        Page<MdmUserRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
        List<MdmUserRespVo> list = mdmUserMapper.pageBasic(page, reqVo);
        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            list.forEach(item -> item.setUserPassword(null));
        }
        return PageResult.<MdmUserRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    @Override
    public List<MdmUserRespVo> listDetail(MdmUserReqVo mdmUserReqVo) {
        List<MdmUserEntity> list = this.lambdaQuery()
                .eq(MdmUserEntity::getDelFlag, CrmDelFlagEnum.NORMAL.getCode())
                .like(!StringUtils.isEmpty(mdmUserReqVo.getUserName()), MdmUserEntity::getUserName, mdmUserReqVo.getUserName())
                .like(!StringUtils.isEmpty(mdmUserReqVo.getUserCode()), MdmUserEntity::getUserCode, mdmUserReqVo.getUserCode())
                .like(!StringUtils.isEmpty(mdmUserReqVo.getFullName()), MdmUserEntity::getFullName, mdmUserReqVo.getFullName())
                .eq(!StringUtils.isEmpty(mdmUserReqVo.getEnableStatus()), MdmUserEntity::getEnableStatus, mdmUserReqVo.getEnableStatus())
                .eq(!StringUtils.isEmpty(mdmUserReqVo.getUserType()), MdmUserEntity::getUserType, mdmUserReqVo.getUserType())
                .list();
        return CrmBeanUtil.copyList(list, MdmUserRespVo.class);
    }

    @Override
    public MdmPositionUserOrgRespVo getCurUserPositionDetail() {
        UserRedis user = UserUtils.getUser();
        Assert.notNull(user, "未登录");
        if (UserTypeEnum.USER.getCode().equals(user.getUsertype())) {
            Assert.hasText(user.getPoscode(), "未登录");
            //企业用户
            List<MdmPositionUserOrgRespVo> positionUserOrgList = mdmPositionService.findPositionUserOrgList(new MdmPositionUserOrgReqVo().setPositionCode(user.getPoscode()));
            if (CollectionUtil.listNotEmptyNotSizeZero(positionUserOrgList)) {
                return positionUserOrgList.get(0);
            }
        } else if (UserTypeEnum.CUSTOMER.getCode().equals(user.getUsertype()) || UserTypeEnum.CUSTOMER_EMPLOYEE.getCode().equals(user.getUsertype())) {
            //客户用户或者外部用户 TODO 临时处理，可能需要更换返回对象或者接口
            Assert.hasText(user.getCustcode(), "未登录");
            MdmUserEntity one = this.lambdaQuery()
                    .eq(MdmUserEntity::getUserName, user.getUsername())
                    .one();
            if (one != null) {
                return CrmBeanUtil.copy(one, MdmPositionUserOrgRespVo.class);
            }
        }
        return null;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void changeUserHeadUrl(MdmUserChangeHeadReqVo reqVo) {
        Assert.hasText(reqVo.getUserName(), "缺失用户登录名称");
        MdmUserEntity userEntity = this.lambdaQuery()
                .eq(MdmUserEntity::getDelFlag, CrmDelFlagEnum.NORMAL.getCode())
                .eq(MdmUserEntity::getUserName, reqVo.getUserName())
                .one();
        Assert.notNull(userEntity, "无效的用户登录名");
        userEntity.setUserHeadUrl(reqVo.getUserHeadUrl());
        this.updateById(userEntity);
    }

    @Override
    public List<MdmPositionSelectRespVo> findUserPositionSelectList(MdmUserPositionSelectReqVo reqVo) {
        reqVo.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
        Integer pageSize = reqVo.getPageSize();
        if (pageSize == null || pageSize == 0) {
            pageSize = Integer.valueOf(ParamUtil.getParameterValue(ParameterParam.SELECT_BOX_OPTION_SIZE));
        }
        Set<String> selectedSet = new HashSet<>(16);
        if (CollectionUtil.listNotEmptyNotSizeZero(reqVo.getSelectedCodeList())) {
            selectedSet.addAll(reqVo.getSelectedCodeList());
        }
        if (StringUtils.isNotEmpty(reqVo.getSelectedCode())) {
            selectedSet.add(reqVo.getSelectedCode());
        }
        reqVo.setSelectedCodeList(new ArrayList<>(selectedSet));
        List<MdmPositionSelectRespVo> list = mdmUserMapper.findUserPositionSelectList(new Page<>(1L, pageSize, false), reqVo);
        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            list.forEach(item -> {
                item.setUnionName(generateUnionName(item.getOrgName(), item.getPositionName(), item.getFullName()));
            });
        }
        return list;
    }

    @Override
    public List<MdmPositionSelectRespVo> findUserParentPositionSelectList(MdmUserParentPositionSelectReqVo reqVo) {
        reqVo.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
        //排除组织的顶点，结果要排除他们的下级组织，不排除它们
        Set<String> excludeTopOrgCode = new HashSet<>(16);
        //排除组织顶点中要去掉的
        Set<String> excludeOrgCodeList = new HashSet<>(16);
        Set<String> includeOrgCodeSet = new HashSet<>(16);

        if (StringUtils.isNotEmpty(reqVo.getChildPositionOrgCode())) {
            excludeTopOrgCode.add(reqVo.getChildPositionOrgCode());
            includeOrgCodeSet.add(reqVo.getChildPositionOrgCode());
        }
        if (StringUtils.isNotEmpty(reqVo.getChildPositionCode())) {
            MdmPositionRespVo position = mdmPositionService.queryByPositionCode(reqVo.getChildPositionCode());
            if (position != null) {
                if (StringUtils.isNotEmpty(position.getOrgCode())) {
                    includeOrgCodeSet.add(position.getOrgCode());
                }
                List<MdmPositionRespVo> allChildrenPositionCodeList = mdmPositionService.findAllChildrenPositionList(null, position.getPositionCode());
                if (CollectionUtil.listNotEmptyNotSizeZero(allChildrenPositionCodeList)) {
                    excludeTopOrgCode.addAll(allChildrenPositionCodeList.stream().filter(item -> StringUtils.isNotEmpty(item.getOrgCode())).map(MdmPositionRespVo::getOrgCode).collect(Collectors.toSet()));
                }
            }
        }
        if (!excludeTopOrgCode.isEmpty()) {
            List<String> excludeChildrenOrgCodeList = mdmOrgService.findAllChildrenOrgCodeListByList(null, new ArrayList<>(excludeTopOrgCode));
            if (CollectionUtil.listNotEmptyNotSizeZero(excludeChildrenOrgCodeList)) {
                excludeOrgCodeList.addAll(excludeChildrenOrgCodeList.stream().filter(item -> !includeOrgCodeSet.contains(item)).collect(Collectors.toList()));
            }
        }

        Integer pageSize = reqVo.getPageSize();
        if (pageSize == null || pageSize == 0) {
            pageSize = Integer.valueOf(ParamUtil.getParameterValue(ParameterParam.SELECT_BOX_OPTION_SIZE));
        }
        Set<String> selectedSet = new HashSet<>(16);
        if (CollectionUtil.listNotEmptyNotSizeZero(reqVo.getSelectedCodeList())) {
            selectedSet.addAll(reqVo.getSelectedCodeList());
        }
        if (StringUtils.isNotEmpty(reqVo.getSelectedCode())) {
            selectedSet.add(reqVo.getSelectedCode());
        }
        reqVo.setSelectedCodeList(new ArrayList<>(selectedSet));
        List<MdmPositionSelectRespVo> list = mdmUserMapper.findUserParentPositionSelectList(new Page<>(1L, pageSize, false), reqVo, new ArrayList<>(excludeOrgCodeList));
        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            list.forEach(item -> {
                item.setUnionName(generateUnionName(item.getOrgName(), item.getPositionName(), item.getFullName()));
            });
        }
        return list;
    }

    @Override
    public List<MdmUserSelectRespVo> findUserSelectList(MdmUserSelectReqVo reqVo) {
        reqVo.setUserType(UserTypeEnum.USER.getCode());
        Integer pageSize = reqVo.getPageSize();
        if (pageSize == null || pageSize == 0) {
            pageSize = Integer.valueOf(ParamUtil.getParameterValue(ParameterParam.SELECT_BOX_OPTION_SIZE));
        }
        List<MdmUserSelectRespVo> list = new ArrayList<>();
        Set<String> codeList = new HashSet<>(16);
        if (StringUtils.isNotEmpty(reqVo.getSelectedCode())) {
            codeList.add(reqVo.getSelectedCode());
        }
        if (CollectionUtil.listNotEmptyNotSizeZero(reqVo.getSelectedCodeList())) {
            codeList.addAll(reqVo.getSelectedCodeList());
        }
        if (!codeList.isEmpty()) {
            reqVo.setSelectedCodeList(new ArrayList<>(codeList));
        }
        if (pageSize > 0) {
            list.addAll(mdmUserMapper.findUserSelectList(new Page<>(1L, pageSize, false), reqVo));
        }
        return list;
    }

    @Override
    public MdmUserRespVo getUserForLogin(String userName, String userPhone, String email) {
        MdmUserEntity user = null;
        if (StringUtils.isNotEmpty(userName)) {
            user = this.lambdaQuery()
                    .eq(MdmUserEntity::getDelFlag, CrmDelFlagEnum.NORMAL.getCode())
                    .eq(MdmUserEntity::getUserName, userName)
                    .one();
            Assert.notNull(user, "该用户不存在");
        } else if (StringUtils.isNotEmpty(userPhone)) {
            List<MdmUserEntity> list = this.lambdaQuery()
                    .eq(MdmUserEntity::getDelFlag, CrmDelFlagEnum.NORMAL.getCode())
                    .eq(MdmUserEntity::getUserPhone, userPhone)
                    .list();
            Assert.notEmpty(list, "该手机号未对应任何用户，请先行注册");
            Assert.isTrue(list.size() == 1, "该手机号关联了多个用户，请联系管理员");
            user = list.get(0);
        } else if (StringUtils.isNotEmpty(email)) {
            List<MdmUserEntity> list = this.lambdaQuery()
                    .eq(MdmUserEntity::getDelFlag, CrmDelFlagEnum.NORMAL.getCode())
                    .eq(MdmUserEntity::getEmail, email)
                    .list();
            Assert.notEmpty(list, "该邮箱未对应任何用户");
            Assert.isTrue(list.size() == 1, "该邮箱关联了多个用户，请联系管理员");
            user = list.get(0);
        } else {
            throw new BusinessException("请传入查询条件");
        }
        return CrmBeanUtil.copy(user, MdmUserRespVo.class);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void setUserForceChangePassword(String userName) {
        if (StringUtils.isNotEmpty(userName)) {
            this.lambdaUpdate()
                    .eq(MdmUserEntity::getUserName, userName)
                    .set(MdmUserEntity::getForceChangePassword, YesNoEnum.yesNoEnum.ONE.getValue())
                    .update();
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void editPassword(String userName, String password) {
        if (StringUtils.isNotEmpty(userName)) {
            this.lambdaUpdate()
                    .eq(MdmUserEntity::getUserName, userName)
                    .set(MdmUserEntity::getUserPassword, Md5EncryptionAndDecryption.encryPwd(password))
                    .update();
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateUserLastLoginDateTime(String userName, String loginDateTime) {
        if (StringUtils.isNotEmpty(userName) && StringUtils.isNotEmpty(loginDateTime)) {
            this.lambdaUpdate()
                    .eq(MdmUserEntity::getUserName, userName)
                    .set(MdmUserEntity::getLastLoginTime, loginDateTime)
                    .update();
        }
    }

    @Override
    public List<MdmPositionPageRespVo> findCustomerEmployeePage(MdmPositionCurrentUserReqVo reqVo) {
        Page<MdmPositionPageRespVo> page = PageUtil.buildPage(reqVo.getPageNum(), reqVo.getPageSize());
        return Optional.ofNullable(UserUtils.getUser())
                .map(UserRedis::getCustcode)
                .filter(x -> !StringUtils.isEmpty(x))
                .map(x -> mdmUserMapper.findCustomerEmployeePage(page, x))
                .orElse(Collections.emptyList());
    }

    /**
     * 生成 组织名称/职位名称/用户名称
     *
     * @param orgName      组织名称
     * @param positionName 职位名称
     * @param fullName     用户名称
     * @return
     */
    private String generateUnionName(String orgName, String positionName, String fullName) {
        return (orgName == null ? "" : orgName) + "/" + (positionName == null ? "" : positionName) + "/" + (fullName == null ? "" : fullName);
    }

    /**
     * 用户列表查询设置角色
     *
     * @param list
     */
    private void setRoleCodeName(List<MdmUserPageRespVo> list) {
        if (CollectionUtil.listEmpty(list)) {
            return;
        }
        List<String> collect = list.stream().filter(item -> StringUtils.isNotEmpty(item.getPositionCode())).map(MdmUserPageRespVo::getPositionCode).collect(Collectors.toList());
        if (CollectionUtil.listNotEmptyNotSizeZero(collect)) {
            final Map<String, List<MdmRoleRespVo>> map = mdmPositionRoleService.getPositionRoleGroupByPositionCodeList(collect);
            if (!map.isEmpty()) {
                list.forEach(item -> {
                    if (StringUtils.isNotEmpty(item.getPositionCode())) {
                        if (map.containsKey(item.getPositionCode())) {
                            List<MdmRoleRespVo> roleList = map.get(item.getPositionCode());
                            if (CollectionUtil.listNotEmptyNotSizeZero(roleList)) {
                                item.setRoleCode(roleList.stream().map(MdmRoleRespVo::getRoleCode).collect(Collectors.joining(",")));
                                item.setRoleName(roleList.stream().map(MdmRoleRespVo::getRoleName).collect(Collectors.joining(",")));
                            }
                        }
                    }
                });
            }
        }
    }

    /**
     * 用户列表查询设置锁定状态
     *
     * @param list
     */
    private void setLockState(List<MdmUserPageRespVo> list) {
        if (CollectionUtil.listEmpty(list)) {
            return;
        }
        final Set<String> lockUserNameSet = LoginHelpUtil.checkAndGetLockUserName(list.stream().map(MdmUserPageRespVo::getUserName).collect(Collectors.toSet()));
        list.forEach(item -> {
            item.setLockState(!lockUserNameSet.contains(item.getUserName()) ? CrmEnableStatusEnum.ENABLE.getCode() : CrmEnableStatusEnum.DISABLE.getCode());
        });
    }
}
