package com.biz.crm.utils;

import com.biz.crm.base.BusinessException;
import com.biz.crm.common.param.ParameterParam;
import com.biz.crm.common.param.RedisParam;
import com.biz.crm.dict.service.MdmDictDataService;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.eunm.mdm.LoginFromTypeEnum;
import com.biz.crm.message.ShortMessage;
import com.biz.crm.nebular.mdm.constant.DictConstant;
import com.biz.crm.nebular.mdm.constant.UserTypeEnum;
import com.biz.crm.nebular.mdm.dict.resp.DictDataVo;
import com.biz.crm.nebular.webservice.mail.CrmMailMessage;
import com.biz.crm.service.RedisService;
import com.biz.crm.util.DateUtil;
import com.biz.crm.util.ParamUtil;
import com.biz.crm.util.Result;
import com.biz.crm.webservice.mail.MailFeign;
import com.biz.crm.webservice.message.ShortMessageFeign;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.Assert;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.*;
import java.util.stream.Collectors;

@Slf4j
@Component
public class LoginHelpUtil {

    public enum AccountType {
        USER_NAME,
        PHONE,
        EMAIL,
        ;
    }

    public enum VerificationCodeType {
        LOGIN,
        LOGIN_AND_RESET,
        ;
    }

    private static RedisService redisService;
    private static ShortMessageFeign shortMessageFeign;
    private static MdmDictDataService mdmDictDataService;
    private static MailFeign mailFeign;

    @Autowired
    public void setRedisService(RedisService redis) {
        LoginHelpUtil.redisService = redis;
    }

    @Autowired
    public void setShortMessageFeign(ShortMessageFeign shortMessageFeign) {
        LoginHelpUtil.shortMessageFeign = shortMessageFeign;
    }

    @Autowired
    public void setMdmDictDataService(MdmDictDataService mdmDictDataService) {
        LoginHelpUtil.mdmDictDataService = mdmDictDataService;
    }

    @Autowired
    public void setMailFeign(MailFeign mailFeign) {
        LoginHelpUtil.mailFeign = mailFeign;
    }

    /**
     * 判断账号是否被锁定（锁定会直接抛异常）
     *
     * @param account         账号关键值
     * @param accountType 账号类型
     */
    public static void checkLock(String account, AccountType accountType) {
        if (StringUtils.isEmpty(account)) {
            return;
        }
        long userLockMinutes = LoginHelpUtil.getLockMinutes(account, accountType);
        if (userLockMinutes != 0L) {
            if (AccountType.USER_NAME == accountType) {
                throw new BusinessException("用户已经被锁定，请" + userLockMinutes + "分钟后再尝试");
            }
            if (AccountType.PHONE == accountType) {
                throw new BusinessException("手机号已经被锁定，请" + userLockMinutes + "分钟后再尝试");
            }
            if (AccountType.EMAIL == accountType) {
                throw new BusinessException("邮箱已经被锁定，请" + userLockMinutes + "分钟后再尝试");
            }
        }
    }


    public static void addErrorAccount(String userName, String phone, String email, String accountContext) {
        if (StringUtils.isEmpty(accountContext)) {
            accountContext = "账号";
        }
        long lastTimes = addErrorAndGetTimes2(userName, phone, email);
        if (lastTimes != 0L) {
            throw new BusinessException(accountContext + "，还有" + lastTimes + "次尝试机会");
        } else {
            throw new BusinessException(accountContext + "，账号被锁定，请" + LoginHelpUtil.getMaxLockMinutes() + "分钟后再尝试");
        }
    }

    /**
     * 添加密码错误次数并抛异常
     *
     * @param userName 登录账号（必填）
     * @param phone    手机号（如果登录账号对应用户有手机号则必填）
     * @param email    邮箱（如果登录账号对应用户有邮箱则必填）
     */
    public static void addError(String userName, String phone, String email) {
        Assert.hasText(userName, "缺失登录账号");
        long lastTimes = addErrorAndGetTimes(userName, phone, email);
        if (lastTimes != 0L) {
            throw new BusinessException("密码错误，还有" + lastTimes + "次尝试机会");
        } else {
            throw new BusinessException("密码错误，账号被锁定，请" + LoginHelpUtil.getMaxLockMinutes() + "分钟后再尝试");
        }
    }

    /**
     * 校验用户是否可用
     *
     * @param enableStatus
     * @param startTime
     * @param endTime
     */
    public static void checkUserLogin(String enableStatus, String startTime, String endTime) {
        Assert.isTrue(CrmEnableStatusEnum.ENABLE.getCode().equals(enableStatus), "用户未启用");
        String now = DateUtil.dateNowHms();
        if (StringUtils.isNotEmpty(startTime)) {
            Assert.isTrue(now.compareTo(startTime) >= 0, "用户未生效");
        }
        if (StringUtils.isNotEmpty(endTime)) {
            Assert.isTrue(now.compareTo(endTime) <= 0, "用户已失效");
        }
    }

    /**
     * 校验用户类型与登录入口是否匹配
     *
     * @param userType
     * @param fromType
     */
    public static void checkUserType(String userType, String fromType) {
        Assert.hasText(userType, "缺失用户类型");
        Assert.hasText(fromType, "缺失登录来源");
        if (UserTypeEnum.USER.getCode().equals(userType) && LoginFromTypeEnum.CONSOLE.getValue().equals(fromType)) {
            return;
        }
        Map<String, DictDataVo> userTypeMap = mdmDictDataService.getDictDataDetailMap(DictConstant.USER_TYPE);
        Assert.isTrue(userTypeMap.containsKey(userType), "未识别的用户类型");
        DictDataVo dictDataVo = userTypeMap.get(userType);
        String usableLoginFromType = dictDataVo.getExtendMap().get("usable_login_from_type");
        Assert.hasText(usableLoginFromType, "用户类型未配置可用登录入口");
        if (StringUtils.isNotEmpty(usableLoginFromType)) {
            Set<String> fromTypeSet = new HashSet<>(Arrays.asList(usableLoginFromType.split(",")));
            Assert.isTrue(fromTypeSet.contains(fromType), UserTypeEnum.getDescByCode(userType) + "不能登录" + LoginFromTypeEnum.getDesc(fromType));
        }
        if (LoginFromTypeEnum.CONSOLE.getValue().equals(fromType)) {
            Assert.isTrue(UserTypeEnum.USER.getCode().equals(userType), "只有企业用户才能登录当前系统");
        } else if (LoginFromTypeEnum.APP_SFA.getValue().equals(fromType)) {
            Assert.isTrue(UserTypeEnum.USER.getCode().equals(userType) || UserTypeEnum.CUSTOMER_EMPLOYEE.getCode().equals(userType), "该用户类型不能登录当前系统");
        } else if (LoginFromTypeEnum.APPLET_SFA.getValue().equals(fromType)) {
            Assert.isTrue(UserTypeEnum.USER.getCode().equals(userType) || UserTypeEnum.CUSTOMER_EMPLOYEE.getCode().equals(userType), "该用户类型不能登录当前系统");
        } else if (LoginFromTypeEnum.WEB_DMS.getValue().equals(fromType)) {
            Assert.isTrue(UserTypeEnum.CUSTOMER.getCode().equals(userType) || UserTypeEnum.CUSTOMER_EMPLOYEE.getCode().equals(userType), "该用户类型不能登录当前系统");
        } else if (LoginFromTypeEnum.APPLET_DMS.getValue().equals(fromType)) {
            Assert.isTrue(UserTypeEnum.CUSTOMER.getCode().equals(userType) || UserTypeEnum.CUSTOMER_EMPLOYEE.getCode().equals(userType), "该用户类型不能登录当前系统");
        } else if (LoginFromTypeEnum.APP_DMS.getValue().equals(fromType)) {
            Assert.isTrue(UserTypeEnum.CUSTOMER.getCode().equals(userType) || UserTypeEnum.CUSTOMER_EMPLOYEE.getCode().equals(userType), "该用户类型不能登录当前系统");
        } else if (LoginFromTypeEnum.OFFICIAL_ACCOUNTS_DMS.getValue().equals(fromType)) {
            Assert.isTrue(UserTypeEnum.CUSTOMER.getCode().equals(userType) || UserTypeEnum.CUSTOMER_EMPLOYEE.getCode().equals(userType), "该用户类型不能登录当前系统");
        } else {

        }
    }

    /**
     * 获取账号锁定时间（分钟）
     *
     * @param account
     * @param accountType
     * @return
     */
    public static long getLockMinutes(String account, AccountType accountType) {
        if (AccountType.USER_NAME == accountType) {
            return getLockMinutes(RedisParam.LOGIN_LOCK_USER, account);
        }
        if (AccountType.PHONE == accountType) {
            return getLockMinutes(RedisParam.LOGIN_LOCK_PHONE, account);
        }
        if (AccountType.EMAIL == accountType) {
            return getLockMinutes(RedisParam.LOGIN_LOCK_EMAIL, account);
        }
        return 0;
    }

    /**
     * 解锁
     *
     * @param userName 登录账号（必填）
     * @param phone    手机号（如果登录账号对应用户有手机号则必填）
     * @param email    邮箱（如果登录账号对应用户有邮箱则必填）
     */
    public static void unlock(String userName, String phone, String email) {
        if (StringUtils.isNotEmpty(userName)) {
            redisService.hdel(RedisParam.LOGIN_LOCK_USER, userName);
            redisService.hdel(RedisParam.LOGIN_LOCK_USER_ERROR_TIMES, userName);
        }
        if (StringUtils.isNotEmpty(phone)) {
            redisService.hdel(RedisParam.LOGIN_LOCK_PHONE, phone);
            redisService.hdel(RedisParam.LOGIN_LOCK_PHONE_ERROR_TIMES, phone);
        }
        if (StringUtils.isNotEmpty(email)) {
            redisService.hdel(RedisParam.LOGIN_LOCK_EMAIL, email);
            redisService.hdel(RedisParam.LOGIN_LOCK_EMAIL_ERROR_TIMES, email);
        }
    }

    /**
     * 获取用户最大锁定时长（分钟）
     *
     * @return
     */
    public static long getMaxLockMinutes() {
        long maxLockSeconds = getMaxLockSeconds();
        BigDecimal divide = new BigDecimal(maxLockSeconds).divide(new BigDecimal(60), 0, BigDecimal.ROUND_UP);
        return divide.longValue();
    }

    /**
     * 生成登录验证码并且缓存和手机号码、用户的对应关系
     *
     * @param fromType 登录入口
     * @param userName 登录账号
     * @param account  手机号或者邮箱
     * @return
     */
    public static String saveLoginVerificationCodeRelUser(String fromType, String userName, String account, AccountType accountType, VerificationCodeType verificationCodeType) {
        String verification = generateVerificationCode();
        if (AccountType.PHONE == accountType) {
            if (VerificationCodeType.LOGIN == verificationCodeType) {
                redisService.hset(RedisParam.LOGIN_PHONE_VERIFICATION + fromType + ":" + account, verification, userName, RedisParam.SECONDS_OF_FIVE_MINUTE);
            }
            if (VerificationCodeType.LOGIN_AND_RESET == verificationCodeType) {
                redisService.hset(RedisParam.LOGIN_PHONE_RESET_VERIFICATION + fromType + ":" + account, verification, userName, RedisParam.SECONDS_OF_FIVE_MINUTE);
            }
        }
        if (AccountType.EMAIL == accountType) {
            if (VerificationCodeType.LOGIN == verificationCodeType) {
                redisService.hset(RedisParam.LOGIN_EMAIL_VERIFICATION + fromType + ":" + account, verification, userName, RedisParam.SECONDS_OF_HOUR);
            }
            if (VerificationCodeType.LOGIN_AND_RESET == verificationCodeType) {
                redisService.hset(RedisParam.LOGIN_EMAIL_RESET_VERIFICATION + fromType + ":" + account, verification, userName, RedisParam.SECONDS_OF_HOUR);
            }
        }
        return verification;
    }

    /**
     * 发送验证码
     *
     * @param fromType             登录入口
     * @param verificationCode     验证码
     * @param account              账号关键值
     * @param accountType          账号类型
     * @param verificationCodeType 验证码用途
     */
    public static void sendVerificationCode(String fromType, String verificationCode, String account, AccountType accountType, VerificationCodeType verificationCodeType) {
        if (AccountType.PHONE == accountType) {
            ShortMessage shortMessage = new ShortMessage();
            shortMessage.setPhoneNumbers(account);
            shortMessage.setTemplateParam("{\"code\":\"" + verificationCode + "\"}");
            if (VerificationCodeType.LOGIN == verificationCodeType) {
                //发送手机号登录验证码
                String template = mdmDictDataService.getDictDataMap(DictConstant.MESSAGE_TEMPLATE).get(DictConstant.MESSAGE_TEMPLATE_SYSTEM_LOGIN);
                Assert.hasText(template, "未配置手机验证码登录短信模板");
                shortMessage.setTemplateCode(template);
            }
            if (VerificationCodeType.LOGIN_AND_RESET == verificationCodeType) {
                //发送手机号忘记密码验证码
                String template = mdmDictDataService.getDictDataMap(DictConstant.MESSAGE_TEMPLATE).get(DictConstant.MESSAGE_TEMPLATE_SYSTEM_LOGIN_AND_RESET);
                Assert.hasText(template, "未配置手机验证码重置密码短信模板");
                shortMessage.setTemplateCode(template);
            }
            shortMessageFeign.sendMessage(shortMessage);
        }
        if (AccountType.EMAIL == accountType) {
            String subject = "";
            StringBuilder sb = new StringBuilder();
            if (VerificationCodeType.LOGIN == verificationCodeType) {
                //发送邮箱登录验证码
                sb.append("<p>您正在尝试通过邮箱验证码登录</p>");
                subject = "登录验证码";
            }
            if (VerificationCodeType.LOGIN_AND_RESET == verificationCodeType) {
                //发送邮箱忘记密码验证码
                sb.append("<p>您正在尝试通过邮箱验证身份并重置您的密码</p>");
                subject = "重置密码验证码";
            }
            sb.append("<p>本次验证码为：<strong>").append(verificationCode).append("</strong></p>");
            sb.append("<p>验证码将在1小时后或者使用1次后自动失效</p>");
            sb.append("<p>请勿将验证码告诉他人</p>");
            sb.append("<p>如非您本人操作，请不予理会</p>");

            CrmMailMessage crmMailMessage = new CrmMailMessage(null, null, Collections.singleton(account), subject, sb.toString());
            Result result = mailFeign.sendMail(crmMailMessage);
            Assert.isTrue(result.isSuccess(), "发送邮件失败：" + result.getMessage());
        }
        log.debug("发送验证码成功：登录系统：{}，邮箱/手机号：{}，验证码：{}", LoginFromTypeEnum.getDesc(fromType), account, verificationCode);
    }

    /**
     * 校验用户是否被锁定，返回锁定的用户名
     *
     * @param userNameSet 用户名集合
     * @return
     */
    public static Set<String> checkAndGetLockUserName(Set<String> userNameSet) {
        Set<String> lockUserNameSet = new HashSet<>(16);
        if (!userNameSet.isEmpty()) {
            Map<?, ?> hmget = redisService.hmget(RedisParam.LOGIN_LOCK_USER);
            if (hmget != null) {
                lockUserNameSet.addAll(hmget.keySet().stream().map(x -> (String) x).filter(userNameSet::contains).collect(Collectors.toSet()));
            }
        }
        return lockUserNameSet;
    }

    /**
     * 校验验证码
     *
     * @param fromType                    登录入口
     * @param verificationCode            验证码
     * @param account                     账号关键值
     * @param accountType             账号类型
     * @param verificationCodeType 验证码用途
     * @return
     */
    public static String checkVerificationCode(String fromType, String verificationCode, String account, AccountType accountType, VerificationCodeType verificationCodeType) {
        String userName = null;
        if (AccountType.PHONE == accountType) {
            if (VerificationCodeType.LOGIN == verificationCodeType) {
                Object hget = redisService.hget(RedisParam.LOGIN_PHONE_VERIFICATION + fromType + ":" + account, verificationCode);
                if (hget != null) {
                    redisService.del(RedisParam.LOGIN_PHONE_VERIFICATION + fromType + ":" + account);
                    userName = (String) hget;
                }
            }
            if (VerificationCodeType.LOGIN_AND_RESET == verificationCodeType) {
                Object hget = redisService.hget(RedisParam.LOGIN_PHONE_RESET_VERIFICATION + fromType + ":" + account, verificationCode);
                if (hget != null) {
                    redisService.del(RedisParam.LOGIN_PHONE_RESET_VERIFICATION + fromType + ":" + account);
                    userName = (String) hget;
                }
            }
        }
        if (AccountType.EMAIL == accountType) {
            if (VerificationCodeType.LOGIN == verificationCodeType) {
                Object hget = redisService.hget(RedisParam.LOGIN_EMAIL_VERIFICATION + fromType + ":" + account, verificationCode);
                if (hget != null) {
                    redisService.del(RedisParam.LOGIN_EMAIL_VERIFICATION + fromType + ":" + account);
                    userName = (String) hget;
                }
            }
            if (VerificationCodeType.LOGIN_AND_RESET == verificationCodeType) {
                Object hget = redisService.hget(RedisParam.LOGIN_EMAIL_RESET_VERIFICATION + fromType + ":" + account, verificationCode);
                if (hget != null) {
                    redisService.del(RedisParam.LOGIN_EMAIL_RESET_VERIFICATION + fromType + ":" + account);
                    userName = (String) hget;
                }
            }

        }
        Assert.hasText(userName, "验证码错误或者验证码已过期");
        return userName;
    }

    /*-------------------------------------------------以下是私有方法-------------------------------------------------*/

    /**
     * 增加一次密码错误次数，并返回剩余次数，0表示被锁定
     *
     * @param userName 登录账号（必填）
     * @param phone    手机号（如果登录账号对应用户有手机号则必填）
     * @param email    邮箱（如果登录账号对应用户有邮箱则必填）
     * @return
     */
    private static long addErrorAndGetTimes(String userName, String phone, String email) {
        Assert.hasText(userName, "缺失登录账号");
        long maxErrorTimes = getMaxErrorTimes();
        long lastTimes = maxErrorTimes;
        if (StringUtils.isNotEmpty(userName)) {
            Object lockObj = redisService.hget(RedisParam.LOGIN_LOCK_USER, userName);
            if (lockObj != null) {
                lastTimes = 0L;
            } else {
                Object errorTimesObj = redisService.hget(RedisParam.LOGIN_LOCK_USER_ERROR_TIMES, userName);
                if (maxErrorTimes == 1) {
                    lock(userName, phone, email);
                    lastTimes = 0L;
                } else {
                    int thisAfterAddTimes = 1;
                    if (errorTimesObj != null) {
                        thisAfterAddTimes += ((Integer) errorTimesObj);
                    }
                    if (thisAfterAddTimes >= maxErrorTimes) {
                        lock(userName, phone, email);
                        lastTimes = 0L;
                    } else {
                        setErrorTimes(userName, phone, email, thisAfterAddTimes);
                        lastTimes = maxErrorTimes - thisAfterAddTimes;
                    }
                }
            }
        }
        return lastTimes;
    }

    private static long addErrorAndGetTimes2(String userName, String phone, String email) {
        long maxErrorTimes = getMaxErrorTimes();
        long lastTimes = maxErrorTimes;

        Object lockObj = null;
        if (StringUtils.isNotEmpty(userName)) {
            lockObj = redisService.hget(RedisParam.LOGIN_LOCK_USER, userName);
        } else if (StringUtils.isNotEmpty(phone)) {
            lockObj = redisService.hget(RedisParam.LOGIN_LOCK_PHONE, phone);
        } else if (StringUtils.isNotEmpty(email)) {
            lockObj = redisService.hget(RedisParam.LOGIN_LOCK_EMAIL, email);
        } else {

        }

        if (lockObj != null) {
            lastTimes = 0L;
        } else {
            Object errorTimesObj = null;
            if (StringUtils.isNotEmpty(userName)) {
                errorTimesObj = redisService.hget(RedisParam.LOGIN_LOCK_USER_ERROR_TIMES, userName);
            } else if (StringUtils.isNotEmpty(phone)) {
                errorTimesObj = redisService.hget(RedisParam.LOGIN_LOCK_PHONE_ERROR_TIMES, phone);
            } else if (StringUtils.isNotEmpty(email)) {
                errorTimesObj = redisService.hget(RedisParam.LOGIN_LOCK_EMAIL_ERROR_TIMES, email);
            } else {

            }
            if (maxErrorTimes == 1) {
                lock(userName, phone, email);
                lastTimes = 0L;
            } else {
                int thisAfterAddTimes = 1;
                if (errorTimesObj != null) {
                    thisAfterAddTimes += ((Integer) errorTimesObj);
                }
                if (thisAfterAddTimes >= maxErrorTimes) {
                    lock(userName, phone, email);
                    lastTimes = 0L;
                } else {
                    setErrorTimes(userName, phone, email, thisAfterAddTimes);
                    lastTimes = maxErrorTimes - thisAfterAddTimes;
                }
            }
        }
        return lastTimes;
    }

    /**
     * 锁定用户
     *
     * @param userName 登录账号（必填）
     * @param phone    手机号（如果登录账号对应用户有手机号则必填）
     * @param email    邮箱（如果登录账号对应用户有邮箱则必填）
     */
    private static void lock(String userName, String phone, String email) {
        //锁定的时间（秒）
        long lockSeconds = getMaxLockSeconds();
        //自动解锁时刻的时间戳
        long unlockDateTime = System.currentTimeMillis() + lockSeconds * 1000L;

        if (StringUtils.isNotEmpty(userName)) {
            redisService.hset(RedisParam.LOGIN_LOCK_USER, userName, unlockDateTime, lockSeconds);
            redisService.hdel(RedisParam.LOGIN_LOCK_USER_ERROR_TIMES, userName);
        }
        if (StringUtils.isNotEmpty(phone)) {
            redisService.hset(RedisParam.LOGIN_LOCK_PHONE, phone, unlockDateTime, lockSeconds);
            redisService.hdel(RedisParam.LOGIN_LOCK_PHONE_ERROR_TIMES, phone);
        }
        if (StringUtils.isNotEmpty(email)) {
            redisService.hset(RedisParam.LOGIN_LOCK_EMAIL, email, unlockDateTime, lockSeconds);
            redisService.hdel(RedisParam.LOGIN_LOCK_EMAIL_ERROR_TIMES, email);
        }
    }

    /**
     * 设置错误次数
     *
     * @param userName 登录账号（必填）
     * @param phone    手机号（如果登录账号对应用户有手机号则必填）
     * @param email    邮箱（如果登录账号对应用户有邮箱则必填）
     * @param times    当前错误次数（必填）
     */
    private static void setErrorTimes(String userName, String phone, String email, int times) {
        long errorResetSeconds = getErrorResetSeconds();
        if (StringUtils.isNotEmpty(userName)) {
            redisService.hset(RedisParam.LOGIN_LOCK_USER_ERROR_TIMES, userName, times, errorResetSeconds);
        }
        if (StringUtils.isNotEmpty(phone)) {
            redisService.hset(RedisParam.LOGIN_LOCK_PHONE_ERROR_TIMES, phone, times, errorResetSeconds);
        }
        if (StringUtils.isNotEmpty(email)) {
            redisService.hset(RedisParam.LOGIN_LOCK_EMAIL_ERROR_TIMES, email, times, errorResetSeconds);
        }
    }

    /**
     * 获取锁定的时间（分钟）
     *
     * @param lockKey 锁定前缀
     * @param name    值
     * @return
     */
    private static long getLockMinutes(String lockKey, String name) {
        if (StringUtils.isNotEmpty(name)) {
            Object obj = redisService.hget(lockKey, name);
            if (obj != null) {
                Long unlockDateTimeLong = (Long) obj;
                long l = unlockDateTimeLong - System.currentTimeMillis();
                if (l > 0) {
                    BigDecimal divide = new BigDecimal(l).divide(new BigDecimal(1000L * 60), 0, BigDecimal.ROUND_UP);
                    return divide.longValue();
                }
            }
        }
        return 0;
    }

    /**
     * 获取密码错误重置时间（秒）
     *
     * @return
     */
    private static long getErrorResetSeconds() {
        try {
            String parameterValue = ParamUtil.getParameterValueNoException(ParameterParam.LOGIN_PASSWORD_ERROR_RESET_MINUTES);
            if (StringUtils.isNotEmpty(parameterValue)) {
                return new BigDecimal(parameterValue).multiply(new BigDecimal("60")).setScale(0, RoundingMode.HALF_UP).longValue();
            }
        } catch (Exception e) {
            log.error("获取密码错误重置时间失败");
        }
        return 60L * 60 * 18;
    }

    /**
     * 获取用户密码最大输入错误次数
     *
     * @return
     */
    private static long getMaxErrorTimes() {
        try {
            String parameterValue = ParamUtil.getParameterValueNoException(ParameterParam.LOGIN_ERROR_FREQUENCY);
            if (StringUtils.isNotEmpty(parameterValue)) {
                return Long.valueOf(parameterValue);
            }
        } catch (Exception e) {
            log.error("获取用户密码最大输入错误次数失败");
        }
        return 5L;
    }

    /**
     * 获取用户最大锁定时长（秒）
     *
     * @return
     */
    private static long getMaxLockSeconds() {
        try {
            String parameterValue = ParamUtil.getParameterValueNoException(ParameterParam.LOGIN_LOCK_TIME);
            if (StringUtils.isNotEmpty(parameterValue)) {
                return new BigDecimal(parameterValue).multiply(new BigDecimal("60")).setScale(0, RoundingMode.HALF_UP).longValue();
            }
        } catch (Exception e) {
            log.error("获取用户最大锁定时长失败");
        }
        return 60L * 15;
    }

    /**
     * 生成随机数字验证码
     *
     * @return
     */
    private static String generateVerificationCode() {
        //长度后续改为配置
        int length = 6;
        double random = Math.random();
        return String.format("%0" + length + "d", new BigDecimal(random).multiply(new BigDecimal(Math.pow(10, length))).setScale(0, RoundingMode.HALF_UP).longValue());
    }

}
