package com.biz.crm.utils;

import com.alibaba.fastjson.JSON;
import lombok.SneakyThrows;
import lombok.ToString;
import org.springframework.util.ReflectionUtils;
import org.springframework.util.StringUtils;

import java.lang.reflect.Field;
import java.util.*;

/**
 * @author zxw
 */
public class TreeUtil {


    /**
     * 将 List 转为树形结构
     *
     * @param origList          : 要转换的 List
     * @param idFieldName       : id字段名
     * @param parentIdFieldName : parentId 字段名
     * @param childrenFieldName : children 字段名
     * @param <T>               : 拥有父子结构的 Entity
     * @return : 树形结果
     * @throws Exception .
     */
    @SneakyThrows
    public static <T> List<T> convert(List<T> origList, String idFieldName,
                                      String parentIdFieldName, String childrenFieldName) {
        // 用于保存当前 id 索引的实体类
        Map<String, T> idMaps = new HashMap<>();
        // 暂存区, 用于保存没有找到父 id 的控件
        List<T> tempList = new ArrayList<>();
        List<T> result = new ArrayList<>();
        for (T entity : origList) {
            // 获取 id, parentId, children
            String id = Objects.toString(getFieldValue(entity, idFieldName), "");
            String parentId = Objects.toString(getFieldValue(entity, parentIdFieldName), "");
            if (StringUtils.isEmpty(id)) {
                throw new Exception("存在id为空的资料");
            }
            idMaps.put(id, entity);
            if (StringUtils.isEmpty(parentId)) {
                // 如果父 id 为空, 则实体类为第一层
                result.add(entity);
            } else {
                // 根据父 id 获取实体类
                T parentEntity = idMaps.get(parentId);
                if (parentEntity == null) {
                    // 没找到先放入暂存区
                    tempList.add(entity);
                } else {
                    // 父组件判断是否存在 children, 不存在新增, 存在则直接假如
                    setChildrenValue(childrenFieldName, entity, parentEntity);
                }
            }
        }
        // 处理暂存区, 暂存区的一定不为根节点, 所以它只要父节点存在, 那么此轮查询一定能找到父节点(上一轮已经将全部节点放入 idMaps)
        for (T entity : tempList) {
            // 获取 parentId
            String parentId = Objects.toString(getFieldValue(entity, parentIdFieldName), "");
            // 根据父id获取实体类
            T parentEntity = idMaps.get(parentId);
            if (parentEntity == null) {
                throw new Exception("存在孤立的子节点");
            } else {
                // 父组件判断是否存在children, 不存在新增, 存在则直接假如
                setChildrenValue(childrenFieldName, entity, parentEntity);
            }
        }
        return result;
    }

    private static <T> void setChildrenValue(String childrenFieldName, T entity, T parentEntity) throws Exception {
        Object children = getFieldValue(parentEntity, childrenFieldName);
        List<T> childrenList;
        if (children == null) {
            childrenList = new ArrayList<>();
            childrenList.add(entity);
            setFieldValue(parentEntity, childrenFieldName, childrenList);
        } else {
            List<T> childrenReal = (List<T>) children;
            childrenReal.add(entity);
        }
    }

    private static <T> Object getFieldValue(T entity, String fieldName) throws Exception {
        Field field = ReflectionUtils.findField(entity.getClass(), fieldName);
        if (field == null) {
            throw new Exception(String.format("字段名称[%s]不存在", fieldName));
        }
        boolean accessible = field.isAccessible();
        field.setAccessible(true);
        Object result = ReflectionUtils.getField(field, entity);
        field.setAccessible(accessible);
        return result;
    }

    private static <T> void setFieldValue(T entity, String fieldName, Object value) throws Exception {
        Field field = ReflectionUtils.findField(entity.getClass(), fieldName);
        if (field == null) {
            throw new Exception(String.format("字段名称[%s]不存在", fieldName));
        }
        boolean accessible = field.isAccessible();
        field.setAccessible(true);
        ReflectionUtils.setField(field, entity, value);
        field.setAccessible(accessible);
    }

    public static void main(String[] args) throws Exception {
        List<Demo> list = new ArrayList<>();
        for (int i = 0; i < 5; i++) {
            Demo demo = new Demo(i, "一级节点" + i);
            list.add(demo);
        }
        for (int i = 5; i < 15; i++) {
            Demo demo = new Demo(i, i % 5, "二级节点" + i);
            list.add(demo);
        }
        for (int i = 15; i < 100; i++) {
            Demo demo = new Demo(i, i % 10 + 5, "三级节点" + i);
            list.add(demo);
        }
        Demo demo = new Demo(100, 102, "非法节点");
        list.add(demo);
        List<Demo> convert = TreeUtil.convert(list, "id", "pid", "children");
        String s = JSON.toJSONString(convert);
        System.out.println(s);
    }

}

@ToString
class Demo {
    private Integer id;
    private Integer pid;
    private String name;
    private List<Demo> children;

    public Demo(Integer id, Integer pid, String name) {
        this.id = id;
        this.pid = pid;
        this.name = name;
    }

    public Demo(Integer id, String name) {
        this.id = id;
        this.name = name;
    }

    public Integer getId() {
        return this.id;
    }

    public Integer getPid() {
        return this.pid;
    }

    public String getName() {
        return this.name;
    }

    public List<Demo> getChildren() {
        return this.children;
    }

    public void setId(Integer id) {
        this.id = id;
    }

    public void setPid(Integer pid) {
        this.pid = pid;
    }

    public void setName(String name) {
        this.name = name;
    }

    public void setChildren(List<Demo> children) {
        this.children = children;
    }

    public boolean equals(final Object o) {
        if (o == this) return true;
        if (!(o instanceof Demo)) return false;
        final Demo other = (Demo) o;
        if (!other.canEqual((Object) this)) return false;
        final Object this$id = this.getId();
        final Object other$id = other.getId();
        if (this$id == null ? other$id != null : !this$id.equals(other$id)) return false;
        final Object this$pid = this.getPid();
        final Object other$pid = other.getPid();
        if (this$pid == null ? other$pid != null : !this$pid.equals(other$pid)) return false;
        final Object this$name = this.getName();
        final Object other$name = other.getName();
        if (this$name == null ? other$name != null : !this$name.equals(other$name)) return false;
        final Object this$children = this.getChildren();
        final Object other$children = other.getChildren();
        if (this$children == null ? other$children != null : !this$children.equals(other$children)) return false;
        return true;
    }

    protected boolean canEqual(final Object other) {
        return other instanceof Demo;
    }

    public int hashCode() {
        final int PRIME = 59;
        int result = 1;
        final Object $id = this.getId();
        result = result * PRIME + ($id == null ? 43 : $id.hashCode());
        final Object $pid = this.getPid();
        result = result * PRIME + ($pid == null ? 43 : $pid.hashCode());
        final Object $name = this.getName();
        result = result * PRIME + ($name == null ? 43 : $name.hashCode());
        final Object $children = this.getChildren();
        result = result * PRIME + ($children == null ? 43 : $children.hashCode());
        return result;
    }
}