package com.biz.crm.utils;

import com.biz.crm.base.BusinessException;
import com.biz.crm.service.RedisService;
import com.google.code.kaptcha.impl.DefaultKaptcha;
import io.jsonwebtoken.lang.Assert;
import java.util.HashMap;
import java.util.Map;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import javax.imageio.ImageIO;
import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletResponse;
import java.awt.image.BufferedImage;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.Optional;

/**
 * @author zxw
 * @date 2021-05-07 14:32
 **/
@Component
@Slf4j
public class VerificationUtil {

    private static RedisService redisService;
    private static DefaultKaptcha defaultKaptcha;
    private static final String VERIFICATION = "verification:";

    private final static Integer MIN_LENGTH = 8;

    // 字母表
    private final static String SMALL_LETTERS = "abcdefghijklmnopqrstuvwxyz";

    private final static String BIG_LETTERS = "ABCDEFGHIJKLMNOPQRSTUVWXYZ";

    // 特殊字符
    private final static String SPECIAL_CHARS = "~!@#$%^&*()_+-={}[],./<>?:|";

    private final static String NUMBERS = "1234567890";

    @Autowired
    public void setRedisService(RedisService redisService) {
        VerificationUtil.redisService = redisService;
    }

    @Autowired
    public void setDefaultKaptcha(DefaultKaptcha defaultKaptcha) {
        VerificationUtil.defaultKaptcha = defaultKaptcha;
    }

    public static String generateRandomStr() {
        String createText = defaultKaptcha.createText();
        Assert.hasText(createText, "生成随机字符串失败");
        return createText;
    }

    public static void generate(String token) {
        Assert.hasText("唯一标识不能为空", "token");
        byte[] captchaChallengeAsJpeg;
        ByteArrayOutputStream jpegOutputStream = new ByteArrayOutputStream();
        //生成验证码字符串
        String createText = generateRandomStr();

        redisService.setMinutes(VERIFICATION + token, createText, 1);
        //使用生成的验证码字符串返回一个BufferedImage对象并转为byte写入到byte数组中
        BufferedImage challenge = defaultKaptcha.createImage(createText);
        try {
            ImageIO.write(challenge, "jpg", jpegOutputStream);
        } catch (IOException e) {
            log.error("生成图形验证码失败", e);
            throw new BusinessException("生成验证码失败");
        }
        //定义response输出类型为image/jpeg类型，使用response输出流输出图片的byte数组
        HttpServletResponse response = Optional.ofNullable((ServletRequestAttributes) RequestContextHolder.getRequestAttributes())
                .map(ServletRequestAttributes::getResponse)
                .orElseThrow(() -> new BusinessException("获取当前线程响应失败"));
        response.setHeader("Cache-Control", "no-store");
        response.setHeader("Pragma", "no-cache");
        response.setDateHeader("Expires", 0);
        response.setContentType("image/jpeg");
        try {
            captchaChallengeAsJpeg = jpegOutputStream.toByteArray();
            ServletOutputStream servletOutputStream = response.getOutputStream();
            servletOutputStream.write(captchaChallengeAsJpeg);
            servletOutputStream.flush();
            servletOutputStream.close();
        } catch (IOException e) {
            log.error("输出验证码失败", e);
            throw new BusinessException("生成验证码失败");
        }
    }

    public static void checkVerification(String token, String verification) {
        Assert.hasText(token, "唯一标识不能为空");
        Assert.hasText(verification, "验证码不能为空");
        Object object = redisService.get(VERIFICATION + token);
        log.info("图形验证码：获取验证码:{}", object);
        Assert.notNull(object, "验证码已失效");
        Assert.isTrue(verification.equals(object), "验证码不正确");
        redisService.del(VERIFICATION + token);
    }


    /**
     *  先判断传入密码的长度是否够12位，其次判断是否包含各个类型的符号.
     *  密码要求: 大小写字母特殊字符以及数字, 且长度大于12位.
     *  @param pwd
     *  @return
     */
    public static void isStrongPwd(String pwd){
        boolean isLongEnough = (pwd.length() >= MIN_LENGTH)?true:false;
        if (!isLongEnough){
            //长度不够
            throw new BusinessException("密码长度必须大于等于8位数");
        }else {
            //长度够了
            boolean hasBigLetters = false;
            boolean hasSmallLetters = false;
            boolean hasNumbers = false;
            boolean hasSpecialChars = false;
            int len = pwd.length();

            for (int i = 0; i < len; i++){
                if(!hasSmallLetters && SMALL_LETTERS.indexOf(pwd.charAt(i)) > -1){
                    hasSmallLetters = true;
                }
                if(!hasBigLetters && BIG_LETTERS.indexOf(pwd.charAt(i)) > -1){
                    hasBigLetters = true;
                }
                if(!hasNumbers && NUMBERS.indexOf(pwd.charAt(i)) > -1){
                    hasNumbers = true;
                }
                if(!hasSpecialChars && SPECIAL_CHARS.indexOf(pwd.charAt(i)) > -1){
                    hasSpecialChars = true;
                }
            }
            if (!hasBigLetters){
                throw new BusinessException("密码里面没有大写字母！");
            }else if(!hasSmallLetters){
                throw new BusinessException("密码里面没有小写字母！");
            }else if(!hasNumbers){
                throw new BusinessException("密码里面没有数字！");

            }else if(!hasSpecialChars){
                throw new BusinessException("密码里面没有特殊字符！");
            }
        }

    }
}
