package com.biz.crm.order.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.nebular.order.OrderDetailVo;
import com.biz.crm.nebular.order.OrderGroupItemVo;
import com.biz.crm.nebular.order.OrderVo;
import com.biz.crm.order.OrderDetailEntity;
import com.biz.crm.order.OrderEntity;
import com.biz.crm.order.mapper.OrderDetailMapper;
import com.biz.crm.order.mapper.OrderDetailPromotionMapper;
import com.biz.crm.order.service.OrderDetailService;
import com.biz.crm.order.utils.OrderDetailUtil;
import com.biz.crm.util.CollectionUtil;
import com.biz.crm.util.CrmBeanUtil;
import com.biz.crm.util.StringUtils;
import com.biz.crm.util.ValidateUtils;
import com.google.common.collect.Lists;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import static com.biz.crm.eunm.dms.OrderEunm.LineTypeEnum.*;

/**
 * @Description:
 * @Author: zhangyuzhu
 * @Date: 2021/1/28 16:49
 **/
@ConditionalOnMissingBean(name = "orderDetailServiceExpandImpl")
@Service(value = "orderDetailService")
public class OrderDetailServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<OrderDetailMapper, OrderDetailEntity>
        implements OrderDetailService {

    @Resource
    private OrderDetailPromotionMapper orderDetailPromotionMapper;

    @Resource
    private OrderDetailMapper orderDetailMapper;


    /**
     * 1、转换模型
     * 2、保存
     * @param orderVo
     * @param orderEntity
     */
    @Transactional
    @Override
    public List<OrderDetailEntity> addBatch(OrderVo orderVo, OrderEntity orderEntity,boolean removeZreoFlag) {
        //1
        List<OrderDetailEntity> orderDetailEntities = OrderDetailUtil.packageOrderDetails(orderVo,orderEntity,removeZreoFlag);

        //2
        this.saveBatch(orderDetailEntities);
        return orderDetailEntities;
    }

    /**
     * 1、删除原有订单行
     * 2、保存新的订单行
     * @param orderVo
     * @param orderEntity
     */
    @Override
    public List<OrderDetailEntity> editBatch(OrderVo orderVo, OrderEntity orderEntity,boolean removeZreoFlag) {
        //1
        QueryWrapper<OrderDetailEntity> wrapper = new QueryWrapper<>();
        wrapper.eq("order_code", orderEntity.getOrderCode());
        orderDetailMapper.delete(wrapper);

        //2
        return this.addBatch(orderVo,orderEntity,removeZreoFlag);
    }

    @Override
    @Transactional
    public void deleteByOrderCode(String orderCode) {
        QueryWrapper<OrderDetailEntity> wrapper = new QueryWrapper<>();
        wrapper.eq("order_code", orderCode);
        orderDetailMapper.delete(wrapper);
    }

    /**
     * 根据订单编码查询分组后的订单详情
     * 1、校验入参
     * 2、查询所有订单详情
     * 3、组装结果记录
     * @param orderCode
     * @return
     */
    @Override
    public List<OrderGroupItemVo> findGroupByOrderCode(String orderCode) {
        //1、
        if(StringUtils.isEmpty(orderCode)) {
            return Lists.newArrayList();
        }
        //2、
        List<OrderDetailVo> detailVos = this.findByOrderCode(orderCode);
        if(CollectionUtil.listEmpty(detailVos)) {
            return Lists.newArrayList();
        }
        //3、
        return this.groupOrderDetails(detailVos);
    }

    /**
     * 订单详情分组
     * 1、校验入参
     * 2、按照分组编码分组
     * 3、分别组转各组本品、赠品、货补商品
     * 4、分组排序
     * @param detailVos
     * @return
     */
    private List<OrderGroupItemVo> groupOrderDetails(List<OrderDetailVo> detailVos) {
        //1、
        if(CollectionUtil.listEmpty(detailVos)) {
            return Lists.newArrayList();
        }
        //2、
        List<OrderGroupItemVo> result = Lists.newArrayList();
        Map<String, List<OrderDetailVo>> groupItemMap = detailVos.stream().collect(Collectors.groupingBy(OrderDetailVo::getGroupCode));
        //3、
        groupItemMap.forEach((k, details) -> {
            List<OrderDetailVo> normalList = details.stream().filter(detail -> SOURCE_PRODUCT.getCode().equals(detail.getLineType()))
                    .sorted(Comparator.comparing(OrderDetailVo::getLineNo)).collect(Collectors.toList());
            List<OrderDetailVo> giftList = details.stream().filter(detail -> GIFT_PRODUCT.getCode().equals(detail.getLineType()))
                    .sorted(Comparator.comparing(OrderDetailVo::getLineNo)).collect(Collectors.toList());
            List<OrderDetailVo> backList = details.stream().filter(detail -> REP_PRODUCT.getCode().equals(detail.getLineType()))
                    .sorted(Comparator.comparing(OrderDetailVo::getLineNo)).collect(Collectors.toList());
            List<OrderDetailVo> reissueList = details.stream().filter(detail -> REISSUE_PRODUCT.getCode().equals(detail.getLineType()))
                    .sorted(Comparator.comparing(OrderDetailVo::getLineNo)).collect(Collectors.toList());
            OrderGroupItemVo groupItemVo = new OrderGroupItemVo();
            groupItemVo.setGroupCode(k);
            groupItemVo.setNormalList(normalList);
            groupItemVo.setGiftList(giftList);
            groupItemVo.setBackList(backList);
            groupItemVo.setReissueList(reissueList);
            result.add(groupItemVo);
        });
        //4、
        return result.stream().sorted(Comparator.comparing(OrderGroupItemVo::getGroupCode)).collect(Collectors.toList());
    }

    /**
     * 根据订单编码查询所有订单详情
     * @param orderCode 订单编码
     * @return
     */
    @Override
    public List<OrderDetailVo> findByOrderCode(String orderCode) {
        if(StringUtils.isEmpty(orderCode)) {
            return Lists.newArrayList();
        }
        List<OrderDetailEntity> entities = this.orderDetailMapper.selectList(Wrappers.<OrderDetailEntity>query().eq("order_code", orderCode)
        .orderByAsc("line_no"));
        if(CollectionUtil.listEmpty(entities)) {
            return Lists.newArrayList();
        }
        return CrmBeanUtil.copyList(entities, OrderDetailVo.class);
    }

    /**
     * 根据订单编码列表删除订单明细
     * @param orderCodes
     */
    @Override
    @Transactional
    public void deleteByOrderCodes(List<String> orderCodes) {
        ValidateUtils.notEmpty(orderCodes, "删除订单明细时，传入订单编码列表不能为空");
        QueryWrapper<OrderDetailEntity> wrapper = new QueryWrapper<>();
        wrapper.in("order_code", orderCodes);
        orderDetailMapper.delete(wrapper);
    }
}
