package com.biz.crm.order.utils;

import com.biz.crm.constant.dms.RedisPathConstant;
import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.eunm.dms.PromotionPolicyEunm;
import com.biz.crm.nebular.order.*;
import com.biz.crm.order.OrderEntity;
import com.biz.crm.service.RedisService;
import com.biz.crm.util.CodeUtil;
import com.biz.crm.util.CollectionUtil;
import com.biz.crm.util.StringUtils;
import java.math.BigDecimal;
import org.springframework.beans.BeanUtils;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.stream.Collectors;

/**
 * @Description:
 * @Author: zhangyuzhu
 * @Date: 2021/3/2 15:18
 **/
@ConditionalOnMissingBean(name = "orderUtilExpand")
@Component("orderUtil")
public class OrderUtil {

    /**
     * 订单计算促销排序字典编码
     */
    public static final String CAL_PROMOTION_SORT_CODE = "cal_promotion_sort";


    /**
     * 汇总订单数据
     * @param orderVo
     * @return
     */
    public static void collect(OrderVo orderVo){
        Map<String, OrderDetailCollectVo> totalMap = new HashMap<>();
        for(OrderGroupItemVo group : orderVo.getGroupItemVos()){
            if(!CollectionUtils.isEmpty(group.getNormalList())){
                for(OrderDetailVo detail : group.getNormalList()){
                    collectProduct(detail,totalMap);
                }
            }

            if(!CollectionUtils.isEmpty(group.getGiftList())){
                for(OrderDetailVo detail : group.getGiftList()){
                    collectProduct(detail,totalMap);
                }
            }

            if(!CollectionUtils.isEmpty(group.getBackList())){
                for(OrderDetailVo detail : group.getBackList()){
                    collectProduct(detail,totalMap);
                }
            }
        }
        orderVo.setOrderDetailCollectVos(CollectionUtils.isEmpty(totalMap)?new ArrayList<>():new ArrayList<>(totalMap.values()));
        orderVo.setGroupItemVos(new ArrayList<>());
    }

    /**
     * 汇总单个产品
     * @param detail
     * @param totalMap
     */
    public static void collectProduct(OrderDetailVo detail,Map<String, OrderDetailCollectVo> totalMap){
        OrderDetailCollectVo vo = totalMap.get(detail.getProductCode());
        if(null == vo){
            vo = new OrderDetailCollectVo();
            BeanUtils.copyProperties(detail,vo);
            totalMap.put(detail.getProductCode(),vo);
            return;
        }
        vo.setAmount(vo.getAmount().add(detail.getAmount()));
        vo.setProductNum(vo.getProductNum().add(detail.getProductNum()));
    }


    /**
     * 为orderVo添加促销
     * 1、为每个组
     * 2、为每个组组装买赠促销
     * @param promotionVos
     * @return
     */
    public static OrderVo packagePromotionToVo(OrderVo orderVo,List<OrderDetailPromotionVo> promotionVos){
        for(OrderGroupItemVo group : orderVo.getGroupItemVos()){
            if(CollectionUtils.isEmpty(group.getNormalList())){
                group.setBuyAndGiveFlag(YesNoEnum.yesNoEnum.NO.getValue());
                continue;
            }
            //1
            ArrayList<OrderDetailPromotionVo> allPromotion = new ArrayList<>();
            for(OrderDetailVo detail : group.getNormalList()){
                ArrayList<OrderDetailPromotionVo> linePromotions = findPromotionByLineNo(detail.getLineNo(),promotionVos);
                detail.setPromotionVos(linePromotions);
                allPromotion.addAll(linePromotions);
            }

            //2
            OrderDetailPromotionVo buyGiftPromotion = findBuyGiftPromotion(allPromotion);
            if(null == buyGiftPromotion){
                group.setBuyAndGiveFlag(YesNoEnum.yesNoEnum.NO.getValue());
                continue;
            }
            group.setBuyAndGiveFlag(YesNoEnum.yesNoEnum.YES.getValue());
            group.setPromotionCode(buyGiftPromotion.getPromotionCode());
            group.setPromotionName(buyGiftPromotion.getPromotionName());
        }
        return orderVo;
    }

    /**
     * 查询买赠促销
     * @param allPromotion
     * @return
     */
    public static OrderDetailPromotionVo findBuyGiftPromotion(ArrayList<OrderDetailPromotionVo> allPromotion){
        if(CollectionUtils.isEmpty(allPromotion)){
            return null;
        }
        for(OrderDetailPromotionVo promotion : allPromotion){
            if(promotion.getPromotionType().equals(PromotionPolicyEunm.PromotionTypeEunm.BUYGIFT.getCode())){
                return promotion;
            }
        }
        return null;
    }

    /**
     * 通过订单行查询促销
     * @param lineNo
     * @return
     */
    public static ArrayList<OrderDetailPromotionVo> findPromotionByLineNo(String lineNo,List<OrderDetailPromotionVo> promotionVos){
        ArrayList<OrderDetailPromotionVo> reList = new ArrayList<>();
        for(OrderDetailPromotionVo promotion : promotionVos){
            if(promotion.getOrderDetailLineNo().equals(lineNo)){
                reList.add(promotion);
            }
        }
        return reList;
    }


    /**
     * 提取订单所有的促销
     * @param orderVo
     * @return
     */
    public static List<OrderDetailPromotionVo> extractPromotion(OrderVo orderVo){
        List<OrderDetailPromotionVo> list = new ArrayList<>();
        for(OrderGroupItemVo group : orderVo.getGroupItemVos()){
            if(CollectionUtils.isEmpty(group.getNormalList())){
                continue;
            }
            for(OrderDetailVo detail : group.getNormalList()){
                if(CollectionUtils.isEmpty(detail.getPromotionVos())){
                    continue;
                }
                list.addAll(detail.getPromotionVos());
            }
        }
        return list;
    }


    /**
     * 判断订单算价是否成功
     * @param orderVo
     * @return
     */
    public static boolean judegeOrder(OrderVo orderVo){
        if(orderVo.getCalSucessFlag().equals(YesNoEnum.yesNoEnum.NO.getValue())){
            return false;
        }
        return true;
    }

    /**
     * 判断订单组算价是否成功
     * @param group
     * @return
     */
    public static boolean judegeOrderGroup(OrderGroupItemVo group){
        if(group.getCalSucessFlag().equals(YesNoEnum.yesNoEnum.NO.getValue())){
            return false;
        }
        return true;
    }

    /**
     * vo转entity 订单头
     *
     * @param orderVo
     * @return
     */
    public OrderEntity packageOrderHead(OrderVo orderVo) {
        OrderEntity orderEntity = new OrderEntity();
        BeanUtils.copyProperties(orderVo, orderEntity);
        orderEntity.setOrderCode(CodeUtil.getCodeDefault());
        return orderEntity;
    }


    /**
     * vo转entity 订单头
     *
     * @param orderVo
     * @return
     */
    public OrderEntity packageOrderHeadForEndit(OrderVo orderVo, OrderEntity orderEntity) {
        String id = orderEntity.getId();
        String orderCode = orderEntity.getOrderCode();
        BeanUtils.copyProperties(orderVo, orderEntity);
        orderEntity.setId(id);
        orderEntity.setOrderCode(orderCode);
        return orderEntity;
    }


    /**
     * 订单行号
     * 1、按照分组排序并设置组号
     * 2、循环设置行号
     *
     * @param orderVo
     * @return
     */
    public OrderVo createLineAndSort(OrderVo orderVo) {
        //1
        int groupNum = 1;
        int lineNum = 1;
        for (OrderGroupItemVo orderGroupItemVo : orderVo.getGroupItemVos()) {
            String groupCode = OrderCodeUtil.fullFour(groupNum);
            orderGroupItemVo.setGroupCode(groupCode);
            if (!CollectionUtils.isEmpty(orderGroupItemVo.getBackList())) {
                for (OrderDetailVo item : orderGroupItemVo.getBackList()) {
                    item.setGroupCode(groupCode);
                    item.setLineNo(OrderCodeUtil.fullFour(lineNum));
                    zPromotionLineNum(OrderCodeUtil.fullFour(lineNum),item);
                    item.setOnlykey(new StringBuilder(item.getLineNo()).append(",").append(item.getProductCode()).toString());
                    lineNum++;
                }
            }

            if (!CollectionUtils.isEmpty(orderGroupItemVo.getNormalList())) {
                for (OrderDetailVo item : orderGroupItemVo.getNormalList()) {
                    item.setGroupCode(groupCode);
                    item.setLineNo(OrderCodeUtil.fullFour(lineNum));
                    zPromotionLineNum(OrderCodeUtil.fullFour(lineNum),item);
                    item.setOnlykey(new StringBuilder(item.getLineNo()).append(",").append(item.getProductCode()).toString());
                    lineNum++;
                }
            }

            if (!CollectionUtils.isEmpty(orderGroupItemVo.getGiftList())) {
                for (OrderDetailVo item : orderGroupItemVo.getGiftList()) {
                    item.setGroupCode(groupCode);
                    item.setLineNo(OrderCodeUtil.fullFour(lineNum));
                    zPromotionLineNum(OrderCodeUtil.fullFour(lineNum),item);
                    item.setOnlykey(new StringBuilder(item.getLineNo()).append(",").append(item.getProductCode()).toString());
                    lineNum++;
                }
            }
            groupNum++;
        }

        //2
        return sort(orderVo);
    }

    /**
     * 给行上促销设置行号
     * @param lineNum
     * @param item
     */
    public static void zPromotionLineNum(String lineNum,OrderDetailVo item){
        if(CollectionUtils.isEmpty(item.getPromotionVos())){
            return;
        }
        for(OrderDetailPromotionVo promotionVo : item.getPromotionVos()){
            promotionVo.setOrderDetailLineNo(lineNum);
        }
    }

    /**
     * 排序
     *
     * @param orderVo
     * @return
     */
    public static OrderVo sort(OrderVo orderVo) {
        List<OrderGroupItemVo> groupItemVos = orderVo.getGroupItemVos();
        groupItemVos = groupItemVos.stream().sorted(Comparator.comparing(OrderGroupItemVo::getGroupCode))
                .collect(Collectors.toList());
        for (OrderGroupItemVo orderGroupItemVo : groupItemVos) {
            if (!CollectionUtils.isEmpty(orderGroupItemVo.getBackList())) {
                List<OrderDetailVo> list = orderGroupItemVo.getBackList().stream().sorted(Comparator.comparing(OrderDetailVo::getLineNo))
                        .collect(Collectors.toList());
                orderGroupItemVo.setBackList(list);
            }

            if (!CollectionUtils.isEmpty(orderGroupItemVo.getNormalList())) {
                List<OrderDetailVo> list = orderGroupItemVo.getNormalList().stream().sorted(Comparator.comparing(OrderDetailVo::getLineNo))
                        .collect(Collectors.toList());
                orderGroupItemVo.setNormalList(list);
            }

            if (!CollectionUtils.isEmpty(orderGroupItemVo.getGiftList())) {
                List<OrderDetailVo> list = orderGroupItemVo.getGiftList().stream().sorted(Comparator.comparing(OrderDetailVo::getLineNo))
                        .collect(Collectors.toList());
                orderGroupItemVo.setGiftList(list);
            }
        }
        orderVo.setGroupItemVos(groupItemVos);
        return orderVo;
    }


    /**
     * 扣除促销政策赠品使用数量(redis)
     * @param orderVo
     */
    public static void reduceGiftCount(OrderVo orderVo, RedisService redisService) {
        if(orderVo == null || CollectionUtil.listEmpty(orderVo.getGroupItemVos())) {
            return;
        }
        orderVo.getGroupItemVos().forEach(group -> {
            List<OrderDetailVo> normalList = group.getNormalList();
            String promotionCode = null;
            if(CollectionUtil.listNotEmpty(normalList)) {
                for (OrderDetailVo n : normalList) {
                    promotionCode = getGiftPromotion(n);
                    if(StringUtils.isNotEmpty(promotionCode)) {
                        break;
                    }
                }
            }
            if(StringUtils.isNotEmpty(promotionCode) && CollectionUtil.listNotEmpty(group.getGiftList())) {
                BigDecimal giftCount = BigDecimal.ZERO;
                for (OrderDetailVo detailVo : group.getGiftList()) {
                    giftCount = giftCount.add(detailVo.getProductNum() == null ? BigDecimal.ZERO : detailVo.getProductNum());
                }
                BigDecimal totalCount = (BigDecimal) redisService.get(RedisPathConstant.PROMOTION_GIFT_REDIS_PATH_BASE.concat(orderVo.getCusCode().concat("-").concat(promotionCode)));
                totalCount = totalCount == null ? BigDecimal.ZERO : totalCount;
                totalCount = totalCount.subtract(giftCount);
                totalCount = totalCount.compareTo(BigDecimal.ZERO) < 0 ? BigDecimal.ZERO : totalCount;
                redisService.set(RedisPathConstant.PROMOTION_GIFT_REDIS_PATH_BASE.concat(orderVo.getCusCode().concat("-").concat(promotionCode)), totalCount);
            }
        });
    }

    /**
     * 保存促销政策赠品使用数量(redis)
     * @param orderVo
     */
    public static void saveGiftCount(OrderVo orderVo, RedisService redisService) {
        if(orderVo == null || CollectionUtil.listEmpty(orderVo.getGroupItemVos())) {
            return;
        }
        orderVo.getGroupItemVos().forEach(group -> {
            List<OrderDetailVo> normalList = group.getNormalList();
            String promotionCode = null;
            if(CollectionUtil.listNotEmpty(normalList)) {
                for (OrderDetailVo n : normalList) {
                    promotionCode = getGiftPromotion(n);
                    if(StringUtils.isNotEmpty(promotionCode)) {
                        break;
                    }
                }
            }
            if(StringUtils.isNotEmpty(promotionCode) && CollectionUtil.listNotEmpty(group.getGiftList())) {
                BigDecimal giftCount = BigDecimal.ZERO;
                for (OrderDetailVo detailVo : group.getGiftList()) {
                    giftCount = giftCount.add(detailVo.getProductNum() == null ? BigDecimal.ZERO : detailVo.getProductNum());
                }
                BigDecimal totalCount = (BigDecimal) redisService.get(RedisPathConstant.PROMOTION_GIFT_REDIS_PATH_BASE.concat(orderVo.getCusCode().concat("-").concat(promotionCode)));
                totalCount = totalCount == null ? BigDecimal.ZERO : totalCount;
                redisService.set(RedisPathConstant.PROMOTION_GIFT_REDIS_PATH_BASE.concat(orderVo.getCusCode().concat("-").concat(promotionCode)), totalCount.add(giftCount));
            }
        });
    }

    /**
     * 获取买赠政策
     * @param detailVo
     * @return
     */
    private static String getGiftPromotion(OrderDetailVo detailVo) {
        if(detailVo == null || CollectionUtil.listEmpty(detailVo.getPromotionVos())) {
            return null;
        }
        for (OrderDetailPromotionVo promotionVo : detailVo.getPromotionVos()) {
            if(Objects.equals(promotionVo.getPromotionType(), PromotionPolicyEunm.PromotionTypeEunm.BUYGIFT.getCode())) {
                return promotionVo.getPromotionCode();
            }
        }
        return null;
    }


}
