package com.biz.crm.mq;

import com.biz.crm.base.BusinessException;
import com.biz.crm.common.param.RedisParam;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.mq.mqlog.MdmMqMessageLogFeign;
import com.biz.crm.service.RedisService;
import com.biz.crm.util.CrmDateUtils;
import com.biz.crm.util.DateUtil;
import com.biz.crm.util.JsonPropertyUtil;
import com.biz.crm.util.UserUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.apache.rocketmq.spring.core.RocketMQListener;
import org.springframework.beans.factory.annotation.Autowired;

import java.time.LocalDateTime;


/**
 *  @author: luoqi
 *  @Date: 2020-10-30 11:33
 *  @version: V1.0
 *  @Description:
 */
@Slf4j
public abstract class AbstractRocketMQConsumer implements RocketMQListener<RocketMQMessageBody> {
    @Autowired
    private MdmMqMessageLogFeign mdmMqMessageLogFeign;
    @Autowired
    private RedisService redisService;

    @Override
    public void onMessage(RocketMQMessageBody message) {
        String token = message.getToken();
        if(StringUtils.isBlank(token)){
            UserUtils.doTokenForNull();
        }else{
            UserUtils.setToken(token);
        }
        if(null == message){
            log.warn("消费MQ消息：RocketMQMessageBody 为空，忽略本次消费");
            throw new BusinessException("消费MQ消息：RocketMQMessageBody 为空，忽略本次消费");
        }

        Exception handleException = null;
        String json = JsonPropertyUtil.toJsonString(message);
        //丢失日志id
        if(StringUtils.isBlank(message.getId())){
            log.warn("消费MQ消息，记录日志失败：message = {}", json);
            //抛出异常
            if(null != handleException){
                throw new BusinessException("消费MQ消息，失败：message = " + json, handleException);
            }
            return;
        }
        Object o = redisService.get(message.getId());
        MdmMqMessageLogVo logVo = new MdmMqMessageLogVo();
        if(o!=null) {
            logVo = (MdmMqMessageLogVo) o;
        }

        logVo.setUpdateDate(DateUtil.formatDate());
        logVo.setUpdateDateSecond(DateUtil.dateNowStrHms());
        // 调用用户实现的 handleMessage 消费消息
        try {
            logVo.setCallbackBegin(LocalDateTime.now().format(CrmDateUtils.yyyyMMddHHmmssSSS));
            Object result = this.handleMessage(message);
            logVo.setCallbackEnd(LocalDateTime.now().format(CrmDateUtils.yyyyMMddHHmmssSSS));
            logVo.setCallbackLog(JsonPropertyUtil.toJsonString(result));
            logVo.setEndStatus(CrmEnableStatusEnum.ENABLE.getCode());
        }catch (Exception e){
            log.error("消费MQ消息，失败：message = {} >>>", json, e);
            logVo.setCallbackLog(ExceptionUtils.getStackTrace(e));
            logVo.setEndStatus(CrmEnableStatusEnum.DISABLE.getCode());
            handleException = e;
        }
        //回写日志到数据行
        try {
            logVo.setId(message.getId());
            redisService.setDays(logVo.getId(),logVo, RedisParam.TIME5);
            mdmMqMessageLogFeign.updateById(message.getId());
        }catch (Exception e){
            log.error("消费MQ消息，记录日志失败：message = {} >>>", json, e);
        }
        //抛出异常
        if(null != handleException){
            throw new BusinessException("消费MQ消息，失败：message = " + json, handleException);
        }

    }

    /**
     * 消费消息
     * @param message
     * @return
     */
    protected abstract Object handleMessage(RocketMQMessageBody message) throws InterruptedException;
}
