package com.biz.crm.base;

import com.biz.crm.common.PageResult;
import com.biz.crm.config.SpringApplicationContextUtil;
import com.biz.crm.eunm.sfa.SfaVisitEnum;
import com.biz.crm.mdm.customer.MdmCustomerMsgFeign;
import com.biz.crm.mdm.terminal.MdmTerminalFeign;
import com.biz.crm.nebular.mdm.customer.MdmCustomerContactPageReqVo;
import com.biz.crm.nebular.mdm.customer.MdmCustomerContactRespVo;
import com.biz.crm.nebular.mdm.customer.MdmCustomerMsgContactPageRespVo;
import com.biz.crm.nebular.mdm.customer.MdmCustomerMsgRespVo;
import com.biz.crm.nebular.mdm.terminal.MdmTerminalContactPageReqVo;
import com.biz.crm.nebular.mdm.terminal.MdmTerminalContactPageVo;
import com.biz.crm.nebular.mdm.terminal.MdmTerminalContactVo;
import com.biz.crm.nebular.mdm.terminal.MdmTerminalVo;
import com.biz.crm.util.CollectionUtil;
import com.biz.crm.util.CrmBeanUtil;
import com.biz.crm.util.Result;
import com.biz.crm.util.StringUtils;
import com.google.common.collect.Lists;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;


/**
 * SFA 客户工具类
 *
 * @author: luoqi
 * @Date: 2021-4-1 16:39
 * @version: V1.0
 * @Description:
 */
public class SfaClientHelper {
    private static MdmTerminalFeign mdmTerminalFeign = SpringApplicationContextUtil.getApplicationContext().getBean(MdmTerminalFeign.class);
    private static MdmCustomerMsgFeign mdmCustomerMsgFeign = SpringApplicationContextUtil.getApplicationContext().getBean(MdmCustomerMsgFeign.class);

    /**
     * 按组织查询分页
     *
     * @param orgCodes
     * @param pageNum
     * @param pageSize
     * @return
     */
    public static PageResult<SfaClientData> loadTerminalClientDataByOrgCodes(List<String> orgCodes, int pageNum, int pageSize) {
        if (CollectionUtils.isEmpty(orgCodes)) {
            return emptyPage();
        }
        MdmTerminalVo vo = new MdmTerminalVo();
        vo.setOrgCodeList(orgCodes);
        vo.setPageNum(pageNum);
        vo.setPageSize(pageSize);
        return loadTerminalClientDataByParams(vo);
    }

    /**
     * 按组织查询分页
     *
     * @param channelCodes
     * @param pageNum
     * @param pageSize
     * @return
     */
    public static PageResult<SfaClientData> loadTerminalClientDataByChannelCodes(List<String> channelCodes, int pageNum, int pageSize) {
        if (CollectionUtils.isEmpty(channelCodes)) {
            return emptyPage();
        }
        MdmTerminalVo vo = new MdmTerminalVo();
        vo.setChannelList(channelCodes);
        vo.setPageNum(pageNum);
        vo.setPageSize(pageSize);
        return loadTerminalClientDataByParams(vo);
    }

    private static PageResult<SfaClientData> emptyPage() {
        PageResult<SfaClientData> sfaClientDataPageResult = PageResult.<SfaClientData>builder().build();
        sfaClientDataPageResult.setData(Lists.newArrayList());
        sfaClientDataPageResult.setCount(0L);
        return sfaClientDataPageResult;
    }

    private static PageResult<SfaClientData> loadTerminalClientDataByParams(MdmTerminalVo vo) {
        PageResult<SfaClientData> sfaClientDataPageResult = PageResult.<SfaClientData>builder().build();
        PageResult<MdmTerminalVo> pageResult = ApiResultUtil.objResult(mdmTerminalFeign.page(vo), true);
        List<MdmTerminalVo> mdmTerminalVos = pageResult.getData();
        List<SfaClientData> sfaClientDataList = mdmTerminalVos.stream().map(v -> buildTerminalClientData(v)).collect(Collectors.toList());
        sfaClientDataPageResult.setData(sfaClientDataList);
        sfaClientDataPageResult.setCount(pageResult.getCount());
        return sfaClientDataPageResult;
    }

    /**
     * 查不到客户会抛异常
     * 加载客户信息
     *
     * @param clientType
     * @param clientCode
     * @return
     */
    public static SfaClientData loadClientData(String clientType, String clientCode) {
        if (StringUtils.isEmpty(clientType)) {
            throw new BusinessException("客户类型为空");
        }


        //加载客户信息
        if (SfaVisitEnum.ClientType.DEALER.getVal().equals(clientType)) {
            return loadDealerClientDataRequired(clientCode);
        }
        return loadTerminalClientDataRequired(clientCode);
    }

    /**
     * 查不到客户返回null
     * 加载客户信息
     *
     * @param clientType
     * @param clientCode
     * @return
     */
    public static SfaClientData loadClientDataNotRequired(String clientType, String clientCode) {
        if (StringUtils.isEmpty(clientType)) {
            throw new BusinessException("客户类型为空");
        }


        //加载客户信息
        if (SfaVisitEnum.ClientType.DEALER.getVal().equals(clientType)) {
            return loadDealerClientDataNotRequired(clientCode);
        }
        return loadTerminalClientDataNotRequired(clientCode);
    }


    /**
     * 加载客户信息
     *
     * @return
     */
    public static List<SfaClientData> loadClientDataList(List<String> customerCodes, List<String> terminalCodes) {
        List<SfaClientData> list = loadDealerClientDataList(customerCodes);
        List<SfaClientData> terminalList = loadTerminalClientDataList(terminalCodes);
        if (CollectionUtil.listNotEmptyNotSizeZero(terminalList)) {
            list.addAll(terminalList);
        }
        return list;
    }

    /**
     * 加载客户信息
     *
     * @return
     */
    public static Map<String, SfaClientData> loadClientDataMap(List<String> customerCodes, List<String> terminalCodes) {
        List<SfaClientData> list = loadDealerClientDataList(customerCodes);
        List<SfaClientData> terminalList = loadTerminalClientDataList(terminalCodes);
        if (CollectionUtil.listNotEmptyNotSizeZero(terminalList)) {
            list.addAll(terminalList);
        }
        Map<String, SfaClientData> map = list.stream().collect(Collectors.toMap(v ->
                        v.getClientType() + v.getClientCode()
                , v -> v, (o, o2) -> o2));
        return map;
    }

    /**
     * 请不要改成 public ,否则会提高后期扩展性优化的成本
     * 加载经销商客户信息
     *
     * @return
     */
    private static List<SfaClientData> loadDealerClientDataList(List<String> customerCodes) {
        if (CollectionUtils.isEmpty(customerCodes)) {
            return Lists.newArrayList();
        }
        Result<List<MdmCustomerMsgContactPageRespVo>> customerResult = mdmCustomerMsgFeign.sfaFindCustomerContactPage(customerCodes);
        List<MdmCustomerMsgContactPageRespVo> list = ApiResultUtil.objResult(customerResult, true);
        return list.stream().map(v -> {
            SfaClientData sfaClientData = new SfaClientData();
            sfaClientData.setClientCode(v.getCustomerCode());

            sfaClientData.setClientType(SfaVisitEnum.ClientType.DEALER.getVal());
            sfaClientData.setClientTypeName(SfaVisitEnum.ClientType.DEALER.getDesc());
            sfaClientData.setClientAddress(v.getRegisteredAddress());
            sfaClientData.setClientPhone(v.getContactPhone());
            sfaClientData.setClientContacts(v.getContactName());
            if (StringUtils.isNotEmpty(v.getLongitude()) && StringUtils.isNotEmpty(v.getLatitude())) {
                sfaClientData.setLongitude(new BigDecimal(v.getLongitude().trim()));
                sfaClientData.setLatitude(new BigDecimal(v.getLatitude().trim()));
            }
            if (org.apache.commons.lang3.StringUtils.isBlank(v.getCustomerType())) {
                //todo 设置默认客户细类，表单也要支持
//                throw new BusinessException("MDM主数据错误，经销商[" + v.getCustomerCode() + "]客户类型未指定，请前往MDM数据中心完善相关数据");
            }
            sfaClientData.setClientName(v.getCustomerName());
            sfaClientData.setClientSubclass(v.getCustomerType());
            sfaClientData.setClientSubclassName(v.getCustomerTypeName());
            return sfaClientData;
        }).collect(Collectors.toList());


    }


    /**
     * 请不要改成 public ,否则会提高后期扩展性优化的成本
     * 加载终端客户信息
     *
     * @return
     */
    private static List<SfaClientData> loadTerminalClientDataList(List<String> terminalCodes) {
        if (CollectionUtils.isEmpty(terminalCodes)) {
            return Lists.newArrayList();
        }
        Result<List<MdmTerminalContactPageVo>> terminalResult = mdmTerminalFeign.sfaFindTerminalContactPage(terminalCodes);
        List<MdmTerminalContactPageVo> list = ApiResultUtil.objResult(terminalResult, true);
        return list.stream().map(v -> {
            SfaClientData sfaClientData = new SfaClientData();
            sfaClientData.setClientCode(v.getTerminalCode());

            sfaClientData.setClientType(SfaVisitEnum.ClientType.TERMINAL.getVal());
            sfaClientData.setClientTypeName(SfaVisitEnum.ClientType.TERMINAL.getDesc());

            sfaClientData.setClientAddress(v.getTerminalAddress());

            sfaClientData.setClientPhone(v.getContactPhone());
            sfaClientData.setClientContacts(v.getContactName());
            sfaClientData.setClientPhoto(v.getShopImagePath());
            if (StringUtils.isNotEmpty(v.getLongitude()) && StringUtils.isNotEmpty(v.getLatitude())) {
                sfaClientData.setLongitude(new BigDecimal(v.getLongitude().trim()));
                sfaClientData.setLatitude(new BigDecimal(v.getLatitude().trim()));
            }
            if (org.apache.commons.lang3.StringUtils.isBlank(v.getTerminalType())) {
                //todo 设置默认客户细类，表单也要支持
//                throw new BusinessException("MDM主数据错误，终端[" + v.getTerminalCode() + "]类型信息未指定，请前往MDM数据中心完善相关数据");
            }
            sfaClientData.setClientName(v.getTerminalName());
            sfaClientData.setClientSubclass(v.getTerminalType());
            sfaClientData.setClientSubclassName(v.getTerminalTypeName());
            return sfaClientData;
        }).collect(Collectors.toList());


    }


    /**
     * 查不到客户会抛异常
     * 请不要改成 public ,否则会提高后期扩展性优化的成本
     * 加载经销商客户信息
     *
     * @param clientCode
     * @return
     */
    private static SfaClientData loadDealerClientDataRequired(String clientCode) {
        if (StringUtils.isEmpty(clientCode)) {
            throw new BusinessException("客户编码为空");
        }
        //加载客户信息
        MdmCustomerMsgRespVo client = ApiResultUtil.objResult(mdmCustomerMsgFeign.query(null, clientCode), true);
        if (null == client) {
            throw new BusinessException("不存在的客户[" + clientCode + "]");
        }
        return buildDealerClientData(client);
    }

    /**
     * 查不到客户返回null
     * 请不要改成 public ,否则会提高后期扩展性优化的成本
     * 加载经销商客户信息
     *
     * @param clientCode
     * @return
     */
    private static SfaClientData loadDealerClientDataNotRequired(String clientCode) {
        if (StringUtils.isEmpty(clientCode)) {
            throw new BusinessException("客户编码为空");
        }
        //加载客户信息
        MdmCustomerMsgRespVo client = ApiResultUtil.objResult(mdmCustomerMsgFeign.query(null, clientCode), true);
        if (null == client) {
            return null;
        }
        return buildDealerClientData(client);
    }

    private static SfaClientData buildDealerClientData(MdmCustomerMsgRespVo client) {
        SfaClientData sfaClientData = CrmBeanUtil.copy(client, SfaClientData.class);
        sfaClientData.setClientCode(client.getCustomerCode());

        sfaClientData.setClientType(SfaVisitEnum.ClientType.DEALER.getVal());
        sfaClientData.setClientTypeName(SfaVisitEnum.ClientType.DEALER.getDesc());


        sfaClientData.setClientAddress(client.getRegisteredAddress());
        List<MdmCustomerContactRespVo> mdmCustomerContactVos = client.getMdmCustomerContactVos();
        if (!CollectionUtils.isEmpty(mdmCustomerContactVos)) {
            sfaClientData.setClientPhone(mdmCustomerContactVos.get(0).getContactPhone());
            sfaClientData.setClientContacts(mdmCustomerContactVos.get(0).getContactName());
        }
        if (StringUtils.isNotEmpty(client.getLongitude()) && StringUtils.isNotEmpty(client.getLatitude())) {
            sfaClientData.setLongitude(new BigDecimal(client.getLongitude().trim()));
            sfaClientData.setLatitude(new BigDecimal(client.getLatitude().trim()));
        }
        sfaClientData.setClientName(client.getCustomerName());
        if (org.apache.commons.lang3.StringUtils.isBlank(client.getCustomerType())) {
            //todo 设置默认客户细类，表单也要支持
//            throw new BusinessException("MDM主数据错误，经销商[" + client.getCustomerCode() + "]客户类型未指定，请前往MDM数据中心完善相关数据");
        }
        sfaClientData.setClientSubclass(client.getCustomerType());
        sfaClientData.setClientSubclassName(client.getCustomerTypeName());
        return sfaClientData;
    }


    /**
     * 查不到客户会抛异常
     * 请不要改成 public ,否则会提高后期扩展性优化的成本
     * 加载终端客户信息
     *
     * @param clientCode
     * @return
     */
    private static SfaClientData loadTerminalClientDataRequired(String clientCode) {
        if (StringUtils.isEmpty(clientCode)) {
            throw new BusinessException("客户编码为空");
        }
        //加载客户信息
        MdmTerminalVo client = ApiResultUtil.objResult(mdmTerminalFeign.query(null, clientCode), true);
        if (null == client) {
            throw new BusinessException("不存在的客户[" + clientCode + "]");
        }
        return buildTerminalClientData(client);
    }

    /**
     * 查不到客户返回null
     * 请不要改成 public ,否则会提高后期扩展性优化的成本
     * 加载终端客户信息
     *
     * @param clientCode
     * @return
     */
    private static SfaClientData loadTerminalClientDataNotRequired(String clientCode) {
        if (StringUtils.isEmpty(clientCode)) {
            throw new BusinessException("客户编码为空");
        }
        //加载客户信息
        MdmTerminalVo client = ApiResultUtil.objResult(mdmTerminalFeign.query(null, clientCode), true);
        if (null == client) {
            return null;
        }
        return buildTerminalClientData(client);
    }

    private static SfaClientData buildTerminalClientData(MdmTerminalVo client) {
        SfaClientData sfaClientData = CrmBeanUtil.copy(client, SfaClientData.class);
        sfaClientData.setClientCode(client.getTerminalCode());
        sfaClientData.setClientType(SfaVisitEnum.ClientType.TERMINAL.getVal());
        sfaClientData.setClientTypeName(SfaVisitEnum.ClientType.TERMINAL.getDesc());

        sfaClientData.setClientAddress(client.getTerminalAddress());

        List<MdmTerminalContactVo> mdmTerminalContactVos = client.getContacts();
        if (!CollectionUtils.isEmpty(mdmTerminalContactVos)) {
            sfaClientData.setClientPhone(mdmTerminalContactVos.get(0).getContactPhone());
            sfaClientData.setClientContacts(mdmTerminalContactVos.get(0).getContactName());
        }
        sfaClientData.setClientPhoto(client.getShopImagePath());
        if (StringUtils.isNotEmpty(client.getLongitude()) && StringUtils.isNotEmpty(client.getLatitude())) {
            sfaClientData.setLongitude(new BigDecimal(client.getLongitude().trim()));
            sfaClientData.setLatitude(new BigDecimal(client.getLatitude().trim()));
        }
        if (org.apache.commons.lang3.StringUtils.isBlank(client.getTerminalType())) {
            //todo 设置默认客户细类，表单也要支持
//            throw new BusinessException("MDM主数据错误，终端[" + client.getTerminalCode() + "]类型信息未指定，请前往MDM数据中心完善相关数据");
        }
        sfaClientData.setClientName(client.getTerminalName());
        sfaClientData.setClientSubclass(client.getTerminalType());
        sfaClientData.setClientSubclassName(client.getTerminalTypeName());
        return sfaClientData;
    }

}
