package com.biz.crm.cache.util;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.biz.crm.base.BusinessException;
import com.biz.crm.calculatesalary.service.ISfaCalculateSalaryDateService;
import com.biz.crm.eunm.CrmDelFlagEnum;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.eunm.sfa.WorkSignEnum;
import com.biz.crm.mdm.org.MdmOrgRedisFeign;
import com.biz.crm.nebular.mdm.org.resp.MdmOrgParentChildrenRedisVo;
import com.biz.crm.nebular.mdm.position.resp.MdmPositionRespVo;
import com.biz.crm.nebular.sfa.worksignrule.req.SfaWorkSignRuleReqVo;
import com.biz.crm.nebular.sfa.worksignrule.resp.SfaWorkSignRuleRespVo;
import com.biz.crm.service.RedisService;
import com.biz.crm.util.CollectionUtil;
import com.biz.crm.util.CommonConstant;
import com.biz.crm.util.DateUtil;
import com.biz.crm.util.StringUtils;
import com.biz.crm.worksignrule.mapper.SfaWorkSignRuleMapper;
import com.biz.crm.worksignrule.model.SfaWorkSignPersonnelEntity;
import com.biz.crm.worksignrule.model.SfaWorkSignRuleEntity;
import com.biz.crm.worksignrule.model.SfaWorkSignSpecialEntity;
import com.biz.crm.worksignrule.service.ISfaWorkSignPersonnelService;
import com.biz.crm.worksignrule.service.ISfaWorkSignRuleService;
import com.biz.crm.worksignrule.service.ISfaWorkSignSpecialService;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.List;

/**
 * @author jianglong
 * @version V1.0
 * @Package com.biz.crm.worksignrule.util
 * @Description: TODO
 * @date 2021/4/6 下午9:13
 */
@Component
public class SfaWorkSignRuleUtil {
    //考勤规则缓存key
    public static final String redisKey = "SFA_WORK_SIGN_RULE:v1:";
    @Autowired
    private RedisService redisService;
    @Resource
    private SfaWorkSignRuleMapper sfaWorkSignRuleMapper;
    @Resource
    private ISfaWorkSignRuleService iSfaWorkSignRuleService;
    @Resource
    private ISfaWorkSignPersonnelService iSfaWorkSignPersonnelService;
    @Resource
    private MdmOrgRedisFeign mdmOrgRedisFeign;
    @Resource
    private ISfaWorkSignSpecialService iSfaWorkSignSpecialService;
    @Resource
    private ISfaCalculateSalaryDateService dateService;

    /**
     * 查询全部已生效的考勤规则
     * @return
     */
    public List<SfaWorkSignRuleEntity> getAllRule(){
        // 查询全部已生效的考勤规则
        QueryWrapper<SfaWorkSignRuleEntity> wrappers = new QueryWrapper<>();
        wrappers.eq("enable_status",CrmEnableStatusEnum.ENABLE.getCode());
        wrappers.eq("del_flag", CrmDelFlagEnum.NORMAL.getCode());
        List<SfaWorkSignRuleEntity> rules = sfaWorkSignRuleMapper.selectList(wrappers);
        rules = rules==null?new ArrayList<>():rules;
        return rules;
    }

    /**
     * 是否设置职位级别
     * @param entity
     * @return
     */
    public boolean checkPositionLevel(SfaWorkSignRuleEntity entity){
        if(entity == null) {
            return false;
        }
        List<SfaWorkSignPersonnelEntity> entities = iSfaWorkSignPersonnelService.findByRuleCodeAndCodeType(entity.getRuleCode(), WorkSignEnum.codeType.POS_LEVEL.getVal());
        return CollectionUtil.listNotEmpty(entities);
    }

    //判断当前用户所有职位对应 职位级别是否存在规则中
    public boolean checkPositionLevel(SfaWorkSignRuleEntity entity, List<MdmPositionRespVo> mdmPositionRespVos){
        if(entity == null) {
            return false;
        }
        List<SfaWorkSignPersonnelEntity> entities = iSfaWorkSignPersonnelService.findByRuleCodeAndCodeType(entity.getRuleCode(), WorkSignEnum.codeType.POS_LEVEL.getVal());
        //如果没有获取到职级范围，则说明不需要校验职级，返回成功
        if(CollectionUtil.listEmpty(entities)) {
            return true;
        }
        Set<String> positionLevelSet = mdmPositionRespVos.stream().map(MdmPositionRespVo::getPositionLevelCode).collect(Collectors.toSet());
        entities = entities.stream().filter(en -> positionLevelSet.contains(en.getWspCode())).collect(Collectors.toList());
        //如果有一条考勤范围记录的职级编码能匹配上一条传入的职级编码，则返回true，反之为false
        return CollectionUtil.listNotEmpty(entities);
    }

    //判断当前用户所有职位对应组织是否存在规则中
    public boolean checkPositionOrg(SfaWorkSignRuleEntity entity, List<MdmPositionRespVo> mdmPositionRespVos){
        if(entity == null) {
            return false;
        }
        List<SfaWorkSignPersonnelEntity> entities = iSfaWorkSignPersonnelService.findByRuleCodeAndCodeType(entity.getRuleCode(), WorkSignEnum.codeType.ORG.getVal());
        //如果没有获取到组织维度的考勤范围，说明数据异常，校验不通过
        if(CollectionUtil.listEmpty(entities)) {
            return false;
        }
        boolean flag = false;
        Set<String> orgCodeSet = mdmPositionRespVos.stream().map(MdmPositionRespVo::getOrgCode).collect(Collectors.toSet());
        //根据已有组织，查出所有下级组织
        List<MdmOrgParentChildrenRedisVo> mdmOrgParentChildrenRedisVos = mdmOrgRedisFeign.findRedisOrgChildrenListByCodeList(Lists.newArrayList(orgCodeSet)).getResult();
        if(mdmOrgParentChildrenRedisVos == null) {
            mdmOrgParentChildrenRedisVos = Lists.newArrayList();
        }
        Set<String> childrenOrgCodes = CollectionUtil.listEmpty(mdmOrgParentChildrenRedisVos) ? Sets.newHashSet()
                : mdmOrgParentChildrenRedisVos.stream().map(MdmOrgParentChildrenRedisVo::getOrgCode).collect(Collectors.toSet());
        for(SfaWorkSignPersonnelEntity en : entities) {
            //是否当前组织为当前组织
            if(Objects.equals(en.getCurrentAble(), CommonConstant.GLOBAL.YesOrNo.Y.getItemCode())) {
                if(orgCodeSet.contains(en.getWspCode())) {
                    flag = true;
                    break;
                }
            }
            //是否当前组织为当前组织及下级组织
            else {
                if(childrenOrgCodes.contains(en.getWspCode())) {
                    flag = true;
                    break;
                }
            }
        }
        return flag;
    }
    //检查今天是否不用打卡3个判断
    public boolean checkMust(SfaWorkSignRuleEntity entity, String executeDate){
        if(entity == null) {
            return false;
        }
        //校验特殊日期
        SfaWorkSignSpecialEntity specialEntity = iSfaWorkSignSpecialService.findByRuleCodeAndWssDate(entity.getRuleCode(), executeDate);
        //如果存在特殊日期，并且该特殊日期类型为‘必须打卡’，则返回true
        if(specialEntity != null && Objects.equals(specialEntity.getWssType(), YesNoEnum.yesNoEnum.ONE.getValue())) {
            return true;
        }
        //如果存在特殊日期，并且特殊日期类型为‘不打卡’，则返回false
        else if (specialEntity != null && Objects.equals(specialEntity.getWssType(), YesNoEnum.yesNoEnum.TWO.getValue())) {
            return false;
        }
        //判断是否节假日，如果是节假日，则返回false
        boolean holiday = dateService.isHoliday(executeDate);
        if(holiday) {
            return false;
        }
        //判断是否工作日，如果时工作日，返回true
        if(StringUtils.isNotEmpty(entity.getWorkingDay()) && entity.getWorkingDay().contains(DateUtil.getWeek(executeDate))) {
            return true;
        }
        return false;
    }

    //获取编码
    public String getKey(String ruleCode){
       if(StringUtils.isEmpty(ruleCode)){
           throw new BusinessException("规则编码不能为空");
       }
       return redisKey+ruleCode;
    }
    //删除缓存
    public void deleteKey(String ruleCode){
        redisService.del(getKey(ruleCode));
    }
    //设置缓存
    public void setObj(SfaWorkSignRuleRespVo sfaWorkSignRuleRespVo){
        String key = getKey(sfaWorkSignRuleRespVo.getRuleCode());
        redisService.set(key,sfaWorkSignRuleRespVo);
    }
    //获取对象
    public SfaWorkSignRuleRespVo getObj(String ruleCode){
        Object o = redisService.get(getKey(ruleCode));
        if(o==null){
            return compensate(ruleCode);
        }
        SfaWorkSignRuleRespVo sfaWorkSignRuleRespVo = (SfaWorkSignRuleRespVo)o;
        if(StringUtils.isEmpty(sfaWorkSignRuleRespVo.getRuleCode())){
            return null;
        }
        return sfaWorkSignRuleRespVo;
    }

    /**
     * 缓存丢失补偿  数据库如果没有设置空对象 编码默认空字符串
     */
    public SfaWorkSignRuleRespVo compensate(String ruleCode){
        if(StringUtils.isEmpty(ruleCode)) {
            this.redisService.set(this.getKey(new String()), null);
        }
        QueryWrapper<SfaWorkSignRuleEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("rule_code",ruleCode);
        SfaWorkSignRuleEntity entity = iSfaWorkSignRuleService.getOne(queryWrapper);
        SfaWorkSignRuleReqVo param = new SfaWorkSignRuleReqVo();
        param.setId(entity.getId());
        SfaWorkSignRuleRespVo vo = iSfaWorkSignRuleService.query(param);
        if(vo == null) {
            this.redisService.set(this.getKey(ruleCode), null);
        }
        this.setObj(vo);
        return vo;
    }

}
