package com.biz.crm.calculatesalary.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.base.BusinessException;
import com.biz.crm.calculatesalary.mapper.SfaCalculateSalaryDateMapper;
import com.biz.crm.calculatesalary.mapper.SfaCalculateSalaryYearMapper;
import com.biz.crm.calculatesalary.model.SfaCalculateSalaryDateEntity;
import com.biz.crm.calculatesalary.model.SfaCalculateSalaryYearEntity;
import com.biz.crm.calculatesalary.service.ISfaCalculateSalaryDateService;
import com.biz.crm.calculatesalary.service.ISfaCalculateSalaryYearService;
import com.biz.crm.common.PageResult;
import com.biz.crm.eunm.CrmDelFlagEnum;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.eunm.sfa.SfaCommonEnum;
import com.biz.crm.nebular.sfa.calculateSalary.req.SfaCalculateSalaryDateReqVo;
import com.biz.crm.nebular.sfa.calculateSalary.req.SfaCalculateSalaryYearReqVo;
import com.biz.crm.nebular.sfa.calculateSalary.req.SfaSalaryDateReqVo;
import com.biz.crm.nebular.sfa.calculateSalary.resp.SfaCalculateSalaryDateRespVo;
import com.biz.crm.nebular.sfa.calculateSalary.resp.SfaCalculateSalaryYearRespVo;
import com.biz.crm.nebular.sfa.calculateSalary.resp.SfaSalaryDateRespVo;
import com.biz.crm.util.*;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.time.LocalDate;
import java.time.Period;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 计算薪资年 接口实现
 *
 * @author lf
 * @date 2020-12-28 15:00:09
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name="SfaCalculateSalaryYearServiceExpandImpl")
public class SfaCalculateSalaryYearServiceImpl<M extends BaseMapper<T>,T> extends ServiceImpl<SfaCalculateSalaryYearMapper, SfaCalculateSalaryYearEntity> implements ISfaCalculateSalaryYearService {

    @Resource
    private SfaCalculateSalaryYearMapper sfaCalculateSalaryYearMapper;
    @Resource
    private ISfaCalculateSalaryDateService sfaCalculateSalaryDateService;
    @Resource
    private SfaCalculateSalaryDateMapper sfaCalculateSalaryDateMapper;

    /**
     * 列表
     * @param reqVo
     * @return
     */
    @Override
    public PageResult<SfaCalculateSalaryYearRespVo> findList(SfaCalculateSalaryYearReqVo reqVo){
        Page<SfaCalculateSalaryYearRespVo> page = new Page<>(reqVo.getPageNum(), reqVo.getPageSize());
        List<SfaCalculateSalaryYearRespVo> list = sfaCalculateSalaryYearMapper.findList(page, reqVo);
        return PageResult.<SfaCalculateSalaryYearRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    /**
     * 通过id查询详情
     * @param id
     * @return
     */
    @Override
    public SfaCalculateSalaryYearRespVo queryDetailById(String id) {
        SfaCalculateSalaryYearEntity yearEntity = this.lambdaQuery()
                .eq(SfaCalculateSalaryYearEntity::getId,id)
                .eq(SfaCalculateSalaryYearEntity::getDelFlag,CrmDelFlagEnum.NORMAL.getCode())
                .one();
        SfaCalculateSalaryYearRespVo yearRespVo = CrmBeanUtil.copy(yearEntity,SfaCalculateSalaryYearRespVo.class);
        List<SfaCalculateSalaryDateRespVo> sfaCalculateSalaryDateRespVos = sfaCalculateSalaryDateMapper.selectListByCondition(new SfaCalculateSalaryDateReqVo(){{
            this.setYear(yearRespVo.getYear());
        }});
        List<SfaSalaryDateRespVo> sfaSalaryDateRespVoList = new ArrayList<>();
        //分组
        Map<String,List<SfaCalculateSalaryDateRespVo>> salaryDateMap = sfaCalculateSalaryDateRespVos.stream().collect(Collectors.groupingBy(SfaCalculateSalaryDateRespVo::getSalary));
        for (Map.Entry<String,List<SfaCalculateSalaryDateRespVo>> map:salaryDateMap.entrySet()) {
            List<SfaCalculateSalaryDateRespVo> salaryDateRespVoList = map.getValue().stream().
                    sorted(Comparator.comparing(SfaCalculateSalaryDateRespVo::getDay)).collect(Collectors.toList());
            salaryDateRespVoList.forEach(data->{
                data.setIsCalculationSalaryDesc(SfaCommonEnum.whether.GETMAP.get(data.getIsCalculationSalary()));
            });
            SfaSalaryDateRespVo sfaSalaryDateRespVo = new SfaSalaryDateRespVo(){{
                this.setSalary(map.getKey());
                this.setSalaryDaysList(salaryDateRespVoList);
            }};
            sfaSalaryDateRespVo.setSalaryStartDate(salaryDateRespVoList.get(0).getDay());
            sfaSalaryDateRespVo.setSalaryEndDate(salaryDateRespVoList.get(salaryDateRespVoList.size()-1).getDay());
            sfaSalaryDateRespVoList.add(sfaSalaryDateRespVo);
        }
        yearRespVo.setSalaryDateRespVoList(sfaSalaryDateRespVoList);
        return yearRespVo;
    }



    /**
     * 查询
     * @param reqVo
     * @return sfaCalculateSalaryYearRespVo
     */
    @Override
    public SfaCalculateSalaryYearRespVo query(SfaCalculateSalaryYearReqVo reqVo){
        return null;
    }

    /**
     * 新增
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void save(SfaCalculateSalaryYearReqVo reqVo){
        UserRedis userRedis = UserUtils.getUser();
        reqVo.setUserName(userRedis.getUsername());
        reqVo.setRealName(userRedis.getRealname());
        reqVo.setOrgCode(userRedis.getOrgcode());
        reqVo.setOrgName(userRedis.getOrgname());
        reqVo.setPosCode(userRedis.getPoscode());
        reqVo.setPosName(userRedis.getPosname());
        SfaCalculateSalaryYearEntity sfaCalculateSalaryYearEntity = this.lambdaQuery()
                .eq(SfaCalculateSalaryYearEntity::getYear,reqVo.getYear())
                .eq(SfaCalculateSalaryYearEntity::getDelFlag,CrmDelFlagEnum.NORMAL.getCode()).one();
        if (sfaCalculateSalaryYearEntity != null){
            throw new BusinessException("已存在计算薪资的年份");
        }
        //校验参数信息
        this.checkParams(reqVo);
        //组装年份参数信息
        this.assembleYearParam(reqVo);
        SfaCalculateSalaryYearEntity entity = CrmBeanUtil.copy(reqVo,SfaCalculateSalaryYearEntity.class);
        entity.setDelFlag(CrmDelFlagEnum.NORMAL.getCode());
        entity.setEnableStatus(CrmEnableStatusEnum.DISABLE.getCode());
        this.save(entity);
        List<SfaCalculateSalaryDateEntity> calculateSalaryDateEntityList = this.assembleDateParam(reqVo.getSfaSalaryDateReqVoList(),entity.getYear());
        sfaCalculateSalaryDateService.saveBatch(calculateSalaryDateEntityList);
    }

    /**
     * 校验参数信息
     * @param reqVo
     */
    private void checkParams(SfaCalculateSalaryYearReqVo reqVo){
        AssertUtils.isNotEmpty(reqVo.getYear(),"年份不能为空");
        AssertUtils.isNotEmpty(reqVo.getSfaSalaryDateReqVoList(),"节假日不能为空");
        Map<String,String> dayMap = new HashMap<>();
        reqVo.getSfaSalaryDateReqVoList().forEach(data->{
            AssertUtils.isNotEmpty(data.getSalary(),"节假日名称不能为空");
            AssertUtils.isNotEmpty(data.getSalaryStartDate(),"节假日开始时间不能为空");
            AssertUtils.isNotEmpty(data.getSalaryEndDate(),"节假日结束时间不能为空");
            AssertUtils.isNotEmpty(data.getSalaryDaysList(),"节假日时间不能为空");
            LocalDate startDate = null;
            LocalDate endDate = null;
            try {
                startDate = LocalDate.parse(data.getSalaryStartDate(),CrmDateUtils.yyyyMMdd);
                endDate = LocalDate.parse(data.getSalaryEndDate(),CrmDateUtils.yyyyMMdd);
            }catch (Exception e){
                throw new BusinessException("日期格式转换错误");
            }
            //计算时间
            long days =  (endDate.toEpochDay() - startDate.toEpochDay()) + 1;
            if (days != data.getSalaryDaysList().size()){
                throw new BusinessException("选择节假日区间不等于节假日天数");
            }
            data.getSalaryDaysList().forEach(da->{
                if (dayMap.containsKey(da.getDay())){
                    throw new BusinessException("存在重复的节假日期");
                }else {
                    dayMap.put(da.getDay(),da.getIsCalculationSalary());
                }
                AssertUtils.isNotEmpty(da.getDay(),"节假日不能为空");
                AssertUtils.isNotEmpty(da.getIsCalculationSalary(),"是否计算薪资选择不能为空");
            });
        });

    }

    /**
     * 组装年份参数
     * @param reqVo
     */
    private void assembleYearParam(SfaCalculateSalaryYearReqVo reqVo){
        //法定节假日天数
        Integer holidayDays = 0;
        //计算薪资天数
        Long salaryDays = 0l;
        for (SfaSalaryDateReqVo data : reqVo.getSfaSalaryDateReqVoList()) {
            holidayDays = holidayDays+data.getSalaryDaysList().size();
            //统计计算薪资天数
            Long num = data.getSalaryDaysList().stream().filter(da->da.getIsCalculationSalary().equals(SfaCommonEnum.whether.YES.getValue())).count();
            salaryDays = salaryDays + num;
        }
        reqVo.setSalaryDays(salaryDays.toString());
        reqVo.setHolidayDays(holidayDays.toString());

    }

    /**
     * 组装日期参数
     * @param sfaSalaryDateReqVoList
     * @param year
     */
    private List<SfaCalculateSalaryDateEntity> assembleDateParam(List<SfaSalaryDateReqVo> sfaSalaryDateReqVoList, String year){
        UserRedis user = UserUtils.getUser();
        List<SfaCalculateSalaryDateEntity> salaryDateEntityList = new ArrayList<>();
        sfaSalaryDateReqVoList.forEach(data->{
            data.getSalaryDaysList().forEach(da->{
                da.setSalary(data.getSalary());
                da.setYear(year);
                da.setUserName(user.getUsername());
                da.setRealName(user.getRealname());
                da.setOrgCode(user.getOrgcode());
                da.setOrgName(user.getOrgname());
                da.setPosCode(user.getPoscode());
                da.setPosName(user.getPosname());
                da.setDelFlag(CrmDelFlagEnum.NORMAL.getCode());
                da.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
                SfaCalculateSalaryDateEntity entity = CrmBeanUtil.copy(da,SfaCalculateSalaryDateEntity.class);
                salaryDateEntityList.add(entity);
            });
        });
        return salaryDateEntityList;
    }

    /**
     * 更新
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(SfaCalculateSalaryYearReqVo reqVo){
        AssertUtils.isNotEmpty(reqVo.getId(),"主键id不能为空");
        //校验参数信息
        this.checkParams(reqVo);
        //组装年份参数信息
        this.assembleYearParam(reqVo);
        SfaCalculateSalaryYearEntity entity = this.getById(reqVo.getId());
        entity.setSalaryDays(reqVo.getSalaryDays());
        entity.setHolidayDays(reqVo.getHolidayDays());
        this.updateById(entity);
        sfaCalculateSalaryDateMapper.deleteSalaryDateByYear(reqVo.getYear());
        List<SfaCalculateSalaryDateEntity> calculateSalaryDateEntityList = this.assembleDateParam(reqVo.getSfaSalaryDateReqVoList(),entity.getYear());
        sfaCalculateSalaryDateService.saveBatch(calculateSalaryDateEntityList);
    }

    /**
     * 删除
     * @param ids
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteBatch(List<String> ids){
        if (!CollectionUtils.isNotEmpty(ids)){
            throw new BusinessException("数据主键不能为空");
        }
        List<SfaCalculateSalaryYearEntity> sfaCalculateSalaryYearEntities = sfaCalculateSalaryYearMapper.selectBatchIds(ids);

        if(CollectionUtils.isNotEmpty(sfaCalculateSalaryYearEntities)){
            List<String> yearIdList = sfaCalculateSalaryYearEntities.stream().map(SfaCalculateSalaryYearEntity::getId).collect(Collectors.toList());
            List<String> yearList = sfaCalculateSalaryYearEntities.stream().map(SfaCalculateSalaryYearEntity::getYear).collect(Collectors.toList());
            List<SfaCalculateSalaryDateEntity> sfaCalculateSalaryDateEntities = sfaCalculateSalaryDateService.lambdaQuery()
                    .in(SfaCalculateSalaryDateEntity::getYear,yearList).list();
            List<String> dateIdList = sfaCalculateSalaryDateEntities.stream().map(SfaCalculateSalaryDateEntity::getId).collect(Collectors.toList());
            this.removeByIds(yearIdList);
            sfaCalculateSalaryDateService.removeByIds(dateIdList);
        }
//        this.updateBatchById(sfaCalculateSalaryYearEntities);
    }

    /**
     * 启用
     * @param ids
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void enableBatch(List<String> ids){
        if (!CollectionUtils.isNotEmpty(ids)){
            throw new BusinessException("数据主键不能为空");
        }
        //设置状态为启用
        List<SfaCalculateSalaryYearEntity> sfaCalculateSalaryYearEntities = sfaCalculateSalaryYearMapper.selectBatchIds(ids);
        if(CollectionUtils.isNotEmpty(sfaCalculateSalaryYearEntities)){
            sfaCalculateSalaryYearEntities.forEach(o -> {
                List<SfaCalculateSalaryYearEntity> entityList = this.lambdaQuery().eq(SfaCalculateSalaryYearEntity::getYear,o.getYear())
                        .eq(SfaCalculateSalaryYearEntity::getEnableStatus,CrmEnableStatusEnum.ENABLE.getCode()).list();
                if (entityList != null && entityList.size()>0){
                    throw new BusinessException("已存在"+o.getYear()+"年计算薪资日期数据");
                }
                o.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
            });
        }
        this.updateBatchById(sfaCalculateSalaryYearEntities);
    }

    /**
     * 禁用
     * @param ids
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void disableBatch(List<String> ids){
        if (!CollectionUtils.isNotEmpty(ids)){
            throw new BusinessException("数据主键不能为空");
        }
        //设置状态为禁用
        List<SfaCalculateSalaryYearEntity> sfaCalculateSalaryYearEntities = sfaCalculateSalaryYearMapper.selectBatchIds(ids);
        if(CollectionUtils.isNotEmpty(sfaCalculateSalaryYearEntities)){
                sfaCalculateSalaryYearEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.DISABLE.getCode());
            });
        }
        this.updateBatchById(sfaCalculateSalaryYearEntities);
    }

    /**
     * 查询年份列表
     * @return
     */
    @Override
    public List<String> getYearList() {
        List<SfaCalculateSalaryYearEntity> entityList = this.lambdaQuery().list();
        List<String> yearList = entityList.stream().map(SfaCalculateSalaryYearEntity::getYear).collect(Collectors.toList());
        return yearList;
    }
}
