package com.biz.crm.checkin.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.base.BusinessException;
import com.biz.crm.checkin.mapper.SfaCheckInGroupMapper;
import com.biz.crm.checkin.model.*;
import com.biz.crm.checkin.service.*;
import com.biz.crm.common.PageResult;
import com.biz.crm.eunm.CrmDelFlagEnum;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.eunm.sfa.VisitStepFromEnum;
import com.biz.crm.nebular.sfa.checkin.req.*;
import com.biz.crm.nebular.sfa.checkin.resp.SfaCheckInGroupRespVo;
import com.biz.crm.nebular.sfa.checkin.resp.SfaCheckInGroupTypeRespVo;
import com.biz.crm.nebular.sfa.checkin.resp.SfaCheckInPlaceRespVo;
import com.biz.crm.util.*;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.time.LocalDate;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 签到组;接口实现
 *
 * @author 任钢
 * @date 2021-01-04 14:18:34
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name="SfaCheckInGroupServiceExpandImpl")
public class SfaCheckInGroupServiceImpl<M extends BaseMapper<T>,T> extends ServiceImpl<SfaCheckInGroupMapper, SfaCheckInGroupEntity> implements ISfaCheckInGroupService {

    @Resource
    private SfaCheckInGroupMapper sfaCheckInGroupMapper;
    @Autowired
    private ISfaCheckInDateService iSfaCheckInDateService;
    @Autowired
    private ISfaCheckInGroupTypeService iSfaCheckInGroupTypeService;
    @Autowired
    private ISfaCheckInTypeService iSfaCheckInTypeService;
    @Autowired
    private ISfaCheckInOrgService iSfaCheckInOrgService;
    @Autowired
    private ISfaCheckInUserService iSfaCheckInUserService;
    @Autowired
    private ISfaCheckInSignGroupReportService iSfaCheckInSignGroupReportService;
    @Resource
    private ISfaCheckInPlaceService iSfaCheckInPlaceService;

    /**
     * 列表
     * @param reqVo
     * @return
     */
    @Override
    public PageResult<SfaCheckInGroupRespVo> findList(SfaCheckInGroupReqVo reqVo){
        Page<SfaCheckInGroupRespVo> page = new Page<>(reqVo.getPageNum(), reqVo.getPageSize());
        List<SfaCheckInGroupRespVo> list = sfaCheckInGroupMapper.findList(page, reqVo);
        if(list != null && list.size() > 0) {
            list.forEach(vo -> {
//                loadingInfo(vo);
                loadingGroupTypeInfo(vo);
                vo.setEnableStatusName(CrmEnableStatusEnum.getDesc(vo.getEnableStatus()));
            });
        }
        return PageResult.<SfaCheckInGroupRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }
    /**
     * 列表
     * @param reqVo
     * @return
     */
    @Override
    public PageResult<SfaCheckInGroupRespVo> myCheckInGroup(SfaCheckInGroupReqVo reqVo){
        Page<SfaCheckInGroupRespVo> page = new Page<>(reqVo.getPageNum(), reqVo.getPageSize());
        List<SfaCheckInGroupRespVo> list = sfaCheckInGroupMapper.findList(page, reqVo);
        if(list != null && list.size() > 0) {
            list.forEach(vo -> {
                this.loadingGroupTypeInfo(vo);
                vo.setEnableStatusName(CrmEnableStatusEnum.getDesc(vo.getEnableStatus()));
            });
        }
        return PageResult.<SfaCheckInGroupRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    /**
     * 查询
     * @param id
     * @return sfaCheckInGroupRespVo
     */
    @Override
    public SfaCheckInGroupRespVo query(String id){
        SfaCheckInGroupEntity entity = sfaCheckInGroupMapper.selectOne(Wrappers.lambdaQuery(SfaCheckInGroupEntity.class).eq(SfaCheckInGroupEntity::getId, id));
        if(entity == null) {
            throw new BusinessException("id错误");
        }

        SfaCheckInGroupRespVo respVo = CrmBeanUtil.copy(entity, SfaCheckInGroupRespVo.class);
        loadingInfo(respVo);
        return respVo;
    }

    /**
     * 根据签到组code获取签到组信息
     * @param groupCode
     * @return
     */
    @Override
    public SfaCheckInGroupRespVo getByGroupCode(String groupCode) {
        SfaCheckInGroupEntity entity = sfaCheckInGroupMapper.selectOne(
                Wrappers.lambdaQuery(SfaCheckInGroupEntity.class).eq(SfaCheckInGroupEntity::getGroupCode, groupCode)
        );
        if(entity == null) {
            throw new BusinessException("编码错误");
        }

        SfaCheckInGroupRespVo respVo = CrmBeanUtil.copy(entity, SfaCheckInGroupRespVo.class);
        loadingInfo(respVo);
        return respVo;
    }

    /**
     * 新增or编辑
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void save(SfaCheckInGroupSaveReqVo reqVo){
        SfaCheckInGroupEntity entity = CrmBeanUtil.copy(reqVo,SfaCheckInGroupEntity.class);
        if(StringUtils.isNotEmpty(reqVo.getId())) {
            SfaCheckInGroupEntity old = sfaCheckInGroupMapper.selectOne(Wrappers.lambdaQuery(SfaCheckInGroupEntity.class).eq(SfaCheckInGroupEntity::getId, reqVo.getId()));
            if(old == null) {
                throw new BusinessException("签到组id错误");
            }
            entity.setGroupCode(old.getGroupCode());
        } else {
            entity.setGroupCode(CodeUtil.getCodeDefault());
        }
        // 详情数据校验
        infoCheck(reqVo);
        // 删除详情信息
        deleteInfo(entity.getGroupCode());
        // 保存
        this.saveOrUpdate(entity);
        // 保存详情信息
        saveInfo(entity.getGroupCode(), reqVo);

        // 初始化报表基础信息
        iSfaCheckInSignGroupReportService.refreshReport(entity.getGroupCode());
    }


    /**
     * 删除
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteBatch(SfaCheckInGroupReqVo reqVo){
        List<SfaCheckInGroupEntity> sfaCheckInGroupEntities = sfaCheckInGroupMapper.selectBatchIds(reqVo.getIds());
        if(CollectionUtils.isNotEmpty(sfaCheckInGroupEntities)){
            sfaCheckInGroupEntities.forEach(o -> {
                 o.setDelFlag(CrmDelFlagEnum.DELETE.getCode());
                 o.setEnableStatus(CrmDelFlagEnum.DELETE.getCode());
            });
        }
        this.updateBatchById(sfaCheckInGroupEntities);
    }

    /**
     * 启用
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void enableBatch(SfaCheckInGroupReqVo reqVo){
        //设置状态为启用
        List<SfaCheckInGroupEntity> sfaCheckInGroupEntities = sfaCheckInGroupMapper.selectBatchIds(reqVo.getIds());
        if(CollectionUtils.isNotEmpty(sfaCheckInGroupEntities)){
            sfaCheckInGroupEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
            });
        }
        this.updateBatchById(sfaCheckInGroupEntities);
    }

    /**
     * 禁用
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void disableBatch(SfaCheckInGroupReqVo reqVo){
        //设置状态为禁用
        List<SfaCheckInGroupEntity> sfaCheckInGroupEntities = sfaCheckInGroupMapper.selectBatchIds(reqVo.getIds());
        if(CollectionUtils.isNotEmpty(sfaCheckInGroupEntities)){
                sfaCheckInGroupEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.DISABLE.getCode());
            });
        }
        this.updateBatchById(sfaCheckInGroupEntities);
    }

    private void loadingGroupTypeInfo(SfaCheckInGroupRespVo vo){
        // 打卡类型
        SfaCheckInGroupTypeReqVo groupTypeReqVo = new SfaCheckInGroupTypeReqVo();
        groupTypeReqVo.setGroupCode(vo.getGroupCode());
        List<SfaCheckInGroupTypeRespVo> groupTypeList = iSfaCheckInGroupTypeService.findList(groupTypeReqVo).getData();
        if(groupTypeList != null && groupTypeList.size() > 0) {
            SfaCheckInTypeReqVo typeReqVo = new SfaCheckInTypeReqVo();
            typeReqVo.setIds(groupTypeList.stream().map(o -> o.getSfaCheckInTypeId()).collect(Collectors.toList()));
            vo.setTypeRespVos(iSfaCheckInTypeService.findList(typeReqVo).getData());
            if(CollectionUtils.isNotEmpty(vo.getTypeRespVos())) {
                String typeNames = vo.getTypeRespVos().stream().map(o -> o.getTypeName()).collect(Collectors.joining(","));
                vo.setTypeNames(typeNames);
            }
        }
    }
    /**
     * 装载详情信息
     * @param vo
     */
    private void loadingInfo(SfaCheckInGroupRespVo vo) {
        // 打卡日期
        SfaCheckInDateReqVo dateReqVo = new SfaCheckInDateReqVo();
        dateReqVo.setGroupCode(vo.getGroupCode());
        vo.setDateRespVos(iSfaCheckInDateService.findList(dateReqVo).getData());
        this.loadingGroupTypeInfo(vo);
        // 打卡组织
        SfaCheckInOrgReqVo orgReqVo = new SfaCheckInOrgReqVo();
        orgReqVo.setGroupCode(vo.getGroupCode());
        vo.setOrgRespVos(iSfaCheckInOrgService.findList(orgReqVo).getData());
        // 打卡人员
        SfaCheckInUserReqVo userReqVo = new SfaCheckInUserReqVo();
        userReqVo.setGroupCode(vo.getGroupCode());
        vo.setUserRespVos(iSfaCheckInUserService.findList(userReqVo).getData());
        //打卡地址
        List<SfaCheckInPlaceEntity> entityList = iSfaCheckInPlaceService.list(Wrappers.lambdaQuery(SfaCheckInPlaceEntity.class).eq(SfaCheckInPlaceEntity::getGroupCode, vo.getGroupCode()));
        List<SfaCheckInPlaceRespVo> placeRespVos = CrmBeanUtil.copyList(entityList, SfaCheckInPlaceRespVo.class);
        vo.setPlaceRespVos(placeRespVos);
    }

    /**
     * 详情数据校验
     * @param reqVo
     */
    private void infoCheck(SfaCheckInGroupSaveReqVo reqVo) {
        AssertUtils.isNotEmpty(reqVo.getGroupName(), "签到组名称不能为空");
        // 非空校验
        AssertUtils.isNotEmpty(reqVo.getTimeType(), "时间类型不能为空");
        AssertUtils.isNotEmpty(reqVo.getGroupTypeReqVos(), "打卡类型不能为空");
        if(CollectionUtils.isEmpty(reqVo.getOrgReqVos()) && CollectionUtils.isEmpty(reqVo.getUserReqVos())) {
            throw new BusinessException("请添加打卡组织或者打卡人员");
        }
        // 打卡日期
        if(VisitStepFromEnum.timeType.DIY_TIME.getValue().equals(reqVo.getTimeType())){
            List<SfaCheckInDateReqVo> dateReqVos = reqVo.getDateReqVos();
            AssertUtils.isNotEmpty(dateReqVos,"时间段不能为空");
            //校验时间段——1.开始时间必定小于等于结束时间。
            dateReqVos.forEach(data->{
                //1.开始时间必定小于等于结束时间
                int compareTo = LocalDate.parse(data.getStartTime()).compareTo(LocalDate.parse(data.getEndTime()));
                if(0<compareTo){
                    throw new BusinessException("开始时间大于结束时间");
                }
            });
        }

        // 打卡类型
        reqVo.getGroupTypeReqVos().forEach(vo -> {
            AssertUtils.isNotEmpty(vo.getSfaCheckInTypeId(), "打卡类型ID不能为空");
            iSfaCheckInTypeService.query(vo.getSfaCheckInTypeId());
        });
        // 打卡组织
        if(CollectionUtils.isNotEmpty(reqVo.getOrgReqVos())) {
            reqVo.getOrgReqVos().forEach(vo -> {
                AssertUtils.isNotEmpty(vo.getOrgCode(), "组织编码不能为空");
                AssertUtils.isNotEmpty(vo.getOrgName(), "组织名称不能为空");
            });
        }
        // 打卡人员
        if(CollectionUtils.isNotEmpty(reqVo.getUserReqVos())) {
            reqVo.getUserReqVos().forEach(vo -> {
                AssertUtils.isNotEmpty(vo.getRealName(), "用户名称不能为空");
                AssertUtils.isNotEmpty(vo.getUserName(), "用户编码不能为空");
            });
        }
        // 打卡地址
        if(CollectionUtils.isNotEmpty(reqVo.getPlaceReqVos())){
            reqVo.getPlaceReqVos().forEach(vo->{
                AssertUtils.isNotEmpty(vo.getPlaceLatin(), "地点纬度不能为空");
                AssertUtils.isNotEmpty(vo.getPlaceLngin(), "地点经度不能为空");
                AssertUtils.isNotEmpty(vo.getAddress(), "地址不能为空");
                AssertUtils.isNotEmpty(vo.getName(), "名称不能为空");
            });
        }
    }

    /**
     * 根据签到组code删除其他信息(物理删除)
     * @param groupCode
     */
    private void deleteInfo(String groupCode) {
        // 打卡日期
        iSfaCheckInDateService.deleteByGroupCode(groupCode);
        // 打卡类型
        iSfaCheckInGroupTypeService.deleteByGroupCode(groupCode);
        // 打卡组织
        iSfaCheckInOrgService.deleteByGroupCode(groupCode);
        // 打卡人员
        iSfaCheckInUserService.deleteByGroupCode(groupCode);
        // 打卡地址
        iSfaCheckInPlaceService.remove(Wrappers.lambdaQuery(SfaCheckInPlaceEntity.class).eq(SfaCheckInPlaceEntity::getGroupCode,groupCode));
    }

    /**
     * 根据签到组code保存其他信息
     * @param groupCode
     * @param reqVo
     */
    private void saveInfo(String groupCode, SfaCheckInGroupSaveReqVo reqVo) {
        List<SfaCheckInDateEntity> listDate = new ArrayList<>();
        // 打卡日期
        if(CollectionUtils.isNotEmpty(reqVo.getDateReqVos())&&VisitStepFromEnum.timeType.DIY_TIME.getValue().equals(reqVo.getTimeType())) {
            listDate = reqVo.getDateReqVos().stream().map(o -> {
                SfaCheckInDateEntity entity = CrmBeanUtil.copy(o, SfaCheckInDateEntity.class);
                entity.setGroupCode(groupCode);
                entity.setTimeType(VisitStepFromEnum.timeType.DIY_TIME.getValue());
                return entity;
            }).collect(Collectors.toList());
        }else if (VisitStepFromEnum.timeType.LONG_TIME.getValue().equals(reqVo.getTimeType())){
            SfaCheckInDateEntity sfaCheckInDateEntity = new SfaCheckInDateEntity();
            sfaCheckInDateEntity.setGroupCode(groupCode);
            sfaCheckInDateEntity.setStartTime(LocalDate.now().format(CrmDateUtils.yyyyMMdd));
            sfaCheckInDateEntity.setTimeType(VisitStepFromEnum.timeType.LONG_TIME.getValue());
            listDate.add(sfaCheckInDateEntity);
        }
        iSfaCheckInDateService.saveOrUpdateBatch(listDate);

        // 打卡类型
        if(CollectionUtils.isNotEmpty(reqVo.getGroupTypeReqVos())) {
            List<SfaCheckInGroupTypeEntity> list = reqVo.getGroupTypeReqVos().stream().map(o -> {
                SfaCheckInGroupTypeEntity entity = CrmBeanUtil.copy(o, SfaCheckInGroupTypeEntity.class);
                entity.setGroupCode(groupCode);
                return entity;
            }).collect(Collectors.toList());
            iSfaCheckInGroupTypeService.saveOrUpdateBatch(list);
        }
        // 打卡组织
        if(CollectionUtils.isNotEmpty(reqVo.getOrgReqVos())) {
            List<SfaCheckInOrgEntity> list = reqVo.getOrgReqVos().stream().map(o -> {
                SfaCheckInOrgEntity entity = CrmBeanUtil.copy(o, SfaCheckInOrgEntity.class);
                entity.setGroupCode(groupCode);
                entity.setId(null);
                return entity;
            }).collect(Collectors.toList());
            iSfaCheckInOrgService.saveOrUpdateBatch(list);
        }
        // 打卡人员
        if(CollectionUtils.isNotEmpty(reqVo.getUserReqVos())) {
            List<SfaCheckInUserEntity> list = reqVo.getUserReqVos().stream().map(o -> {
                SfaCheckInUserEntity entity = CrmBeanUtil.copy(o, SfaCheckInUserEntity.class);
                entity.setGroupCode(groupCode);
                entity.setId(null);
                return entity;
            }).collect(Collectors.toList());
            iSfaCheckInUserService.saveOrUpdateBatch(list);
        }
        // 打卡地址
        if(CollectionUtils.isNotEmpty(reqVo.getPlaceReqVos())){
            List<SfaCheckInPlaceEntity> list = reqVo.getPlaceReqVos().stream().map(o -> {
                SfaCheckInPlaceEntity entity = CrmBeanUtil.copy(o, SfaCheckInPlaceEntity.class);
                entity.setGroupCode(groupCode);
                return entity;
            }).collect(Collectors.toList());
            iSfaCheckInPlaceService.saveOrUpdateBatch(list);
        }
    }
}
