package com.biz.crm.collection.service.impl;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.aop.CrmLog;
import com.biz.crm.base.BusinessException;
import com.biz.crm.collection.mapper.SfaDirectoryConfigMapper;
import com.biz.crm.collection.mapper.SfaDirectoryInputMapper;
import com.biz.crm.collection.model.SfaDirectoryConfigEntity;
import com.biz.crm.collection.model.SfaDirectoryInputEntity;
import com.biz.crm.collection.service.ISfaDirectoryConfigService;
import com.biz.crm.collection.service.ISfaDirectoryInputService;
import com.biz.crm.common.PageResult;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.nebular.sfa.collection.req.SfaDirectoryConfigReqVo;
import com.biz.crm.nebular.sfa.collection.req.SfaDirectoryInputReqVo;
import com.biz.crm.nebular.sfa.collection.resp.SfaDirectoryConfigRespVo;
import com.biz.crm.util.*;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * 拜访步骤表单配置 接口实现
 *
 * @author liuhongming
 * @date 2020-09-18 10:33:32
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name="SfaDirectoryConfigServiceExpandImpl")
public class SfaDirectoryConfigServiceImpl extends ServiceImpl<SfaDirectoryConfigMapper, SfaDirectoryConfigEntity> implements ISfaDirectoryConfigService {

    @Resource
    private SfaDirectoryConfigMapper sfaDirectoryConfigMapper;
    @Resource
    private SfaDirectoryInputMapper sfaDirectoryInputMapper;

    @Autowired
    private ISfaDirectoryInputService iSfaDirectoryInputService;

    /**
     * 列表
     * @param reqVo
     * @return
     */
    @Override
    @CrmLog
    public PageResult<SfaDirectoryConfigRespVo> findList(SfaDirectoryConfigReqVo reqVo){
        Page<SfaDirectoryConfigRespVo> page = new Page<>(reqVo.getPageNum(), reqVo.getPageSize());
        List<SfaDirectoryConfigRespVo> list = sfaDirectoryConfigMapper.findList(page, reqVo);
        return PageResult.<SfaDirectoryConfigRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    /**
     * 查询
     * @param reqVo
     * @return sfaDirectoryConfigRespVo
     */
    @Override
    @CrmLog
    public SfaDirectoryConfigRespVo query(SfaDirectoryConfigReqVo reqVo){
        if (null==reqVo||(StringUtils.isEmpty(reqVo.getId())&& CollectionUtil.listNotEmptyNotSizeZero(reqVo.getIds())&&StringUtils.isEmpty(reqVo.getDirectoryCode()))){
            return new SfaDirectoryConfigRespVo();
        }
        List<SfaDirectoryConfigRespVo> list = this.findList(reqVo).getData();
        if (CollectionUtil.listEmpty(list)){
            return new SfaDirectoryConfigRespVo();
        }
        SfaDirectoryConfigRespVo respVo = list.get(0);
        return respVo;
    }

    /**
     * 新增,编辑,校验
     * @param reqVo
     */
    public void saveCheck(SfaDirectoryConfigReqVo reqVo){
        AssertUtils.isNotEmpty(reqVo.getDirectoryName(),"请输入步骤名称");
        AssertUtils.isNotEmpty(reqVo.getSfacusType(),"请选择客户类型");
        SimpleDateFormat ymd_hms  =new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        AssertUtils.isNotEmpty(reqVo.getBeginDate(),"请选择有效开始时间");
        Date beginDate = null;
        try{
            beginDate = (Date) ymd_hms.parse(reqVo.getBeginDate());
        }catch(Exception e){
            throw new BusinessException("有效开始时间格式错误");
        }
        AssertUtils.isNotEmpty(reqVo.getEndDate(), "有效结束时间不能为空");
        Date endDate = null;
        try{
            endDate = (Date) ymd_hms.parse(reqVo.getEndDate());
        }catch(Exception e){
            throw new BusinessException("有效结束时间格式错误");
        }
        if (!(beginDate.getTime() < endDate.getTime())){
            throw new BusinessException("有效开始时间不能有效大于结束时间");
        }
        if (CollectionUtil.listEmpty(reqVo.getDirectoryInputReqVos())){
            throw new BusinessException("请选择拜访控件");
        }
    }


    /**
     * 新增
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @CrmLog
    public void save(SfaDirectoryConfigReqVo reqVo){
        this.saveCheck(reqVo);
        SfaDirectoryConfigEntity entity = CrmBeanUtil.copy(reqVo,SfaDirectoryConfigEntity.class);
        //TODO 步骤编码
        String code = CodeUtil.createCode();
        entity.setDirectoryCode(code);
        entity.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
        this.save(entity);
        this.saveData(reqVo,entity);
    }

    /**
     * 保存业务明细数据
     * @param reqVo
     * @param entity
     */
    public void saveData(SfaDirectoryConfigReqVo reqVo, SfaDirectoryConfigEntity entity){
        if (CollectionUtil.listNotEmptyNotSizeZero(reqVo.getDirectoryInputReqVos())){
            List<SfaDirectoryInputEntity> directoryInputEntityList = reqVo.getDirectoryInputReqVos().stream().map(o -> {
                SfaDirectoryInputEntity directoryInputEntity = new SfaDirectoryInputEntity();
                CrmBeanUtil.copyProperties(o, directoryInputEntity);
                directoryInputEntity.setDirectoryId(entity.getId());
                directoryInputEntity.setDirectoryCode(entity.getDirectoryCode());
                directoryInputEntity.setDirectoryName(entity.getDirectoryName());
                return directoryInputEntity;
            }).collect(Collectors.toList());
            iSfaDirectoryInputService.saveBatch(directoryInputEntityList);
        }
    }

    /**
     * 更新
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @CrmLog
    public void update(SfaDirectoryConfigReqVo reqVo){
        if(StringUtils.isEmpty(reqVo.getId())){
            throw new BusinessException("数据主键不能为空");
        }
        SfaDirectoryConfigEntity entity = this.getById(reqVo.getId());
        if(Objects.isNull(entity)){
            throw new BusinessException("拜访步骤表单配置数据不存在");
        }
        this.saveCheck(reqVo);
        CrmBeanUtil.copyPropertiesIgnoreEmpty(reqVo,entity);
        this.updateById(entity);
        this.deleteBatchDate(reqVo);
        this.saveData(reqVo,entity);
    }

    /**
     * 删除
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @CrmLog
    public void deleteBatch(SfaDirectoryConfigReqVo reqVo){
        if(StringUtils.isEmpty(reqVo.getId())&&CollectionUtil.listEmpty(reqVo.getIds())){
            throw new BusinessException("数据主键不能为空");
        }
        reqVo.setPageSize(-1);
        List<SfaDirectoryConfigRespVo> list = this.findList(reqVo).getData();
        if (CollectionUtil.listEmpty(list)){
            throw new BusinessException("拜访步骤表单配置数据不存在");
        }
        sfaDirectoryConfigMapper.deleteProductsByParams(reqVo);
        this.deleteBatchDate(reqVo);
    }

    public void deleteBatchDate(SfaDirectoryConfigReqVo reqVo){
        if(StringUtils.isEmpty(reqVo.getId())&&CollectionUtil.listEmpty(reqVo.getIds())){
            throw new BusinessException("数据主键不能为空");
        }
        SfaDirectoryInputReqVo directoryInputReqVo = new SfaDirectoryInputReqVo();
        directoryInputReqVo.setDirectoryId(reqVo.getId());
        directoryInputReqVo.setDirectoryIds(reqVo.getIds());
        sfaDirectoryInputMapper.deleteProductsByParams(directoryInputReqVo);
    }
    /**
     * 启用
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @CrmLog
    public void enableBatch(SfaDirectoryConfigReqVo reqVo){
        if(StringUtils.isEmpty(reqVo.getId())&&CollectionUtil.listEmpty(reqVo.getIds())){
            throw new BusinessException("数据主键不能为空");
        }
        if (StringUtils.isNotEmpty(reqVo.getId())){
            List<String> ids = new ArrayList<>();
            if (CollectionUtil.listNotEmptyNotSizeZero(reqVo.getIds())){
                ids = reqVo.getIds();
            }
            ids.add(reqVo.getId());
            reqVo.setIds(ids);
        }
        //设置状态为启用
        List<SfaDirectoryConfigEntity> sfaDirectoryConfigEntities = sfaDirectoryConfigMapper.selectBatchIds(reqVo.getIds());
        if(CollectionUtils.isNotEmpty(sfaDirectoryConfigEntities)){
            sfaDirectoryConfigEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
            });
        }
        this.updateBatchById(sfaDirectoryConfigEntities);
    }

    /**
     * 禁用
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @CrmLog
    public void disableBatch(SfaDirectoryConfigReqVo reqVo){
        if(StringUtils.isEmpty(reqVo.getId())&&CollectionUtil.listEmpty(reqVo.getIds())){
            throw new BusinessException("数据主键不能为空");
        }
        if (StringUtils.isNotEmpty(reqVo.getId())){
            List<String> ids = new ArrayList<>();
            if (CollectionUtil.listNotEmptyNotSizeZero(reqVo.getIds())){
                ids = reqVo.getIds();
            }
            ids.add(reqVo.getId());
            reqVo.setIds(ids);
        }
        //设置状态为禁用
        List<SfaDirectoryConfigEntity> sfaDirectoryConfigEntities = sfaDirectoryConfigMapper.selectBatchIds(reqVo.getIds());
        if(CollectionUtils.isNotEmpty(sfaDirectoryConfigEntities)){
                sfaDirectoryConfigEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.DISABLE.getCode());
            });
        }
        this.updateBatchById(sfaDirectoryConfigEntities);
    }
}
