package com.biz.crm.collection.service.impl;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.aop.CrmLog;
import com.biz.crm.base.BusinessException;
import com.biz.crm.collection.mapper.SfaInputConfigMapper;
import com.biz.crm.collection.model.SfaInputConfigEntity;
import com.biz.crm.collection.service.ISfaInputConfigService;
import com.biz.crm.common.PageResult;
import com.biz.crm.eunm.CrmDelFlagEnum;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.nebular.sfa.collection.req.SfaInputConfigReqVo;
import com.biz.crm.nebular.sfa.collection.resp.SfaInputConfigRespVo;
import com.biz.crm.util.*;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

/**
 * 拜访控件 接口实现
 *
 * @author liuhongming
 * @date 2020-09-18 10:33:32
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name="SfaInputConfigServiceExpandImpl")
public class SfaInputConfigServiceImpl extends ServiceImpl<SfaInputConfigMapper, SfaInputConfigEntity> implements ISfaInputConfigService {

    @Resource
    private SfaInputConfigMapper sfaInputConfigMapper;

    /**
     * 列表
     * @param reqVo
     * @return
     */
    @Override
    @CrmLog
    public PageResult<SfaInputConfigRespVo> findList(SfaInputConfigReqVo reqVo){
        Page<SfaInputConfigRespVo> page = new Page<>(reqVo.getPageNum(), reqVo.getPageSize());
        List<SfaInputConfigRespVo> list = sfaInputConfigMapper.findList(page, reqVo);
        return PageResult.<SfaInputConfigRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    /**
     * 查询
     * @param reqVo
     * @return sfaInputConfigRespVo
     */
    @Override
    @CrmLog
    public SfaInputConfigRespVo query(SfaInputConfigReqVo reqVo){
        if (null==reqVo||(StringUtils.isEmpty(reqVo.getId())&& CollectionUtil.listNotEmptyNotSizeZero(reqVo.getIds())&&StringUtils.isEmpty(reqVo.getLableCode()))){
            return new SfaInputConfigRespVo();
        }
        List<SfaInputConfigRespVo> list = this.findList(reqVo).getData();
        if (CollectionUtil.listEmpty(list)){
            return new SfaInputConfigRespVo();
        }
        return list.get(0);
    }

    /**
     * 新增,编辑,校验
     * @param reqVo
     */
    public void saveCheck(SfaInputConfigReqVo reqVo){
        AssertUtils.isNotEmpty(reqVo.getLableName(),"请输入控件名称");
        AssertUtils.isNotEmpty(reqVo.getInputType(),"请选择控件类型");
    }

    /**
     * 新增
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @CrmLog
    public void save(SfaInputConfigReqVo reqVo){
        this.saveCheck(reqVo);
        SfaInputConfigEntity entity = CrmBeanUtil.copy(reqVo,SfaInputConfigEntity.class);
        //TODO 控件编码
        String code = CodeUtil.createCode();
        entity.setLableCode(code);
        entity.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
        this.save(entity);
    }

    /**
     * 更新
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @CrmLog
    public void update(SfaInputConfigReqVo reqVo){
        if(StringUtils.isEmpty(reqVo.getId())){
            throw new BusinessException("数据主键不能为空");
        }
        SfaInputConfigEntity entity = this.getById(reqVo.getId());
        if(Objects.isNull(entity)){
            throw new BusinessException("拜访控件数据不存在");
        }
        this.saveCheck(reqVo);
        CrmBeanUtil.copyPropertiesIgnoreEmpty(reqVo,entity);
        this.updateById(entity);
    }

    /**
     * 删除
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @CrmLog
    public void deleteBatch(SfaInputConfigReqVo reqVo){
        if(StringUtils.isEmpty(reqVo.getId())&&CollectionUtil.listEmpty(reqVo.getIds())){
            throw new BusinessException("数据主键不能为空");
        }
        List<SfaInputConfigRespVo> list = this.findList(reqVo).getData();
        if(CollectionUtil.listEmpty(list)){
            throw new BusinessException("公告数据不存在");
        }
        sfaInputConfigMapper.deleteProductsByParams(reqVo);
    }

    /**
     * 启用
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @CrmLog
    public void enableBatch(SfaInputConfigReqVo reqVo){
        if(StringUtils.isEmpty(reqVo.getId())&&CollectionUtil.listEmpty(reqVo.getIds())){
            throw new BusinessException("数据主键不能为空");
        }
        if (StringUtils.isNotEmpty(reqVo.getId())){
            List<String> ids = new ArrayList<>();
            if (CollectionUtil.listNotEmptyNotSizeZero(reqVo.getIds())){
                ids = reqVo.getIds();
            }
            ids.add(reqVo.getId());
            reqVo.setIds(ids);
        }
        //设置状态为启用
        List<SfaInputConfigEntity> sfaInputConfigEntities = sfaInputConfigMapper.selectBatchIds(reqVo.getIds());
        if(CollectionUtils.isNotEmpty(sfaInputConfigEntities)){
            sfaInputConfigEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
            });
        }
        this.updateBatchById(sfaInputConfigEntities);
    }

    /**
     * 禁用
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @CrmLog
    public void disableBatch(SfaInputConfigReqVo reqVo){
        if(StringUtils.isEmpty(reqVo.getId())&&CollectionUtil.listEmpty(reqVo.getIds())){
            throw new BusinessException("数据主键不能为空");
        }
        if (StringUtils.isNotEmpty(reqVo.getId())){
            List<String> ids = new ArrayList<>();
            if (CollectionUtil.listNotEmptyNotSizeZero(reqVo.getIds())){
                ids = reqVo.getIds();
            }
            ids.add(reqVo.getId());
            reqVo.setIds(ids);
        }
        //设置状态为禁用
        List<SfaInputConfigEntity> sfaInputConfigEntities = sfaInputConfigMapper.selectBatchIds(reqVo.getIds());
        if(CollectionUtils.isNotEmpty(sfaInputConfigEntities)){
                sfaInputConfigEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.DISABLE.getCode());
            });
        }
        this.updateBatchById(sfaInputConfigEntities);
    }
}
