package com.biz.crm.helpdefense.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.base.BusinessException;
import com.biz.crm.collection.service.ISfaVisitRoleDirectoryService;
import com.biz.crm.common.PageResult;
import com.biz.crm.config.CrmDictMethod;
import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.eunm.sfa.SfaVisitEnum;
import com.biz.crm.helpdefense.mapper.SfaVisitHelpDefenseMapper;
import com.biz.crm.helpdefense.model.SfaHelpDefenseDetailEntity;
import com.biz.crm.helpdefense.model.SfaHelpDefenseEntity;
import com.biz.crm.helpdefense.model.SfaVisitHelpDefenseDetailEntity;
import com.biz.crm.helpdefense.model.SfaVisitHelpDefenseEntity;
import com.biz.crm.helpdefense.service.ISfaHelpDefenseDetailService;
import com.biz.crm.helpdefense.service.ISfaHelpDefenseService;
import com.biz.crm.helpdefense.service.ISfaVisitHelpDefenseDetailService;
import com.biz.crm.helpdefense.service.ISfaVisitHelpDefenseService;
import com.biz.crm.nebular.sfa.collection.req.SfaVisitRoleConfigReqVo;
import com.biz.crm.nebular.sfa.helpdefense.resp.step.HelpDefenseStepCheckInfoVo;
import com.biz.crm.nebular.sfa.helpdefense.resp.step.HelpDefenseStepInOutStoreInfoVo;
import com.biz.crm.nebular.sfa.helpdefense.resp.step.HelpDefenseStepInfoVo;
import com.biz.crm.nebular.sfa.helpdefense.resp.step.LoadHelpDefenseExecutionInfoRespVo;
import com.biz.crm.nebular.sfa.collection.resp.SfaVisitRoleDirectoryRespVo;
import com.biz.crm.nebular.sfa.helpdefense.req.*;
import com.biz.crm.nebular.sfa.helpdefense.resp.FindHelpDefenseScheduleRespVo;
import com.biz.crm.nebular.sfa.helpdefense.resp.SfaVisitHelpDefenseDetailRespVo;
import com.biz.crm.nebular.sfa.helpdefense.resp.SfaVisitHelpDefenseRespVo;
import com.biz.crm.nebular.sfa.picture.resp.SfaVisitPictureRespVo;
import com.biz.crm.picture.model.SfaVisitPictureEntity;
import com.biz.crm.picture.service.ISfaVisitPictureService;
import com.biz.crm.util.*;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 拜访-协访执行;接口实现
 *
 * @author lf
 * @date 2021-01-11 15:42:32
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "SfaVisitHelpDefenseServiceExpandImpl")
@Transactional
public class SfaVisitHelpDefenseServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<SfaVisitHelpDefenseMapper, SfaVisitHelpDefenseEntity> implements ISfaVisitHelpDefenseService {

    @Resource
    private SfaVisitHelpDefenseMapper sfaVisitHelpDefenseMapper;
    @Autowired
    private ISfaHelpDefenseDetailService sfaHelpDefenseDetailService;
    @Autowired
    private ISfaVisitPictureService sfaVisitPictureService;
    @Autowired
    private ISfaVisitRoleDirectoryService sfaVisitRoleDirectoryService;
    @Autowired
    private ISfaHelpDefenseService sfaHelpDefenseService;
    @Autowired
    private ISfaVisitHelpDefenseDetailService sfaVisitHelpDefenseDetailService;

    /**
     * 列表
     *
     * @param reqVo
     * @return
     */
    @Override
    public PageResult<SfaVisitHelpDefenseRespVo> findList(SfaVisitHelpDefenseReqVo reqVo) {
        Page<SfaVisitHelpDefenseRespVo> page = new Page<>(reqVo.getPageNum(), reqVo.getPageSize());
        List<SfaVisitHelpDefenseRespVo> list = sfaVisitHelpDefenseMapper.findList(page, reqVo);
        return PageResult.<SfaVisitHelpDefenseRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    /**
     * 查询协访明细报表
     *
     * @param reqVo
     * @return
     */
    @Override
    public PageResult<SfaVisitHelpDefenseRespVo> findVisitHelpDefenseReport(SfaVisitHelpDefenseReqVo reqVo) {
        Page<SfaVisitHelpDefenseRespVo> page = new Page<>(reqVo.getPageNum(), reqVo.getPageSize());
        List<SfaVisitHelpDefenseRespVo> list = sfaVisitHelpDefenseMapper.findReportList(page, reqVo);
        list.forEach(data -> {
            List<SfaVisitHelpDefenseDetailEntity> sfaVisitHelpDefenseDetailEntities = sfaVisitHelpDefenseDetailService.lambdaQuery()
                    .eq(SfaVisitHelpDefenseDetailEntity::getVisitHelpDefenseId, data.getId()).list();
            if (CollectionUtil.listNotEmptyNotSizeZero(sfaVisitHelpDefenseDetailEntities)) {
                Long num = sfaVisitHelpDefenseDetailEntities.stream().filter(da -> da.getIsQualified().equals(SfaVisitEnum.SfaVisitQualified.NOT_QUALIFIED.getVal())).count();
                BigDecimal qualifiedNum = new BigDecimal(num);
                BigDecimal total = new BigDecimal(sfaVisitHelpDefenseDetailEntities.size());
                qualifiedNum.divide(total, 2, BigDecimal.ROUND_DOWN);
                data.setQualifiedNum(qualifiedNum);
            } else {
                data.setQualifiedNum(BigDecimal.ZERO);
            }
            List<SfaVisitPictureEntity> sfaVisitPictureEntities = sfaVisitPictureService.lambdaQuery()
                    .eq(SfaVisitPictureEntity::getBusinessId, data.getId()).list();
            Map<String, List<SfaVisitPictureEntity>> picMap = sfaVisitPictureEntities.stream().collect(Collectors.groupingBy(SfaVisitPictureEntity::getPicType));
            List<String> inStorePicList = new ArrayList<>();
            List<String> outStorePicList = new ArrayList<>();
            for (Map.Entry<String, List<SfaVisitPictureEntity>> pic : picMap.entrySet()) {
                List<SfaVisitPictureEntity> picList = pic.getValue();
                if (pic.getKey().equals(SfaVisitEnum.visitPicType.DEALER_IN_STORE_PIC.getVal())) {
                    picList.forEach(p -> {
                        String picUrl = p.getPicUrl() + p.getPicPath();
                        inStorePicList.add(picUrl);
                    });
                } else if (pic.getKey().equals(SfaVisitEnum.visitPicType.DEALER_OUT_STORE_PIC.getVal())) {
                    picList.forEach(p -> {
                        String picUrl = p.getPicUrl() + p.getPicPath();
                        outStorePicList.add(picUrl);
                    });
                }
            }
            data.setInStorePicList(inStorePicList);
            data.setOutStorePicList(outStorePicList);
        });
        return PageResult.<SfaVisitHelpDefenseRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    /**
     * 查看协访报表详细信息
     *
     * @param id
     * @return
     */
    @Override
    @CrmDictMethod
    public SfaVisitHelpDefenseRespVo queryVisitHelpDefenseReportDetail(String id) {
        SfaVisitHelpDefenseReqVo reqVo = new SfaVisitHelpDefenseReqVo() {{
            this.setPageNum(1);
            this.setPageSize(1);
            this.setId(id);
        }};
        Page<SfaVisitHelpDefenseRespVo> page = new Page<>(reqVo.getPageNum(), reqVo.getPageSize());
        List<SfaVisitHelpDefenseRespVo> list = sfaVisitHelpDefenseMapper.findReportList(page, reqVo);
        if (!CollectionUtil.listNotEmptyNotSizeZero(list)) {
            throw new BusinessException("未查询到协访报表详细信息");
        }
        SfaVisitHelpDefenseRespVo respVo = list.get(0);
        List<SfaVisitHelpDefenseDetailEntity> sfaVisitHelpDefenseDetailEntities = sfaVisitHelpDefenseDetailService.lambdaQuery()
                .eq(SfaVisitHelpDefenseDetailEntity::getVisitHelpDefenseId, respVo.getId()).list();
        List<SfaVisitHelpDefenseDetailRespVo> detailRespVos = CrmBeanUtil.copyList(sfaVisitHelpDefenseDetailEntities, SfaVisitHelpDefenseDetailRespVo.class);
        Long num = sfaVisitHelpDefenseDetailEntities.stream().filter(da -> da.getIsQualified().equals(SfaVisitEnum.SfaVisitQualified.NOT_QUALIFIED.getVal())).count();
        BigDecimal qualifiedNum = new BigDecimal(num);
        BigDecimal total = new BigDecimal(sfaVisitHelpDefenseDetailEntities.size());
        qualifiedNum.divide(total, 2, BigDecimal.ROUND_DOWN);
        respVo.setQualifiedNum(qualifiedNum);
        respVo.setSfaVisitHelpDefenseDetailList(detailRespVos);
        List<SfaVisitPictureEntity> sfaVisitPictureEntities = sfaVisitPictureService.lambdaQuery()
                .eq(SfaVisitPictureEntity::getBusinessId, respVo.getId()).list();
        if (CollectionUtil.listNotEmptyNotSizeZero(sfaVisitPictureEntities)) {
            Map<String, List<SfaVisitPictureEntity>> picMap = sfaVisitPictureEntities.stream().collect(Collectors.groupingBy(SfaVisitPictureEntity::getPicType));
            List<String> inStorePicList = Lists.newArrayList();
            List<String> outStorePicList = Lists.newArrayList();
            for (Map.Entry<String, List<SfaVisitPictureEntity>> pic : picMap.entrySet()) {
                if (pic.getKey().equals(SfaVisitEnum.visitPicType.DEALER_IN_STORE_PIC.getVal())) {
                    pic.getValue().forEach(p -> {
                        String url = p.getPicUrl() + p.getPicPath();
                        inStorePicList.add(url);
                    });
                } else if (pic.getKey().equals(SfaVisitEnum.visitPicType.DEALER_OUT_STORE_PIC.getVal())) {
                    pic.getValue().forEach(p -> {
                        String url = p.getPicUrl() + p.getPicPath();
                        outStorePicList.add(url);
                    });
                }
            }
            respVo.setInStorePicList(inStorePicList);
            respVo.setOutStorePicList(outStorePicList);
        }
        return respVo;
    }

    /**
     * 查询拜访-协访执行列表
     *
     * @param findVisitHelpDefenseListReqVo
     * @return
     */
    @Override
    public PageResult<SfaVisitHelpDefenseRespVo> findVisitHelpDefenseList(FindVisitHelpDefenseListReqVo findVisitHelpDefenseListReqVo ) {

        AssertUtils.isNotEmpty(findVisitHelpDefenseListReqVo.getDefenseDate(), "协访日期为空");
        AssertUtils.isNotEmpty(findVisitHelpDefenseListReqVo.getLongitude(), "经度为空");
        AssertUtils.isNotEmpty(findVisitHelpDefenseListReqVo.getLatitude(), "纬度为空");
        if (!StringUtils.isNotEmpty(findVisitHelpDefenseListReqVo.getUserName())) {
            UserRedis userRedis = UserUtils.getUser();
            findVisitHelpDefenseListReqVo.setUserName(userRedis.getUsername());
            findVisitHelpDefenseListReqVo.setPosCode(userRedis.getPoscode());
        }

        SfaVisitHelpDefenseReqVo reqVo = CrmBeanUtil.copy(findVisitHelpDefenseListReqVo, SfaVisitHelpDefenseReqVo.class);
        Page<SfaVisitHelpDefenseRespVo> page = new Page<>(reqVo.getPageNum(), reqVo.getPageSize());


        reqVo.setHelpDefenseStatus(SfaVisitEnum.HelpVisitStatus.ALREADY_OVERDUE.getVal());
        List<SfaVisitHelpDefenseRespVo> sfaVisitHelpDefenseRespVos = sfaVisitHelpDefenseMapper.findVisitHelpDefenseList(page, reqVo);
        if (CollectionUtil.listNotEmptyNotSizeZero(sfaVisitHelpDefenseRespVos)) {
            sfaVisitHelpDefenseRespVos.forEach(data -> {
                if (data.getDistance() != null) {
                    data.setDistance(data.getDistance().divide(new BigDecimal(1000)).setScale(2, BigDecimal.ROUND_DOWN));
                }
                if (StringUtils.isNotEmpty(reqVo.getUserName())) {
                    Map<String, String> buttons = this.getButtons(reqVo.getUserName(), data.getHelpDefenseStatus(), data.getDefenseDate());
                    data.setButtons(buttons);
                }
            });
        }
        return PageResult.<SfaVisitHelpDefenseRespVo>builder()
                .data(sfaVisitHelpDefenseRespVos)
                .count(page.getTotal())
                .build();
    }

    /**
     * 查询协访进度
     *
     * @param findHelpDefenseScheduleVo
     * @return
     */
    @Override
    public FindHelpDefenseScheduleRespVo findHelpDefenseSchedule(FindHelpDefenseScheduleReqVo findHelpDefenseScheduleVo) {
        SfaVisitHelpDefenseReqVo reqVo = CrmBeanUtil.copy(findHelpDefenseScheduleVo, SfaVisitHelpDefenseReqVo.class);
        FindHelpDefenseScheduleRespVo respVo = new FindHelpDefenseScheduleRespVo();
        AssertUtils.isNotEmpty(reqVo.getDefenseDate(), "协访日期为空");
//        AssertUtils.isNotEmpty(reqVo.getClientType(), "客户类型为空");
        if (StringUtils.isNotEmpty(reqVo.getUserName())) {
            UserRedis userRedis = UserUtils.getUser();
            reqVo.setUserName(userRedis.getUsername());
            reqVo.setPosCode(userRedis.getPoscode());
        }
//        reqVo.setHelpDefenseStatus(SfaVisitEnum.helpDefenseStatus.ALREADY_OVERDUE.getVal());
        List<SfaVisitHelpDefenseRespVo> sfaVisitHelpDefenseRespVos = sfaVisitHelpDefenseMapper.findHelpDefenseSchedule(reqVo);
        respVo.setHelpDefenseTotal(sfaVisitHelpDefenseRespVos.size());
        Long num = sfaVisitHelpDefenseRespVos.stream().filter(data -> data.getHelpDefenseStatus().equals(SfaVisitEnum.HelpVisitStatus.ALREADY_SUCCESS.getVal())).count();
        respVo.setHelpDefenseSuccess(Integer.valueOf(num.toString()));
        return respVo;
    }

    /**
     * 协访-进店打卡
     *
     * @param helpDefenseInStoreReqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public String helpDefenseInStore(HelpDefenseInStoreReqVo helpDefenseInStoreReqVo) {
        //检查必要参数并使用前端传的明细id加载协访计划明细
        SfaHelpDefenseDetailEntity sfaHelpDefenseDetailEntity = this.checkInStore(helpDefenseInStoreReqVo);
        //协访计划
        SfaHelpDefenseEntity sfaHelpDefenseEntity = sfaHelpDefenseService.lambdaQuery()
                .eq(SfaHelpDefenseEntity::getId, sfaHelpDefenseDetailEntity.getHelpDefenseId()).one();
        if(null == sfaHelpDefenseEntity){
            throw new BusinessException("未查询到协访计划");
        }
        SfaVisitHelpDefenseEntity oldEntity = this.lambdaQuery().eq(SfaVisitHelpDefenseEntity :: getHelpDefenseDetailId, sfaHelpDefenseDetailEntity.getId()).one();
        //组装拜访协访执行数据
        SfaVisitHelpDefenseEntity entity = this.buildSfaVisitHelpDefenseEntity(helpDefenseInStoreReqVo, sfaHelpDefenseDetailEntity, sfaHelpDefenseEntity);
        if(null == oldEntity){
            //新增
            this.save(entity);
        }else{
            //更新
            this.sfaVisitPictureService.remove(Wrappers.lambdaUpdate(SfaVisitPictureEntity.class).eq(SfaVisitPictureEntity :: getBusinessId, oldEntity.getId()));
            entity.setId(oldEntity.getId());
            this.updateById(entity);
        }

        String id = entity.getId();
        //打卡图片
        if (CollectionUtil.listNotEmptyNotSizeZero(helpDefenseInStoreReqVo.getHelpDefensePicList())) {
            List<SfaVisitPictureEntity> sfaVisitPictureEntities = CrmBeanUtil.copyList(helpDefenseInStoreReqVo.getHelpDefensePicList(), SfaVisitPictureEntity.class);
            sfaVisitPictureEntities.forEach(data -> {
                data.setBusinessId(id);
                data.setPicType(SfaVisitEnum.visitPicType.DEALER_IN_STORE_PIC.getVal());
            });
            sfaVisitPictureService.saveBatch(sfaVisitPictureEntities);
        }
        SfaHelpDefenseDetailEntity updateSfaHelpDefenseDetailEntity = new SfaHelpDefenseDetailEntity();
        updateSfaHelpDefenseDetailEntity.setId(sfaHelpDefenseDetailEntity.getId());
        //修改协访计划明细状态
        updateSfaHelpDefenseDetailEntity.setHelpDefenseStatus(SfaVisitEnum.HelpVisitStatus.IN_HELP_DEFENSE.getVal());
        sfaHelpDefenseDetailService.updateById(updateSfaHelpDefenseDetailEntity);
        return id;
    }

    /**
     * 组装协访执行
     * @param sfaHelpDefenseDetailEntity
     * @param sfaHelpDefenseEntity
     * @return
     */
    private SfaVisitHelpDefenseEntity buildSfaVisitHelpDefenseEntity(HelpDefenseInStoreReqVo helpDefenseInStoreReqVo, SfaHelpDefenseDetailEntity sfaHelpDefenseDetailEntity, SfaHelpDefenseEntity sfaHelpDefenseEntity){
        SfaVisitHelpDefenseEntity entity = new SfaVisitHelpDefenseEntity();
        entity.setId(null);
        UserRedis userRedis = UserUtils.getUser();
        entity.setUserName(userRedis.getUsername());
        entity.setRealName(userRedis.getRealname());
        entity.setOrgName(userRedis.getOrgname());
        entity.setOrgCode(userRedis.getOrgcode());
        entity.setPosCode(userRedis.getPoscode());
        entity.setPosName(userRedis.getPosname());
        entity.setInStore(LocalDateTime.now().format(CrmDateUtils.yyyyMMddHHmmss));
        entity.setInStoreAddress(helpDefenseInStoreReqVo.getInStoreAddress());
        entity.setInStoreLatitude(helpDefenseInStoreReqVo.getInStoreLatitude());
        entity.setInStoreLongitude(helpDefenseInStoreReqVo.getInStoreLongitude());
        entity.setRemarks(helpDefenseInStoreReqVo.getRemarks());
        entity.setClientId(sfaHelpDefenseDetailEntity.getClientId());
        entity.setClientType(sfaHelpDefenseDetailEntity.getClientType());
        entity.setClientCode(sfaHelpDefenseDetailEntity.getClientCode());
        entity.setClientAddress(sfaHelpDefenseDetailEntity.getClientAddress());
        entity.setClientName(sfaHelpDefenseDetailEntity.getClientName());
        entity.setClientPhone(sfaHelpDefenseDetailEntity.getClientPhone());
        entity.setDefenseDate(sfaHelpDefenseEntity.getHelpDefenseDate());
        entity.setHelpDefenseDetailId(sfaHelpDefenseDetailEntity.getId());
        entity.setHelpDefenseStatus(SfaVisitEnum.HelpVisitStatus.IN_HELP_DEFENSE.getVal());
        return entity;
    }

    /**
     * 查询协访步骤列表
     *
     * @param id
     * @return
     */
    @Override
    public List<SfaVisitRoleDirectoryRespVo> findHelpDefenseROleDirectory(String id) {
        SfaHelpDefenseDetailEntity sfaHelpDefenseDetailEntity = sfaHelpDefenseDetailService.lambdaQuery()
                .eq(SfaHelpDefenseDetailEntity::getId, id).one();
        if (sfaHelpDefenseDetailEntity == null) {
            throw new BusinessException("未查询到协访明细信息");
        }
        SfaVisitHelpDefenseEntity sfaVisitHelpDefenseEntity = this.lambdaQuery()
                .eq(SfaVisitHelpDefenseEntity::getHelpDefenseDetailId, sfaHelpDefenseDetailEntity.getId()).one();
        if (sfaVisitHelpDefenseEntity == null) {
            throw new BusinessException("未查询到协访明细信息");
        }

        SfaVisitRoleConfigReqVo reqVo = new SfaVisitRoleConfigReqVo() {{
            this.setSfacusType(sfaHelpDefenseDetailEntity.getClientType());
            this.setVisitType(SfaVisitEnum.sfaVisitType.VISIT.getVal());
            this.setVisitPosCode(sfaHelpDefenseDetailEntity.getCoverHelpPosCode());
            this.setVisitUserName(sfaHelpDefenseDetailEntity.getCoverHelpUserName());
        }};
        List<SfaVisitRoleDirectoryRespVo> sfaVisitRoleDirectoryRespVos = sfaVisitRoleDirectoryService.findRoleDirectoryList(reqVo);
        List<SfaVisitHelpDefenseDetailEntity> visitHelpDefenseDetailEntityList = sfaVisitHelpDefenseDetailService.lambdaQuery()
                .eq(SfaVisitHelpDefenseDetailEntity::getVisitHelpDefenseId, sfaVisitHelpDefenseEntity.getId()).list();
        sfaVisitRoleDirectoryRespVos.forEach(data -> {
            data.setIsSuccess(YesNoEnum.YesNoCodeNumberEnum.NO.getCode().toString());
            data.setIsSuccessDesc(YesNoEnum.YesNoCodeNumberEnum.NO.getValue());
        });
        if (CollectionUtil.listNotEmptyNotSizeZero(visitHelpDefenseDetailEntityList)) {
            Map<String, String> roleMap = visitHelpDefenseDetailEntityList.stream().
                    collect(Collectors.toMap(SfaVisitHelpDefenseDetailEntity::getDirectoryCode, SfaVisitHelpDefenseDetailEntity::getIsQualified));
            sfaVisitRoleDirectoryRespVos.forEach(data -> {
                if (roleMap.containsKey(data.getDirectoryCode())) {
                    data.setIsSuccessDesc(SfaVisitEnum.SfaVisitQualified.GETMAP.get(roleMap.get(data.getDirectoryCode())));
                    data.setIsSuccess(roleMap.get(data.getDirectoryCode()));
                }
            });
        }
        return sfaVisitRoleDirectoryRespVos;
    }

    /**
     * 加载协访任务执行信息
     *
     * @param id
     * @return
     */
    @Override
    public LoadHelpDefenseExecutionInfoRespVo loadHelpDefenseExecutionInfo(String id) {
        SfaHelpDefenseDetailEntity sfaHelpDefenseDetailEntity = sfaHelpDefenseDetailService.lambdaQuery()
                .eq(SfaHelpDefenseDetailEntity::getId, id).one();
        if (sfaHelpDefenseDetailEntity == null) {
            throw new BusinessException("未查询到协访明细信息");
        }
        SfaHelpDefenseEntity helpDefenseEntity = sfaHelpDefenseService.lambdaQuery()
                .eq(SfaHelpDefenseEntity::getId, sfaHelpDefenseDetailEntity.getHelpDefenseId()).one();
        if (sfaHelpDefenseDetailEntity == null) {
            throw new BusinessException("未查询到协访信息");
        }

        SfaVisitHelpDefenseEntity sfaVisitHelpDefenseEntity = this.lambdaQuery()
                .eq(SfaVisitHelpDefenseEntity::getHelpDefenseDetailId, sfaHelpDefenseDetailEntity.getId()).one();

        if (sfaVisitHelpDefenseEntity == null) {
            sfaVisitHelpDefenseEntity = new SfaVisitHelpDefenseEntity();
//            throw new BusinessException("未查询到协访执行数据信息");
        }
        LoadHelpDefenseExecutionInfoRespVo executionInfoRespVo = CrmBeanUtil.copy(sfaHelpDefenseDetailEntity, LoadHelpDefenseExecutionInfoRespVo.class);
        executionInfoRespVo.setRealName(helpDefenseEntity.getHelpRealName());
        //组装进离店数据
        this.buildInOutStoreInfo(executionInfoRespVo, sfaVisitHelpDefenseEntity);
        executionInfoRespVo.setVisitInfo(this.buildVisitInfo(sfaHelpDefenseDetailEntity, sfaVisitHelpDefenseEntity));
        return executionInfoRespVo;
    }

    /**
     * 组装拜访内容
     * @param sfaHelpDefenseDetailEntity
     * @param sfaVisitHelpDefenseEntity
     * @return
     */
    private LoadHelpDefenseExecutionInfoRespVo.VisitInfo buildVisitInfo(SfaHelpDefenseDetailEntity sfaHelpDefenseDetailEntity, SfaVisitHelpDefenseEntity sfaVisitHelpDefenseEntity){
        SfaVisitRoleConfigReqVo reqVo = new SfaVisitRoleConfigReqVo() {{
            this.setSfacusType(sfaHelpDefenseDetailEntity.getClientType());
            this.setVisitType(SfaVisitEnum.sfaVisitType.VISIT.getVal());
            this.setVisitPosCode(sfaHelpDefenseDetailEntity.getCoverHelpPosCode());
            this.setVisitUserName(sfaHelpDefenseDetailEntity.getCoverHelpUserName());
        }};

        List<SfaVisitRoleDirectoryRespVo> sfaVisitRoleDirectoryRespVos = sfaVisitRoleDirectoryService.findRoleDirectoryList(reqVo);
        List<HelpDefenseStepCheckInfoVo> checkInfoVosTemp = CrmBeanUtil.copyList(sfaVisitRoleDirectoryRespVos, HelpDefenseStepCheckInfoVo.class);

        Map<String, String> helpDefenseDetailMap;
        if(org.apache.commons.lang3.StringUtils.isNotBlank(sfaVisitHelpDefenseEntity.getId())){
            helpDefenseDetailMap = sfaVisitHelpDefenseDetailService.lambdaQuery()
                    .eq(SfaVisitHelpDefenseDetailEntity::getVisitHelpDefenseId, sfaVisitHelpDefenseEntity.getId()).list().stream().
                            collect(Collectors.toMap(SfaVisitHelpDefenseDetailEntity::getDirectoryCode, SfaVisitHelpDefenseDetailEntity::getIsQualified));
        }else{
            helpDefenseDetailMap = Maps.newHashMap();
        }

        checkInfoVosTemp.forEach(data -> {
            if (helpDefenseDetailMap.containsKey(data.getDirectoryCode())) {
                data.setIsQualified(helpDefenseDetailMap.get(data.getDirectoryCode()));
                data.setIsQualifiedDesc(SfaVisitEnum.SfaVisitQualified.GETMAP.get(helpDefenseDetailMap.get(data.getDirectoryCode())));
            }
        });
        LoadHelpDefenseExecutionInfoRespVo.VisitInfo visitInfo = new LoadHelpDefenseExecutionInfoRespVo.VisitInfo();
//        SfaVisitHelpDefenseRespVo sfaVisitHelpDefenseRespVo = this.queryHelpDefenseDetail()
        List<HelpDefenseStepCheckInfoVo> checkInfoVos = checkInfoVosTemp.stream().sorted(Comparator.comparing(HelpDefenseStepCheckInfoVo::getSort)).collect(Collectors.toList());
        visitInfo.setHelpDefenseStepInfoVos(checkInfoVos);
        visitInfo.setHelpDefenseSummary(sfaVisitHelpDefenseEntity.getHelpDefenseSummary());
        visitInfo.setKeyQuestions(sfaVisitHelpDefenseEntity.getKeyQuestions());
        return visitInfo;
    }

    /**
     * 组装进离店数据
     * @param executionInfoRespVo
     * @param sfaVisitHelpDefenseEntity
     */
    private void buildInOutStoreInfo(LoadHelpDefenseExecutionInfoRespVo executionInfoRespVo, SfaVisitHelpDefenseEntity sfaVisitHelpDefenseEntity){
        Map<String, List<SfaVisitPictureEntity>> picMap = this.loadVisitHelpDefensePic(sfaVisitHelpDefenseEntity);
        String in = SfaVisitEnum.visitPicType.DEALER_IN_STORE_PIC.getVal()
                , out = SfaVisitEnum.visitPicType.DEALER_OUT_STORE_PIC.getVal();
        //进店
        HelpDefenseStepInOutStoreInfoVo inStoreInfoVo = new HelpDefenseStepInOutStoreInfoVo();
        if(org.apache.commons.lang3.StringUtils.isNotBlank(sfaVisitHelpDefenseEntity.getInStore())){
            inStoreInfoVo.setAddress(sfaVisitHelpDefenseEntity.getInStoreAddress());
            inStoreInfoVo.setInOutDateTime(sfaVisitHelpDefenseEntity.getInStore());
            inStoreInfoVo.setLatitude(sfaVisitHelpDefenseEntity.getInStoreLatitude());
            inStoreInfoVo.setLongitude(sfaVisitHelpDefenseEntity.getInStoreLongitude());
        }
        //进店图片
        List<SfaVisitPictureRespVo> pictureRespVos = picMap.containsKey(in) ? CrmBeanUtil.copyList(picMap.get(in), SfaVisitPictureRespVo.class) : Lists.newArrayList();
        inStoreInfoVo.setPicList(pictureRespVos);
        executionInfoRespVo.setHelpDefenseStepInfoVoAsInStore(inStoreInfoVo);
        //离店
        HelpDefenseStepInOutStoreInfoVo outStoreInfoVo = new HelpDefenseStepInOutStoreInfoVo();
        if(org.apache.commons.lang3.StringUtils.isNotBlank(sfaVisitHelpDefenseEntity.getOutStore())){
            outStoreInfoVo.setAddress(sfaVisitHelpDefenseEntity.getOutStoreAddress());
            outStoreInfoVo.setInOutDateTime(sfaVisitHelpDefenseEntity.getOutStore());
            outStoreInfoVo.setLatitude(sfaVisitHelpDefenseEntity.getOutStoreLatitude());
            outStoreInfoVo.setLongitude(sfaVisitHelpDefenseEntity.getOutStoreLongitude());
        }
        //离店图片
        pictureRespVos = picMap.containsKey(in) ? CrmBeanUtil.copyList(picMap.get(out), SfaVisitPictureRespVo.class) : Lists.newArrayList();
        outStoreInfoVo.setPicList(pictureRespVos);
        executionInfoRespVo.setHelpDefenseStepInfoVoAsOutStore(outStoreInfoVo);
    }

    /**
     * 加载协访进离店图片
     * @param sfaVisitHelpDefenseEntity
     * @return
     */
    private Map<String, List<SfaVisitPictureEntity>> loadVisitHelpDefensePic(SfaVisitHelpDefenseEntity sfaVisitHelpDefenseEntity){
        Map<String, List<SfaVisitPictureEntity>> picMap;
        if(org.apache.commons.lang3.StringUtils.isNotBlank(sfaVisitHelpDefenseEntity.getId())){
            String in = SfaVisitEnum.visitPicType.DEALER_IN_STORE_PIC.getVal()
                    , out = SfaVisitEnum.visitPicType.DEALER_OUT_STORE_PIC.getVal();
            picMap = sfaVisitPictureService.lambdaQuery().eq(SfaVisitPictureEntity :: getBusinessId, sfaVisitHelpDefenseEntity.getId()).in(SfaVisitPictureEntity :: getPicType
                    , Lists.newArrayList(in, out)).list()
                    .stream().collect(Collectors.groupingBy(SfaVisitPictureEntity :: getPicType));
        }else{
            picMap = Maps.newHashMap();
        }
        return picMap;
    }

    /**
     * 协访-离店打卡
     *
     * @param reqVo
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void helpDefenseOutStore(SfaVisitHelpDefenseReqVo reqVo) {
        SfaHelpDefenseDetailEntity sfaHelpDefenseDetailEntity = this.checkOutStore(reqVo);


        SfaVisitHelpDefenseEntity entity = this.lambdaQuery()
                .eq(SfaVisitHelpDefenseEntity::getHelpDefenseDetailId, sfaHelpDefenseDetailEntity.getId()).one();
        if (entity == null) {
            throw new BusinessException("未查询到协访执行数据");
        }
        SfaVisitHelpDefenseEntity updateEntity = new SfaVisitHelpDefenseEntity();
        updateEntity.setId(entity.getId());
        updateEntity.setHelpDefenseStatus(SfaVisitEnum.HelpVisitStatus.ALREADY_SUCCESS.getVal());
        updateEntity.setOutStore(LocalDateTime.now().format(CrmDateUtils.yyyyMMddHHmmss));
        updateEntity.setOutStoreAddress(reqVo.getOutStoreAddress());
        updateEntity.setOutStoreLatitude(reqVo.getOutStoreLatitude());
        updateEntity.setOutStoreLongitude(reqVo.getOutStoreLongitude());

        if (CollectionUtil.listNotEmptyNotSizeZero(reqVo.getHelpDefensePicList())) {
            List<SfaVisitPictureEntity> sfaVisitPictureEntities = CrmBeanUtil.copyList(reqVo.getHelpDefensePicList(), SfaVisitPictureEntity.class);
            sfaVisitPictureEntities.forEach(data -> {
                data.setBusinessId(entity.getId());
                data.setPicType(SfaVisitEnum.visitPicType.DEALER_OUT_STORE_PIC.getVal());
            });
            sfaVisitPictureService.saveBatch(sfaVisitPictureEntities);
        }
        this.updateById(updateEntity);
        SfaHelpDefenseDetailEntity updateSfaHelpDefenseDetailEntity = new SfaHelpDefenseDetailEntity();
        updateSfaHelpDefenseDetailEntity.setId(sfaHelpDefenseDetailEntity.getId());
        //修改协访计划明细状态
        updateSfaHelpDefenseDetailEntity.setHelpDefenseStatus(SfaVisitEnum.HelpVisitStatus.ALREADY_SUCCESS.getVal());
        sfaHelpDefenseDetailService.updateById(updateSfaHelpDefenseDetailEntity);
    }

    /**
     * 7018;协访模块接口-协访步骤执行
     *
     * @param reqVo
     */
    @Override
    @Transactional
    public void saveHelpDefenseDetail(SfaVisitHelpDefenseDetailReqVo reqVo) {
        this.checkHelpDefenseStep(reqVo);
        SfaHelpDefenseDetailEntity sfaHelpDefenseDetailEntity = this.sfaHelpDefenseDetailService.lambdaQuery().eq(SfaHelpDefenseDetailEntity :: getId, reqVo.getId()).one();
        if(null == sfaHelpDefenseDetailEntity){
            throw new BusinessException("未查询到协访计划明细数据");
        }
        SfaVisitHelpDefenseEntity sfaVisitHelpDefenseEntity = this.lambdaQuery()
                .eq(SfaVisitHelpDefenseEntity::getHelpDefenseDetailId, sfaHelpDefenseDetailEntity.getId()).one();
        if (sfaVisitHelpDefenseEntity == null) {
            throw new BusinessException("未查询到协防执行数据");
        }
        //先删除
        this.sfaVisitHelpDefenseDetailService.remove(Wrappers.lambdaQuery(SfaVisitHelpDefenseDetailEntity.class)
                .eq(SfaVisitHelpDefenseDetailEntity :: getVisitHelpDefenseId, sfaVisitHelpDefenseEntity.getId())
                .eq(SfaVisitHelpDefenseDetailEntity :: getDirectoryCode, reqVo.getDirectoryCode()));
        //再保存
        SfaVisitHelpDefenseDetailEntity sfaVisitHelpDefenseDetailEntity = CrmBeanUtil.copy(reqVo, SfaVisitHelpDefenseDetailEntity.class);
        sfaVisitHelpDefenseDetailEntity.setVisitHelpDefenseId(sfaVisitHelpDefenseEntity.getId());
        sfaVisitHelpDefenseDetailEntity.setId(null);
        sfaVisitHelpDefenseDetailService.save(sfaVisitHelpDefenseDetailEntity);
    }

    /**
     * 协访-保存协访总结
     *
     * @param reqVo
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void saveHelpDefenseSummary(SfaVisitHelpDefenseReqVo reqVo) {
        AssertUtils.isNotEmpty(reqVo.getHelpDefenseSummary(), "协访总结为空");
        AssertUtils.isNotEmpty(reqVo.getKeyQuestions(), "重大问题记录为空");
        AssertUtils.isNotEmpty(reqVo.getId(), "协访明细id为空");
        SfaHelpDefenseDetailEntity defenseDetailEntity = this.sfaHelpDefenseDetailService.getById(reqVo.getId());
        if(null == defenseDetailEntity){
            throw new BusinessException("未查询的协访明细数据");
        }
        SfaVisitHelpDefenseEntity sfaVisitHelpDefenseEntity = this.lambdaQuery()
                .eq(SfaVisitHelpDefenseEntity::getHelpDefenseDetailId, defenseDetailEntity.getId()).one();
        if (sfaVisitHelpDefenseEntity == null) {
            throw new BusinessException("未查询到协访执行数据");
        }
        SfaVisitHelpDefenseEntity updateEntity = new SfaVisitHelpDefenseEntity();
        updateEntity.setId(sfaVisitHelpDefenseEntity.getId());
        updateEntity.setHelpDefenseSummary(reqVo.getHelpDefenseSummary());
        updateEntity.setKeyQuestions(reqVo.getKeyQuestions());
        this.updateById(updateEntity);
    }

    /**
     * 协访-查询协访详细信息
     *
     * @param id
     * @return
     */
    @Override
    @CrmDictMethod
    public SfaVisitHelpDefenseRespVo queryHelpDefenseDetail(String id) {
        SfaHelpDefenseDetailEntity sfaHelpDefenseDetailEntity = sfaHelpDefenseDetailService.lambdaQuery()
                .eq(SfaHelpDefenseDetailEntity::getId, id).one();
        if (sfaHelpDefenseDetailEntity == null) {
            throw new BusinessException("未查询到协访计划明细信息");
        }
        SfaVisitHelpDefenseEntity sfaVisitHelpDefenseEntity = this.lambdaQuery()
                .eq(SfaVisitHelpDefenseEntity::getHelpDefenseDetailId, sfaHelpDefenseDetailEntity.getId()).one();
        if (sfaVisitHelpDefenseEntity == null) {
            throw new BusinessException("未查询到协访执行数据信息");
        }
        SfaVisitHelpDefenseRespVo respVo = CrmBeanUtil.copy(sfaVisitHelpDefenseEntity, SfaVisitHelpDefenseRespVo.class);
        List<SfaVisitHelpDefenseDetailEntity> sfaVisitHelpDefenseDetailEntities = sfaVisitHelpDefenseDetailService.lambdaQuery()
                .eq(SfaVisitHelpDefenseDetailEntity::getVisitHelpDefenseId, sfaVisitHelpDefenseEntity.getId()).list();
        List<SfaVisitHelpDefenseDetailRespVo> sfaVisitHelpDefenseDetailRespVos = CrmBeanUtil.copyList(sfaVisitHelpDefenseDetailEntities, SfaVisitHelpDefenseDetailRespVo.class);

        this.loadOutInStorePicList(respVo, sfaVisitHelpDefenseEntity.getId());

        respVo.setSfaVisitHelpDefenseDetailList(sfaVisitHelpDefenseDetailRespVos);
        return respVo;
    }

    /**
     * 加载进店离店打卡照片
     * @param respVo
     * @param id
     */
    private void loadOutInStorePicList(SfaVisitHelpDefenseRespVo respVo, String id){
        List<SfaVisitPictureEntity> pictureEntities = sfaVisitPictureService.lambdaQuery().eq(SfaVisitPictureEntity :: getBusinessId, id).list();
        Map<String, List<SfaVisitPictureEntity>> stringListMap = pictureEntities.stream().collect(Collectors.groupingBy(SfaVisitPictureEntity :: getPicType));

        //进店
        List<SfaVisitPictureEntity> entities  = stringListMap.get(SfaVisitEnum.visitPicType.DEALER_IN_STORE_PIC.getVal());
        List<String> picUrls;
        if(null != entities){
            picUrls = entities.stream().map(v -> {return v.getPicUrl() + v.getPicPath();}).collect(Collectors.toList());
        }else{
            picUrls = Lists.newArrayList();
        }
        respVo.setInStorePicList(picUrls);
        //离店
        entities  = stringListMap.get(SfaVisitEnum.visitPicType.DEALER_OUT_STORE_PIC.getVal());

        if(null != entities){
            picUrls = entities.stream().map(v -> {return v.getPicUrl() + v.getPicPath();}).collect(Collectors.toList());
        }else{
            picUrls = Lists.newArrayList();
        }
        respVo.setOutStorePicList(picUrls);
    }

    /**
     * 进店参数检查
     *
     * @param helpDefenseInStoreReqVo
     */
    private SfaHelpDefenseDetailEntity checkInStore(HelpDefenseInStoreReqVo helpDefenseInStoreReqVo) {

        AssertUtils.isNotEmpty(helpDefenseInStoreReqVo.getInStoreAddress(), "进店地址为空");
        if (helpDefenseInStoreReqVo.getInStoreLatitude() == null && helpDefenseInStoreReqVo.getInStoreLongitude() == null) {
            throw new BusinessException("进店经纬度为空");
        }
        AssertUtils.isNotEmpty(helpDefenseInStoreReqVo.getId(), "协访明细id为空");

        SfaHelpDefenseDetailEntity sfaHelpDefenseDetailEntity = sfaHelpDefenseDetailService.lambdaQuery()
                .eq(SfaHelpDefenseDetailEntity::getId, helpDefenseInStoreReqVo.getId()).one();
        if (sfaHelpDefenseDetailEntity == null) {
            throw new BusinessException("未查询到协访明细信息");
        }
        if(!SfaVisitEnum.HelpVisitStatus.READY_HELP_DEFENSE.getVal().equals(sfaHelpDefenseDetailEntity.getHelpDefenseStatus())){
            throw new BusinessException("待协访时才可进店打卡");
        }

        SfaVisitHelpDefenseReqVo reqVo = new SfaVisitHelpDefenseReqVo();
        UserRedis userRedis = UserUtils.getUser();
        reqVo.setUserName(userRedis.getUsername());
        reqVo.setPosCode(userRedis.getPoscode());
        reqVo.setClientType(sfaHelpDefenseDetailEntity.getClientType());
        reqVo.setHelpDefenseStatus(SfaVisitEnum.HelpVisitStatus.IN_HELP_DEFENSE.getVal());
        reqVo.setDefenseDate(LocalDateTime.now().format(CrmDateUtils.yyyyMMdd));
        List<SfaVisitHelpDefenseRespVo> sfaVisitHelpDefenseRespVos = sfaVisitHelpDefenseMapper.findHelpDefenseSchedule(reqVo);
        if (sfaVisitHelpDefenseRespVos.size() > 0) {
            throw new BusinessException("已有正在协访的客户");
        }
        return sfaHelpDefenseDetailEntity;
    }

    /**
     * 离店参数校验
     *
     * @param reqVo
     */
    private SfaHelpDefenseDetailEntity checkOutStore(SfaVisitHelpDefenseReqVo reqVo) {
//        AssertUtils.isNotEmpty(reqVo.getOutStore(), "离店时间为空");
        AssertUtils.isNotEmpty(reqVo.getOutStoreAddress(), "离店地址为空");
        if (reqVo.getOutStoreLatitude() == null && reqVo.getOutStoreLongitude() == null) {
            throw new BusinessException("离店经纬度为空");
        }
        AssertUtils.isNotEmpty(reqVo.getId(), "协访明细id为空");

        SfaHelpDefenseDetailEntity sfaHelpDefenseDetailEntity = sfaHelpDefenseDetailService.lambdaQuery()
                .eq(SfaHelpDefenseDetailEntity::getId, reqVo.getId()).one();
        if (sfaHelpDefenseDetailEntity == null) {
            throw new BusinessException("未查询到协访计划明细信息");
        }
        return sfaHelpDefenseDetailEntity;
    }

    /**
     * 协访-步骤协访参数校验
     *
     * @param reqVo
     */
    private void checkHelpDefenseStep(SfaVisitHelpDefenseDetailReqVo reqVo) {
        AssertUtils.isNotEmpty(reqVo.getId(), "协访计划明细id为空");
        AssertUtils.isNotEmpty(reqVo.getDirectoryName(), "协访步骤名称为空");
        AssertUtils.isNotEmpty(reqVo.getDirectoryCode(), "协访步骤编码为空");
        AssertUtils.isNotEmpty(reqVo.getIsQualified(), "协访是否合格为空");
    }


    /**
     * 配置按钮
     *
     * @param helpDefenseStatus
     * @param userName
     * @return
     */
    private Map<String, String> getButtons(String userName, String helpDefenseStatus, String defenseDate) {
        LocalDate now = LocalDate.now();
        LocalDate localVisitDate = LocalDate.parse(defenseDate, CrmDateUtils.yyyyMMdd);
        Map<String, String> map = new HashMap<>();
        String username = UserUtils.getUser().getUsername();
        AssertUtils.isNotEmpty(username, "未获取到登陆人信息");
        map.put(SfaVisitEnum.visitButton.NAVIGATION.getVal(), SfaVisitEnum.visitButton.NAVIGATION.getDesc());
        map.put(SfaVisitEnum.visitButton.PHONE.getVal(), SfaVisitEnum.visitButton.PHONE.getDesc());
        map.put(SfaVisitEnum.visitButton.DETAIL.getVal(), SfaVisitEnum.visitButton.DETAIL.getDesc());
        map.put(SfaVisitEnum.visitButton.VISIT.getVal(), SfaVisitEnum.visitButton.VISIT.getDesc());
        if (!now.isEqual(localVisitDate)) {
            map.remove(SfaVisitEnum.visitButton.DETAIL.getVal());
            map.remove(SfaVisitEnum.visitButton.VISIT.getVal());
        }
        if (helpDefenseStatus.equals(SfaVisitEnum.visitStatus.V1.getVal())) {
            map.remove(SfaVisitEnum.visitButton.DETAIL.getVal());
            //判断是否为当前登陆人
            if (!userName.equals(username)) {
                map.remove(SfaVisitEnum.visitButton.VISIT.getVal());
            }
        } else if (helpDefenseStatus.equals(SfaVisitEnum.visitStatus.V2.getVal())) {
            map.remove(SfaVisitEnum.visitButton.DETAIL.getVal());
            if (!userName.equals(username)) {
                map.remove(SfaVisitEnum.visitButton.VISIT.getVal());
            }
        } else if (helpDefenseStatus.equals(SfaVisitEnum.visitStatus.V3.getVal())) {
            map.remove(SfaVisitEnum.visitButton.VISIT.getVal());
        } else if (helpDefenseStatus.equals(SfaVisitEnum.visitStatus.V4.getVal())) {
            map.remove(SfaVisitEnum.visitButton.VISIT.getVal());
        }
        return map;
    }

    /**
     * 查询
     *
     * @param reqVo
     * @return sfaVisitHelpDefenseRespVo
     */
    @Override
    public SfaVisitHelpDefenseRespVo query(SfaVisitHelpDefenseReqVo reqVo) {
        return null;
    }

    /**
     * 新增
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void save(SfaVisitHelpDefenseReqVo reqVo) {
        SfaVisitHelpDefenseEntity entity = CrmBeanUtil.copy(reqVo, SfaVisitHelpDefenseEntity.class);
        this.save(entity);
    }

    /**
     * 更新
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(SfaVisitHelpDefenseReqVo reqVo) {
        SfaVisitHelpDefenseEntity entity = this.getById(reqVo.getId());
        this.updateById(entity);
    }
}
