package com.biz.crm.integral.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.base.ApiResultUtil;
import com.biz.crm.common.PageResult;
import com.biz.crm.eunm.sfa.IntegralEnum;
import com.biz.crm.integral.mapper.SfaIntegralDetailMapper;
import com.biz.crm.integral.model.SfaIntegralDetailEntity;
import com.biz.crm.integral.model.SfaIntegralRangeEntity;
import com.biz.crm.integral.model.SfaIntegralRuleEntity;
import com.biz.crm.integral.model.SfaIntegralTaskEntity;
import com.biz.crm.integral.req.SfaIntegralDetailAddReq;
import com.biz.crm.integral.service.SfaIntegralDetailService;
import com.biz.crm.integral.service.SfaIntegralRangeService;
import com.biz.crm.integral.service.SfaIntegralRuleService;
import com.biz.crm.integral.service.SfaIntegralTaskService;
import com.biz.crm.mdm.org.MdmOrgFeign;
import com.biz.crm.mdm.position.MdmPositionFeign;
import com.biz.crm.nebular.mdm.org.req.MdmOrgReqVo;
import com.biz.crm.nebular.mdm.org.resp.MdmOrgRespVo;
import com.biz.crm.nebular.mdm.position.resp.MdmPositionRespVo;
import com.biz.crm.nebular.mdm.position.resp.MdmPositionUserOrgRespVo;
import com.biz.crm.nebular.sfa.integral.req.SfaIntegralDetailReqVo;
import com.biz.crm.nebular.sfa.integral.req.SfaIntegralDetailUserReqVo;
import com.biz.crm.nebular.sfa.integral.req.SfaIntegralRankAppReqVo;
import com.biz.crm.nebular.sfa.integral.req.SfaIntegralRankReqVo;
import com.biz.crm.nebular.sfa.integral.resp.SfaIntegralDetailRespVo;
import com.biz.crm.nebular.sfa.integral.resp.SfaIntegralRankRespVo;
import com.biz.crm.nebular.sfa.integral.resp.SfaIntegralUserRankRespVo;
import com.biz.crm.util.*;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 积分明细
 *
 * @author YuanZiJian
 * @date 2021/1/12 10:04
 * @version: V1.0
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "SfaIntegralDetailServiceExpandImpl")
public class SfaIntegralDetailServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<SfaIntegralDetailMapper, SfaIntegralDetailEntity> implements SfaIntegralDetailService {

    @Resource
    MdmPositionFeign mdmPositionFeign;

    @Resource
    SfaIntegralRangeService sfaIntegralRangeService;

    @Resource
    SfaIntegralTaskService sfaIntegralTaskService;

    @Resource
    SfaIntegralRuleService sfaIntegralRuleService;

    @Resource
    SfaIntegralDetailMapper sfaIntegralDetailMapper;

    @Resource
    MdmOrgFeign mdmOrgFeign;


    /**
     * 分页查询积分明细
     *
     * @param sfaIntegralDetailReqVo
     * @return com.biz.crm.common.PageResult<com.biz.crm.nebular.sfa.integral.resp.SfaIntegralDetailRespVo>
     * @method pageIntegralDetail
     * @date: 2021/1/12 10:53
     * @author: YuanZiJian
     */
    @Override
    public PageResult<SfaIntegralDetailRespVo> integralDetailReport(SfaIntegralDetailReqVo sfaIntegralDetailReqVo) {
        Page<SfaIntegralDetailReqVo> page = new Page(sfaIntegralDetailReqVo.getPageNum(), sfaIntegralDetailReqVo.getPageSize());
        List<SfaIntegralDetailEntity> records = sfaIntegralDetailMapper.integralDetailReport(page, sfaIntegralDetailReqVo);
        List<SfaIntegralDetailRespVo> sfaIntegralDetailRespVos = CrmBeanUtil.copyList(records, SfaIntegralDetailRespVo.class);
        return PageResult.<SfaIntegralDetailRespVo>builder()
                .data(sfaIntegralDetailRespVos)
                .count(page.getTotal())
                .build();
    }

    /**
     * 完成任务，获得积分
     *
     * @param reqVo
     * @return void
     * @method integralDetailAdd
     * @date: 2021/1/12 13:40
     * @author: YuanZiJian
     */
    @Override
    @Transactional
    public void integralDetailAdd(SfaIntegralDetailAddReq reqVo) {
        if (!this.checkForIntegralDetailAdd(reqVo)) {
            return;
        }
        //获取积分新增时的用户信息
        UserRedis user = this.userRedisForIntegralAdd(reqVo);
        //匹配该用户的积分规则
        SfaIntegralRuleEntity ruleEntity = this.matchingIntegralRule(reqVo, user);
        if(ruleEntity==null){
            return;
        }
        //通过规则Id获得对应的计算规则没有获得则说明没有被计算规则覆盖
        SfaIntegralTaskEntity task = sfaIntegralTaskService.getOne(Wrappers.lambdaQuery(SfaIntegralTaskEntity.class)
                .eq(SfaIntegralTaskEntity::getIntegralRuleId, ruleEntity.getId())
                .eq(SfaIntegralTaskEntity::getIntegralTaskType, reqVo.getIntegralTaskType()));
        if (task == null) {
            log.info("没有相应的计算规则，不加积分");
            return;
        }
        //封装一条积分记录
        SfaIntegralDetailEntity sfaDetailEntity = this.buildSfaDetailEntity(reqVo, ruleEntity, task, user);
        SfaIntegralDetailEntity sfaDetailEntityOld = this.lambdaQuery().eq(SfaIntegralDetailEntity :: getBizId , sfaDetailEntity.getBizId()).one();
        if(null != sfaDetailEntityOld){
            sfaDetailEntity.setId(sfaDetailEntityOld.getId());
        }
        this.saveOrUpdate(sfaDetailEntity);
    }

    protected boolean checkForIntegralDetailAdd(SfaIntegralDetailAddReq reqVo){
        if (org.apache.commons.lang3.StringUtils.isBlank(reqVo.getIntegralTaskType())) {
            log.warn("积分类型类型为空, reqVo={}", JsonPropertyUtil.toJsonString(reqVo));
            return false;
        }
        if (org.apache.commons.lang3.StringUtils.isBlank(reqVo.getBizId())) {
            log.warn("积分业务id为空, reqVo={}", JsonPropertyUtil.toJsonString(reqVo));
            return false;
        }
        if (org.apache.commons.lang3.StringUtils.isBlank(reqVo.getPosCode())) {
            log.warn("积分职位编码为空, reqVo={}", JsonPropertyUtil.toJsonString(reqVo));
            return false;
        }
        return true;
    }


    /**
     * 获取积分新增时的用户信息
     * @param reqVo
     * @return
     */
    protected UserRedis userRedisForIntegralAdd (SfaIntegralDetailAddReq reqVo){
        String posCode = reqVo.getPosCode();
        //获取登录用户对象的所属组织和所属职位等级的编码
        UserRedis user = UserUtils.getUser();
        if (!StringUtils.isEmpty(posCode)){
            if(user==null){
                UserUtils.doTokenForNull();
                UserRedis userTemp = UserUtils.getUser();
                if(null == userTemp){
                    log.error("积分新增异常：设置会话失败");
                }
            }
            //根据职位编码查询职位及关联用户、组织、上级信息
            MdmPositionUserOrgRespVo mdmPositionUserOrgRespVo = ApiResultUtil.objResult(mdmPositionFeign.getPositionUserOrgByPositionCode(posCode), true);
            if (!ObjectUtils.isEmpty(mdmPositionUserOrgRespVo)){
                user.setPoscode(mdmPositionUserOrgRespVo.getPositionCode())
                        .setUsername(mdmPositionUserOrgRespVo.getUserName())
                        .setPosname(mdmPositionUserOrgRespVo.getPositionName())
                        .setOrgcode(mdmPositionUserOrgRespVo.getOrgCode())
                        .setOrgname(mdmPositionUserOrgRespVo.getOrgName())
                        .setRealname(mdmPositionUserOrgRespVo.getFullName());
            }
        }
        return user;
    }
    /**
     * 匹配该用户的积分规则
     * @param userRedis
     * @return
     */
    protected SfaIntegralRuleEntity matchingIntegralRule(SfaIntegralDetailAddReq reqVo, UserRedis userRedis){

        List<SfaIntegralRangeEntity> sfaIntegralRangeEntities = Lists.newArrayList();
        MdmPositionRespVo positionRespVo = PositionUtil.getPositionByCode(userRedis.getPoscode());
        if (!ObjectUtils.isEmpty(positionRespVo)){
            String positionLevelCode = positionRespVo.getPositionLevelCode();
            String orgCode = userRedis.getOrgcode();
            //通过两个编码查出对应的规则,没有则没有被规则覆盖
            sfaIntegralRangeEntities = sfaIntegralRangeService.list(Wrappers.lambdaQuery(SfaIntegralRangeEntity.class)
                    .in(SfaIntegralRangeEntity::getCode, Lists.newArrayList(positionLevelCode, orgCode)));
        }
        if (CollectionUtil.listEmpty(sfaIntegralRangeEntities) ) {
            log.info("当前用户没有被积分规则覆盖");
            return null;
        }
        //获得对应的规则Id
        String integralRuleId = sfaIntegralRangeEntities.get(0).getIntegralRuleId();
        SfaIntegralRuleEntity ruleEntity = sfaIntegralRuleService.getOne(Wrappers.lambdaQuery(SfaIntegralRuleEntity.class).eq(SfaIntegralRuleEntity::getId, integralRuleId));
        return ruleEntity;
    }

    /**
     * 分页查询积分排名——实现分页和模糊
     *
     * @param sfaIntegralRankReqVo
     * @return com.biz.crm.common.PageResult<com.biz.crm.nebular.sfa.integral.resp.SfaIntegralRankRespVo>
     * @method pageIntegralRank
     * @date: 2021/1/12 16:09
     * @author: YuanZiJian
     */
    @Override
    public PageResult<SfaIntegralRankRespVo> pageIntegralRank(SfaIntegralRankReqVo sfaIntegralRankReqVo) {
        Page<SfaIntegralDetailEntity> page = new Page(sfaIntegralRankReqVo.getPageNum(), sfaIntegralRankReqVo.getPageSize());
        List<SfaIntegralDetailEntity> sfaIntegralDetailEntities = sfaIntegralDetailMapper.pageDetailNum(page, sfaIntegralRankReqVo);
        //获取用户的信息Fullname
        List<SfaIntegralRankRespVo> sfaIntegralRankRespVos = CrmBeanUtil.copyList(sfaIntegralDetailEntities, SfaIntegralRankRespVo.class);
        int rank = 0;
        if (sfaIntegralRankReqVo.getPageNum() > 0){
            rank = (sfaIntegralRankReqVo.getPageNum()-1) * sfaIntegralRankReqVo.getPageSize();
        }
        for (SfaIntegralRankRespVo sfaIntegralRankRespVo : sfaIntegralRankRespVos) {
            MdmPositionRespVo result = mdmPositionFeign.getPrimaryPositionByUserName(sfaIntegralRankRespVo.getUserName()).getResult();
            if (!ObjectUtils.isEmpty(result)){
                sfaIntegralRankRespVo.setFullName(result.getFullName());
                sfaIntegralRankRespVo.setPosName(result.getPositionName());
                Result<MdmPositionRespVo> detail = mdmPositionFeign.detail(null, result.getPositionCode());
                sfaIntegralRankRespVo.setOrgName(result.getOrgName());
                sfaIntegralRankRespVo.setPosLevelName(detail.getResult().getPositionLevelName());
                sfaIntegralRankRespVo.setRank(++rank);
            }
        }
        return PageResult.<SfaIntegralRankRespVo>builder()
                .data(sfaIntegralRankRespVos)
                .count(page.getTotal())
                .build();
    }

    /**
     * 查询个人的积分明细
     * 查询方式userName不模糊
     * 当userName为Null时则默认取登陆用户的userName
     * @param sfaIntegralDetailUserReqVo 积分明细Vo 用于查询个人的积分明细
     * @return com.biz.crm.common.PageResult<com.biz.crm.nebular.sfa.integral.resp.SfaIntegralDetailRespVo>
     * @method pageDetailByLoginUser
     * @date: 2021/1/15 10:13
     * @author: YuanZiJian
     */
    @Override
    public PageResult<SfaIntegralDetailRespVo> pageDetailByLoginUser(SfaIntegralDetailUserReqVo sfaIntegralDetailUserReqVo) {
        if(StringUtils.isEmpty(sfaIntegralDetailUserReqVo.getUserName())){
            UserRedis user = UserUtils.getUser();
            sfaIntegralDetailUserReqVo.setUserName(user.getUsername());
        }
        Page<SfaIntegralDetailEntity> page = new Page(sfaIntegralDetailUserReqVo.getPageNum(),sfaIntegralDetailUserReqVo.getPageSize());
        Page<SfaIntegralDetailEntity> integralDetailEntityPage = this.page(page, Wrappers.lambdaQuery(SfaIntegralDetailEntity.class).eq(SfaIntegralDetailEntity::getUserName, sfaIntegralDetailUserReqVo.getUserName()));
        List<SfaIntegralDetailEntity> detailEntityPageRecords = integralDetailEntityPage.getRecords();
        List<SfaIntegralDetailRespVo> sfaIntegralDetailRespVos = CrmBeanUtil.copyList(detailEntityPageRecords, SfaIntegralDetailRespVo.class);
        return PageResult.<SfaIntegralDetailRespVo>builder()
                .data(sfaIntegralDetailRespVos)
                .count(page.getTotal())
                .build();
    }

    /**
     * 查询userName的用户的总积分
     * 当userName为Null时则默认取登陆用户的userName
     * @return java.lang.Integer
     * @method countIntegralNumByLoginUser
     * @date: 2021/1/15 10:15
     * @author: YuanZiJian
     */
    @Override
    public SfaIntegralUserRankRespVo countIntegralNumByUserNameRedis(String userName) {
        if (StringUtils.isEmpty(userName)){
            UserRedis user = UserUtils.getUser();
            userName= user.getUsername();
        }
        List<SfaIntegralUserRankRespVo> sfaIntegralUserRankRespVos = sfaIntegralDetailMapper.listIntegralRank();
        String finalUserName = userName;
        List<SfaIntegralUserRankRespVo> collect = sfaIntegralUserRankRespVos.stream().filter(data -> finalUserName.equals(data.getUserName())).collect(Collectors.toList());
        //collect为空 则没有明细相关数据
        if(CollectionUtil.listEmpty(collect)){
            return new SfaIntegralUserRankRespVo().setUserName(userName).setIntegralNum(0).setRank(0);
        }
        return collect.get(0);
    }

    /**
     * 分页查询积分排名——实现分页和组织类型查询
     *
     * @param sfaIntegralRankAppReqVo 小程序端查询条件封装
     * @return com.biz.crm.common.PageResult<com.biz.crm.nebular.sfa.integral.resp.SfaIntegralRankRespVo>
     * @method pageIntegralRankToApp
     * @date: 2021/1/15 13:55
     * @author: YuanZiJian
     */
    @Override
    public PageResult<SfaIntegralRankRespVo> pageIntegralRankToApp(SfaIntegralRankAppReqVo sfaIntegralRankAppReqVo) {
        String userOrgCode = UserUtils.getUser().getOrgcode();
        SfaIntegralRankReqVo sfaIntegralRankReqVo = CrmBeanUtil.copy(sfaIntegralRankAppReqVo, SfaIntegralRankReqVo.class);
        if(StringUtils.isNotEmpty(sfaIntegralRankAppReqVo.getOrgRange())){
            //封装查询条件
            sfaIntegralRankReqVo = getSearchVo(userOrgCode, sfaIntegralRankReqVo, sfaIntegralRankAppReqVo.getOrgRange());
        }
        return pageIntegralRank(sfaIntegralRankReqVo);
    }

    /**
     * 封装需要查询的组织范围
     * @method getSearchVo
     * @date: 2021/1/15 15:37
     * @author: YuanZiJian
     * @param userOrgCode
     * @param sfaIntegralRankReqVo
     * @param orgRange
     * @return com.biz.crm.nebular.sfa.integral.req.SfaIntegralRankReqVo
     */
    private SfaIntegralRankReqVo getSearchVo(String userOrgCode,SfaIntegralRankReqVo sfaIntegralRankReqVo, String orgRange){
        //封装查条件——本部门
        if(IntegralEnum.orgSearchRange.NOW_ORG.getDes().equals(orgRange)){
            List<String> orgCodeList = new ArrayList<>();
            orgCodeList.add(userOrgCode);
            sfaIntegralRankReqVo.setOrgCodeList(orgCodeList);
        }
        //封装查条件——本公司
        if(IntegralEnum.orgSearchRange.ALL_ORG.getDes().equals(orgRange)){
            MdmOrgRespVo topParentOrgByCode = OrgUtil.getTopParentOrgByCode(userOrgCode);
            List<String> childrenOrgCodeListIncludeSelf = OrgUtil.getChildrenOrgCodeListIncludeSelf(topParentOrgByCode.getOrgCode());
            sfaIntegralRankReqVo.setOrgCodeList(childrenOrgCodeListIncludeSelf);
        }
        //封装查条件——当前组织及下级组织
        if(IntegralEnum.orgSearchRange.AFTER_ORG.getDes().equals(orgRange)){
            Result<List<MdmOrgRespVo>> result = mdmOrgFeign.findAllChildrenOrgList(new MdmOrgReqVo().setOrgCode(userOrgCode));
            List<String> allChildrenOrgList = result.getResult().stream().map(MdmOrgRespVo::getOrgCode).collect(Collectors.toList());
            sfaIntegralRankReqVo.setOrgCodeList(allChildrenOrgList);
        }
        return sfaIntegralRankReqVo;
    }




    /**
     * 封装积分明细实体
     *
     * @param ruleEntity   规则实体
     * @param task         计算规则实体
     * @param user         登陆用户
     * @return com.biz.crm.integral.model.SfaIntegralDetailEntity
     * @method getSfaDetailEntity
     * @date: 2021/1/12 15:01
     * @author: YuanZiJian
     */
    protected SfaIntegralDetailEntity buildSfaDetailEntity(SfaIntegralDetailAddReq reqVo, SfaIntegralRuleEntity ruleEntity, SfaIntegralTaskEntity task, UserRedis user) {
        SfaIntegralDetailEntity sfaIntegralDetailEntity = new SfaIntegralDetailEntity();
        sfaIntegralDetailEntity.setBizId(reqVo.getBizId());
        sfaIntegralDetailEntity.setIntegralNum(task.getIntegralTaskReward());
        sfaIntegralDetailEntity.setIntegralRuleId(ruleEntity.getId());
        sfaIntegralDetailEntity.setIntegralTaskType(task.getIntegralTaskType());
        sfaIntegralDetailEntity.setIntegralRuleName(ruleEntity.getIntegralRuleName());
        sfaIntegralDetailEntity.setPosName(user.getPosname());
        sfaIntegralDetailEntity.setPosCode(user.getPoscode());
        sfaIntegralDetailEntity.setUserName(user.getUsername());
        sfaIntegralDetailEntity.setFullName(user.getRealname());
        sfaIntegralDetailEntity.setOrgCode(user.getOrgcode());
        sfaIntegralDetailEntity.setOrgName(user.getOrgname());
        sfaIntegralDetailEntity.setIntegralRuleCode(ruleEntity.getIntegralRuleCode());
        return sfaIntegralDetailEntity;
    }


    /**
     * 模糊查询相关字段拼接
     * @method getWrapper
     * @date: 2021/1/13 18:43
     * @author: YuanZiJian
     * @param sfaIntegralDetailReqVo
     * @return com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper
     */
    private LambdaQueryWrapper getWrapper(SfaIntegralDetailReqVo sfaIntegralDetailReqVo){
        LambdaQueryWrapper<SfaIntegralDetailEntity> lambdaQueryWrapper = Wrappers.lambdaQuery(SfaIntegralDetailEntity.class);
        if (StringUtils.isNotEmpty(sfaIntegralDetailReqVo.getFullName())){
            lambdaQueryWrapper.like(SfaIntegralDetailEntity::getFullName, sfaIntegralDetailReqVo.getFullName());
        }
        if (StringUtils.isNotEmpty(sfaIntegralDetailReqVo.getIntegralRuleId())){
            lambdaQueryWrapper.like(SfaIntegralDetailEntity::getIntegralRuleId, sfaIntegralDetailReqVo.getIntegralRuleId());
        }
        if (StringUtils.isNotEmpty(sfaIntegralDetailReqVo.getIntegralRuleName())){
            lambdaQueryWrapper.like(SfaIntegralDetailEntity::getIntegralRuleName, sfaIntegralDetailReqVo.getIntegralRuleName());
        }
        if (StringUtils.isNotEmpty(sfaIntegralDetailReqVo.getFullName())){
            lambdaQueryWrapper.like(SfaIntegralDetailEntity::getFullName, sfaIntegralDetailReqVo.getFullName());
        }
        if (StringUtils.isNotEmpty(sfaIntegralDetailReqVo.getPosName())){
            lambdaQueryWrapper.like(SfaIntegralDetailEntity::getPosName, sfaIntegralDetailReqVo.getPosName());
        }
        lambdaQueryWrapper.orderByDesc(SfaIntegralDetailEntity::getCreateDateAll);
        return lambdaQueryWrapper;
    }

}
