package com.biz.crm.integral.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.base.BusinessException;
import com.biz.crm.common.PageResult;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.eunm.sfa.IntegralEnum;
import com.biz.crm.integral.mapper.SfaIntegralRuleMapper;
import com.biz.crm.integral.model.SfaIntegralRangeEntity;
import com.biz.crm.integral.model.SfaIntegralRuleEntity;
import com.biz.crm.integral.model.SfaIntegralTaskEntity;
import com.biz.crm.integral.service.SfaIntegralRangeService;
import com.biz.crm.integral.service.SfaIntegralRuleService;
import com.biz.crm.integral.service.SfaIntegralTaskService;
import com.biz.crm.mdm.position.MdmPositionFeign;
import com.biz.crm.nebular.mdm.position.resp.MdmPositionRespVo;
import com.biz.crm.nebular.sfa.integral.req.SfaIntegralRuleReqVo;
import com.biz.crm.nebular.sfa.integral.resp.SfaIntegralRangeRespVo;
import com.biz.crm.nebular.sfa.integral.resp.SfaIntegralRulePageRespVo;
import com.biz.crm.nebular.sfa.integral.resp.SfaIntegralRuleRespVo;
import com.biz.crm.nebular.sfa.integral.resp.SfaIntegralTaskRespVo;
import com.biz.crm.nebular.sfa.notice.resp.SfaNoticeFileRespVo;
import com.biz.crm.util.*;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.List;
import java.util.StringJoiner;
import java.util.stream.Collectors;

/**
 * 积分规则实现类
 *
 * @author YuanZiJian
 * @date 2021/1/8 14:26
 * @version: V1.0
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "SfaIntegralRuleServiceExpandImpl")
public class SfaIntegralRuleServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<SfaIntegralRuleMapper, SfaIntegralRuleEntity> implements SfaIntegralRuleService {

    @Autowired
    SfaIntegralRangeService sfaIntegralRangeService;

    @Autowired
    SfaIntegralTaskService sfaIntegralTaskService;

    @Autowired
    SfaIntegralRuleMapper sfaIntegralRuleMapper;

    @Autowired
    MdmPositionFeign mdmPositionFeign;

    /**
     * 新增积分规则
     *
     * @param sfaIntegralRuleReqVo
     * @return void
     * @method saveIntegralRule
     * @date: 2021/1/8 14:24
     * @author: YuanZiJian
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void saveIntegralRule(SfaIntegralRuleReqVo sfaIntegralRuleReqVo) {
        //数据校验
        checkRuleVo(sfaIntegralRuleReqVo);
        //更新时额外校验参数
        if (StringUtils.isNotEmpty(sfaIntegralRuleReqVo.getId())) {
            throw new BusinessException("新增时积分规则id必须为空");
        }
        //职位级别
        List<SfaIntegralRangeEntity> sfaIntegralRangeEntities = CrmBeanUtil.copyList(sfaIntegralRuleReqVo.getPosLevel(), SfaIntegralRangeEntity.class);
        //组织
        List<SfaIntegralRangeEntity> sfaIntegralRangeEntitiesOrg = CrmBeanUtil.copyList(sfaIntegralRuleReqVo.getOrgs(), SfaIntegralRangeEntity.class);
        //设置职位级别类型
        sfaIntegralRangeEntities.stream().forEach(data->data.setType(IntegralEnum.ruleType.POS_LEVEL.getValue()));
        //设置组织类型
        sfaIntegralRangeEntitiesOrg.stream().forEach(data->data.setType(IntegralEnum.ruleType.ORG.getValue()));
        //校验组织和角色是否与已存在规则有交叉部分
        checkOrgAndRole(sfaIntegralRangeEntitiesOrg,sfaIntegralRangeEntities,sfaIntegralRuleReqVo.getId());
        //通过上面的校验则说明可以新增职位
        SfaIntegralRuleEntity sfaIntegralRuleEntity = CrmBeanUtil.copy(sfaIntegralRuleReqVo, SfaIntegralRuleEntity.class);
        sfaIntegralRuleEntity.setIntegralRuleCode(CodeUtil.generateCode(IntegralEnum.ruleCode.INTEGRAL_RULE_CODE.getValue()));
        List<SfaIntegralTaskEntity> sfaIntegralTaskEntities = CrmBeanUtil.copyList(sfaIntegralRuleReqVo.getTasks(), SfaIntegralTaskEntity.class);
        this.save(sfaIntegralRuleEntity);
        //将角色组织覆盖范围和积分计算规则(积分任务) 与积分规则Id关联
        sfaIntegralRangeEntities.stream().forEach(data -> data.setIntegralRuleId(sfaIntegralRuleEntity.getId()));
        sfaIntegralTaskEntities.stream().forEach(data -> data.setIntegralRuleId(sfaIntegralRuleEntity.getId()));
        //存入数据库
        sfaIntegralRangeService.saveBatch(sfaIntegralRangeEntities);
        sfaIntegralTaskService.saveBatch(sfaIntegralTaskEntities);
    }

    /**
     * 积分规则删除——物理删除
     *
     * @param id 积分规则主键Id
     * @return void
     * @method deleteIntegralRule
     * @date: 2021/1/8 16:37
     * @author: YuanZiJian
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteIntegralRule(String id) {
        //删除积分计算规则
        sfaIntegralTaskService.remove(Wrappers.lambdaQuery(SfaIntegralTaskEntity.class).eq(SfaIntegralTaskEntity::getIntegralRuleId, id));
        //删除积分覆盖的组织和角色
        sfaIntegralRangeService.remove(Wrappers.lambdaQuery(SfaIntegralRangeEntity.class).eq(SfaIntegralRangeEntity::getIntegralRuleId, id));
        //删除积分规则主表的信息
        this.removeById(id);
    }

    /**
     * 更新积分规则
     *
     * @param sfaIntegralRuleReqVo
     * @return void
     * @method updateIntegralRule
     * @date: 2021/1/8 16:42
     * @author: YuanZiJian
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateIntegralRule(SfaIntegralRuleReqVo sfaIntegralRuleReqVo) {
        //数据校验
        checkRuleVo(sfaIntegralRuleReqVo);
        //更新时额外校验参数
        if (StringUtils.isEmpty(sfaIntegralRuleReqVo.getId())) {
            throw new BusinessException("积分规则id为空");
        }
        List<SfaIntegralRangeEntity> sfaIntegralRangeEntities = CrmBeanUtil.copyList(sfaIntegralRuleReqVo.getPosLevel(), SfaIntegralRangeEntity.class);
        List<SfaIntegralRangeEntity> sfaIntegralRangeEntitiesOrg = CrmBeanUtil.copyList(sfaIntegralRuleReqVo.getOrgs(), SfaIntegralRangeEntity.class);
        sfaIntegralRangeEntities.stream().forEach(data->data.setType(IntegralEnum.ruleType.POS_LEVEL.getValue()));
        sfaIntegralRangeEntitiesOrg.stream().forEach(data->data.setType(IntegralEnum.ruleType.ORG.getValue()));
        //校验组织和角色是否与已存在规则有交叉部分
        checkOrgAndRole(sfaIntegralRangeEntitiesOrg,sfaIntegralRangeEntities,sfaIntegralRuleReqVo.getId());
        //通过上面的校验则说明可以新增职位
        SfaIntegralRuleEntity sfaIntegralRuleEntity = CrmBeanUtil.copy(sfaIntegralRuleReqVo, SfaIntegralRuleEntity.class);
        List<SfaIntegralTaskEntity> sfaIntegralTaskEntities = CrmBeanUtil.copyList(sfaIntegralRuleReqVo.getTasks(), SfaIntegralTaskEntity.class);
        //将角色组织覆盖范围和积分计算规则(积分任务) 与积分规则Id关联
        sfaIntegralRangeEntities.stream().forEach(data->data.setIntegralRuleId(sfaIntegralRuleReqVo.getId()));
        sfaIntegralTaskEntities.stream().forEach(data->data.setIntegralRuleId(sfaIntegralRuleReqVo.getId()));
        this.updateById(sfaIntegralRuleEntity);
        sfaIntegralRangeService.remove(Wrappers.lambdaQuery(SfaIntegralRangeEntity.class).eq(SfaIntegralRangeEntity::getIntegralRuleId, sfaIntegralRuleReqVo.getId()));
        sfaIntegralTaskService.remove(Wrappers.lambdaQuery(SfaIntegralTaskEntity.class).eq(SfaIntegralTaskEntity::getIntegralRuleId, sfaIntegralRuleReqVo.getId()));
        sfaIntegralRangeService.saveBatch(sfaIntegralRangeEntities);
        sfaIntegralTaskService.saveBatch(sfaIntegralTaskEntities);
    }

    /**
     * 根据Id查询某个积分规则的详细信息
     *
     * @param id 积分规则id
     * @return com.biz.crm.nebular.sfa.Integral.resp.SfaIntegralRuleRespVo
     * @method query
     * @date: 2021/1/8 17:34
     * @author: YuanZiJian
     */
    @Override
    public SfaIntegralRuleRespVo queryById(String id) {
        List<SfaIntegralRangeEntity> sfaIntegralRangeEntities = sfaIntegralRangeService.list(Wrappers.lambdaQuery(SfaIntegralRangeEntity.class).eq(SfaIntegralRangeEntity::getIntegralRuleId, id));
        List<SfaIntegralTaskEntity> sfaIntegralTaskEntities = sfaIntegralTaskService.list(Wrappers.lambdaQuery(SfaIntegralTaskEntity.class).eq(SfaIntegralTaskEntity::getIntegralRuleId, id));
        SfaIntegralRuleEntity sfaIntegralRuleEntity = this.getOne(Wrappers.lambdaQuery(SfaIntegralRuleEntity.class).eq(SfaIntegralRuleEntity::getId, id));
        //封装Vo
        SfaIntegralRuleRespVo ruleRespVo = getRuleRespVo(sfaIntegralRangeEntities, sfaIntegralTaskEntities, sfaIntegralRuleEntity);
        return ruleRespVo;
    }

    /**
     * 根据条件查出所有的积分规则
     * 条件：积分规则Id、积分规则名称、覆盖的角色编码和角色名、覆盖的组织编码和组织名
     * 分页实现
     *
     * @param sfaIntegralRuleReqVo
     * @return com.biz.crm.common.PageResult<com.biz.crm.nebular.sfa.Integral.resp.SfaIntegralRuleRespVo>
     * @method pageIntegralRule
     * @date: 2021/1/11 11:11
     * @author: YuanZiJian
     */
    @Override
    public PageResult<SfaIntegralRulePageRespVo> pageIntegralRule(SfaIntegralRuleReqVo sfaIntegralRuleReqVo) {
        Page<SfaIntegralRuleEntity> page = new Page(sfaIntegralRuleReqVo.getPageNum(),sfaIntegralRuleReqVo.getPageSize());
        List<SfaIntegralRulePageRespVo> pageRule = sfaIntegralRuleMapper.pageRule(page,sfaIntegralRuleReqVo);
        pageRule.stream().forEach(data->data.setEnableStatusName(CrmEnableStatusEnum.GETMAP.get(data.getEnableStatus())));
        return PageResult.<SfaIntegralRulePageRespVo>builder()
                .data(pageRule)
                .count(page.getTotal())
                .build();
    }

    /**
     * 根据传入ID列表批量删除
     *
     * @param ids 积分规则的Id集合
     * @return void
     * @method deleteByIds
     * @date: 2021/1/11 10:38
     * @author: YuanZiJian
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteByIds(List<String> ids) {
        //批量删除积分计算规则
        sfaIntegralTaskService.remove(Wrappers.lambdaQuery(SfaIntegralTaskEntity.class).in(SfaIntegralTaskEntity::getIntegralRuleId, ids));
        //批量删除积分覆盖的组织和角色
        sfaIntegralRangeService.remove(Wrappers.lambdaQuery(SfaIntegralRangeEntity.class).in(SfaIntegralRangeEntity::getIntegralRuleId, ids));
        //批量删除积分规则主表的信息
        this.removeByIds(ids);
    }

    /**
     * 根据传入ID列表批量启用
     *
     * @param ids 积分规则的Id集合
     * @return void
     * @method enableByIds
     * @date: 2021/1/11 10:39
     * @author: YuanZiJian
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void enableByIds(List<String> ids) {
        this.update(Wrappers.lambdaUpdate(SfaIntegralRuleEntity.class)
                .in(SfaIntegralRuleEntity::getId, ids)
                .set(SfaIntegralRuleEntity::getEnableStatus, CrmEnableStatusEnum.ENABLE.getCode()));
    }

    /**
     * 根据传入ID列表批量禁用
     *
     * @param ids 积分规则的id集合
     * @return void
     * @method disableByIds
     * @date: 2021/1/11 10:40
     * @author: YuanZiJian
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void disableByIds(List<String> ids) {
        this.update(Wrappers.lambdaUpdate(SfaIntegralRuleEntity.class)
                .in(SfaIntegralRuleEntity::getId, ids)
                .set(SfaIntegralRuleEntity::getEnableStatus, CrmEnableStatusEnum.DISABLE.getCode()));
    }

    /**
     * 获取当前用户对应的积分规则的详情
     *
     * @return com.biz.crm.nebular.sfa.integral.resp.SfaIntegralRuleRespVo
     * @method getTaskByUserName
     * @date: 2021/1/15 9:59
     * @author: YuanZiJian
     */
    @Override
    public SfaIntegralRuleRespVo getTaskByLoginUser() {
        //获取登录用户对象的所属组织和所属职位等级的编码
        UserRedis user = UserUtils.getUser();
        Result<MdmPositionRespVo> detail = mdmPositionFeign.detail(null, user.getPoscode());
        String positionLevelCode = detail.getResult().getPositionLevelCode();
        String orgcode = user.getOrgcode();
        //通过两个编码查出对应的规则,没有则没有被规则覆盖
        List<SfaIntegralRangeEntity> sfaIntegralRangeEntities = sfaIntegralRangeService.list(Wrappers.lambdaQuery(SfaIntegralRangeEntity.class)
                .eq(SfaIntegralRangeEntity::getCode, positionLevelCode)
                .or()
                .eq(SfaIntegralRangeEntity::getCode, orgcode));
        if (sfaIntegralRangeEntities.size() == 0 || sfaIntegralRangeEntities.size() > 2) {
            return new SfaIntegralRuleRespVo().setTasks(new ArrayList<>());
        }
        //获得对应的规则Id
        String integralRuleId = sfaIntegralRangeEntities.get(0).getIntegralRuleId();
        return queryById(integralRuleId);
    }


    /**
     * 校验新增规则的组织和角色是否与已存在规则有交叉部分
     *
     * @return void
     * @method checkOrgAndRole
     * @date: 2021/1/8 15:05
     * @author: YuanZiJian
     */
    private void checkOrgAndRole(List<SfaIntegralRangeEntity> sfaIntegralOrgRanges,List<SfaIntegralRangeEntity> sfaIntegralPosRanges,String id) {
        //将组织和角色合并到一个list中
        sfaIntegralPosRanges.addAll(sfaIntegralOrgRanges);
        List<String> integralRangeCodes = sfaIntegralPosRanges.stream().map(SfaIntegralRangeEntity::getCode).collect(Collectors.toList());
        LambdaQueryWrapper<SfaIntegralRangeEntity> queryWrapper = Wrappers.lambdaQuery(SfaIntegralRangeEntity.class);
        //判断是新增还是修改——新增无需排除自身，修改需要排除自身
        if(StringUtils.isNotEmpty(id)){
            queryWrapper.ne(SfaIntegralRangeEntity::getIntegralRuleId,id);
        }
        List<SfaIntegralRangeEntity> rangeEntities = sfaIntegralRangeService.list(queryWrapper.in(SfaIntegralRangeEntity::getCode, integralRangeCodes));
        //若没有查出，说明组织和规则都没有被使用
        if(CollectionUtil.listEmpty(rangeEntities)){
            return;
        }
        //按照组织分类得到组织关联的规则Id
        List<String> orgRuleId = rangeEntities.stream().filter(data -> data.getType().equals(IntegralEnum.ruleType.ORG.getValue()))
                .map(SfaIntegralRangeEntity::getIntegralRuleId)
                .collect(Collectors.toList());
        //按照职位级别分类得到职位级别关联的规则Id
        List<String> posRuleId = rangeEntities.stream().filter(data -> data.getType().equals(IntegralEnum.ruleType.POS_LEVEL.getValue()))
                .map(SfaIntegralRangeEntity::getIntegralRuleId)
                .collect(Collectors.toList());
        //取组织和职位关联规则Id的交集——只要有交集则说明重复
        List<String> ruleId = orgRuleId.stream().filter(data -> posRuleId.contains(data)).collect(Collectors.toList());
        if(CollectionUtil.listNotEmpty(ruleId)){
            List<SfaIntegralRuleEntity> ruleEntities = this.list(Wrappers.lambdaQuery(SfaIntegralRuleEntity.class).in(SfaIntegralRuleEntity::getId, ruleId));
            StringJoiner sj = new StringJoiner("、");
            ruleEntities.forEach(data->sj.add(data.getIntegralRuleName()));
            throw new BusinessException("与其他积分规则重复："+sj.toString());
        }
    }

    /**
     * Vo字段基本校验
     *
     * @param sfaIntegralRuleReqVo
     * @return void
     * @method checkRuleVo
     * @date: 2021/1/8 15:48
     * @author: YuanZiJian
     */
    private void checkRuleVo(SfaIntegralRuleReqVo sfaIntegralRuleReqVo) {
        if (StringUtils.isEmpty(sfaIntegralRuleReqVo.getIntegralRuleName())) {
            throw new BusinessException("规则名称为空");
        }
        if (StringUtils.isEmpty(sfaIntegralRuleReqVo.getStartTime()) || StringUtils.isEmpty(sfaIntegralRuleReqVo.getEndTime())) {
            throw new BusinessException("有效期参数错误");
        }
        if(CollectionUtil.listEmpty(sfaIntegralRuleReqVo.getPosLevel())){
            throw new BusinessException("职位列表为空");
        }
        if(CollectionUtil.listEmpty(sfaIntegralRuleReqVo.getOrgs())){
            throw new BusinessException("组织列表为空");
        }
    }

    /**
     * 封装积分规则RespVo
     *
     * @param sfaIntegralRangeEntities 积分覆盖的组织和角色
     * @param sfaIntegralTaskEntities  积分计算规则
     * @param sfaIntegralRuleEntity    积分规则基本字段
     * @return com.biz.crm.nebular.sfa.Integral.resp.SfaIntegralRuleRespVo 积分规则Vo返回实体
     * @method getRuleRespVo
     * @date: 2021/1/8 17:59
     * @author: YuanZiJian
     */
    private SfaIntegralRuleRespVo getRuleRespVo(List<SfaIntegralRangeEntity> sfaIntegralRangeEntities
            , List<SfaIntegralTaskEntity> sfaIntegralTaskEntities, SfaIntegralRuleEntity sfaIntegralRuleEntity) {
        List<SfaIntegralRangeRespVo> sfaIntegralRangeRespVos = CrmBeanUtil.copyList(sfaIntegralRangeEntities, SfaIntegralRangeRespVo.class);
        List<SfaIntegralTaskRespVo> sfaIntegralTaskRespVos = CrmBeanUtil.copyList(sfaIntegralTaskEntities, SfaIntegralTaskRespVo.class);
        List<SfaIntegralRangeRespVo> orgVo = sfaIntegralRangeRespVos.stream().filter(data -> IntegralEnum.ruleType.ORG.getValue().equals(data.getType())).collect(Collectors.toList());
        List<SfaIntegralRangeRespVo> posVo = sfaIntegralRangeRespVos.stream().filter(data -> IntegralEnum.ruleType.POS_LEVEL.getValue().equals(data.getType())).collect(Collectors.toList());
        SfaIntegralRuleRespVo sfaIntegralRuleRespVo = CrmBeanUtil.copy(sfaIntegralRuleEntity, SfaIntegralRuleRespVo.class);
        sfaIntegralRuleRespVo.setOrgs(orgVo);
        sfaIntegralRuleRespVo.setPos(posVo);
        sfaIntegralRuleRespVo.setTasks(sfaIntegralTaskRespVos);
        return sfaIntegralRuleRespVo;
    }


}
