package com.biz.crm.moblie.controller.visit.component;

import com.alibaba.fastjson.JSONObject;
import com.biz.crm.base.BusinessException;
import com.biz.crm.base.CrmAttachment;
import com.biz.crm.base.SfaClientData;
import com.biz.crm.base.SfaClientHelper;
import com.biz.crm.collection.controller.resp.VisitStepResp;
import com.biz.crm.collection.service.ISfaVisitRoleDirectoryService;
import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.eunm.sfa.SfaVisitEnum;
import com.biz.crm.moblie.controller.visit.component.VisitStepListener.VisitStepListenerCommittedData;
import com.biz.crm.moblie.controller.visit.req.ClientReq;
import com.biz.crm.moblie.controller.visit.req.VisitStepExecuteReq;
import com.biz.crm.moblie.controller.visit.req.step.ExecutorLoadReq;
import com.biz.crm.moblie.controller.visit.req.step.ExecutorWorkbenchLoadReq;
import com.biz.crm.moblie.controller.visit.resp.StepExecuteDataResp;
import com.biz.crm.service.RedisService;
import com.biz.crm.util.CrmDateUtils;
import com.biz.crm.util.LocationUtils;
import com.biz.crm.util.UserRedis;
import com.biz.crm.util.UserUtils;
import com.biz.crm.visitinfo.model.SfaVisitPlanInfoEntity;
import com.biz.crm.visitinfo.model.SfaVisitPlanInfoRedisData;
import com.biz.crm.visitinfo.service.ISfaVisitPlanInfoService;
import com.biz.crm.visitstep.model.SfaVisitStepActivityExecutionRedisData;
import com.biz.crm.visitstep.model.SfaVisitStepFromEntity;
import com.biz.crm.visitstep.resp.SfaVisitStepFromRespVo;
import com.biz.crm.visitstep.service.ISfaVisitStepFromService;
import com.google.common.collect.Lists;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.poi.ss.formula.functions.T;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import javax.annotation.PostConstruct;
import javax.annotation.Resource;
import java.lang.reflect.Field;
import java.math.BigDecimal;
import java.time.LocalDate;
import java.util.HashMap;
import java.util.List;
import java.util.Map;


/**
 * 步骤执行器-抽象
 *
 * @author: luoqi
 * @since: 2021-3-5 20:37:00
 * @version: V1.0
 */
@Slf4j
public abstract class AbstractVisitStepRedisExecutor<ExecuteReq extends VisitStepExecuteReq.StepExecuteDataReq, Resp extends StepExecuteDataResp, LoadParam extends ExecutorLoadReq>
        implements VisitStepExecutor<ExecuteReq, Resp, LoadParam> {
    /**
     * 步骤模板注册容器
     */
    public static final Map<String, Class<? extends AbstractVisitStepRedisExecutor>> VISIT_STEP_EXECUTOR_CLASS_MAPPING = new HashMap<>();

    @Resource
    protected RedisService redisService;
    @Resource
    protected ISfaVisitStepFromService sfaVisitStepFromService;
    @Resource
    protected ISfaVisitRoleDirectoryService sfaVisitRoleDirectoryService;
    @Resource
    protected ISfaVisitPlanInfoService sfaVisitPlanInfoService;

    protected VisitDataDurabilityService getVisitDataDurabilityService() {
        return null;
    }


    @PostConstruct
    public final void constructed() {
        String visitStep = this.visitStepRegistry();
        if (StringUtils.isBlank(visitStep)) {
            throw new BusinessException("拜访步骤模板注册失败: 请指定步骤模板执行器[" + this.getClass().getName()
                    + "]的拜访步骤模板编码");
        }
        if (VISIT_STEP_EXECUTOR_CLASS_MAPPING.containsKey(visitStep)) {
            throw new BusinessException("拜访步骤模板注册失败: 步骤模板执行器[" + this.getClass().getName()
                    + "]与[" + VISIT_STEP_EXECUTOR_CLASS_MAPPING.get(visitStep).getName()
                    + "]注册了相同的步骤模板编码[" + visitStep + "]");
        }
        VISIT_STEP_EXECUTOR_CLASS_MAPPING.put(visitStep, this.getClass());
    }

    /**
     * 拜访步骤模板页面注册
     * 请返回该执行器对应的拜访步骤模板编码(WEB后台，数据字典搜【拜访步骤模板】)
     *
     * @return 拜访步骤模板编码
     */
    protected abstract String visitStepRegistry();

    /**
     * 步骤执行数据加载
     *
     * @param loadParam 查询参数
     */
    public Resp load(LoadParam loadParam) {
        if (StringUtils.isBlank(loadParam.getFormId())) {
            throw new BusinessException("表单id为空");
        }
        Resp resp = this.doLoad(loadParam);

        return resp;
    }

    /**
     * 步骤执行数据加载
     *
     * @param loadParam 查询参数
     * @return 查询执行表单需要的数据
     */
    protected abstract Resp doLoad(LoadParam loadParam);

    /**
     * 加载工作台编辑页面数据
     *
     * @param workbenchLoadReq 工作台的查询参数
     * @return 工作台表单需要展示的业务对象数据
     */
    public Resp loadEditPageForWorkbench(ExecutorWorkbenchLoadReq workbenchLoadReq) {

        //默认使用拜访模板
        if (StringUtils.isBlank(workbenchLoadReq.getVisitBigType())) {
            workbenchLoadReq.setVisitBigType(SfaVisitEnum.VisitBigType.VISIT.getVal());
        }
        Resp resp = this.doLoadEditPageForWorkbench(workbenchLoadReq);

        return resp;
    }

    /**
     * 加载工作台编辑页面数据
     *
     * @param loadParam 查询参数
     * @return 工作台表单展示的业务数据
     */
    protected abstract Resp doLoadEditPageForWorkbench(ExecutorWorkbenchLoadReq loadParam);

    /**
     * 查询ES的执行数据
     *
     * @param id 业务id
     * @return 返回该步骤ES执行数据
     */
    public abstract StepExecuteDataResp getEsDataById(String id);

    /**
     * 查询ES的执行数据
     *
     * @param visitPlanInfoId 拜访id
     * @param stepCode        步骤编码
     * @return 步骤执行数据
     */
    public abstract StepExecuteDataResp getEsDataByVisitPlanInfo(String visitPlanInfoId, String stepCode);


    protected List<? extends VisitStepListener.VisitStepListenerCommittedData> loadRedisDataList(ExecutorLoadReq loadParam) {
        return Lists.newArrayList();
    }

    /**
     * 步骤执行数据持久化
     *
     * @param loadParam              查询参数
     * @param sfaVisitPlanInfoEntity 拜访计划
     * @return 持久化后的业务对象
     */
    @Transactional
    public List<? extends VisitStepListener.VisitStepListenerCommittedData> dataDurability(ExecutorLoadReq loadParam, SfaVisitPlanInfoEntity sfaVisitPlanInfoEntity) {
        List<? extends VisitStepListener.VisitStepListenerCommittedData> entities = this.loadRedisDataList(loadParam);
        if (CollectionUtils.isEmpty(entities)) {
            log.warn("持久化步骤数据：失败，步骤模板执行器[" + this.getClass().getName()
                    + "]未获取到执行数据");
            return Lists.newArrayList();
        }
        for (VisitStepListener.VisitStepListenerCommittedData entity : entities) {
            entity.setClientCode(sfaVisitPlanInfoEntity.getClientCode());
            entity.setClientName(sfaVisitPlanInfoEntity.getClientName());
            entity.setClientSubclass(sfaVisitPlanInfoEntity.getClientSubclass());
            entity.setClientSubclassName(sfaVisitPlanInfoEntity.getClientSubclassName());
            entity.setClientType(sfaVisitPlanInfoEntity.getClientType());
            entity.setClientTypeName(sfaVisitPlanInfoEntity.getClientTypeName());
        }
        return this.dataDurability(entities, this.getFormData(loadParam.getFormId()));
    }

    /**
     * 组装客户数据
     *
     * @param data       步骤业务对象
     * @param clientCode 客户编码
     * @param clientType 客户大类
     */
    protected void buildClientData(VisitStepListener.VisitStepListenerCommittedData data, String clientCode, String clientType) {
        SfaClientData clientData = SfaClientHelper.loadClientData(clientType, clientCode);
        VisitStepListener.VisitStepListenerCommittedData.copyClientData(data, clientData);
    }

    /**
     * 组装客户数据
     *
     * @param data                   步骤业务对象
     * @param sfaVisitPlanInfoEntity 拜访计划
     */
    protected void buildClientData(VisitStepListener.VisitStepListenerCommittedData data, SfaVisitPlanInfoEntity sfaVisitPlanInfoEntity) {
        if (null == data || null == sfaVisitPlanInfoEntity) {
            return;
        }
        String clientCode = sfaVisitPlanInfoEntity.getClientCode(), clientType = sfaVisitPlanInfoEntity.getClientType();
        //陌拜不用去mdm查询客户信息
        if (SfaVisitEnum.VisitBigType.UNFAMILIAR_VISIT.getVal().equals(sfaVisitPlanInfoEntity.getVisitBigType())) {
            data.setClientCode(clientCode);
            data.setClientName(sfaVisitPlanInfoEntity.getClientName());
            data.setClientType(clientType);
            data.setClientTypeName(sfaVisitPlanInfoEntity.getClientTypeName());
            data.setClientPhoto(sfaVisitPlanInfoEntity.getClientPhoto());
            data.setClientPhone(sfaVisitPlanInfoEntity.getClientPhone());
            return;
        }
        SfaClientData clientData = SfaClientHelper.loadClientData(clientType, clientCode);
        VisitStepListener.VisitStepListenerCommittedData.copyClientData(data, clientData);
    }

    /**
     * 步骤执行数据持久化for陈列检查
     *
     * @param entities   需要持久化的业务对象
     * @param fromRespVo 表单
     * @return 持久化后的业务对象
     */
    @Transactional
    public List<? extends VisitStepListener.VisitStepListenerCommittedData> dataDurabilityNew(List<? extends VisitStepListener.VisitStepListenerCommittedData> entities, SfaVisitStepFromRespVo fromRespVo, Boolean doListenerFlag) {
        VisitDataDurabilityService dataDurabilityService = this.getVisitDataDurabilityService();
        if (null == dataDurabilityService) {
            log.warn("持久化步骤数据：失败，步骤模板执行器[" + this.getClass().getName()
                + "]未指定持久化服务实例，请重写[AbstractVisitStepRedisExecutor.getVisitDataDurabilityService]并指定持久化服务实例！");
            return Lists.newArrayList();
        }
        entities = dataDurabilityService.dataDurability(entities, fromRespVo);
        if(doListenerFlag){
            this.doListener(fromRespVo.getStepCode(), entities);
        }
        return entities;
    }

    /**
     * 步骤执行数据持久化
     *
     * @param entities   需要持久化的业务对象
     * @param fromRespVo 表单
     * @return 持久化后的业务对象
     */
    @Transactional
    public List<? extends VisitStepListener.VisitStepListenerCommittedData> dataDurability(List<? extends VisitStepListener.VisitStepListenerCommittedData> entities, SfaVisitStepFromRespVo fromRespVo) {
        VisitDataDurabilityService dataDurabilityService = this.getVisitDataDurabilityService();
        if (null == dataDurabilityService) {
            log.warn("持久化步骤数据：失败，步骤模板执行器[" + this.getClass().getName()
                    + "]未指定持久化服务实例，请重写[AbstractVisitStepRedisExecutor.getVisitDataDurabilityService]并指定持久化服务实例！");
            return Lists.newArrayList();
        }
        entities = dataDurabilityService.dataDurability(entities, fromRespVo);
        this.doListener(fromRespVo.getStepCode(), entities);
        return entities;
    }

    /**
     * 事件数据转换
     *
     * @param entities 需要转换的业务对象
     * @return 转换后的业务对象
     */
    protected List<? extends VisitStepListener.VisitStepListenerCommittedData> convertToListenerCommittedData(List<? extends VisitStepListener.VisitStepListenerCommittedData> entities) {

        return entities;
    }

    /**
     * 调用监听器
     *
     * @param visitStepCode 步骤编码
     * @param entities      步骤业务对象
     */
    private void doListener(String visitStepCode, List<? extends VisitStepListener.VisitStepListenerCommittedData> entities) {
        List<AbstractVisitStepListener> listeners = AbstractVisitStepListener.getListeners(visitStepCode);
        listeners = this.customerListeners(listeners);
        if (CollectionUtils.isEmpty(listeners)) {
            return;
        }
        List<? extends VisitStepListener.VisitStepListenerCommittedData> visitStepListenerCommittedData = this.convertToListenerCommittedData(entities);
        for (AbstractVisitStepListener listener : listeners) {
            if (null == listener) {
                continue;
            }
            VisitStepListener.VisitStepListenerCommittedEvent committedEvent = new VisitStepListener.VisitStepListenerCommittedEvent();
//            //新拷贝一份数据，防止多个listener操作同一个数据对象
//            List<? extends VisitStepListener.VisitStepListenerCommittedData> committedDataOfListenerTemp = visitStepListenerCommittedData.stream().map(v -> CrmBeanUtil.copy(v, v.getClass())).collect(Collectors.toList());
            committedEvent.setEntities(visitStepListenerCommittedData);
            listener.committed(committedEvent);
        }
    }

    /**
     * 对监听器列表的个性化扩展
     *
     * @param listeners 监听器列表
     * @return 扩展后的监听器列表
     */
    protected List<AbstractVisitStepListener> customerListeners(List<AbstractVisitStepListener> listeners) {
        return listeners;
    }


    /**
     * 2021-04-08 已废弃, 变更为实现 VisitDataDurabilityService 接口
     * 将数据传输到ES
     *
     * @param loadParam 查询参数
     */
    @Deprecated
    public void transToEs(ExecutorLoadReq loadParam) {
    }

    /**
     * 步骤执行数据加载
     *
     * @param formId 表单id
     * @param resp   执行数据
     */
    protected void loadFormData(String formId, StepExecuteDataResp resp) {
        SfaVisitStepFromRespVo fromRespVo = this.sfaVisitStepFromService.queryById(formId);
        if (null == fromRespVo) {
            throw new BusinessException("未查询到表单配置数据");
        }
        resp.setSfaVisitStepFrom(fromRespVo);
    }

    /**
     * 表单数据加载
     *
     * @param formId 表单id
     * @return 表单
     */
    protected SfaVisitStepFromRespVo getFormData(String formId) {
        SfaVisitStepFromRespVo fromRespVo = this.sfaVisitStepFromService.queryById(formId);
        if (null == fromRespVo) {
            throw new BusinessException("未查询到表单[" + formId + "]配置数据");
        }
        return fromRespVo;
    }

    /**
     * 查询当前登录人的表单
     *
     * @param clientType     客户大类
     * @param stepCode       步骤编码
     * @param visitBigType   拜访大类
     * @param clientSubclass 客户细类
     * @return 表单
     */
    protected SfaVisitStepFromRespVo getFormData(String clientType, String stepCode, String visitBigType, String clientSubclass) {
        UserRedis userRedis = UserUtils.getUser();
        Map<String, Map<String, VisitStepResp>> visitStepMap = this.sfaVisitRoleDirectoryService
                .findRoleVisitStepMapForClientTypes(userRedis.getUsername(), userRedis.getPoscode(), Lists.newArrayList(clientType), visitBigType, Lists.newArrayList(clientSubclass));
        Map<String, VisitStepResp> ofClientType = visitStepMap.get(visitBigType + clientType);
        if(null == ofClientType || !ofClientType.containsKey(stepCode)){
            if(log.isInfoEnabled()){
                log.info("未加载到该用户[" + stepCode + "]步骤配置，尝试加载该步骤的全局表单");
            }
            String pageCode = null;
            Class c = this.getClass();
            for(Map.Entry<String, Class<? extends AbstractVisitStepRedisExecutor>> entry : VISIT_STEP_EXECUTOR_CLASS_MAPPING.entrySet()){
                if(c.equals(entry.getValue())){
                    pageCode = entry.getKey();
                }
            }
            SfaVisitStepFromRespVo fromRespVo = this.sfaVisitStepFromService.loadOrAddGlobalForm(pageCode, stepCode);
//            if(null == fromRespVo){
//                throw new BusinessException("未加载到[" + stepCode + "]步骤全局表单配置，请前往[拜访步骤表单配置]页面配置相关数据！");
//            }
            return fromRespVo;
//            throw new BusinessException("未加载到该用户[" + stepCode + "]步骤配置，请前往[拜访步骤配置]页面配置相关数据！");
        }
        VisitStepResp visitStepResp = ofClientType.get(stepCode);
        return this.sfaVisitStepFromService.queryById(visitStepResp.getFormId());
    }


    /**
     * 更新步骤状态
     *
     * @param redisHashKey redisHashKey
     * @param stepCode     步骤编码
     * @param visitBigType 拜访大类
     */
    protected void updateStepStatus(String redisHashKey, String stepCode, String visitBigType) {
        this.sfaVisitPlanInfoService.updateStepStatus(redisHashKey, stepCode, visitBigType);
    }

    /**
     * 修改步骤执行状态
     *
     * @param sfaVisitPlanInfoRedisData 拜访计划明细执行数据
     * @param stepCode                  步骤编码
     */
    protected void changeStepStatus(SfaVisitPlanInfoRedisData sfaVisitPlanInfoRedisData, String stepCode) {
        this.sfaVisitPlanInfoService.changeStepStatus(sfaVisitPlanInfoRedisData, stepCode);
    }

    /**
     * 加载计划明细数据
     *
     * @param redisHashKey redisHashKey
     * @param visitBigType 拜访大类
     * @return 拜访计划明细
     */
    protected SfaVisitPlanInfoEntity loadAndCheckSfaVisitPlanInfoEntity(String redisHashKey, String visitBigType) {
        String nowDate = LocalDate.now().format(CrmDateUtils.yyyyMMdd);
        SfaVisitPlanInfoEntity sfaVisitPlanInfoEntity = (SfaVisitPlanInfoEntity) this.redisService
                .hmget(SfaVisitPlanInfoEntity.getInstance().redisHashCurrent(nowDate, visitBigType).toString(), redisHashKey);
        if (null == sfaVisitPlanInfoEntity) {
            throw new BusinessException("不存在的拜访计划，或该计划不在今日计划内！");
        }
        UserRedis userRedis = UserUtils.getUser();
        if (!org.apache.commons.lang3.StringUtils.equals(sfaVisitPlanInfoEntity.getVisitPosCode(), userRedis.getPoscode())) {
            throw new BusinessException("你没有权限访问该数据，请重新登陆或切换职位再尝试！");
        }
        return sfaVisitPlanInfoEntity;
    }

    /**
     * 判断拜访客户经纬度不为空
     *
     * @param visitStepExecuteReq 拜访执行参数
     * @param entity              拜访计划明细
     * @param longitude           纬度
     * @param latitude            经度
     */
    protected void checkLongitudeAndLatitude(VisitStepExecuteReq<? extends ExecuteReq> visitStepExecuteReq, SfaVisitPlanInfoEntity entity, String longitude, String latitude) {
        if (entity != null && entity.getLatitude() != null && entity.getLatitude() != null) {
            SfaVisitStepFromEntity stepFromEntity = sfaVisitStepFromService.lambdaQuery().eq(SfaVisitStepFromEntity::getId, visitStepExecuteReq.getFormId()).one();
            if (stepFromEntity != null && stepFromEntity.getLocateType().equals(YesNoEnum.yesNoEnum.Y.getValue())) {
                if (com.biz.crm.util.StringUtils.isNotEmpty(stepFromEntity.getDistance())) {
                    if (StringUtils.isBlank(longitude)) {
                        throw new BusinessException("经度为空");
                    }
                    if (StringUtils.isBlank(latitude)) {
                        throw new BusinessException("纬度为空");
                    }
                    BigDecimal distance = new BigDecimal(stepFromEntity.getDistance()).setScale(2, BigDecimal.ROUND_HALF_DOWN);
                    log.info("当前步骤配置的距离:{}", distance);
                    log.info("当前位置的经度:{},纬度:{},拜访客户的经度:{},纬度:{}", longitude, latitude, entity.getLongitude(), entity.getLatitude());
                    double data = LocationUtils.getDistance(new BigDecimal(longitude),
                            new BigDecimal(latitude), entity.getLongitude(), entity.getLatitude());
                    BigDecimal nowDistance = new BigDecimal(data).setScale(2, BigDecimal.ROUND_HALF_DOWN);
                    log.info("当前位置距离拜访客户的距离:{}", nowDistance);
                    if (distance.compareTo(nowDistance) == -1) {
                        throw new BusinessException("您未在客户距离" + distance + "M内，请前往客户地址进行操作！");
                    }
                }
            }
        }
    }

    /**
     * 步骤执行校验抽象
     *
     * @param visitStepExecuteReq 执行数据
     */
    protected void check(VisitStepExecuteReq<? extends ExecuteReq> visitStepExecuteReq) {

        if (StringUtils.isBlank(visitStepExecuteReq.getFormId())) {
            throw new BusinessException("请指定表单id");
        }
//        if(StringUtils.isBlank(visitStepExecuteReq.getDirectoryCode())){
//            throw new BusinessException("拜访步骤编码为空");
//        }
        if (StringUtils.isBlank(visitStepExecuteReq.getRedisHashKey())) {
            throw new BusinessException("请指定redisHashKey");
        }
    }

    /**
     * 针对使用控件的数据格式转换
     *
     * @param visitStepExecuteReq
     */
    @SneakyThrows
    protected void controlDataTrans(VisitStepExecuteReq<? extends ExecuteReq> visitStepExecuteReq) {
        Object t = visitStepExecuteReq.getStepExecuteData();
        Class userCla = t.getClass();
        Field[] fs = userCla.getFields();
        for (int i = 0; i < fs.length; i++) {
            Field f = fs[i];
            // 设置些属性是可以访问的
            f.setAccessible(true);
            // 得到此属性的值
            Object val = f.get(t);
            //获取属性名称
            String name = f.getName();
            //判断对象类型object时，需要进行转换为json字符串存入数据库，是为了避免存入对象时发生错误
            if (name.contains("obj") && val != null && !("").equals(val)) {
                log.warn("获取到obj扩展控件字段的值:{}", val);
                //进行转换为json字符串处理
                f.set(t, JSONObject.toJSONString(val));
            }
        }
    }

    /**
     * 数据转换为obj
     * @param t
     */
    @SneakyThrows
    protected void controlDataToObj(Object t){
        Class userCla = t.getClass();
        Field[] fs = userCla.getFields();
        for (int i = 0; i < fs.length; i++) {
            Field f = fs[i];
            // 设置些属性是可以访问的
            f.setAccessible(true);
            // 得到此属性的值
            Object val = f.get(t);
            //获取属性名称
            String name = f.getName();
            //判断对象类型object时，需要进行转换为json字符串存入数据库，是为了避免存入对象时发生错误
            if (name.contains("obj") && val != null && !("").equals(val)) {
                log.warn("获取到obj扩展控件字段的值:{}", val);
                //进行转换为json字符串处理
                f.set(t, JSONObject.parse((String) val));
            }
        }
    }

    /**
     * 校验照片
     *
     * @param sfaVisitPics 照片
     * @param min          最少限制
     * @param max          最多限制
     * @param bizTag       错误提示
     */
    protected void checkPics(List<CrmAttachment> sfaVisitPics, int min, int max, String bizTag) {
        min = min < 0 ? 0 : min;
        if (max < min) {
            max = min;
        }
        bizTag = null == bizTag ? "" : bizTag;
        if (null == sfaVisitPics) {
            sfaVisitPics = Lists.newArrayList();
        }
        if (sfaVisitPics.size() < min) {
            throw new BusinessException(bizTag + "图片数量不能少于[" + min + "]张");
        }
        if (sfaVisitPics.size() > max) {
            throw new BusinessException(bizTag + "图片数量不能超过[" + max + "]张");
        }
        for (CrmAttachment sfaVisitPictureReqVo : sfaVisitPics) {
            if (StringUtils.isBlank(sfaVisitPictureReqVo.getUrlPathPrefix())) {
                throw new BusinessException(bizTag + "照片域名为空");
            }
            if (StringUtils.isBlank(sfaVisitPictureReqVo.getUrlPathPrefix())) {
                throw new BusinessException(bizTag + "照片相对路径为空");
            }
        }
    }

    /**
     * 校验客户参数
     *
     * @param req 客户参数
     */
    protected void checkClientReq(ClientReq req) {
        if (StringUtils.isBlank(req.getClientCode())) {
            throw new BusinessException("客户编码为空");
        }
        if (StringUtils.isBlank(req.getClientName())) {
            throw new BusinessException("客户名称为空");
        }
        if (StringUtils.isBlank(req.getClientType())) {
            throw new BusinessException("客户类型为空");
        }
    }


}
