package com.biz.crm.moblie.service.impl;

import com.biz.crm.base.ApiResultUtil;
import com.biz.crm.base.BusinessException;
import com.biz.crm.base.utils.PointUtil;
import com.biz.crm.common.PageResult;
import com.biz.crm.enums.SfaCustomerDistanceEnum;
import com.biz.crm.eunm.sfa.SfaVisitEnum;
import com.biz.crm.gaode.vo.GaodeNearbyVo;
import com.biz.crm.gaode.vo.QueryAroundVo;
import com.biz.crm.map.model.req.SfaTencentMapData;
import com.biz.crm.map.model.resp.SfaTencentMapDataRespVo;
import com.biz.crm.map.service.SfaTencentMapDataService;
import com.biz.crm.mdm.map.MdmTerminalCustomerElasticsearchFeign;
import com.biz.crm.mdm.poi.MdmAmapPoiFeign;
import com.biz.crm.moblie.controller.visit.req.GetVisitListReq;
import com.biz.crm.moblie.controller.visit.resp.SfaVisitResp;
import com.biz.crm.moblie.service.ISfaNearbyCustomerService;
import com.biz.crm.nebular.mdm.dict.resp.DictDataVo;
import com.biz.crm.nebular.mdm.poi.req.MdmAmapPoiDistanceReqVo;
import com.biz.crm.nebular.mdm.poi.resp.MdmAmapPoiDistanceRespVo;
import com.biz.crm.nebular.mdm.terminal.req.MdmTerminalCustomerElasticsearchPageReqVo;
import com.biz.crm.nebular.mdm.terminal.resp.MdmTerminalCustomerElasticsearchRespVo;
import com.biz.crm.nebular.sfa.moblie.req.SfaNearbyCustomerReqVo;
import com.biz.crm.sfa.map.GaoDeFeign;
import com.biz.crm.util.*;
import com.biz.crm.visitinfo.model.SfaVisitPlanInfoEntity;
import com.biz.crm.visitinfo.service.ISfaVisitPlanInfoService;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.MapUtils;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.LocalDate;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 附近客户 实现
 *
 * @author yangshan
 * @date 2021/3/9 14:47
 */

@Service
@Slf4j
@ConditionalOnMissingBean(name = "sfaNearbyCustomerServiceExpandImpl")
public class SfaNearbyCustomerServiceImpl implements ISfaNearbyCustomerService {

    @Resource
    private GaoDeFeign gaoDeFeign;
    @Resource
    private MdmTerminalCustomerElasticsearchFeign mdmTerminalCustomerElasticsearchFeign;

    @Resource
    private ISfaVisitPlanInfoService sfaVisitPlanInfoService;
    @Resource
    private SfaTencentMapDataService sfaTencentMapDataService;
    @Resource
    private MdmAmapPoiFeign mdmAmapPoiFeign;

    protected List<SfaVisitResp.SfaVisitPlanInfoResp> loadCustomers(SfaNearbyCustomerReqVo reqVo){
        //默认高德地图
        if(StringUtils.isEmpty(reqVo.getMapData())){
            reqVo.setMapData(SfaVisitEnum.mapData.GAODE_MAP_DATA.getCode());
        }
        List<SfaVisitResp.SfaVisitPlanInfoResp> respList;
        //地图数据
        if(SfaVisitEnum.mapData.GAODE_MAP_DATA.getCode().equals(reqVo.getMapData())){
            respList = this.gaodeDataConvert(this.getGaodeNearbyData(reqVo));
        }else if(SfaVisitEnum.mapData.GAODE_MAP_POI_DATA.getCode().equals(reqVo.getMapData())){
            respList = this.gaodeDataConvert(this.getGaodePoiNearbyData(reqVo));
        }else {
            respList = this.tencentDataConvert(this.getTencentData(reqVo));
        }
        return respList;
    }

    private List<MdmTerminalCustomerElasticsearchRespVo> getAllClientFromMdm(String[] point){
        List<MdmTerminalCustomerElasticsearchRespVo> result = Lists.newArrayList();
        MdmTerminalCustomerElasticsearchPageReqVo mdmTerminalCustomerElasticsearchPageReqVo = new MdmTerminalCustomerElasticsearchPageReqVo();
        PageDataAdviser pageDataAdviser = new PageDataAdviser(500);
        mdmTerminalCustomerElasticsearchPageReqVo.setPageNum(pageDataAdviser.nextPage());
        mdmTerminalCustomerElasticsearchPageReqVo.setPageSize(500);
        PageResult<MdmTerminalCustomerElasticsearchRespVo> mdmResult = this.getDataFromMdm(mdmTerminalCustomerElasticsearchPageReqVo, point);
        if (mdmResult != null ) {
            result = mdmResult.getData();
            while (pageDataAdviser.hasNext(mdmResult.getCount())) {
                mdmTerminalCustomerElasticsearchPageReqVo.setPageSize(500);
                mdmTerminalCustomerElasticsearchPageReqVo.setPageNum(pageDataAdviser.nextPage());
                PageResult<MdmTerminalCustomerElasticsearchRespVo> list = this.getDataFromMdm(mdmTerminalCustomerElasticsearchPageReqVo, point);
                if (null != result) {
                    result.addAll(list.getData());
                }
            }
        }
        return result;
    }

    /**
     * 调用地图API传入location，radius查询范围类所有客户和经销商，调用mdm接口传入location，radius查询出范围内已合作的客户和经销商
     * 比对两个list，去重，返回未合作的list
     *
     * @param reqVo
     * @return
     */
    @Override
    public SfaVisitResp findNotCooperationCustomerList(SfaNearbyCustomerReqVo reqVo) {
        String[] point = this.doCheckParam(reqVo);
        List<SfaVisitResp.SfaVisitPlanInfoResp> respList = this.loadCustomers(reqVo);


        //数据转化
        LinkedHashMap<String, SfaVisitResp.SfaVisitPlanInfoResp> collect = respList.stream().collect(Collectors.toMap(SfaVisitResp.SfaVisitPlanInfoResp::getClientCode,
                Function.identity(), (oldVal, newVal) -> oldVal, LinkedHashMap::new));
        //调用mdm接口,查询范围内已合作的所有客户
        List<MdmTerminalCustomerElasticsearchRespVo> list = this.getAllClientFromMdm(point);
        //去重，返回未合作list
        if (CollectionUtil.listNotEmpty(list)){
            list.forEach(o -> {
                if (collect.containsKey(o.getName())){
                    collect.remove(o.getAmapId(), collect.get(o.getAmapId()));
                }
            });
        }

        //获取今日陌拜数据,获取未合作拜访的客户，按钮改变
        List<SfaVisitResp.SfaVisitPlanInfoResp> planInfoEntities = this.visitPlanToday(point,
                SfaVisitEnum.VisitBigType.UNFAMILIAR_VISIT.getVal(), reqVo.getVisitTime());
        if (CollectionUtil.listNotEmpty(planInfoEntities)){
            planInfoEntities.forEach(plan -> {
                if (collect.containsKey(plan.getClientCode())){
                    SfaVisitResp.SfaVisitPlanInfoResp sfaVisitPlanInfoResp = collect.get(plan.getClientCode());
                    plan.setDistance(sfaVisitPlanInfoResp.getDistance());
                    plan.setClientAddress(sfaVisitPlanInfoResp.getClientAddress());
                    plan.setClientPhone(sfaVisitPlanInfoResp.getClientPhone());
                    plan.setLongitude(sfaVisitPlanInfoResp.getLongitude());
                    plan.setLatitude(sfaVisitPlanInfoResp.getLatitude());
                    plan.setBusiType(sfaVisitPlanInfoResp.getBusiType());
                    plan.setBusiLabel(sfaVisitPlanInfoResp.getBusiLabel());
                    plan.setChannel(sfaVisitPlanInfoResp.getChannel());
                    collect.put(plan.getClientCode(), plan);
                }
            });
        }
        List<SfaVisitResp.SfaVisitPlanInfoResp> respVos = collect.values().stream().collect(Collectors.toList());

        SfaVisitResp sfaVisitResp = new SfaVisitResp();
        sfaVisitResp.setPlanInfoEntities(respVos);
        return sfaVisitResp;
    }

    /**
     * 调用高德地图接口查询客户数据
     */
    public List<GaodeNearbyVo> getGaodeNearbyData(SfaNearbyCustomerReqVo reqVo){
        QueryAroundVo queryAroundVo = new QueryAroundVo();
        queryAroundVo.setLocation(reqVo.getLocation())
                .setRadius(null)
                .setKeywords(null)
                .setOffset(reqVo.getPageSize().toString())
                .setPage(reqVo.getPageNum().toString());
        //从数据字典获取距离范围，只取第一条数据，如果为空，默认配置2000
        List<DictDataVo> distanceList = DictUtil.list(SfaCustomerDistanceEnum.SFA_CUSTOMER_DISTANCE_ENUM.getDesc());
        if(CollectionUtil.listNotEmpty(distanceList)){
            String radius = distanceList.get(0).getDictValue();
            queryAroundVo.setRadius(radius);
        }
        if (StringUtils.isNotEmpty(reqVo.getClientName())){
            queryAroundVo.setKeywords(reqVo.getClientName());
        }else {
            //从数据字典获取关键字，如果为空，默认配置超市|KTV
            List<DictDataVo> keywordList = DictUtil.list(SfaVisitEnum.customerKeyword.CUSTOMER_KEYWORD_ENUM.getDesc());
            if(CollectionUtil.listNotEmpty(keywordList)){
                String keywords = keywordList.stream().map(e -> e.getDictValue()).collect(Collectors.joining("|"));
                queryAroundVo.setKeywords(keywords);
            }
        }

        //调用高德地图接口
        PageResult<GaodeNearbyVo> result = ApiResultUtil.objResult(gaoDeFeign.queryAround(queryAroundVo));
        if (ObjectUtils.isEmpty(result)){
            if (StringUtils.isNotEmpty(reqVo.getClientName())){
                return Lists.newArrayList();
            }
            throw new BusinessException("获取地图数据失败！");
        }
        return result.getData();
    }

    public List<GaodeNearbyVo> getGaodePoiNearbyData(SfaNearbyCustomerReqVo reqVo) {
        //从数据字典获取距离范围，只取第一条数据，如果为空，默认配置2000
        List<DictDataVo> distanceList = DictUtil.list(SfaCustomerDistanceEnum.SFA_CUSTOMER_DISTANCE_ENUM.getDesc());
        String radius = "20000";
        if (CollectionUtil.listNotEmpty(distanceList)) {
            radius = distanceList.get(0).getDictValue();
        }
        MdmAmapPoiDistanceReqVo distanceReqVo = new MdmAmapPoiDistanceReqVo();
        distanceReqVo.setPageNum(reqVo.getPageNum());
        distanceReqVo.setPageSize(reqVo.getPageSize());
        distanceReqVo.setDistance(radius);
        distanceReqVo.setPoiName(reqVo.getClientName());
        String location = reqVo.getLocation();
        distanceReqVo.setLongitude(location.substring(0, location.indexOf(",")));
        distanceReqVo.setLatitude(location.substring(location.indexOf(",") + 1));
        Result<PageResult<MdmAmapPoiDistanceRespVo>> result = mdmAmapPoiFeign.findAmapPoiDistancePageList(distanceReqVo);
        if (!result.isSuccess()) {
            throw new BusinessException("查询失败：" + result.getMessage());
        }
        if (CollectionUtil.listEmpty(result.getResult().getData())){
            return Lists.newArrayList();
        }
        return result.getResult().getData().stream().map(x -> {
            GaodeNearbyVo vo = CrmBeanUtil.copy(x, GaodeNearbyVo.class);
            vo.setLocation(x.getLatitude() + "," + x.getLongitude());
            vo.setName(x.getPoiName());
            vo.setPcode(x.getAmapProvinceCode());
            vo.setPname(x.getAmapProvinceName());
            vo.setCitycode(x.getAmapCityCode());
            vo.setCityname(x.getAmapCityName());
            vo.setAdcode(x.getAmapDistrictCode());
            vo.setAdname(x.getAmapDistrictName());
            vo.setAddress(x.getAddress());
            vo.setType(x.getTypeName());
            vo.setTypecode(x.getTypeCode());
            vo.setDistance(new BigDecimal(x.getDistance()).divide(new BigDecimal("1000"), 3, RoundingMode.HALF_UP).toString());
            return vo;
        }).collect(Collectors.toList());
    }

    /**
     * 高得地图数据转换
     */
    public List<SfaVisitResp.SfaVisitPlanInfoResp> gaodeDataConvert(List<GaodeNearbyVo> gaodeData){
        if (CollectionUtil.listEmpty(gaodeData)){
            return Lists.newArrayList();
        }
        List<SfaVisitResp.SfaVisitPlanInfoResp> respList = gaodeData.stream().map(m -> {
            SfaVisitResp.SfaVisitPlanInfoResp o = new SfaVisitResp.SfaVisitPlanInfoResp();
            o.setClientCode(m.getId());
            o.setRedisHashKey(SfaVisitPlanInfoEntity.VisitRedisHashKey.DEF);
            o.setLongitude(new BigDecimal(m.getLongitude()));
            o.setLatitude(new BigDecimal(m.getLatitude()));
            o.setClientAddress(m.getAddress());
            o.setClientPhone(m.getTel());
            o.setClientName(m.getName());
            o.setDistance(new BigDecimal(m.getDistance()).divide(new BigDecimal("1000"), 3, RoundingMode.HALF_UP));
            o.setPcode(m.getPcode());
            o.setCitycode("510600");
            o.setAdcode(m.getAdcode());
            return o;
        }).collect(Collectors.toList());
        //配置按钮
        respList.forEach(o -> {
            Map<String, String> map = SfaVisitEnum.visitButton.buildVisitUnfamiliarButton(UserUtils.getUser().getUsername(), LocalDate.now().toString(), SfaVisitEnum.visitStatus.V1.getVal());
            o.setButtons(map);
        });
        return respList;
    }

    /**
     * sfa 查询今日拜访计划列表
     */
    public List<SfaVisitResp.SfaVisitPlanInfoResp> visitPlanToday(String[] point, String visitBigType, String visitTime){
        //获取拜访数据
        UserRedis userRedis = UserUtils.getUser();
        GetVisitListReq req = new GetVisitListReq();
        req.setVisitUserName(userRedis.getUsername());
        req.setVisitPositionCode(userRedis.getPoscode());
        //默认今日
        req.setVisitDate(LocalDate.now().toString());
        if (StringUtils.isNotEmpty(visitTime)){
            try {
                LocalDate.parse(visitTime, CrmDateUtils.yyyyMMdd);
            }catch (Exception e){
                throw new BusinessException("时间格式错误");
            }
            req.setVisitDate(visitTime);
        }
        req.setLongitude(new BigDecimal(point[0]));
        req.setLatitude(new BigDecimal(point[1]));
        req.setVisitBigType(visitBigType);
        List<SfaVisitResp.SfaVisitPlanInfoResp> planInfoEntities = sfaVisitPlanInfoService.getVisitInfoList(req).getPlanInfoEntities();
        return planInfoEntities;
    }

    /**
     * mdm返回当前位置的已合作客户
     */
    public PageResult<MdmTerminalCustomerElasticsearchRespVo> getDataFromMdm(MdmTerminalCustomerElasticsearchPageReqVo mdmReqVo,String[] split){
        //从数据字典获取距离范围，只取第一条数据,如果为空，默认配置2000
        List<DictDataVo> distanceList = DictUtil.list(SfaCustomerDistanceEnum.SFA_CUSTOMER_DISTANCE_ENUM.getDesc());
        AssertUtils.isNotEmpty(distanceList, "缺失数据字典");
        String radius = distanceList.get(0).getDictValue();
        mdmReqVo.setLongitude(split[0])
                .setLatitude(split[1])
                .setDistance(Double.parseDouble(radius));
        PageResult<MdmTerminalCustomerElasticsearchRespVo> mdmResult = ApiResultUtil.objResult(mdmTerminalCustomerElasticsearchFeign.list(mdmReqVo), true);
        return mdmResult;
    }

    /**
     * 调用mdm接口传入location，radius查询出范围内已合作的客户和经销商，获取idList，
     * 查询每一条记录的拜访状态，配置按钮
     * @param reqVo
     * @return
     */
    @Override
    public SfaVisitResp findCooperationCustomerList(SfaNearbyCustomerReqVo reqVo){
        String[] point = this.doCheckParam(reqVo);
        //mdm返回当前位置的已合作客户
        UserRedis userRedis = UserUtils.getUser();
        MdmTerminalCustomerElasticsearchPageReqVo mdmTerminalCustomerElasticsearchPageReqVo = new MdmTerminalCustomerElasticsearchPageReqVo();
        mdmTerminalCustomerElasticsearchPageReqVo.setPositionCodes(userRedis.getPoscode())
                .setPageNum(reqVo.getPageNum())
                .setPageSize(reqVo.getPageSize())
                .setName(reqVo.getClientName());
        PageResult<MdmTerminalCustomerElasticsearchRespVo> list = this.getDataFromMdm(mdmTerminalCustomerElasticsearchPageReqVo, point);
        if (CollectionUtil.listEmpty(list.getData())){
            return new SfaVisitResp();
        }
        //比对列表，返回已合作客户数据配置按钮
        Map<String, MdmTerminalCustomerElasticsearchRespVo> clientMap = list.getData().stream()
                .collect(Collectors.toMap(MdmTerminalCustomerElasticsearchRespVo::getCode, a -> a, (k1, k2) -> k1));

        //获取今日拜访数据
        List<SfaVisitResp.SfaVisitPlanInfoResp> planInfoEntities = this.visitPlanToday(point,
                SfaVisitEnum.VisitBigType.VISIT.getVal(), reqVo.getVisitTime());
        Map<String, SfaVisitResp.SfaVisitPlanInfoResp> collect = new HashMap<>();
        if (CollectionUtil.listNotEmpty(planInfoEntities)){
            collect = planInfoEntities.stream().collect(Collectors.toMap(SfaVisitResp.SfaVisitPlanInfoResp::getClientCode, a -> a, (k1, k2) -> k1));
        }
        //比对，取出今日，当前范围内的已合作客户
        HashMap<String, SfaVisitResp.SfaVisitPlanInfoResp> map = new HashMap<>();
        for (String o : clientMap.keySet()){
            if (!CollectionUtils.isEmpty(collect)){
                if(collect.containsKey(o)){
                    map.put(o, collect.get(o));
                    continue;
                }
            }
            //已合作，未在今日拜访计划中客户
            Map<String, String> mapButton = SfaVisitEnum.visitButton.buildVisitButton(userRedis.getUsername(), LocalDate.now().toString(), SfaVisitEnum.visitStatus.V1.getVal());
            //过滤异常按钮
            mapButton.remove(SfaVisitEnum.visitButton.EXCEPTION.getVal());
            //数据转化
            SfaVisitResp.SfaVisitPlanInfoResp resp = this.setParam(clientMap.get(o), mapButton);
            map.put(o, resp);
        }
//        //根据客户名称查询
//        if (StringUtils.isEmpty(reqVo.getClientName())){
//            //已合作客户和拜访计划客户没有交集
//            if (map.size() == clientMap.size()){
//                map.putAll(collect);
//            }
//            if (MapUtils.isEmpty(map)){
//                return new SfaVisitResp();
//            }
//        }
        List<SfaVisitResp.SfaVisitPlanInfoResp> sfaVisitPlanInfoRespVos = map.values().stream().collect(Collectors.toList());
        //按照距离排序
        List<SfaVisitResp.SfaVisitPlanInfoResp> result = sfaVisitPlanInfoRespVos.stream()
                .sorted(Comparator.nullsLast(Comparator.comparing(SfaVisitResp.SfaVisitPlanInfoResp::getDistance, Comparator.nullsLast(BigDecimal::compareTo))))
                .collect(Collectors.toList());
        SfaVisitResp sfaVisitResp = new SfaVisitResp();
        sfaVisitResp.setPlanInfoEntities(result);
        return sfaVisitResp;
    }

    /**
     * 数据转换
     */
    public SfaVisitResp.SfaVisitPlanInfoResp setParam(MdmTerminalCustomerElasticsearchRespVo m, Map<String, String> mapButton){
        //数据转化
        SfaVisitResp.SfaVisitPlanInfoResp resp = CrmBeanUtil.copy(m, SfaVisitResp.SfaVisitPlanInfoResp.class);
        resp.setDistance(new BigDecimal(m.getDistance()).divide(new BigDecimal("1000"), 3, RoundingMode.HALF_UP));
        resp.setLongitude(new BigDecimal(m.getLongitude()));
        resp.setLatitude(new BigDecimal(m.getLatitude()));
        resp.setClientContacts(m.getContactName());
        resp.setClientCode(m.getCode());
        resp.setClientName(m.getName());
        resp.setClientPhone(m.getContactPhone());
        resp.setClientAddress(m.getAddress());
        resp.setButtons(mapButton);
        resp.setRedisHashKey(SfaVisitPlanInfoEntity.VisitRedisHashKey.DEF);
        resp.setVisitStatus(SfaVisitEnum.visitStatus.V1.getVal());
        resp.setVisitStatusName(SfaVisitEnum.visitStatus.V1.getDesc());
        return resp;
    }
    /**
     * 参数校验
     *
     * @param reqVo
     */
    public String[] doCheckParam(SfaNearbyCustomerReqVo reqVo){
        AssertUtils.isNotEmpty(reqVo.getPageNum().toString(), "每页数据的数量不能为空");
        AssertUtils.isNotEmpty(reqVo.getPageSize().toString(), "页码不能为空");
        return PointUtil.splitString(reqVo.getLocation());
    }

    /**
     * 腾讯地图数据
     * @param reqVo
     * @return
     */
    public List<SfaTencentMapDataRespVo> getTencentData(SfaNearbyCustomerReqVo reqVo) {
        String[] split = this.doCheckParam(reqVo);
        SfaTencentMapData.SfaTencentMapDataReqVo dataReqVo = new SfaTencentMapData.SfaTencentMapDataReqVo();
        dataReqVo.setPageNum(reqVo.getPageNum());
        dataReqVo.setPageSize(reqVo.getPageSize());
        dataReqVo.setLng(split[0]);
        dataReqVo.setLat(split[1]);
        if (StringUtils.isNotEmpty(reqVo.getClientName())){
            dataReqVo.setPoiTitle(reqVo.getClientName());
        }
        //从数据字典获取距离范围，只取第一条数据，如果为空，默认配置2000
        List<DictDataVo> distanceList = DictUtil.list(SfaCustomerDistanceEnum.SFA_CUSTOMER_DISTANCE_ENUM.getDesc());
        if(CollectionUtil.listNotEmpty(distanceList)){
            String radius = distanceList.get(0).getDictValue();
            dataReqVo.setDistance(radius);
        }
        PageResult<SfaTencentMapDataRespVo> mapDataServiceList = sfaTencentMapDataService.findList(dataReqVo);
        if (ObjectUtils.isEmpty(mapDataServiceList)){
            if (StringUtils.isNotEmpty(reqVo.getClientName())){
                return Lists.newArrayList();
            }
            throw new BusinessException("获取地图数据失败！");
        }
        return mapDataServiceList.getData();
    }

    /**
     * 腾讯地图数据转化
     * @param data
     * @return
     */
    public List<SfaVisitResp.SfaVisitPlanInfoResp> tencentDataConvert(List<SfaTencentMapDataRespVo> data){
        if (CollectionUtil.listEmpty(data)){
            return Lists.newArrayList();
        }
        List<SfaVisitResp.SfaVisitPlanInfoResp> respList = data.stream().map(m -> {
            SfaVisitResp.SfaVisitPlanInfoResp o = new SfaVisitResp.SfaVisitPlanInfoResp();
            o.setClientCode(m.getMapId());
            o.setRedisHashKey(SfaVisitPlanInfoEntity.VisitRedisHashKey.DEF);
            o.setLongitude(new BigDecimal(m.getLongitude()));
            o.setLatitude(new BigDecimal(m.getLatitude()));
            o.setClientAddress(m.getAddress());
            o.setClientName(m.getName());
            o.setBusiLabel(m.getBusiLabel());
            o.setChannel(m.getChannel());
            o.setBusiType(m.getBusiType());
            o.setDistance(new BigDecimal(m.getDistance()).divide(new BigDecimal("1000"), 3, RoundingMode.HALF_UP));
            return o;
        }).collect(Collectors.toList());
        //配置按钮
        respList.forEach(o -> {
            Map<String, String> map = SfaVisitEnum.visitButton.buildVisitUnfamiliarButton(UserUtils.getUser().getUsername(),
                    LocalDate.now().toString(), SfaVisitEnum.visitStatus.V1.getVal());
            o.setButtons(map);
        });
        return respList;
    }
}
