package com.biz.crm.moblie.service.impl;

import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.biz.crm.activiti.mobile.ActivitiMobileFeign;
import com.biz.crm.base.ApiResultUtil;
import com.biz.crm.base.BusinessException;
import com.biz.crm.common.PageResult;
import com.biz.crm.eunm.ActivitiEnum;
import com.biz.crm.eunm.sfa.SfaCommonEnum;
import com.biz.crm.moblie.service.ISfaWorkSignAuditService;
import com.biz.crm.nebular.activiti.act.req.ReadCopyTaskReqVO;
import com.biz.crm.nebular.activiti.act.req.TaskOptReqVO;
import com.biz.crm.nebular.activiti.start.req.OptRecordReqVO;
import com.biz.crm.nebular.activiti.start.req.ProcessInfoReqVO;
import com.biz.crm.nebular.activiti.task.req.CopyTaskReqVO;
import com.biz.crm.nebular.activiti.task.req.TaskQueryReqVO;
import com.biz.crm.nebular.activiti.task.resp.*;
import com.biz.crm.nebular.sfa.worksign.req.*;
import com.biz.crm.nebular.sfa.worksign.resp.*;
import com.biz.crm.util.CrmBeanUtil;
import com.biz.crm.util.Result;
import com.biz.crm.util.UserUtils;
import com.biz.crm.worksign.mapper.*;
import com.biz.crm.worksign.model.*;
import com.biz.crm.worksign.service.ISfaExceptionReportService;
import com.biz.crm.worksign.service.ISfaLeaveCancelService;
import com.biz.crm.worksign.service.ISfaLeaveService;
import com.biz.crm.worksign.service.ISfaTravelService;
import com.biz.crm.worksign.service.ISfaWorkOvertimeService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author ren.gang
 * @ClassName SfaWorkSignAuditServiceImpl.java
 * @Description 考勤审批业务处理
 * @createTime 2020年11月26日 15:45:00
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "SfaWorkSignAuditServiceExpandImpl")
public class SfaWorkSignAuditServiceImpl implements ISfaWorkSignAuditService {

    @Autowired
    private ActivitiMobileFeign mobileFeign;
    @Autowired
    private ISfaLeaveService leaveService;
    @Autowired
    private ISfaWorkOvertimeService overtimeService;
    @Autowired
    private ISfaExceptionReportService exceptionService;
    @Autowired
    private ISfaLeaveCancelService leaveCancelService;

    @Resource
    private SfaLeaveMapper leaveMapper;
    @Resource
    private SfaWorkOvertimeMapper overtimeMapper;
    @Resource
    private SfaExceptionReportMapper exceptionMapper;
    @Resource
    private SfaLeaveCancelMapper leaveCancelMapper;
    @Resource
    private SfaTravelMapper travelMapper;
    @Autowired
    private ISfaTravelService travelService;

    /**
     * 待审批汇总
     *
     * @return
     */
    @Override
    public Map<String, Integer> backlogCount() {
        // 查询我的待办信息
        TaskQueryReqVO reqVO = new TaskQueryReqVO();
        reqVO.setUserCode(UserUtils.getUser().getUsername());
        reqVO.setPositionCode(UserUtils.getUser().getPoscode());
        reqVO.setFormType(ActivitiEnum.getFormType());
        PageResult<TaskRspVO> result = ApiResultUtil.objResult(mobileFeign.findCurrentTask(reqVO), true);

        // 获取表单类型列表
        Map<String, Integer> backlogCount = new HashMap<>();

        // 统计待办数量。查询指定状态(已提交、审批中)
        List<String> bpmStatusList = Arrays.asList(SfaCommonEnum.dataBpmStatus.COMMIT.getValue(), SfaCommonEnum.dataBpmStatus.APPROVAL.getValue());

        // 请假待办数量
        Integer leaveNum = leaveMapper.selectCount(Wrappers.lambdaQuery(SfaLeaveEntity.class)
                .in(SfaLeaveEntity::getBpmStatus, bpmStatusList)
                .in(SfaLeaveEntity::getAuditTaskId, getFormNoList(result, ActivitiEnum.FormTypeEnum.LEAVE))
        );
        backlogCount.put("leave", leaveNum == null ? 0 : leaveNum);

        // 工作日调整待办数量
        Integer overtimeNum = overtimeMapper.selectCount(Wrappers.lambdaQuery(SfaWorkOvertimeEntity.class)
                .in(SfaWorkOvertimeEntity::getBpmStatus, bpmStatusList)
                .in(SfaWorkOvertimeEntity::getAuditTaskId, getFormNoList(result, ActivitiEnum.FormTypeEnum.OVERTIME))
        );
        backlogCount.put("overtime", overtimeNum == null ? 0 : overtimeNum);

        // 异常报备待办数量
        Integer exceptionNum = exceptionMapper.selectCount(Wrappers.lambdaQuery(SfaExceptionReportEntity.class)
                .in(SfaExceptionReportEntity::getBpmStatus, bpmStatusList)
                .in(SfaExceptionReportEntity::getAuditTaskId, getFormNoList(result, ActivitiEnum.FormTypeEnum.EXCEPTION))
        );
        backlogCount.put("exception", exceptionNum == null ? 0 : exceptionNum);

        // 销假待办数量
        Integer leaveCancelNum = leaveCancelMapper.selectCount(Wrappers.lambdaQuery(SfaLeaveCancelEntity.class)
                .in(SfaLeaveCancelEntity::getBpmStatus, bpmStatusList)
                .in(SfaLeaveCancelEntity::getAuditTaskId, getFormNoList(result, ActivitiEnum.FormTypeEnum.LEAVECANCEL))
        );
        backlogCount.put("leaveCancel", leaveCancelNum == null ? 0 : leaveCancelNum);

        // 出差待办数量
        Integer travelNum = travelMapper.selectCount(Wrappers.lambdaQuery(SfaTravelEntity.class)
                .in(SfaTravelEntity::getBpmStatus, bpmStatusList)
                .in(SfaTravelEntity::getAuditTaskId, getFormNoList(result, ActivitiEnum.FormTypeEnum.TRAVEL))
        );
        backlogCount.put("travel", travelNum == null ? 0 : travelNum);

        // 抄送数量
        backlogCount.put("copyTask", 0);
        CopyTaskReqVO copyTaskReqVO = new CopyTaskReqVO();
        copyTaskReqVO.setUserCode(UserUtils.getUser().getUsername());
        copyTaskReqVO.setFormType(ActivitiEnum.getFormType());
        copyTaskReqVO.setStatus(1);
        Result<PageResult<CopyTaskRspVO>> copyTaskRsp = mobileFeign.findMyCopyTask(copyTaskReqVO);
        if (copyTaskRsp != null && copyTaskRsp.getResult() != null) {
            backlogCount.put("copyTask", Integer.parseInt(copyTaskRsp.getResult().getCount().toString()));
        }

        return backlogCount;
    }

    /**
     * 获取指定类型表单号集合
     *
     * @param taskRspVOPageResult
     * @param formTypeEnum
     * @return
     */
    private List<String> getFormNoList(PageResult<TaskRspVO> taskRspVOPageResult, ActivitiEnum.FormTypeEnum formTypeEnum) {
        if (taskRspVOPageResult == null || taskRspVOPageResult.getCount() == 0) {
            return Arrays.asList("");
        }
        List<String> formNoList = taskRspVOPageResult.getData().stream().filter(vo -> vo.getCostType().equals(formTypeEnum.getCostType()))
                .map(vo -> vo.getFormNo()).collect(Collectors.toList());
        if (formNoList == null || formNoList.size() == 0) {
            formNoList = Arrays.asList("");
        }
        return formNoList;
    }


    /**
     * 提交审核信息
     *
     * @param reqVo
     * @return
     */
    @Override
    public Result auditCommit(SfaAuditReqVo reqVo) {
        // 请求信息
        TaskOptReqVO optReqVO = new TaskOptReqVO();
        optReqVO.setTaskId(reqVo.getTaskId());
        optReqVO.setOptBtnCode(reqVo.getResult());
        optReqVO.setRemark(reqVo.getRemark());

        Result result = new Result();
        // 审批后修改审核状态为审核中
        if (ActivitiEnum.FormTypeEnum.LEAVE.getCostType().equals(reqVo.getCostType())) {
            result = leaveService.auditCommit(reqVo.getBusinessId());
        } else if (ActivitiEnum.FormTypeEnum.OVERTIME.getCostType().equals(reqVo.getCostType())) {
            result = overtimeService.auditCommit(reqVo.getBusinessId());
        } else if (ActivitiEnum.FormTypeEnum.EXCEPTION.getCostType().equals(reqVo.getCostType())) {
            result = exceptionService.auditCommit(reqVo.getBusinessId());
        } else if (ActivitiEnum.FormTypeEnum.LEAVECANCEL.getCostType().equals(reqVo.getCostType())) {
            result = leaveCancelService.auditCommit(reqVo.getBusinessId());
        } else if (ActivitiEnum.FormTypeEnum.TRAVEL.getCostType().equals(reqVo.getCostType())) {
            result = travelService.auditCommit(reqVo.getBusinessId());
        } else {
            result.error500("表单类型错误");
        }
        if (result.isSuccess()) {
            // 判断为那种操作
            if (ActivitiEnum.OperationType.ISSUE.getVal().equals(reqVo.getResult())) {
                // 审核通过
                result = mobileFeign.taskComplete(optReqVO);
            } else if (ActivitiEnum.OperationType.REJECT.getVal().equals(reqVo.getResult())) {
                // 审核驳回
                result = mobileFeign.taskReject(optReqVO);
            } else {
                result.error500("错误审批结果");
            }
        }

        return result;
    }

    /**
     * 请假审批-审批列表
     *
     * @return
     */
    @Override
    public PageResult<SfaAuditListLeaveRespVo> auditListLeave(SfaLeaveListReqVo reqVo) {

        // 获取工作流列表
        Result<PageResult<TaskRspVO>> mobileResult = findTaskRspList(reqVo.getAuditType()
                , ActivitiEnum.FormTypeEnum.LEAVE.getCostType());
        return leaveService.findAuditList(mobileResult.getResult(), reqVo);
    }

    /**
     * 工作日调整-审批列表
     *
     * @return
     */
    @Override
    public PageResult<SfaAuditListOvertimeRespVo> auditListOvertime(SfaWorkOvertimeListReqVo reqVo) {
        // 获取工作流列表
        Result<PageResult<TaskRspVO>> mobileResult = findTaskRspList(reqVo.getAuditType()
                , ActivitiEnum.FormTypeEnum.OVERTIME.getCostType());
        return overtimeService.findAuditList(mobileResult.getResult(), reqVo);
    }

    /**
     * 异常报备-审批列表
     *
     * @return
     */
    @Override
    public PageResult<SfaAuditListExceptionRespVo> auditListException(SfaExceptionReportListReqVo reqVo) {
        // 获取工作流列表
        Result<PageResult<TaskRspVO>> mobileResult = findTaskRspList(reqVo.getAuditType()
                , ActivitiEnum.FormTypeEnum.EXCEPTION.getCostType());
        return exceptionService.findAuditList(mobileResult.getResult(), reqVo);
    }

    /**
     * 销假-审批列表
     *
     * @param reqVo
     * @return
     */
    @Override
    public PageResult<SfaAuditListLeaveCancelRespVo> auditListLeaveCancel(SfaAuditListLeaveCancelReqVo reqVo) {
        // 获取工作流列表
        Result<PageResult<TaskRspVO>> mobileResult = findTaskRspList(reqVo.getAuditType()
                , ActivitiEnum.FormTypeEnum.LEAVECANCEL.getCostType());

        return leaveCancelService.findAuditList(mobileResult.getResult(), reqVo);
    }

    /**
     * 出差申请-待审批列表
     *
     * @param reqVo
     * @return
     */
    @Override
    public PageResult<SfaAuditListTravelRespVo> auditListTravel(SfaTravelReqVo reqVo) {
        // 获取工作流列表
        Result<PageResult<TaskRspVO>> mobileResult = findTaskRspList(reqVo.getAuditType()
                , ActivitiEnum.FormTypeEnum.TRAVEL.getCostType());
        return travelService.findAuditList(mobileResult.getResult(), reqVo);
    }

    /**
     * 抄送给我的-列表
     *
     * @param reqVO
     * @return
     */
    @Override
    public PageResult<SfaCopyTaskRespVo> copyTaskList(SfaCopyTaskReqVo reqVO) {
        CopyTaskReqVO copyTaskReqVO = CrmBeanUtil.copy(reqVO, CopyTaskReqVO.class);
        copyTaskReqVO.setUserCode(UserUtils.getUser().getUsername());
        copyTaskReqVO.setFormType(ActivitiEnum.getFormType());

        Result<PageResult<CopyTaskRspVO>> result = mobileFeign.findMyCopyTask(copyTaskReqVO);
        if (result != null && result.getResult() != null && result.getResult().getCount() > 0) {
            // 1、获取抄送列表
            List<CopyTaskRspVO> taskResult = result.getResult().getData();
            // 2、抄送列表转 map（formNO:CopyTaskRspVO）
            Map<String, CopyTaskRspVO> copyTaskMap = taskResult.stream().collect(Collectors.toMap(CopyTaskRspVO::getFormNo, vo -> vo, (k1, k2) -> k2));
            // 3、获取请假formNo集合、工作日调整formNo集合、异常报备formNo集合
            List<String> leaveAuditIds = taskResult.stream().filter(vo -> vo.getCostType().equals(ActivitiEnum.FormTypeEnum.LEAVE.getCostType())).collect(Collectors.toList()).stream().map(vo -> vo.getFormNo()).collect(Collectors.toList());
            List<String> overtimeAuditIds = taskResult.stream().filter(vo -> vo.getCostType().equals(ActivitiEnum.FormTypeEnum.OVERTIME.getCostType())).collect(Collectors.toList()).stream().map(vo -> vo.getFormNo()).collect(Collectors.toList());
            List<String> exceptionAuditIds = taskResult.stream().filter(vo -> vo.getCostType().equals(ActivitiEnum.FormTypeEnum.EXCEPTION.getCostType())).collect(Collectors.toList()).stream().map(vo -> vo.getFormNo()).collect(Collectors.toList());
            List<String> travelAuditIds = taskResult.stream().filter(vo -> vo.getCostType().equals(ActivitiEnum.FormTypeEnum.TRAVEL.getCostType())).collect(Collectors.toList()).stream().map(vo -> vo.getFormNo()).collect(Collectors.toList());
            // 4、分别查询3业务数据，并转成map(auditTaskId:Object)
            Map<String, SfaLeaveRespVo> leaveMap = new HashMap<>();
            if (leaveAuditIds != null && leaveAuditIds.size() > 0 && leaveAuditIds.get(0) != null) {
                SfaLeaveListReqVo findVo = new SfaLeaveListReqVo();
                findVo.setAuditTaskIdList(leaveAuditIds);
                PageResult<SfaLeaveRespVo> businessList = leaveService.findList(findVo);
                if (businessList != null && businessList.getCount() > 0) {
                    leaveMap = businessList.getData().stream().collect(Collectors.toMap(SfaLeaveRespVo::getAuditTaskId, vo -> vo, (k1, k2) -> k2));
                }
            }
            Map<String, SfaWorkOvertimeRespVo> overtimeMap = new HashMap<>();
            if (overtimeAuditIds != null && overtimeAuditIds.size() > 0 && overtimeAuditIds.get(0) != null) {
                SfaWorkOvertimeListReqVo findVo = new SfaWorkOvertimeListReqVo();
                findVo.setAuditTaskIdList(overtimeAuditIds);
                PageResult<SfaWorkOvertimeRespVo> businessList = overtimeService.findList(findVo);
                if (businessList != null && businessList.getCount() > 0) {
                    overtimeMap = businessList.getData().stream().collect(Collectors.toMap(SfaWorkOvertimeRespVo::getAuditTaskId, vo -> vo, (k1, k2) -> k2));
                }
            }
            Map<String, SfaExceptionReportRespVo> exceptionMap = new HashMap<>();
            if (exceptionAuditIds != null && exceptionAuditIds.size() > 0 && exceptionAuditIds.get(0) != null) {
                SfaExceptionReportListReqVo findVo = new SfaExceptionReportListReqVo();
                findVo.setAuditTaskIdList(exceptionAuditIds);
                PageResult<SfaExceptionReportRespVo> businessList = exceptionService.findList(findVo);
                if (businessList != null && businessList.getCount() > 0) {
                    exceptionMap = businessList.getData().stream().collect(Collectors.toMap(SfaExceptionReportRespVo::getAuditTaskId, vo -> vo, (k1, k2) -> k2));
                }
            }
            Map<String, SfaTravelRespVo> travelMap = new HashMap<>();
            if (travelAuditIds != null && travelAuditIds.size() > 0 && travelAuditIds.get(0) != null) {
                SfaTravelReqVo findVo = new SfaTravelReqVo();
                findVo.setAuditTaskIdList(travelAuditIds);
                PageResult<SfaTravelRespVo> sfaTravelRespVoPageResult = travelService.findList(findVo);
                if (sfaTravelRespVoPageResult != null && sfaTravelRespVoPageResult.getCount() > 0) {
                    travelMap = sfaTravelRespVoPageResult.getData().stream().collect(Collectors.toMap(SfaTravelRespVo::getAuditTaskId, vo -> vo, (k1, k2) -> k2));
                }
            }

            // 5、根据1的列表生成返回对象
            List<SfaCopyTaskRespVo> list = new ArrayList<>();
            for (CopyTaskRspVO vo : taskResult) {
                // 5.1、创建返回对象
                SfaCopyTaskRespVo respVo = new SfaCopyTaskRespVo();
                list.add(respVo);
                // 5.2、填充抄送返回结果数据
                respVo.setCopyTaskId(vo.getId());
                respVo.setAuditTaskId(vo.getFormNo());
                respVo.setTitle(vo.getTitle());
                respVo.setCostType(vo.getCostType());
                respVo.setCostTypeName(ActivitiEnum.getTitle(vo.getCostType()));
                respVo.setStatus(vo.getStatus());
                // 5.3、根据类型，根据formNo 查找对应业务数据并填充
                if (ActivitiEnum.FormTypeEnum.LEAVE.getCostType().equals(vo.getCostType())) { // 请假
                    SfaLeaveRespVo object = leaveMap.get(vo.getFormNo());
                    if (object != null) {
                        CrmBeanUtil.copyProperties(object, respVo);
//                        respVo.setId(object.getId());
//                        respVo.setRealName(object.getRealName());
//                        respVo.setPosName(object.getPosName());
//                        respVo.setOrgName(object.getOrgName());
//                        respVo.setApplicationDate(object.getApplicationDate());
//                        respVo.setBeginTime(object.getBeginTime());
//                        respVo.setEndTime(object.getEndTime());
//                        respVo.setBpmStatus(object.getBpmStatus());
                        respVo.setBpmStatusName(SfaCommonEnum.dataBpmStatus.getDesc(object.getBpmStatus()));
                    }
                } else if (ActivitiEnum.FormTypeEnum.OVERTIME.getCostType().equals(vo.getCostType())) { // 工作日调整
                    SfaWorkOvertimeRespVo object = overtimeMap.get(vo.getFormNo());
                    if (object != null) {
                        CrmBeanUtil.copyProperties(object, respVo);
                        respVo.setBpmStatusName(SfaCommonEnum.dataBpmStatus.getDesc(object.getBpmStatus()));
                    }
                } else if (ActivitiEnum.FormTypeEnum.EXCEPTION.getCostType().equals(vo.getCostType())) { // 异常报备
                    SfaExceptionReportRespVo object = exceptionMap.get(vo.getFormNo());
                    if (object != null) {
                        CrmBeanUtil.copyProperties(object, respVo);
                        respVo.setBpmStatusName(SfaCommonEnum.dataBpmStatus.getDesc(object.getBpmStatus()));
                    }
                } else if (ActivitiEnum.FormTypeEnum.TRAVEL.getCostType().equals(vo.getCostType())) { // 出差报备
                    SfaTravelRespVo object = travelMap.get(vo.getFormNo());
                    if (object != null) {
                        CrmBeanUtil.copyProperties(object, respVo);
                        respVo.setRealName(object.getApplyRealName());
                        respVo.setPosName(object.getApplyPosName());
                        respVo.setOrgName(object.getApplyOrgName());
                        respVo.setBpmStatusName(SfaCommonEnum.dataBpmStatus.getDesc(object.getBpmStatus()));
                    }
                }
            }

            return PageResult.<SfaCopyTaskRespVo>builder()
                    .data(list)
                    .count(result.getResult().getCount())
                    .build();
        }
        return null;
    }


    /**
     * 抄送节点已阅
     *
     * @param readCopyTaskReqVO
     * @return
     */
    @Override
    public Result readCopyTask(ReadCopyTaskReqVO readCopyTaskReqVO) {
        return mobileFeign.readCopyTask(readCopyTaskReqVO);
    }

    /**
     * 查询流程审批记录
     *
     * @param reqVo
     * @return
     */
    @Override
    public List<OptRecordRspVO> findAuditRecords(SfaAuditReportListReqVo reqVo) {
        OptRecordReqVO optRecordReqVO = new OptRecordReqVO();
        optRecordReqVO.setFormType(ActivitiEnum.getFormType());
        optRecordReqVO.setCostType(reqVo.getCostType());
        optRecordReqVO.setFormNo(reqVo.getAuditTaskId());

        Result<PageResult<OptRecordRspVO>> result = mobileFeign.getOptRecords(optRecordReqVO);
        if (result == null || result.getResult() == null || result.getResult().getCount() == 0) {
            return null;
        }
        return result.getResult().getData();
    }

    /**
     * 查询流程详细信息
     *
     * @param reqVo
     * @return
     */
    @Override
    public ProcessInfoRspVO getProcessInfo(SfaAuditReportListReqVo reqVo) {
        ProcessInfoReqVO processInfoReqVO = new ProcessInfoReqVO();
        processInfoReqVO.setFormType(ActivitiEnum.getFormType());
        processInfoReqVO.setCostType(reqVo.getCostType());
        processInfoReqVO.setFormNo(reqVo.getAuditTaskId());
        Result<ProcessInfoRspVO> result = mobileFeign.getProcessInfo(processInfoReqVO);
        if (!result.isSuccess()) {
            throw new BusinessException(result.getMessage());
        }
        return result.getResult();
    }

    /**
     * 审核完成回调
     *
     * @return
     */
    @Override
    public void auditFinish(SfaAuditFinishReqVo reqVo) {
        if (ActivitiEnum.FormTypeEnum.LEAVE.getCostType().equals(reqVo.getCostType())) {
            // 请假业务处理
            leaveService.auditFinish(reqVo);
        } else if (ActivitiEnum.FormTypeEnum.OVERTIME.getCostType().equals(reqVo.getCostType())) {
            // 工作日调整业务处理
            overtimeService.auditFinish(reqVo);
        } else if (ActivitiEnum.FormTypeEnum.EXCEPTION.getCostType().equals(reqVo.getCostType())) {
            // 异常报备业务处理
            exceptionService.auditFinish(reqVo);
        } else if (ActivitiEnum.FormTypeEnum.LEAVECANCEL.getCostType().equals(reqVo.getCostType())) {
            // 销假业务处理
            leaveService.auditCancelFinish(reqVo);
        } else if (ActivitiEnum.FormTypeEnum.TRAVEL.getCostType().equals(reqVo.getCostType())) {
            // 出差业务处理
            travelService.auditFinish(reqVo);
        }

    }


    private final static String TASK_TYPE = "2";

    /**
     * 查询当前登录人员审核任务流程列表,默认查询待审核
     *
     * @param type     1待审核，2已审核
     * @param costType 审核表单类型
     * @return
     */
    private Result<PageResult<TaskRspVO>> findTaskRspList(String type, String costType) {
        TaskQueryReqVO taskQueryReqVO = new TaskQueryReqVO();
        taskQueryReqVO.setUserCode(UserUtils.getUser().getUsername());
        taskQueryReqVO.setPositionCode(UserUtils.getUser().getPoscode());
        taskQueryReqVO.setCostType(costType);
        taskQueryReqVO.setFormType(ActivitiEnum.getFormType());

        Result<PageResult<TaskRspVO>> mobileResult;
        // 默认查询待审核
        if (TASK_TYPE.equals(type)) {
            mobileResult = mobileFeign.findDoneTask(taskQueryReqVO);
        } else {
            mobileResult = mobileFeign.findCurrentTask(taskQueryReqVO);
        }
        return mobileResult;
    }


}
