package com.biz.crm.visitinfo.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.base.ApiResultUtil;
import com.biz.crm.base.CrmBaseEntity;
import com.biz.crm.common.PageResult;
import com.biz.crm.common.param.RedisParam;
import com.biz.crm.eunm.CrmDelFlagEnum;
import com.biz.crm.eunm.sfa.SfaVisitEnum;
import com.biz.crm.mdm.position.MdmPositionFeign;
import com.biz.crm.mdm.user.MdmUserFeign;
import com.biz.crm.mq.RocketMQConstant;
import com.biz.crm.mq.RocketMQMessageBody;
import com.biz.crm.mq.RocketMQProducer;
import com.biz.crm.nebular.mdm.position.req.MdmPositionPageReqVo;
import com.biz.crm.nebular.mdm.position.resp.MdmPositionPageRespVo;
import com.biz.crm.nebular.mdm.user.req.MdmUserReqVo;
import com.biz.crm.nebular.mdm.user.resp.MdmUserRespVo;
import com.biz.crm.util.*;
import com.biz.crm.visitinfo.mapper.SfaVisitCompleteInfoMapper;
import com.biz.crm.visitinfo.model.SfaVisitCompleteInfoEntity;
import com.biz.crm.visitinfo.model.SfaVisitPlanInfoEntity;
import com.biz.crm.visitinfo.model.SfaVisitPlanInfoExecuteRedisData;
import com.biz.crm.visitinfo.model.SfaVisitPlanInfoRedisData;
import com.biz.crm.visitinfo.repositories.SfaVisitPlanInfoEntityRepositories;
import com.biz.crm.visitinfo.repositories.SfaVisitPlanInfoRedisDataRepositories;
import com.biz.crm.visitinfo.req.VisitCompleteInfoSearchReqVo;
import com.biz.crm.visitinfo.resp.report.SfaVisitCompleteInfoRespVo;
import com.biz.crm.visitinfo.resp.report.SfaVisitPlanRouteRespVo;
import com.biz.crm.visitinfo.resp.report.SfaVisitRealRouteRespVo;
import com.biz.crm.visitinfo.resp.report.SfaVisitRouteRespVo;
import com.biz.crm.visitinfo.service.ISfaVisitCompleteInfoService;
import com.biz.crm.visitinfo.service.ISfaVisitPlanInfoService;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.data.domain.Sort;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 拜访完成进度报表
 *
 * @author YuanZiJian
 * @date 2021/3/26 17:02
 * @version: V1.0
 */
@Service
@Slf4j
@ConditionalOnMissingBean(name = "SfaVisitCompleteInfoServiceExpandImpl")
public class SfaVisitCompleteInfoServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<SfaVisitCompleteInfoMapper, SfaVisitCompleteInfoEntity> implements JobHelper.JobExecutor, ISfaVisitCompleteInfoService {
    @Resource
    private SfaVisitPlanInfoEntityRepositories sfaVisitPlanInfoEntityRepositories;
    @Resource
    private SfaVisitCompleteInfoMapper sfaVisitCompleteInfoMapper;
    @Resource
    private SfaVisitPlanInfoRedisDataRepositories sfaVisitPlanInfoRedisDataRepositories;
    @Resource
    private MdmUserFeign mdmUserFeign;
    @Resource
    private MdmPositionFeign mdmPositionFeign;
    @Resource
    private JobHelper jobHelper;
    @Resource
    private RocketMQProducer rocketMQProducer;
    @Resource
    private ISfaVisitPlanInfoService sfaVisitPlanInfoService;

    /**
     * 后台——拜访完成报表——获取每日的完成报表进度
     *
     * @param visitCompleteInfoSearchReqVo 查询条件
     * @return com.biz.crm.common.PageResult<com.biz.crm.visitinfo.resp.report.SfaVisitCompleteInfoRespVo>
     * @method getVisitCompleteInfoList
     * @date: 2021/3/26 17:12
     * @author: YuanZiJian
     */
    @Override
    public PageResult<SfaVisitCompleteInfoRespVo> getVisitCompleteInfoDailyList(VisitCompleteInfoSearchReqVo visitCompleteInfoSearchReqVo) {
        Page<VisitCompleteInfoSearchReqVo> page = new Page<>(visitCompleteInfoSearchReqVo.getPageNum(), visitCompleteInfoSearchReqVo.getPageSize());
        List<SfaVisitCompleteInfoEntity> infoEntities = this.sfaVisitCompleteInfoMapper.getVisitCompleteInfoDailyList(page, visitCompleteInfoSearchReqVo);
        List<SfaVisitCompleteInfoRespVo> result = CrmBeanUtil.copyList(infoEntities, SfaVisitCompleteInfoRespVo.class);
        return PageResult.<SfaVisitCompleteInfoRespVo>builder()
                .count(page.getTotal())
                .data(result)
                .build();
    }

    /**
     * 后台——拜访完成报表——获取每月的完成报表进度
     *
     * @param visitCompleteInfoSearchReqVo
     * @return com.biz.crm.common.PageResult<com.biz.crm.visitinfo.resp.report.SfaVisitCompleteInfoRespVo>
     * @method getVisitCompleteInfoMonthList
     * @date: 2021/3/26 17:46
     * @author: YuanZiJian
     */
    @Override
    public PageResult<SfaVisitCompleteInfoRespVo> getVisitCompleteInfoMonthList(VisitCompleteInfoSearchReqVo visitCompleteInfoSearchReqVo) {
        Page<SfaVisitCompleteInfoRespVo> page = new Page<>(visitCompleteInfoSearchReqVo.getPageNum(), visitCompleteInfoSearchReqVo.getPageSize());
        List<SfaVisitCompleteInfoEntity> visitCompleteInfoMonthList = sfaVisitCompleteInfoMapper.getVisitCompleteInfoMonthList(page, visitCompleteInfoSearchReqVo);
        visitCompleteInfoMonthList.forEach(x -> {
            if (x.getPlanCompleteVisitNum() != null && x.getPlanCompleteVisitNum() > 0) {
                x.setCompleteVisitPercent(new BigDecimal(x.getRealCompleteVisitNum())
                        .divide(new BigDecimal(x.getPlanCompleteVisitNum()), 2, BigDecimal.ROUND_HALF_UP)
                        .multiply(new BigDecimal(100)).toString());
            }
            if (x.getPlanCompleteHelpVisitNum() != null && x.getPlanCompleteHelpVisitNum() > 0) {
                x.setCompleteHelpVisitPercent(new BigDecimal(x.getRealCompleteHelpVisitNum())
                        .divide(new BigDecimal(x.getPlanCompleteHelpVisitNum()), 2, BigDecimal.ROUND_HALF_UP)
                        .multiply(new BigDecimal(100)).toString());
            }
        });
        List<SfaVisitCompleteInfoRespVo> sfaVisitCompleteInfoRespVos = CrmBeanUtil.copyList(visitCompleteInfoMonthList, SfaVisitCompleteInfoRespVo.class);
        return PageResult.<SfaVisitCompleteInfoRespVo>builder()
                .count(page.getTotal())
                .data(sfaVisitCompleteInfoRespVos)
                .build();
    }

    @Override
    public void executeJob(Object jobParam) {
        this.doPlanInfoSettlementJob((String) jobParam);
    }

    /**
     * 结算给定日期的拜访计划明细 WEB端
     *
     * @param
     */
    @Override
    public void planInfoSettlementForWeb(String visitDateTemp) {
        this.planInfoSettlementJob(visitDateTemp);
    }

    /**
     * 结算给定日期的拜访计划明细 任务
     *
     * @param
     */
    @Override
    public void planInfoSettlementForJob(String visitDateTemp) {
        this.planInfoSettlementJob(visitDateTemp);
    }

    /**
     * 启动任务
     *
     * @param
     */
    protected void planInfoSettlementJob(String visitDateTemp) {

        //结算昨天的拜访计划
        if (StringUtils.isBlank(visitDateTemp)) {
            visitDateTemp = LocalDate.now().plusDays(-1).format(CrmDateUtils.yyyyMMdd);
        }
        JobHelper.JobContext context = new JobHelper.JobContext();
        context.setJob(JobHelper.Job.VISIT_DAY_RATE_JOB);
        context.setExecutorBeanClass(SfaVisitCompleteInfoServiceImpl.class);
        context.setJobParam(visitDateTemp);
        this.jobHelper.executeJobSaveContext(context);
    }

    /**
     * 执行任务
     *
     * @param visitDate
     */
    protected void doPlanInfoSettlementJob(String visitDate) {
        PageDataAdviser pageDataAdviser = new PageDataAdviser(jobHelper.getJobBizDataPageSize());
        long total = jobHelper.getJobBizDataPageSize();
        while (true) {
            if (pageDataAdviser.hasNext(total)) {
                MdmPositionPageReqVo reqVo = new MdmPositionPageReqVo();
                reqVo.setPageNum(pageDataAdviser.nextPage());
                reqVo.setPageSize(pageDataAdviser.getPageSize());
                PageResult<MdmPositionPageRespVo> pageResult = ApiResultUtil.objResult(this.mdmPositionFeign.positionAndUserPage(reqVo), true);
                List<MdmPositionPageRespVo> users = pageResult.getData();
                this.sendMq(users, visitDate);
                total = pageResult.getCount();
            } else {
                break;
            }
        }

    }

    /**
     * 发送mq
     *
     * @param users
     * @param visitDate
     */
    protected void sendMq(List<MdmPositionPageRespVo> users, String visitDate) {
        RocketMQMessageBody mqMessageBody = new RocketMQMessageBody();
        List<SfaVisitCompleteInfoEntity> entityList = new ArrayList<>();
        for (MdmPositionPageRespVo mdmPositionPageRespVo : users) {
            SfaVisitCompleteInfoEntity completeInfoEntity = new SfaVisitCompleteInfoEntity();
            completeInfoEntity.setVisitDate(visitDate);
            String visitUserName = mdmPositionPageRespVo.getUserName(),
                    visitPositionCode = mdmPositionPageRespVo.getPositionCode(),
                    visitPositionName = mdmPositionPageRespVo.getPositionName();
            if (StringUtils.isBlank(visitUserName)) {
                continue;
            }
            if (StringUtils.isBlank(visitPositionCode)) {
                visitPositionCode = visitUserName;
                visitPositionName = mdmPositionPageRespVo.getFullName();
            }
            completeInfoEntity.setVisitUserName(visitUserName);
            completeInfoEntity.setVisitRealName(mdmPositionPageRespVo.getFullName());
            completeInfoEntity.setVisitPosCode(visitPositionCode);
            completeInfoEntity.setVisitPosName(visitPositionName);
            completeInfoEntity.setVisitOrgName(mdmPositionPageRespVo.getOrgName());
            completeInfoEntity.setVisitOrgCode(mdmPositionPageRespVo.getOrgCode());
            completeInfoEntity.setVisitParentOrgName(mdmPositionPageRespVo.getParentOrgName());
            completeInfoEntity.setVisitParentOrgCode(mdmPositionPageRespVo.getParentOrgCode());
            entityList.add(completeInfoEntity);
        }
        if (CollectionUtil.listNotEmptyNotSizeZero(entityList)) {
            mqMessageBody.setMsgBody(JsonPropertyUtil.toJsonStringNotEmptyVal(entityList));
            mqMessageBody.setTag(RocketMQConstant.CRM_MQ_TAG.VISIT_DAY_RATE_SETTLEMENT);
            this.rocketMQProducer.convertAndSend(mqMessageBody);
        }

    }


    /**
     * 拜访完成进度报表日进度详情
     *
     * @param date
     * @return void
     * @method dailyCompleteInfo
     * @date: 2021/3/29 16:16
     * @author: YuanZiJian
     */
    @Override
    public Result<SfaVisitRouteRespVo> dailyCompleteInfo(String date, String posCode) {
        AssertUtils.isNotEmpty(date, "入参date为空");
        AssertUtils.isNotEmpty(posCode, "入参posCode为空");
        //去es找到当天计划拜访的终端_封装基础数据
        List<SfaVisitPlanInfoEntity> visitPlanInfoEntities = sfaVisitPlanInfoEntityRepositories.findByVisitPosCodeAndVisitDateAndVisitBigType(posCode, date, SfaVisitEnum.VisitBigType.VISIT.getVal(), Sort.by("visitSort").ascending());
        List<SfaVisitPlanRouteRespVo> sfaVisitPlanRouteRespVos = CrmBeanUtil.copyList(visitPlanInfoEntities, SfaVisitPlanRouteRespVo.class);
        //封装电话信息
        if (CollectionUtil.listNotEmpty(sfaVisitPlanRouteRespVos)) {
            SfaVisitPlanRouteRespVo sfaVisitPlanRouteRespVo = sfaVisitPlanRouteRespVos.get(0);
            MdmUserRespVo mdmUserRespVo = ApiResultUtil.objResult(mdmUserFeign.query(new MdmUserReqVo().setUserName(sfaVisitPlanRouteRespVo.getVisitUserName())));
            if (mdmUserRespVo != null) {
                String visitUserPhone = mdmUserRespVo.getUserPhone();
                sfaVisitPlanRouteRespVos.forEach(data -> data.setVisitUserPhone(visitUserPhone));
            }
        }
        //去es找到当天实际拜访的终端
        List<SfaVisitRealRouteRespVo> sfaVisitRealRouteRespVos = new ArrayList<>();
        List<SfaVisitPlanInfoRedisData> visitRealInfoEntities = sfaVisitPlanInfoRedisDataRepositories.findByVisitPosCodeAndVisitDateAndVisitBigTypeAndVisitStatus(posCode, date, SfaVisitEnum.VisitBigType.VISIT.getVal(), SfaVisitEnum.visitStatus.V3.getVal());
        if (CollectionUtil.listNotEmpty(visitRealInfoEntities)) {
            visitRealInfoEntities.forEach(item -> {
                SfaVisitPlanInfoExecuteRedisData sfaVisitPlanInfoExecuteRedisData = item.getSfaVisitPlanInfoExecuteRedisData();
                SfaVisitRealRouteRespVo routeRespVo = CrmBeanUtil.copy(sfaVisitPlanInfoExecuteRedisData, SfaVisitRealRouteRespVo.class);
                //封装基本数据_为方便前端进行转换
                routeRespVo.setLatitude(sfaVisitPlanInfoExecuteRedisData.getOutStoreLatitude());
                routeRespVo.setLongitude(sfaVisitPlanInfoExecuteRedisData.getOutStoreLongitude());
                routeRespVo.setVisitDate(sfaVisitPlanInfoExecuteRedisData.getInStore());
                //求拜访时长
                DateTimeFormatter df = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");
                LocalDateTime inStore = LocalDateTime.parse(sfaVisitPlanInfoExecuteRedisData.getInStore(), df);
                LocalDateTime outStore = LocalDateTime.parse(sfaVisitPlanInfoExecuteRedisData.getOutStore(), df);
                //转换为时间戳
                long inStoreMilli = inStore.toInstant(ZoneOffset.ofHours(8)).toEpochMilli();
                long outStoreMilli = outStore.toInstant(ZoneOffset.ofHours(8)).toEpochMilli();
                LocalDateTime inStoreTimeOut = Instant.ofEpochMilli(outStoreMilli - inStoreMilli).atZone(ZoneOffset.ofHours(8)).toLocalDateTime();
                //手动封装拜访时长
                routeRespVo.setInStoreTime(inStoreTimeOut.format(CrmDateUtils.HHmmss));
                //求距离
                //可能存在对应终端没有配置经纬度的情况——故做判断
                if (item.getLatitude() != null && item.getLongitude() != null
                        && null != sfaVisitPlanInfoExecuteRedisData.getOutStoreLatitude()
                        && null != sfaVisitPlanInfoExecuteRedisData.getOutStoreLongitude()) {
                    //网点纬度和网点经度
                    double latitude = item.getLatitude().doubleValue();
                    double longitude = item.getLongitude().doubleValue();
                    //离店纬度和离店经度
                    double outStoreLatitude = sfaVisitPlanInfoExecuteRedisData.getOutStoreLatitude().doubleValue();
                    double outStoreLongitude = sfaVisitPlanInfoExecuteRedisData.getOutStoreLongitude().doubleValue();
                    double distance = DistanceUtil.getDistance(latitude, longitude, outStoreLatitude, outStoreLongitude);
                    routeRespVo.setDistanceToPlan(distance + "");
                }
                //手动封装与终端的距离
                sfaVisitRealRouteRespVos.add(routeRespVo);
            });
        }
        //封装电话信息
        if (CollectionUtil.listNotEmpty(sfaVisitRealRouteRespVos)) {
            SfaVisitRealRouteRespVo sfaVisitRealRouteRespVo = sfaVisitRealRouteRespVos.get(0);
            MdmUserRespVo mdmUserRespVo = ApiResultUtil.objResult(mdmUserFeign.query(new MdmUserReqVo().setUserName(sfaVisitRealRouteRespVo.getVisitUserName())));
            if (mdmUserRespVo != null) {
                String visitUserPhone = mdmUserRespVo.getUserPhone();
                sfaVisitRealRouteRespVos.forEach(data -> data.setVisitUserPhone(visitUserPhone));
            }
        }
        //按照离店时间降序排
        if (CollectionUtil.listNotEmpty(sfaVisitRealRouteRespVos)) {
            sfaVisitRealRouteRespVos.sort(Comparator.comparing(SfaVisitRealRouteRespVo::getOutStore));
        }
        SfaVisitRouteRespVo sfaVisitRouteRespVo = new SfaVisitRouteRespVo();
        sfaVisitRouteRespVo.setPlanInfoList(sfaVisitPlanRouteRespVos);
        sfaVisitRouteRespVo.setRealInfoList(sfaVisitRealRouteRespVos);
        return Result.ok(sfaVisitRouteRespVo);
    }


    /**
     * 刷新拜访完成进度数据报表
     */
    @Override
    public void brushVisitCompleteInfo() {
        String now = LocalDate.now().format(CrmDateUtils.yyyyMMdd);
        //状态正常并当天（包含）之前的拜访计划，key-职位编码+人员账号+组织编码
        Map<String, List<SfaVisitPlanInfoEntity>> visitPlanInfoMap = sfaVisitPlanInfoService.lambdaQuery()
                .eq(SfaVisitPlanInfoEntity::getDelFlag, CrmDelFlagEnum.NORMAL.getCode())
                .lt(SfaVisitPlanInfoEntity::getVisitDate, now)
                .list().stream().collect(Collectors.groupingBy(x -> x.getVisitPosCode() + RedisParam.DELIMITER + x.getVisitUserName() + RedisParam.DELIMITER + x.getVisitOrgCode()));
        visitPlanInfoMap.forEach((k, v) -> this.planInfoComplete(v));
    }


    /**
     * 进行计算出拜访结算信息
     *
     * @param planInfoEntityList
     * @return
     */
    private List<SfaVisitCompleteInfoEntity> planInfoComplete(List<SfaVisitPlanInfoEntity> planInfoEntityList) {
        List<SfaVisitCompleteInfoEntity> visitCompleteInfoEntityList = new ArrayList<>();
        //通过拜访大类型进行分组(拜访/协访)
        Map<String, List<SfaVisitPlanInfoEntity>> visitBigTypeMap = planInfoEntityList.stream().collect(Collectors.groupingBy(SfaVisitPlanInfoEntity::getVisitBigType));
        visitBigTypeMap.forEach((k, v) -> {
            //k->visitDate
            Map<String, List<SfaVisitPlanInfoEntity>> visitDateMap = v.stream().collect(Collectors.groupingBy(SfaVisitPlanInfoEntity::getVisitDate));
            visitDateMap.forEach((key, value) -> {
                SfaVisitPlanInfoEntity infoEntity = value.get(0);
                SfaVisitCompleteInfoEntity completeInfoEntity = new SfaVisitCompleteInfoEntity() {{
                    this.setVisitUserName(infoEntity.getVisitUserName());
                    this.setVisitRealName(infoEntity.getVisitRealName());
                    this.setVisitPosCode(infoEntity.getVisitPosCode());
                    this.setVisitPosName(infoEntity.getVisitPosName());
                    this.setVisitOrgCode(infoEntity.getVisitOrgCode());
                    this.setVisitOrgName(infoEntity.getVisitOrgName());
                    this.setVisitParentOrgCode(infoEntity.getParentOrgCode());
                    this.setVisitParentOrgName(infoEntity.getParentOrgName());
                    this.setVisitDate(infoEntity.getVisitDate());
                    this.setVisitYearMonth(infoEntity.getVisitDateOfYearMonth());
                }};
                //判断为拜访
                if (SfaVisitEnum.VisitBigType.VISIT.getVal().equals(k)) {
                    SfaVisitCompleteInfoEntity.countingInfoForVisit(completeInfoEntity, value);
                } else if (SfaVisitEnum.VisitBigType.HELP_VISIT.getVal().equals(k)) {
                    //判断为协访
                    SfaVisitCompleteInfoEntity.countingInfoForHelpVisit(completeInfoEntity, value);
                } else if (SfaVisitEnum.VisitBigType.UNFAMILIAR_VISIT.getVal().equals(k)) {
                    //判断为陌拜
                    SfaVisitCompleteInfoEntity.countingInfoForMoVisit(completeInfoEntity, value);
                } else {
                    return;
                }
                visitCompleteInfoEntityList.add(completeInfoEntity);
            });
        });
        if (CollectionUtil.listNotEmptyNotSizeZero(visitCompleteInfoEntityList)) {
            this.saveBatch(visitCompleteInfoEntityList);
        }
        return visitCompleteInfoEntityList;
    }

}
