package com.biz.crm.visitinfo.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.collection.controller.resp.VisitStepResp;
import com.biz.crm.eunm.sfa.SfaVisitEnum;
import com.biz.crm.nebular.sfa.visitinfo.resp.SfaVisitPlanInfoStepFormRespVo;
import com.biz.crm.util.AssertUtils;
import com.biz.crm.util.CollectionUtil;
import com.biz.crm.util.CrmBeanUtil;
import com.biz.crm.visitinfo.mapper.SfaVisitInfoStepFormMapper;
import com.biz.crm.visitinfo.mapper.SfaVisitPlanInfoMapper;
import com.biz.crm.visitinfo.model.SfaVisitInfoStepFormEntity;
import com.biz.crm.visitinfo.model.SfaVisitPlanInfoEntity;
import com.biz.crm.visitinfo.service.ISfaVisitInfoStepFormService;
import com.biz.crm.visitinfo.stepformcomponent.SfaVisitPlanInfoStepFormComponent;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 拜访明细步骤表单接口实现
 *
 * @author lf
 * @date 2021-07-18 15:00:34
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "SfaVisitInfoStepFormServiceExpandImpl")
public class SfaVisitInfoStepFormServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<SfaVisitInfoStepFormMapper, SfaVisitInfoStepFormEntity> implements ISfaVisitInfoStepFormService {

    @Resource
    private SfaVisitInfoStepFormMapper sfaVisitInfoStepFormMapper;
    @Resource
    private SfaVisitPlanInfoMapper sfaVisitPlanInfoMapper;
    @Resource
    private SfaVisitPlanInfoStepFormComponent sfaVisitPlanInfoStepFormComponent;

    /**
     * 分页数据大小
     */
    private final static Integer PAGE_SIZE = 500;

    /**
     * 保存当前拜访明细的拜访步骤
     *
     * @param list
     * @param sfaVisitPlanInfoEntity
     */
    @Override
    public void saveSfaVisitInfoForm(List<VisitStepResp> list, SfaVisitPlanInfoEntity sfaVisitPlanInfoEntity) {
        String visitPlanInfoId = sfaVisitPlanInfoEntity.getId();
        String visitPlanInfoCode = sfaVisitPlanInfoEntity.getVisitPlanCode();
        LambdaQueryWrapper<SfaVisitInfoStepFormEntity> wrapper = Wrappers.lambdaQuery(SfaVisitInfoStepFormEntity.class)
                .eq(SfaVisitInfoStepFormEntity::getVisitPlanInfoId, visitPlanInfoId)
                .eq(SfaVisitInfoStepFormEntity::getVisitPlanCode, visitPlanInfoCode);
        List<SfaVisitInfoStepFormEntity> formList = sfaVisitInfoStepFormMapper.selectList(wrapper);
        if (CollectionUtil.listEmpty(formList)) {
            //过滤到进店/离店打卡的步骤
            List<SfaVisitInfoStepFormEntity> entities = list.stream()
                    .filter(l -> !l.getStepCode().equals(SfaVisitEnum.SfaStepCode.SFA_STEP_CODE_IN_STORE.getVal()) &&
                            !l.getStepCode().equals(SfaVisitEnum.SfaStepCode.SFA_STEP_CODE_OUT_STORE.getVal()))
                    .map(l -> {
                        SfaVisitInfoStepFormEntity entity = CrmBeanUtil.copy(l, SfaVisitInfoStepFormEntity.class);
                        entity.setVisitPlanInfoId(visitPlanInfoId);
                        entity.setVisitPlanCode(visitPlanInfoCode);
                        entity.setSortNum(l.getSort());
                        return entity;
                    }).collect(Collectors.toList());
            this.saveBatch(entities);
        }
    }

    /**
     * 更新拜访步骤状态
     *
     * @param stepCode
     * @param visitPlanInfoId
     */
    @Override
    public void updateVisitStepStatus(String stepCode, String visitPlanInfoId) {
        SfaVisitInfoStepFormEntity formEntity = this.lambdaQuery()
                .eq(SfaVisitInfoStepFormEntity::getVisitPlanInfoId, visitPlanInfoId)
                .eq(SfaVisitInfoStepFormEntity::getStepCode, stepCode).one();
        if (formEntity != null) {
            formEntity.setIsSuccessDesc(SfaVisitEnum.isSuccess.ALREADY_SUCCESS.getDesc());
            formEntity.setIsSuccess(SfaVisitEnum.isSuccess.ALREADY_SUCCESS.getVal());
            this.updateById(formEntity);
        }
    }


    @Override
    public void resolvingVisitPlanInfoStepForm(String visitDate) {
        AssertUtils.isNotEmpty(visitDate, "解析过去拜访明细数据的日期为空");
        //第一次查询 获取count
        Page<SfaVisitPlanInfoStepFormRespVo> page = new Page<>(1, PAGE_SIZE);
        List<SfaVisitPlanInfoStepFormRespVo> formList = sfaVisitPlanInfoMapper.findVisitPlanInfoStepFormList(page, visitDate);
        if (CollectionUtil.listNotEmptyNotSizeZero(formList)) {
            List<SfaVisitPlanInfoStepFormRespVo> list = formList.stream()
                    .collect(Collectors.groupingBy(SfaVisitPlanInfoStepFormRespVo::getVisitPlanInfoId))
                    .entrySet().stream().map(data -> {
                        return data.getValue().get(0);
                    }).collect(Collectors.toList());
            //进行步骤编码的解析
            list = sfaVisitPlanInfoStepFormComponent.getVisitPlanInfoRedisDataByEs(list);
            sfaVisitPlanInfoStepFormComponent.resolvingStepForm(list);
        } else {
            return;
        }
        Integer size = 0;
        if (page.getTotal() > PAGE_SIZE) {
            size = (int) page.getTotal() / PAGE_SIZE;
            if ((int) page.getTotal() % PAGE_SIZE > 0) {
                size = size + 2;
            } else {
                size++;
            }
        }
        for (int i = 2; i < size; i++) {
            List<SfaVisitPlanInfoStepFormRespVo> list = this.getVisitPlanInfoList(i, visitDate);
            if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
                list = formList.stream()
                        .collect(Collectors.groupingBy(SfaVisitPlanInfoStepFormRespVo::getVisitPlanInfoId))
                        .entrySet().stream().map(data -> {
                            return data.getValue().get(0);
                        }).collect(Collectors.toList());
                //进行步骤编码的解析
                list = sfaVisitPlanInfoStepFormComponent.getVisitPlanInfoRedisDataByEs(list);
                sfaVisitPlanInfoStepFormComponent.resolvingStepForm(list);
            } else {
                break;
            }
        }
    }

    /**
     * 查询解析拜访明细数据步骤的数据列表
     *
     * @param num
     * @param visitDate
     * @return
     */
    private List<SfaVisitPlanInfoStepFormRespVo> getVisitPlanInfoList(Integer num, String visitDate) {
        Page<SfaVisitPlanInfoStepFormRespVo> page = new Page<>(num, PAGE_SIZE);
        return sfaVisitPlanInfoMapper.findVisitPlanInfoStepFormList(page, visitDate);
    }
}
