package com.biz.crm.visitinfo.service.impl;

import com.biz.crm.base.ApiResultUtil;
import com.biz.crm.base.BusinessException;
import com.biz.crm.common.PageResult;
import com.biz.crm.eunm.sfa.SfaVisitEnum;
import com.biz.crm.mdm.position.MdmPositionFeign;
import com.biz.crm.mq.RocketMQConstant;
import com.biz.crm.mq.RocketMQMessageBody;
import com.biz.crm.mq.RocketMQProducer;
import com.biz.crm.nebular.mdm.position.req.MdmPositionPageReqVo;
import com.biz.crm.nebular.mdm.position.resp.MdmPositionPageRespVo;
import com.biz.crm.nebular.mdm.position.resp.MdmPositionRespVo;
import com.biz.crm.nebular.sfa.visitnote.resp.SfaVisitGroupRespVo;
import com.biz.crm.nebular.sfa.visitnote.resp.SfaVisitPlanRespVo;
import com.biz.crm.service.RedisService;
import com.biz.crm.util.*;
import com.biz.crm.visitinfo.model.SfaVisitPlanInfoEntity;
import com.biz.crm.visitinfo.req.PlanInfoSettlementReq;
import com.biz.crm.visitnote.model.SfaVisitPlanEntity;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.time.LocalDate;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

@Slf4j
@Component
public class VisitPlanInfoHelper implements JobHelper.JobExecutor {
    @Resource
    private RedisService redisService;
    @Resource
    private SfaVisitPlanInfoServiceEsImpl sfaVisitPlanInfoServiceEsImpl;
    @Resource
    private MdmPositionFeign mdmPositionFeign;

    @Resource
    private JobHelper jobHelper;
    @Resource
    private RocketMQProducer rocketMQProducer;

    @Override
    public void executeJob(Object jobParam) {
        PlanInfoSettlementReq req = JsonPropertyUtil.toObject((String) jobParam, PlanInfoSettlementReq.class);
        this.doPlanInfoSettlementJob(req);
    }

    /**
     * 结算给定日期的拜访计划明细 WEB端
     *
     * @param
     */
    public void planInfoSettlementForWeb(PlanInfoSettlementReq req) {
        this.planInfoSettlementJob(req);
    }

    /**
     * 结算给定日期的拜访计划明细 任务
     *
     * @param
     */
    public void planInfoSettlementForJob(PlanInfoSettlementReq req) {
        this.planInfoSettlementJob(req);
    }

    /**
     * 启动任务
     *
     * @param
     */
    protected void planInfoSettlementJob(PlanInfoSettlementReq req) {
        if (null == req) {
            req = new PlanInfoSettlementReq();
        }
        String visitDateTemp = req.getVisitDate();

        //结算昨天的拜访计划
        if (StringUtils.isBlank(visitDateTemp)) {
            visitDateTemp = LocalDate.now().plusDays(-1).format(CrmDateUtils.yyyyMMdd);
            req.setVisitDate(visitDateTemp);
        }
        JobHelper.JobContext context = new JobHelper.JobContext();
        context.setJob(JobHelper.Job.PLAN_INFO_SETTLEMENT_JOB);
        context.setExecutorBeanClass(VisitPlanInfoHelper.class);
        context.setJobParam(JsonPropertyUtil.toJsonStringNotEmptyVal(req));
        this.jobHelper.executeJobSaveContext(context);
    }


    /**
     * 执行任务
     *
     * @param req
     */
    protected void doPlanInfoSettlementJob(PlanInfoSettlementReq req) {
        String visitDate = req.getVisitDate();
        PageDataAdviser pageDataAdviser = new PageDataAdviser(jobHelper.getJobBizDataPageSize());
        long total = jobHelper.getJobBizDataPageSize();
        while (true) {
            if (pageDataAdviser.hasNext(total)) {
                MdmPositionPageReqVo reqVo = new MdmPositionPageReqVo();
                reqVo.setPageNum(pageDataAdviser.nextPage());
                reqVo.setPageSize(pageDataAdviser.getPageSize());
                reqVo.setUserName(req.getVisitUserName());
                PageResult<MdmPositionPageRespVo> pageResult = ApiResultUtil.objResult(this.mdmPositionFeign.positionAndUserPage(reqVo), true);
                List<MdmPositionPageRespVo> users = pageResult.getData();
                this.sendMq(users, visitDate);
                total = pageResult.getCount();
            } else {
                break;
            }
        }

    }


    /**
     * 发送mq
     *
     * @param users
     * @param visitDate
     */
    protected void sendMq(List<MdmPositionPageRespVo> users, String visitDate) {
        RocketMQMessageBody mqMessageBody = new RocketMQMessageBody();

        PlanInfoSettlementReq settlementReq = new PlanInfoSettlementReq();
        settlementReq.setVisitDate(visitDate);
        //转换对象，过滤用户为空的
        List<PlanInfoSettlementReq.VisitUserPosition> visitUserPositionList = users.stream().map(x -> {
            PlanInfoSettlementReq.VisitUserPosition userPosition = new PlanInfoSettlementReq.VisitUserPosition();
            String visitUserName = x.getUserName(),
                    visitPositionCode = x.getPositionCode();
            if (StringUtils.isBlank(visitUserName)) {
                return userPosition;
            }
            if (StringUtils.isBlank(visitPositionCode)) {
                visitPositionCode = visitUserName;
            }
            userPosition.setVisitPositionCode(visitPositionCode);
            userPosition.setVisitUserName(visitUserName);
            return userPosition;
        }).filter(x -> x != null && StringUtils.isNotBlank(x.getVisitUserName())).collect(Collectors.toList());

        if (CollectionUtil.listNotEmptyNotSizeZero(visitUserPositionList)) {
            settlementReq.setVisitUserPositionList(visitUserPositionList);
            mqMessageBody.setMsgBody(JsonPropertyUtil.toJsonStringNotEmptyVal(settlementReq));
            mqMessageBody.setTag(RocketMQConstant.CRM_MQ_TAG.VISIT_PLAN_INFO_SETTLEMENT);
            this.rocketMQProducer.convertAndSend(mqMessageBody);
        }

//        for (MdmPositionPageRespVo mdmPositionPageRespVo : users) {
//            PlanInfoSettlementReq settlementReq = new PlanInfoSettlementReq();
//            settlementReq.setVisitDate(visitDate);
//            String visitUserName = mdmPositionPageRespVo.getUserName(),
//                    visitPositionCode = mdmPositionPageRespVo.getPositionCode();
//            if(StringUtils.isBlank(visitUserName)){
//                continue;
//            }
//            if(StringUtils.isBlank(visitPositionCode)){
//                visitPositionCode = visitUserName;
//            }
//            settlementReq.setVisitPositionCode(visitPositionCode);
//            settlementReq.setVisitUserName(visitUserName);
//            mqMessageBody.setMsgBody(JsonPropertyUtil.toJsonStringNotEmptyVal(settlementReq));
//            mqMessageBody.setTag(RocketMQConstant.CRM_MQ_TAG.VISIT_PLAN_INFO_SETTLEMENT);
//            this.rocketMQProducer.convertAndSend(mqMessageBody);
//        }

    }


    /**
     * 拜访计划刷新计算
     * 使用给定的最新拜访计划列表，计算出需要删除的计划，需要新增的计划
     * 如果当前计划里存在已经执行过的计划，那么忽略该计划的刷新操作（newPlan.remove(oldValueRedisHashKey)）
     *
     * @param daysPlanInfoTemp 最新拜访计划列表
     * @return
     */
    public Map<String, Map<String, List<SfaVisitPlanInfoEntity>>> flushCounting(Map<String, Map<String, SfaVisitPlanInfoEntity>> daysPlanInfoTemp) {
        //es可能有重复数据（比如redis非正常全量清理缓存时），所以用List装
        Map<String, Map<String, List<SfaVisitPlanInfoEntity>>> needDelete = Maps.newHashMap();
        for (Map.Entry<String, Map<String, SfaVisitPlanInfoEntity>> entry : daysPlanInfoTemp.entrySet()) {
            String redisHash = entry.getKey();
            SfaVisitPlanInfoEntity.VisitRedisHash visitRedisHash = new SfaVisitPlanInfoEntity.VisitRedisHash(redisHash);
            //es老数据
            List<SfaVisitPlanInfoEntity> oldPlanOfEs = this.sfaVisitPlanInfoServiceEsImpl.findByVisitPosCodeAndVisitDateAndVisitBigType(visitRedisHash.getVisitPosCode(), visitRedisHash.getVisitDate(), visitRedisHash.getVisitBigType());
            Map<String, SfaVisitPlanInfoEntity> oldPlanOfRedis = (Map<String, SfaVisitPlanInfoEntity>) this.redisService.hmget(redisHash);
            // 比较redis与es的数据，并补偿redis
            this.compareAndCompensateForRedis(redisHash, oldPlanOfEs, oldPlanOfRedis);

            Map<String, List<SfaVisitPlanInfoEntity>> needDeletePlanInfos = needDelete.get(redisHash);
            if (null == needDeletePlanInfos) {
                needDeletePlanInfos = Maps.newHashMap();
            }
            Map<String, SfaVisitPlanInfoEntity> newPlan = entry.getValue();
            for (Map.Entry<String, SfaVisitPlanInfoEntity> oldValueEntryOfRedis : oldPlanOfRedis.entrySet()) {
                //计算新计划与老计划差异
                this.countingDifference(oldValueEntryOfRedis, newPlan, needDeletePlanInfos);
            }
            needDelete.put(redisHash, needDeletePlanInfos);
        }

        return needDelete;

    }

    /**
     * 计算新计划与老计划差异
     *
     * @param oldValueEntryOfRedis
     * @param newPlan
     * @param needDeletePlanInfos
     */
    protected void countingDifference(Map.Entry<String, SfaVisitPlanInfoEntity> oldValueEntryOfRedis, Map<String, SfaVisitPlanInfoEntity> newPlan, Map<String, List<SfaVisitPlanInfoEntity>> needDeletePlanInfos) {
        SfaVisitPlanInfoEntity oldValue = oldValueEntryOfRedis.getValue();
        String oldValueRedisHashKey = oldValue.getRedisHashKey();
        //老计划被拜访执行过，则忽略该计划的刷新操作
        if (!SfaVisitEnum.visitStatus.V1.getVal().equals(oldValue.getVisitStatus())) {
            newPlan.remove(oldValueRedisHashKey);
            return;
        }
        //如果老计划是临时拜访
        if (SfaVisitEnum.visitType.TEMP_VISIT.getVal().equals(oldValue.getVisitType())) {
            //如果新计划存在该计划，删除老计划，使用新计划
            if (newPlan.containsKey(oldValueRedisHashKey)) {
                this.putDeletePlanInfos(needDeletePlanInfos, oldValue);
                return;
            }
            //新计划不存在该计划，则保持之前用户添加的临时计划不变
            return;
        }
        if (newPlan.containsKey(oldValueRedisHashKey)) {
            //新老列表都存在该计划，则不用刷新
            newPlan.remove(oldValueRedisHashKey);
        } else {
            //新列表，不存在该计划，删除老计划
            this.putDeletePlanInfos(needDeletePlanInfos, oldValue);
        }
    }

    /**
     * 比较redis与es的数据，并补偿redis
     * 解决因redis缓存丢失导致的ES与redis数据差异问题，手动刷新计划后ES数据重复问题
     *
     * @param redisHash
     * @param oldPlanOfEs
     * @param oldPlanOfRedis
     */
    protected void compareAndCompensateForRedis(String redisHash, List<SfaVisitPlanInfoEntity> oldPlanOfEs, Map<String, SfaVisitPlanInfoEntity> oldPlanOfRedis) {
        if (null == oldPlanOfRedis) {
            oldPlanOfRedis = Maps.newHashMap();
        }
        Map<String, SfaVisitPlanInfoEntity> compensateOfRedis = Maps.newHashMap();
        for (SfaVisitPlanInfoEntity oldPlanOfE : oldPlanOfEs) {
            if (oldPlanOfRedis.containsKey(oldPlanOfE.getRedisHashKey())) {
                continue;
            }
            compensateOfRedis.put(oldPlanOfE.getRedisHashKey(), oldPlanOfE);
            oldPlanOfRedis.put(oldPlanOfE.getRedisHashKey(), oldPlanOfE);
        }
        this.saveToRedis(redisHash, compensateOfRedis, null);
    }

    /**
     * 保存到es
     */
    public void saveToRedis(String redisHash, Map<String, SfaVisitPlanInfoEntity> objectObjectMap, final Set<String> needDeleteRedisHashKeys) {
//        Map<String, SfaVisitPlanInfoEntity> dbData = (Map<String, SfaVisitPlanInfoEntity>) this.redisService.hmget(redisHash);

//        Set<String> needDelKeys = dbData.keySet();
        if (!org.springframework.util.CollectionUtils.isEmpty(needDeleteRedisHashKeys)) {
            String[] redisHashKeys = needDeleteRedisHashKeys.toArray(new String[needDeleteRedisHashKeys.size()]);
            this.redisService.hdel(redisHash, redisHashKeys);
        }
        if (objectObjectMap.size() > 0) {
            boolean redis = this.redisService.hmset(redisHash, objectObjectMap, SfaVisitPlanInfoEntity.CACHE_TIME);
            if (!redis) {
                throw new BusinessException("解析拜访计划：保存redis失败！");
            }
        }
    }

    protected void putDeletePlanInfos(Map<String, List<SfaVisitPlanInfoEntity>> needDeletePlanInfos, SfaVisitPlanInfoEntity oldValue) {
        String oldValueRedisHashKey = oldValue.getRedisHashKey();
        List<SfaVisitPlanInfoEntity> deletePlanInfos = needDeletePlanInfos.get(oldValueRedisHashKey);
        if (null == deletePlanInfos) {
            deletePlanInfos = Lists.newArrayList();
        }
        deletePlanInfos.add(oldValue);
        needDeletePlanInfos.put(oldValueRedisHashKey, deletePlanInfos);
    }


    /**
     * 实时去mdm查询职位对应的用户，避免mdm职位用户关系变更后与sfa数据不同步的问题
     *
     * @param list
     */
    public void loadUserDataForVisitGroup(List<SfaVisitGroupRespVo> list) {
        Set<String> posCodes = list.stream().filter(v -> !Objects.equals(v.getUserName(), v.getPosCode())).map(SfaVisitGroupRespVo::getPosCode).collect(Collectors.toSet());
        Map<String, MdmPositionRespVo> positionRespVoMap;
        if (posCodes.size() > 0) {
            positionRespVoMap = PositionUtil.getPositionByCodeList(Lists.newArrayList(posCodes))
                    .stream().collect(Collectors.toMap(MdmPositionRespVo::getPositionCode, v -> v, (t, t2) -> t2));
        } else {
            positionRespVoMap = Maps.newHashMap();
        }
        for (SfaVisitGroupRespVo sfaVisitPlanRespVo : list) {

            MdmPositionRespVo positionRespVo = positionRespVoMap.get(sfaVisitPlanRespVo.getPosCode());
            if (null != positionRespVo && org.apache.commons.lang3.StringUtils.isNotBlank(positionRespVo.getUserName())) {
                sfaVisitPlanRespVo.setUserName(positionRespVo.getUserName());
                sfaVisitPlanRespVo.setFullName(positionRespVo.getFullName());
            }
        }
    }

    /**
     * 实时去mdm查询职位对应的用户，避免mdm职位用户关系变更后与sfa数据不同步的问题
     *
     * @param visitGroupRespVo
     */
    public void loadUserDataForVisitGroup(SfaVisitGroupRespVo visitGroupRespVo) {
        if (Objects.equals(visitGroupRespVo.getUserName(), visitGroupRespVo.getPosCode())) {
            return;
        }
        MdmPositionRespVo positionRespVo = PositionUtil.getPositionByCode(visitGroupRespVo.getPosCode());
        if (null != positionRespVo && org.apache.commons.lang3.StringUtils.isNotBlank(positionRespVo.getUserName())) {
            visitGroupRespVo.setUserName(positionRespVo.getUserName());
            visitGroupRespVo.setFullName(positionRespVo.getFullName());
        }
    }

    /**
     * 实时去mdm查询职位对应的用户，避免mdm职位用户关系变更后与sfa数据不同步的问题
     *
     * @param list
     */
    public void loadUserDataForVisitPlanVo(List<SfaVisitPlanRespVo> list) {
        Set<String> posCodes = list.stream().filter(v -> !Objects.equals(v.getVisitUserName(), v.getVisitPosCode())).map(SfaVisitPlanRespVo::getVisitPosCode).collect(Collectors.toSet());
        Map<String, MdmPositionRespVo> positionRespVoMap;
        if (posCodes.size() > 0) {
            positionRespVoMap = PositionUtil.getPositionByCodeList(Lists.newArrayList(posCodes))
                    .stream().collect(Collectors.toMap(MdmPositionRespVo::getPositionCode, v -> v, (t, t2) -> t2));
        } else {
            positionRespVoMap = Maps.newHashMap();
        }
        for (SfaVisitPlanRespVo sfaVisitPlanRespVo : list) {

            MdmPositionRespVo positionRespVo = positionRespVoMap.get(sfaVisitPlanRespVo.getVisitPosCode());
            if (null != positionRespVo && org.apache.commons.lang3.StringUtils.isNotBlank(positionRespVo.getUserName())) {
                sfaVisitPlanRespVo.setVisitUserName(positionRespVo.getUserName());
                sfaVisitPlanRespVo.setVisitRealName(positionRespVo.getFullName());
            }
        }
    }

    /**
     * 实时去mdm查询职位对应的用户，避免mdm职位用户关系变更后与sfa数据不同步的问题
     *
     * @param list
     */
    public void loadUserDataForVisitPlanEntity(List<SfaVisitPlanEntity> list) {
        Set<String> posCodes = list.stream().filter(v -> !Objects.equals(v.getVisitUserName(), v.getVisitPosCode())).map(SfaVisitPlanEntity::getVisitPosCode).collect(Collectors.toSet());
        Map<String, MdmPositionRespVo> positionRespVoMap;
        if (posCodes.size() > 0) {
            positionRespVoMap = PositionUtil.getPositionByCodeList(Lists.newArrayList(posCodes))
                    .stream().collect(Collectors.toMap(MdmPositionRespVo::getPositionCode, v -> v, (t, t2) -> t2));
        } else {
            positionRespVoMap = Maps.newHashMap();
        }
        for (SfaVisitPlanEntity sfaVisitPlanRespVo : list) {

            MdmPositionRespVo positionRespVo = positionRespVoMap.get(sfaVisitPlanRespVo.getVisitPosCode());
            if (null != positionRespVo && org.apache.commons.lang3.StringUtils.isNotBlank(positionRespVo.getUserName())) {
                sfaVisitPlanRespVo.setVisitUserName(positionRespVo.getUserName());
                sfaVisitPlanRespVo.setVisitRealName(positionRespVo.getFullName());
            }
        }
    }
}
