package com.biz.crm.visitnote.service.impl;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.base.BusinessException;
import com.biz.crm.base.SfaClientData;
import com.biz.crm.base.SfaClientHelper;
import com.biz.crm.base.config.ThreadLocalUtil;
import com.biz.crm.common.GlobalParam;
import com.biz.crm.common.PageResult;
import com.biz.crm.crmlog.handle.util.CrmLogSendUtil;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.eunm.sfa.SfaCodeEnum;
import com.biz.crm.eunm.sfa.SfaVisitEnum;
import com.biz.crm.nebular.mdm.constant.UserTypeEnum;
import com.biz.crm.nebular.sfa.visitnote.req.SfaVisitGroupReqVo;
import com.biz.crm.nebular.sfa.visitnote.req.SfaVisitPlanReqVo;
import com.biz.crm.nebular.sfa.visitnote.req.SfaVisitclientRelReqVo;
import com.biz.crm.nebular.sfa.visitnote.resp.SfaVisitGroupRespVo;
import com.biz.crm.nebular.sfa.visitnote.resp.SfaVisitclientRelRespVo;
import com.biz.crm.util.*;
import com.biz.crm.visitinfo.service.impl.VisitPlanInfoHelper;
import com.biz.crm.visitnote.mapper.SfaVisitGroupMapper;
import com.biz.crm.visitnote.mapper.SfaVisitPlanMapper;
import com.biz.crm.visitnote.mapper.SfaVisitclientRelMapper;
import com.biz.crm.visitnote.model.SfaVisitGroupEntity;
import com.biz.crm.visitnote.model.SfaVisitclientRelEntity;
import com.biz.crm.visitnote.service.ISfaVisitGroupService;
import com.biz.crm.visitnote.service.ISfaVisitclientRelService;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.time.LocalDate;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * 线路组管理 接口实现
 *
 * @author liuhongming
 * @date 2020-09-18 16:53:00
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "SfaVisitGroupServiceExpandImpl")
@Transactional
public class SfaVisitGroupServiceImpl extends ServiceImpl<SfaVisitGroupMapper, SfaVisitGroupEntity> implements ISfaVisitGroupService {

    @Resource
    private SfaVisitGroupMapper sfaVisitGroupMapper;
    @Resource
    private SfaVisitclientRelMapper sfaVisitclientRelMapper;
    @Resource
    private ISfaVisitclientRelService iSfaVisitclientRelService;
    @Resource
    private SfaVisitPlanMapper sfaVisitPlanMapper;
    @Autowired
    private CrmLogSendUtil crmLogSendUtil;

    @Resource
    private VisitPlanInfoHelper visitPlanInfoHelper;

    /**
     * 列表
     *
     * @param reqVo
     * @return
     */
    @Override
    public PageResult<SfaVisitGroupRespVo> findList(SfaVisitGroupReqVo reqVo) {
        Page<SfaVisitGroupRespVo> page = new Page<>(reqVo.getPageNum(), reqVo.getPageSize());
        List<SfaVisitGroupRespVo> list = sfaVisitGroupMapper.findList(page, reqVo);
        List<String> groupCodes = list.stream().map(SfaVisitGroupRespVo::getGroupCode).collect(Collectors.toList());
        SfaVisitclientRelReqVo sfaVisitclientRelReqVo = new SfaVisitclientRelReqVo() {{
            this.setGroupCodes(groupCodes);
            this.setPageSize(-1);
        }};
        List<SfaVisitclientRelRespVo> sfaVisitclientRelRespVos = iSfaVisitclientRelService.findList(sfaVisitclientRelReqVo).getData();
        Map<String, List<SfaVisitclientRelRespVo>> map = sfaVisitclientRelRespVos.stream().collect(Collectors.groupingBy(SfaVisitclientRelRespVo::getGroupCode));
        list.forEach(data -> {
            data.setEnableStatusName(CrmEnableStatusEnum.getDesc(data.getEnableStatus()));
            List<SfaVisitclientRelRespVo> clientRelRespVoList = map.get(data.getGroupCode());
            data.setClientRelNum(clientRelRespVoList.size());
            StringBuffer clientNames = new StringBuffer();
            for (SfaVisitclientRelRespVo clientRel : clientRelRespVoList) {
                if (clientNames.length() > 0) {
                    clientNames.append(",");
                }
                clientNames.append(clientRel.getClientName());
            }
            data.setClientNames(clientNames.toString());
        });
        this.visitPlanInfoHelper.loadUserDataForVisitGroup(list);
        return PageResult.<SfaVisitGroupRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    /**
     * 通过id查询详情
     *
     * @param id
     * @return
     */
    @Override
    public SfaVisitGroupRespVo queryDetailById(String id) {
        AssertUtils.isNotEmpty(id, "数据主键不能为空");
        SfaVisitGroupReqVo reqVo = new SfaVisitGroupReqVo() {{
            this.setId(id);
        }};
        List<SfaVisitGroupRespVo> list = this.findList(reqVo).getData();
        if (CollectionUtil.listEmpty(list)) {
            return new SfaVisitGroupRespVo();
        }
        SfaVisitGroupRespVo respVo = list.get(0);
        //查询对应客户信息
        SfaVisitclientRelReqVo sfaVisitclientRelReqVo = new SfaVisitclientRelReqVo();
        sfaVisitclientRelReqVo.setGroupCode(respVo.getGroupCode());
        sfaVisitclientRelReqVo.setPageSize(-1);
        List<SfaVisitclientRelRespVo> visitclientRelRespVos = iSfaVisitclientRelService.findList(sfaVisitclientRelReqVo).getData();
        respVo.setSfaVisitclientRelRespVos(visitclientRelRespVos);
        return respVo;
    }

    /**
     * 查询
     *
     * @param reqVo
     * @return sfaVisitGroupRespVo
     */
    @Override
    public SfaVisitGroupRespVo query(SfaVisitGroupReqVo reqVo) {
        if (null == reqVo || (StringUtils.isEmpty(reqVo.getId()) && CollectionUtil.listEmpty(reqVo.getIds()))) {
            return new SfaVisitGroupRespVo();
        }
        List<SfaVisitGroupRespVo> list = this.findList(reqVo).getData();
        if (CollectionUtil.listEmpty(list)) {
            return new SfaVisitGroupRespVo();
        }
        SfaVisitGroupRespVo respVo = list.get(0);
        //查询对应客户信息
        SfaVisitclientRelReqVo sfaVisitclientRelReqVo = new SfaVisitclientRelReqVo();
        sfaVisitclientRelReqVo.setGroupCode(respVo.getGroupCode());
        sfaVisitclientRelReqVo.setPageSize(-1);
        List<SfaVisitclientRelRespVo> visitclientRelRespVos = sfaVisitclientRelMapper.findSfaVisitclientRelList(sfaVisitclientRelReqVo);
        respVo.setSfaVisitclientRelRespVos(visitclientRelRespVos);
        return respVo;
    }

    /**
     * 新增
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional
    public void save(SfaVisitGroupReqVo reqVo) {
        UserRedis userRedis = UserUtils.getUser();
        if (userRedis.getUsertype().equals(UserTypeEnum.CUSTOMER_EMPLOYEE.getCode())) {
            userRedis.setPoscode(userRedis.getUsername());
            userRedis.setPosname(userRedis.getRealname());
        }
        if (StringUtils.isEmpty(reqVo.getUserName())) {
            reqVo.setUserName(userRedis.getUsername());
            reqVo.setFullName(userRedis.getRealname());
            reqVo.setOrgCode(userRedis.getOrgcode());
            reqVo.setOrgName(userRedis.getOrgname());
            reqVo.setPosCode(userRedis.getPoscode());
            reqVo.setPosName(userRedis.getPosname());
        }


        this.saveCheck(reqVo);
        SfaVisitGroupEntity entity = CrmBeanUtil.copy(reqVo, SfaVisitGroupEntity.class);
        String code = CodeUtil.getCodeDefault();
        entity.setGroupCode(code);
        entity.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
        this.save(entity);
        this.saveDate(reqVo, entity);
        //添加日志
        Object menuCodeObj = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        if (org.springframework.util.StringUtils.isEmpty(menuCodeObj)){
            throw new BusinessException("缺失菜单编码,无法记录日志");
        }
        crmLogSendUtil.sendForAdd(menuCodeObj.toString(), entity.getId(), reqVo.getUserName(), reqVo);
    }


    public void saveCheck(SfaVisitGroupReqVo reqVo) {
        AssertUtils.isNotEmpty(reqVo.getUserName(), "请选择人员姓名");
        AssertUtils.isNotEmpty(reqVo.getFullName(), "请选择人员姓名");
        AssertUtils.isNotEmpty(reqVo.getGroupName(), "请填写线路组名称");
        AssertUtils.isNotEmpty(reqVo.getOrgCode(), "选择组织编码为空");
        AssertUtils.isNotEmpty(reqVo.getOrgName(), "选择组织名称为空");
        AssertUtils.isNotEmpty(reqVo.getPosCode(), "选择职位编码为空");
        AssertUtils.isNotEmpty(reqVo.getPosName(), "选择职位名称为空");
        //查询当前账号是否存在相同的线路组名称
        SfaVisitGroupEntity visitGroupEntity = this.lambdaQuery()
                .eq(SfaVisitGroupEntity::getUserName, reqVo.getUserName())
                .eq(SfaVisitGroupEntity::getGroupName, reqVo.getGroupName()).one();
        if (visitGroupEntity != null) {
            throw new BusinessException(reqVo.getFullName() + "存在相同的线路组名称[" + visitGroupEntity.getGroupCode() + "]");
        }
        List<SfaVisitclientRelReqVo> sfaVisitclientRelReqVos = reqVo.getSfaVisitclientRelReqVos();
        if (CollectionUtil.listEmpty(sfaVisitclientRelReqVos)) {
            throw new BusinessException("请选择对应客户信息");
        } else {
            List<String> customerCodes = Lists.newArrayList(), terminalCodes = Lists.newArrayList();
            for (SfaVisitclientRelReqVo sfaVisitclientRelReqVo : sfaVisitclientRelReqVos) {
                AssertUtils.isNotEmpty(sfaVisitclientRelReqVo.getClientType(), "客户类型为空");
                AssertUtils.isNotEmpty(sfaVisitclientRelReqVo.getClientName(), "客户名称为空");
                AssertUtils.isNotEmpty(sfaVisitclientRelReqVo.getClientCode(), "客户编码为空");
                if (SfaVisitEnum.ClientType.TERMINAL.getVal().equals(sfaVisitclientRelReqVo.getClientType())) {
                    terminalCodes.add(sfaVisitclientRelReqVo.getClientCode());
                } else {
                    customerCodes.add(sfaVisitclientRelReqVo.getClientCode());
                }
            }
            Map<String, SfaClientData> clientDataMap = SfaClientHelper.loadClientDataMap(customerCodes, terminalCodes);
            sfaVisitclientRelReqVos.stream().forEach(v -> {
                SfaClientData clientData = clientDataMap.get(v.getClientType() + v.getClientCode());
                if (null == clientData) {
                    throw new BusinessException("不存在的客户数据[" + v.getClientCode() + "]");
                }
                v.setClientSubclass(clientData.getClientSubclass());
                v.setClientSubclassName(clientData.getClientSubclassName());
            });
        }
    }

    public void saveDate(SfaVisitGroupReqVo reqVo, SfaVisitGroupEntity entity) {
        if (CollectionUtil.listNotEmptyNotSizeZero(reqVo.getSfaVisitclientRelReqVos())) {
            List<SfaVisitclientRelEntity> sfaVisitclientRelEntities = reqVo.getSfaVisitclientRelReqVos().stream().map(o -> {
                if (!SfaVisitEnum.ClientType.GETMAP.containsKey(o.getClientType())) {
                    throw new BusinessException("不存在的客户类型[" + o.getClientType() + "]");
                }
                SfaVisitclientRelEntity sfaVisitclientRelEntity = new SfaVisitclientRelEntity();
                CrmBeanUtil.copyProperties(o, sfaVisitclientRelEntity);
                sfaVisitclientRelEntity.setId(null);
                sfaVisitclientRelEntity.setGroupCode(entity.getGroupCode());
                return sfaVisitclientRelEntity;
            }).collect(Collectors.toList());
            iSfaVisitclientRelService.saveBatch(sfaVisitclientRelEntities);
        }
    }

    /**
     * 更新
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(SfaVisitGroupReqVo reqVo) {
        if (StringUtils.isEmpty(reqVo.getId())) {
            throw new BusinessException("数据主键不能为空");
        }
        SfaVisitGroupEntity entity = this.getById(reqVo.getId());
        if (Objects.isNull(entity)) {
            throw new BusinessException("线路组数据不存在");
        }
        AssertUtils.isNotEmpty(reqVo.getGroupName(), "请填写线路组名称");
        entity.setGroupName(reqVo.getGroupName());
        //查询没修改之前的线路组信息
        SfaVisitGroupRespVo oldVisitGroupVo = this.queryDetailById(reqVo.getId());
        this.updateById(entity);
        if (CollectionUtil.listEmpty(reqVo.getSfaVisitclientRelReqVos())) {
            throw new BusinessException("请选择对应客户信息");
        }
        reqVo.setGroupCode(entity.getGroupCode());
        this.deleteBatchDate(reqVo);
        this.saveDate(reqVo, entity);
        //查询修改后的线路组信息
        SfaVisitGroupRespVo newVisitGroupVo = this.queryDetailById(reqVo.getId());
        Object menuCodeObj = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        crmLogSendUtil.sendForUpdate(menuCodeObj.toString(),
                entity.getId(), entity.getUserName(), oldVisitGroupVo, newVisitGroupVo);
    }

    /**
     * 删除
     *
     * @param ids
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteBatch(List<String> ids) {
        if (!CollectionUtil.listNotEmptyNotSizeZero(ids)) {
            throw new BusinessException("数据主键不能为空");
        }
        SfaVisitGroupReqVo reqVo = new SfaVisitGroupReqVo() {{
            this.setIds(ids);
        }};
        reqVo.setPageSize(-1);
        List<SfaVisitGroupRespVo> list = this.findList(reqVo).getData();
        if (CollectionUtil.listEmpty(list)) {
            throw new BusinessException("数据不存在");
        }
        List<String> idList = list.stream().map(SfaVisitGroupRespVo::getId).collect(Collectors.toList());
        SfaVisitPlanReqVo sfaVisitPlanReqVo = new SfaVisitPlanReqVo() {{
            this.setVisitDate(LocalDate.now().format(CrmDateUtils.yyyyMMdd));
            this.setVisitGroupIdList(idList);
        }};
        //查询是否存在于周期性拜访计划中
        Integer count = sfaVisitPlanMapper.findSfaVisitPlanGroupCount(sfaVisitPlanReqVo);
        if (count > 0) {
            throw new BusinessException("删除线路组已存在周期性拜访计划");
        }
        sfaVisitGroupMapper.deleteProductsByParams(reqVo);
        List<String> groupCodes = list.stream().map(SfaVisitGroupRespVo::getGroupCode).collect(Collectors.toList());
        reqVo.setGroupCodes(groupCodes);
        this.deleteBatchDate(reqVo);
    }

    public void deleteBatchDate(SfaVisitGroupReqVo reqVo) {
        if (StringUtils.isEmpty(reqVo.getGroupCode()) && CollectionUtil.listEmpty(reqVo.getGroupCodes())) {
            throw new BusinessException("数据主键不能为空");
        }
        SfaVisitclientRelReqVo sfaVisitclientRelReqVo = new SfaVisitclientRelReqVo();
        sfaVisitclientRelReqVo.setGroupCode(reqVo.getGroupCode());
        sfaVisitclientRelReqVo.setGroupCodes(reqVo.getGroupCodes());
        sfaVisitclientRelMapper.deleteProductsByParams(sfaVisitclientRelReqVo);
    }

    /**
     * 启用
     *
     * @param ids
     * @return
     */
    @Override
    public void enableBatch(List<String> ids) {
        if (!CollectionUtil.listNotEmptyNotSizeZero(ids)) {
            throw new BusinessException("数据主键不能为空");
        }
        //设置状态为启用
        List<SfaVisitGroupEntity> sfaVisitGroupEntities = sfaVisitGroupMapper.selectBatchIds(ids);
        if (CollectionUtils.isNotEmpty(sfaVisitGroupEntities)) {
            sfaVisitGroupEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
            });
        }
        this.updateBatchById(sfaVisitGroupEntities);
    }

    /**
     * 禁用
     *
     * @param ids
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void disableBatch(List<String> ids) {
        if (!CollectionUtil.listNotEmptyNotSizeZero(ids)) {
            throw new BusinessException("数据主键不能为空");
        }
        //设置状态为禁用
        List<SfaVisitGroupEntity> sfaVisitGroupEntities = sfaVisitGroupMapper.selectBatchIds(ids);

        List<String> idList = sfaVisitGroupEntities.stream().map(SfaVisitGroupEntity::getId).collect(Collectors.toList());
        SfaVisitPlanReqVo sfaVisitPlanReqVo = new SfaVisitPlanReqVo() {{
            this.setVisitDate(LocalDate.now().format(CrmDateUtils.yyyyMMdd));
            this.setVisitGroupIdList(idList);
        }};
        //查询是否存在于周期性拜访计划中
        Integer count = sfaVisitPlanMapper.findSfaVisitPlanGroupCount(sfaVisitPlanReqVo);
        if (count > 0) {
            throw new BusinessException("禁用线路组已存在周期性拜访计划");
        }
        if (CollectionUtils.isNotEmpty(sfaVisitGroupEntities)) {
            sfaVisitGroupEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.DISABLE.getCode());
            });
        }
        this.updateBatchById(sfaVisitGroupEntities);
    }


}
