package com.biz.crm.visitnote.service.impl;

import cn.hutool.json.JSONUtil;
import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.base.BusinessException;
import com.biz.crm.base.SfaClientData;
import com.biz.crm.base.SfaClientHelper;
import com.biz.crm.base.config.ThreadLocalUtil;
import com.biz.crm.common.GlobalParam;
import com.biz.crm.common.PageResult;
import com.biz.crm.common.param.RedisParam;
import com.biz.crm.crmlog.handle.util.CrmLogSendUtil;
import com.biz.crm.enums.SfaVisitRoleEnum;
import com.biz.crm.eunm.CrmDelFlagEnum;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.eunm.sfa.IntegralEnum;
import com.biz.crm.eunm.sfa.SfaVisitEnum;
import com.biz.crm.integral.controller.SfaIntegralSaveUtil;
import com.biz.crm.integral.req.SfaIntegralDetailAddReq;
import com.biz.crm.nebular.mdm.constant.UserTypeEnum;
import com.biz.crm.nebular.sfa.visitnote.req.SfaVisitPlanRangeReqVo;
import com.biz.crm.nebular.sfa.visitnote.req.SfaVisitPlanReqVo;
import com.biz.crm.nebular.sfa.visitnote.resp.SfaVisitPlanRangeRespVo;
import com.biz.crm.nebular.sfa.visitnote.resp.SfaVisitPlanRespVo;
import com.biz.crm.service.RedisService;
import com.biz.crm.util.*;
import com.biz.crm.visitinfo.mapper.SfaVisitPlanInfoMapper;
import com.biz.crm.visitinfo.model.SfaVisitPlanInfoEntity;
import com.biz.crm.visitinfo.service.ISfaVisitPlanInfoService;
import com.biz.crm.visitinfo.service.impl.SfaVisitPlanInfoServiceEsImpl;
import com.biz.crm.visitinfo.service.impl.VisitPlanInfoHelper;
import com.biz.crm.visitnote.mapper.SfaVisitGroupMapper;
import com.biz.crm.visitnote.mapper.SfaVisitPlanMapper;
import com.biz.crm.visitnote.mapper.SfaVisitPlanRangeMapper;
import com.biz.crm.visitnote.model.SfaVisitPlanEntity;
import com.biz.crm.visitnote.model.SfaVisitPlanRangeEntity;
import com.biz.crm.visitnote.service.ISfaVisitPlanRangeService;
import com.biz.crm.visitnote.service.ISfaVisitPlanService;
import com.biz.crm.visitnote.service.component.SfaVisitPlanExecuteContext;
import com.biz.crm.visitnote.service.component.resolver.SfaVisitPlanResolver;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.time.LocalDate;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

/**
 * 拜访计划制定 接口实现
 *
 * @author liuhongming
 * @date 2020-09-21 14:35:05
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "sfaVisitPlanServiceExpandImpl")
public class SfaVisitPlanServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<SfaVisitPlanMapper, SfaVisitPlanEntity> implements ISfaVisitPlanService {

    @Resource
    private SfaVisitPlanMapper sfaVisitPlanMapper;
    @Resource
    private SfaVisitPlanRangeMapper sfaVisitPlanRangeMapper;

    @Resource
    private ISfaVisitPlanRangeService iSfaVisitPlanRangeService;
    @Resource
    private RedisService redisService;


    @Resource
    private ISfaVisitPlanInfoService iSfaVisitPlanInfoService;
    @Resource
    private SfaVisitGroupMapper sfaVisitGroupMapper;

    @Resource
    private SfaVisitPlanInfoMapper sfaVisitPlanInfoMapper;
    @Resource
    private Map<String, SfaVisitPlanResolver> sfaVisitPlanResolverMap;
    @Resource
    private SfaVisitPlanInfoServiceEsImpl sfaVisitPlanInfoServiceEsImpl;

    @Resource
    private VisitPlanInfoHelper visitPlanInfoHelper;
    @Autowired
    private CrmLogSendUtil crmLogSendUtil;


    /**
     * 列表
     *
     * @param reqVo
     * @return
     */
    @Override
    public PageResult<SfaVisitPlanRespVo> findList(SfaVisitPlanReqVo reqVo) {
        Page<SfaVisitPlanRespVo> page = new Page<>(reqVo.getPageNum(), reqVo.getPageSize());
        List<SfaVisitPlanRespVo> list = sfaVisitPlanMapper.findList(page, reqVo);
        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            this.visitPlanInfoHelper.loadUserDataForVisitPlanVo(list);
            List<String> visitPlanCodes = list.stream().map(SfaVisitPlanRespVo::getVisitPlanCode).collect(Collectors.toList());
            SfaVisitPlanRangeReqVo sfaVisitPlanRangeReqVo = new SfaVisitPlanRangeReqVo();
            sfaVisitPlanRangeReqVo.setVisitPlanCodes(visitPlanCodes);
            sfaVisitPlanRangeReqVo.setPageSize(-1);
            List<SfaVisitPlanRangeRespVo> sfaVisitPlanRangeRespVos = iSfaVisitPlanRangeService.findList(sfaVisitPlanRangeReqVo).getData();
            Map<String, List<SfaVisitPlanRangeRespVo>> readRespMap = sfaVisitPlanRangeRespVos.stream()
                    .collect(Collectors.groupingBy(SfaVisitPlanRangeRespVo::getVisitPlanCode));

            for (SfaVisitPlanRespVo sfaVisitPlanRespVo : list) {

                sfaVisitPlanRespVo.setEnableStatusName(CrmEnableStatusEnum.getDesc(sfaVisitPlanRespVo.getEnableStatus()));
                List<SfaVisitPlanRangeRespVo> visitPlanRangeRespVos = readRespMap.get(sfaVisitPlanRespVo.getVisitPlanCode());
                if (visitPlanRangeRespVos != null && visitPlanRangeRespVos.size() > 0) {
                    StringBuffer groupNameBuffer = new StringBuffer();
                    StringBuffer groupIdBuffer = new StringBuffer();
                    for (SfaVisitPlanRangeRespVo visitPlanRangeRespVo : visitPlanRangeRespVos) {
                        if (groupNameBuffer.length() > 0) {
                            groupNameBuffer.append(",");
                        }
                        if (StringUtils.isNotEmpty(visitPlanRangeRespVo.getVisitGroupName())) {
                            groupNameBuffer.append(visitPlanRangeRespVo.getVisitGroupName());
                        }
                        if (groupIdBuffer.length() > 0) {
                            groupIdBuffer.append(",");
                        }
                        if (StringUtils.isNotEmpty(visitPlanRangeRespVo.getVisitGroupId())) {
                            groupIdBuffer.append(visitPlanRangeRespVo.getVisitGroupId());
                        }
                    }
                    sfaVisitPlanRespVo.setVisitGroupIds(groupIdBuffer.toString());
                    sfaVisitPlanRespVo.setVisitGroupNames(groupNameBuffer.toString());
                }
            }
        }
        return PageResult.<SfaVisitPlanRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    @Override
    public SfaVisitPlanRespVo queryDetailById(String id) {
        AssertUtils.isNotEmpty(id, "数据主键不能为空");
        SfaVisitPlanReqVo reqVo = new SfaVisitPlanReqVo() {{
            this.setId(id);
            this.setPlanType(SfaVisitEnum.visitType.PLAN_VISIT.getVal());
        }};
        List<SfaVisitPlanRespVo> list = this.findList(reqVo).getData();
        if (CollectionUtil.listEmpty(list)) {
            return new SfaVisitPlanRespVo();
        }
        SfaVisitPlanRespVo respVo = list.get(0);
        //查询对应客户信息
        SfaVisitPlanRangeReqVo sfaVisitPlanRangeReqVo = new SfaVisitPlanRangeReqVo();
        sfaVisitPlanRangeReqVo.setVisitPlanCode(respVo.getVisitPlanCode());
        List<SfaVisitPlanRangeRespVo> sfaVisitPlanRangeRespVos = iSfaVisitPlanRangeService.findList(sfaVisitPlanRangeReqVo).getData();
        sfaVisitPlanRangeRespVos.forEach(data -> {
            //判断为线路组
            if (respVo.getRouteType().equals(SfaVisitEnum.routeTypeEnum.R1.getVal())) {
                data.setClientRelNum(sfaVisitGroupMapper.getCountByVisitGroupId(data.getVisitGroupId()));
            }
        });
        respVo.setSfaVisitPlanRangeRespVos(sfaVisitPlanRangeRespVos);
        return respVo;
    }


    /**
     * 查询
     *
     * @param reqVo
     * @return sfaVisitPlanRespVo
     */
    @Override
    public SfaVisitPlanRespVo query(SfaVisitPlanReqVo reqVo) {
        if (null == reqVo || (StringUtils.isEmpty(reqVo.getId()) && CollectionUtil.listEmpty(reqVo.getIds()))) {
            return new SfaVisitPlanRespVo();
        }
        List<SfaVisitPlanRespVo> list = this.findList(reqVo).getData();
        if (CollectionUtil.listEmpty(list)) {
            return new SfaVisitPlanRespVo();
        }
        SfaVisitPlanRespVo respVo = list.get(0);
        //查询对应客户信息
        SfaVisitPlanRangeReqVo sfaVisitPlanRangeReqVo = new SfaVisitPlanRangeReqVo();
        sfaVisitPlanRangeReqVo.setVisitPlanCode(respVo.getVisitPlanCode());
        List<SfaVisitPlanRangeRespVo> sfaVisitPlanRangeRespVos = iSfaVisitPlanRangeService.findList(sfaVisitPlanRangeReqVo).getData();
        sfaVisitPlanRangeRespVos.forEach(data -> {
            if (StringUtils.isNotEmpty(data.getClientType())) {
                data.setClientTypeName(SfaVisitRoleEnum.getSfaVisitRole(data.getClientType()).getDesc());
            }
            //判断为线路组
            if (respVo.getRouteType().equals(SfaVisitEnum.routeTypeEnum.R1.getVal())) {
                data.setClientRelNum(sfaVisitGroupMapper.getCountByVisitGroupId(data.getVisitGroupId()));
            }
        });
        respVo.setSfaVisitPlanRangeRespVos(sfaVisitPlanRangeRespVos);
        return respVo;
    }

    protected void saveCheck(SfaVisitPlanReqVo reqVo) {
        AssertUtils.isNotEmpty(reqVo.getPlanType(), "拜访类型不能为空");
        if (!SfaVisitEnum.visitType.GETMAP.containsKey(reqVo.getPlanType())) {
            throw new BusinessException("拜访类型有误,不存在该拜访类型(临时拜访、计划拜访、协访)");
        }
        AssertUtils.isNotEmpty(reqVo.getRouteType(), "维度类型不能为空(线路组、网点、频率)");
        if (!SfaVisitEnum.routeTypeEnum.GETMAP.containsKey(reqVo.getRouteType())) {
            throw new BusinessException("维度类型有误,不存在该维度(线路组、网点、频率)");
        }
        AssertUtils.isNotEmpty(reqVo.getVisitUserName(), "人员编码不能为空");
        AssertUtils.isNotEmpty(reqVo.getVisitRealName(), "请选择人员姓名");
        AssertUtils.isNotEmpty(reqVo.getVisitPosCode(), "人员职位编码不能为空");
        AssertUtils.isNotEmpty(reqVo.getVisitPosName(), "人员职位名称不能为空");
        AssertUtils.isNotEmpty(reqVo.getVisitOrgCode(), "人员所属组织编码不能为空");
        AssertUtils.isNotEmpty(reqVo.getVisitOrgName(), "人员所属组织名称不能为空");
        AssertUtils.isNotEmpty(reqVo.getVisitStartDate(), "循环开始日期不能为空");
        LocalDate beginDate = null;
        LocalDate now = LocalDate.now();
        try {
            beginDate = LocalDate.parse(reqVo.getVisitStartDate(), CrmDateUtils.yyyyMMdd);
        } catch (Exception e) {
            throw new BusinessException("循环开始日期格式错误");
        }
        AssertUtils.isNotEmpty(reqVo.getVisitEndDate(), "循环结算日期不能为空");
        LocalDate endDate = null;
        try {
            endDate = LocalDate.parse(reqVo.getVisitEndDate(), CrmDateUtils.yyyyMMdd);
        } catch (Exception e) {
            throw new BusinessException("循环结算日期格式错误");
        }
        if (beginDate.isBefore(now)) {
            throw new BusinessException("循环开始日期必须大于当前日期");
        }
        if (beginDate.isAfter(endDate)) {
            throw new BusinessException("循环开始日期不能大于循环结算日期");
        }
//        Integer days = Period.between(beginDate,endDate).getDays()+1;
//        //判断为线路组
//        if (reqVo.getRouteType().equals(SfaVisitEnum.routeTypeEnum.R1.getVal())){
//            if (reqVo.getSfaVisitPlanRangeReqVos().size()>days){
//                throw new BusinessException("时间范围区间");
//            }
//        }
        this.checkVisitPlanRange(reqVo, beginDate, endDate);
    }

    protected void checkVisitPlanRange(SfaVisitPlanReqVo reqVo, LocalDate beginDate, LocalDate endDate) {
        if (!CollectionUtil.listNotEmpty(reqVo.getSfaVisitPlanRangeReqVos())) {
            throw new BusinessException("拜访计划制定范围不能为空");
        }
        int i = 1;
        for (SfaVisitPlanRangeReqVo sfaVisitPlanRangeReqVo : reqVo.getSfaVisitPlanRangeReqVos()) {
            if (SfaVisitEnum.routeTypeEnum.R1.getVal().equals(reqVo.getRouteType())) {
                AssertUtils.isNotEmpty(sfaVisitPlanRangeReqVo.getVisitGroupId(), "线路组" + i + ",线路组id不能为空");
                AssertUtils.isNotEmpty(sfaVisitPlanRangeReqVo.getVisitGroupName(), "线路组" + i + ",线路组名称不能为空");


            } else {
                AssertUtils.isNotEmpty(sfaVisitPlanRangeReqVo.getClientId(), "第" + i + "行,客户id不能为空");
                AssertUtils.isNotEmpty(sfaVisitPlanRangeReqVo.getClientCode(), "第" + i + "行,客户编码不能为空");
                AssertUtils.isNotEmpty(sfaVisitPlanRangeReqVo.getClientName(), "第" + i + "行,客户名称不能为空");
                AssertUtils.isNotEmpty(sfaVisitPlanRangeReqVo.getClientType(), "第" + i + "行,客户类型不能为空");
                //判断为频率拜访是必须有首次拜访日期
                if (SfaVisitEnum.routeTypeEnum.R3.getVal().equals(reqVo.getRouteType())) {
                    AssertUtils.isNotEmpty(sfaVisitPlanRangeReqVo.getFirstVisitDate(), "第" + i + "行,首次拜访日期不能为空");
                    LocalDate firstVisitDate = null;
                    try {
                        firstVisitDate = LocalDate.parse(sfaVisitPlanRangeReqVo.getFirstVisitDate(), CrmDateUtils.yyyyMMdd);
                    } catch (Exception e) {
                        throw new BusinessException("第" + i + "行,首次拜访日期格式错误");
                    }
                    if (beginDate.isAfter(firstVisitDate) || firstVisitDate.isAfter(endDate)) {
                        throw new BusinessException("第" + i + "行,首次拜访日不在循环日期内");
                    }
                    if (null == sfaVisitPlanRangeReqVo.getVisitFrequency()) {
                        throw new BusinessException("第" + i + "行,拜访频率不能为空");
                    }
                    try {
                        String regex = "^[1-9]+[0-9]*$";
                        Pattern p = Pattern.compile(regex);
                        Matcher m = p.matcher(sfaVisitPlanRangeReqVo.getVisitFrequency().toString());
                        if (!m.find()) {
                            System.out.println("falsss");
                            throw new BusinessException("第" + i + "行,拜访频率必须是大于0的整数");
                        }
                    } catch (Exception e) {
                        e.printStackTrace();
                        throw new BusinessException("第" + i + "行,拜访频率必须是大于0的整数");
                    }
                    if (sfaVisitPlanRangeReqVo.getVisitFrequency() <= 0) {
                        throw new BusinessException("第" + i + "行,拜访频率必须是大于0的整数");
                    }
                }
            }
            i++;
        }
    }

    /**
     * 新增
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void save(SfaVisitPlanReqVo reqVo) {
        log.info("执行计划:【{}】", JSONUtil.toJsonPrettyStr(reqVo));
        if (StringUtils.isEmpty(reqVo.getVisitUserName())) {
            UserRedis userRedis = UserUtils.getUser();
            if (userRedis.getUsertype().equals(UserTypeEnum.CUSTOMER_EMPLOYEE.getCode())) {
                userRedis.setPoscode(userRedis.getUsername());
                userRedis.setPosname(userRedis.getRealname());
            }
            reqVo.setVisitUserName(userRedis.getUsername());
            reqVo.setVisitRealName(userRedis.getRealname());
            reqVo.setVisitOrgCode(userRedis.getOrgcode());
            reqVo.setVisitOrgName(userRedis.getOrgname());
            reqVo.setVisitPosCode(userRedis.getPoscode());
            reqVo.setVisitPosName(userRedis.getPosname());
        }
        this.saveCheck(reqVo);
        SfaVisitPlanEntity entity = CrmBeanUtil.copy(reqVo, SfaVisitPlanEntity.class);
        String code = CodeUtil.getCodeDefault();
        entity.setVisitPlanCode(code);
        entity.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
        entity.setDelFlag(CrmDelFlagEnum.NORMAL.getCode());
        this.save(entity);
        this.saveData(reqVo, entity.getVisitPlanCode());
        //添加日志
        Object menuCodeObj = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        if (menuCodeObj == null || menuCodeObj.equals("")) {
            throw new BusinessException("菜单编码为空");
        }
        crmLogSendUtil.sendForAdd(menuCodeObj.toString(), entity.getId(), reqVo.getVisitUserName(), reqVo);
        //新增积分
        SfaIntegralDetailAddReq integralDetailAddReq = new SfaIntegralDetailAddReq();
        integralDetailAddReq.setBizId(entity.getId());
        integralDetailAddReq.setIntegralTaskType(IntegralEnum.taskType.PLANING_VISIT_PLAN.getValue());
        integralDetailAddReq.setPosCode(reqVo.getVisitPosCode());
        integralDetailAddReq.setUserName(reqVo.getVisitUserName());
        integralDetailAddReq.setExtParam(null);
        SfaIntegralSaveUtil.saveIntegralDetail(integralDetailAddReq);
    }

    public void saveData(SfaVisitPlanReqVo reqVo, String visitPlanCode) {
        if (CollectionUtil.listNotEmptyNotSizeZero(reqVo.getSfaVisitPlanRangeReqVos())) {
            //判断线路组是否有重复数据
            if (reqVo.getRouteType().equals(SfaVisitEnum.routeTypeEnum.R1.getVal())) {
                Map<String, List<SfaVisitPlanRangeReqVo>> visitPlanRageMap = reqVo.getSfaVisitPlanRangeReqVos().stream().collect(Collectors.groupingBy(SfaVisitPlanRangeReqVo::getVisitGroupId));
                for (Map.Entry<String, List<SfaVisitPlanRangeReqVo>> map : visitPlanRageMap.entrySet()) {
                    if (map.getValue().size() > 1) {
                        throw new BusinessException("线路组添加重复");
                    }
                }
            }
            int order = 0;
            List<SfaVisitPlanRangeEntity> visitPlanRangeEntities=Lists.newArrayList();
            for (SfaVisitPlanRangeReqVo o : reqVo.getSfaVisitPlanRangeReqVos()) {
                //判断不是线路组的时候，新增客户/终端类型是否正确
                if (!reqVo.getRouteType().equals(SfaVisitEnum.routeTypeEnum.R1.getVal())) {
                  if (!SfaVisitEnum.ClientType.GETMAP.containsKey(o.getClientType())) {
                    throw new BusinessException("不存在的客户类型[" + o.getClientType() + "]");
                  }
                }
                SfaVisitPlanRangeEntity receivingEntity = new SfaVisitPlanRangeEntity();
                CrmBeanUtil.copyProperties(o, receivingEntity);
                receivingEntity.setVisitPlanCode(visitPlanCode);
                receivingEntity.setId(null);
                receivingEntity.setIdx(++order);
                visitPlanRangeEntities.add(receivingEntity);
            }
            iSfaVisitPlanRangeService.saveBatch(visitPlanRangeEntities);
        }
    }

    /**
     * 更新
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional
    public void update(SfaVisitPlanReqVo reqVo) {
        SfaVisitPlanEntity old = this.updateCheck(reqVo);
        SfaVisitPlanEntity updateEntity = new SfaVisitPlanEntity();
        updateEntity.setId(reqVo.getId());
        updateEntity.setVisitStartDate(reqVo.getVisitStartDate());
        updateEntity.setVisitEndDate(reqVo.getVisitEndDate());
        updateEntity.setNotWeekSet(reqVo.getNotWeekSet());
        updateEntity.setNotWeekSetDesc(reqVo.getNotWeekSetDesc());
        //查询未修改之前的数据
        SfaVisitPlanRespVo oldObject = queryDetailById(reqVo.getId());
        this.updateById(updateEntity);
        reqVo.setVisitPlanCode(old.getVisitPlanCode());
        this.deleteBatchDate(reqVo);
        this.saveData(reqVo, old.getVisitPlanCode());
        //查询修改后的数据
        SfaVisitPlanRespVo newObject = queryDetailById(reqVo.getId());
        //添加日志
        Object menuCodeObj = ThreadLocalUtil.getObj(GlobalParam.MENU_CODE);
        crmLogSendUtil.sendForUpdate(menuCodeObj.toString(), reqVo.getId(), reqVo.getVisitUserName(), oldObject, newObject);
    }

    protected SfaVisitPlanEntity updateCheck(SfaVisitPlanReqVo reqVo) {
        if (StringUtils.isEmpty(reqVo.getId())) {
            throw new BusinessException("数据主键不能为空");
        }
        SfaVisitPlanEntity entity = this.getById(reqVo.getId());
        if (Objects.isNull(entity)) {
            throw new BusinessException("数据不存在");
        }
        String visitEndDate = entity.getVisitEndDate();
        String visitStartDate = entity.getVisitStartDate();
        String visitEndDateUpdate = reqVo.getVisitEndDate();
        LocalDate beginDate, endDate, endDateUpdate;
        try {
            beginDate = LocalDate.parse(visitStartDate, CrmDateUtils.yyyyMMdd);
            endDate = LocalDate.parse(visitEndDate, CrmDateUtils.yyyyMMdd);
            endDateUpdate = LocalDate.parse(visitEndDateUpdate, CrmDateUtils.yyyyMMdd);
        } catch (Exception e) {
            throw new BusinessException("循环日期格式错误");
        }
        if (endDate.compareTo(LocalDate.now()) <= 0) {
            throw new BusinessException("当前日期已大于或等于原计划结束日期,不允许编辑");
        }
        if (endDateUpdate.compareTo(LocalDate.now()) < 0) {
            throw new BusinessException("修改后的计划结束日期不能小于当前日期");
        }

        this.checkVisitPlanRange(reqVo, beginDate, endDateUpdate);
//        this.saveCheck(reqVo);
        return entity;
    }

    /**
     * 删除
     *
     * @param ids
     * @return
     */
    @Override
    public void deleteBatch(List<String> ids) {
        if (!CollectionUtil.listNotEmptyNotSizeZero(ids)) {
            throw new BusinessException("数据主键不能为空");
        }
        SfaVisitPlanReqVo reqVo = new SfaVisitPlanReqVo() {{
            this.setIds(ids);
            this.setPageSize(-1);
            this.setPlanType(SfaVisitEnum.visitType.PLAN_VISIT.getVal());
        }};
        List<SfaVisitPlanRespVo> list = this.findList(reqVo).getData();
        if (CollectionUtil.listEmpty(list)) {
            throw new BusinessException("数据不存在");
        }
        for (SfaVisitPlanRespVo data : list) {
            if (data.getEnableStatus().equals(CrmEnableStatusEnum.ENABLE.getCode())) {
                throw new BusinessException("启用状态的拜访计划规则不能删除");
            }
            LocalDate startDate = null;
            LocalDate endDate = null;
            LocalDate now = LocalDate.now();
            LocalDate createDate = LocalDate.parse(data.getCreateDate(), CrmDateUtils.yyyyMMdd);
            try {
                startDate = LocalDate.parse(data.getVisitStartDate(), CrmDateUtils.yyyyMMdd);
                endDate = LocalDate.parse(data.getVisitEndDate(), CrmDateUtils.yyyyMMdd);
            } catch (Exception e) {
                throw new BusinessException("删除前循环日期格式错误");
            }
            if (startDate.isBefore(now) && endDate.isAfter(now)) {
                throw new BusinessException("已有历史拜访记录,不可删除");
            } else if (now.isAfter(endDate)) {
                throw new BusinessException("已完成拜访计划不可删除");
            } else if (!createDate.isEqual(now)) {
                throw new BusinessException("不可删除非当天创建的拜访计划规则");
            }
        }
        sfaVisitPlanMapper.deleteProductsByParams(reqVo);
        List<String> visitPlanCodes = list.stream().map(SfaVisitPlanRespVo::getVisitPlanCode).collect(Collectors.toList());
        reqVo.setVisitPlanCodes(visitPlanCodes);
        this.deleteBatchDate(reqVo);
        this.deleteSfaVisitPlanInfo(visitPlanCodes);
    }

    public void deleteBatchDate(SfaVisitPlanReqVo reqVo) {
        if (StringUtils.isEmpty(reqVo.getVisitPlanCode()) && CollectionUtil.listEmpty(reqVo.getVisitPlanCodes())) {
            throw new BusinessException("数据主键不能为空");
        }
        SfaVisitPlanRangeReqVo visitPlanRangeReqVo = new SfaVisitPlanRangeReqVo();
        visitPlanRangeReqVo.setVisitPlanCode(reqVo.getVisitPlanCode());
        visitPlanRangeReqVo.setVisitPlanCodes(reqVo.getVisitPlanCodes());
        sfaVisitPlanRangeMapper.deleteProductsByParams(visitPlanRangeReqVo);
    }

    /**
     * 删除拜访计划明细
     *
     * @param visitPlanCodes
     */
    public void deleteSfaVisitPlanInfo(List<String> visitPlanCodes) {
        String now = LocalDate.now().format(CrmDateUtils.yyyyMMdd);
        sfaVisitPlanInfoMapper.deleteVisitPlanInfo(visitPlanCodes, now);
    }

    /**
     * 启用
     *
     * @param ids
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void enableBatch(List<String> ids) {
        if (!CollectionUtil.listNotEmptyNotSizeZero(ids)) {
            throw new BusinessException("数据主键不能为空");
        }
        //设置状态为启用
        List<SfaVisitPlanEntity> sfaVisitPlanEntities = sfaVisitPlanMapper.selectBatchIds(ids);
        if (CollectionUtils.isNotEmpty(sfaVisitPlanEntities)) {
            sfaVisitPlanEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
            });
        }
        this.updateBatchById(sfaVisitPlanEntities);
    }

    /**
     * 禁用
     *
     * @param ids
     * @return
     */
    @Override
    public void disableBatch(List<String> ids) {
        //设置状态为禁用
        List<SfaVisitPlanEntity> sfaVisitPlanEntities = sfaVisitPlanMapper.selectBatchIds(ids);
        if (CollectionUtils.isNotEmpty(sfaVisitPlanEntities)) {
            sfaVisitPlanEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.DISABLE.getCode());
            });
            List<String> visitPlanCodeList = sfaVisitPlanEntities.stream().map(SfaVisitPlanEntity::getVisitPlanCode).collect(Collectors.toList());
            this.deleteSfaVisitPlanInfo(visitPlanCodeList);
            this.updateBatchById(sfaVisitPlanEntities);
        }
    }


    /**
     * 初始化拜访计划解析器上下文
     *
     * @param visitPlanEntities
     * @return
     */
    private SfaVisitPlanExecuteContext initVisitPlanResolverContext(List<SfaVisitPlanEntity> visitPlanEntities) {
        //设置拜访计划人员的账号信息
        this.visitPlanInfoHelper.loadUserDataForVisitPlanEntity(visitPlanEntities);
        SfaVisitPlanExecuteContext context = new SfaVisitPlanExecuteContext();
        Map<String, List<SfaVisitPlanEntity>> userPosMapPlan = visitPlanEntities.stream().collect(Collectors.groupingBy(v -> {
            //按用户名 + 用户职位分组收集，以便后面在解析器里集中处理
            return new StringJoiner(RedisParam.DELIMITER).add(v.getVisitUserName()).add(v.getVisitPosCode()).toString();
        }));
        context.setUserPosMapPlan(userPosMapPlan);
        List<SfaVisitPlanRangeEntity> visitPlanRangeEntities = iSfaVisitPlanRangeService.list(Wrappers.lambdaQuery(SfaVisitPlanRangeEntity.class)
                .in(SfaVisitPlanRangeEntity::getVisitPlanCode, visitPlanEntities.stream().map(SfaVisitPlanEntity::getVisitPlanCode).collect(Collectors.toList())).orderByAsc(SfaVisitPlanRangeEntity::getIdx));

        Map<String, List<SfaVisitPlanRangeEntity>> planCodeMapPlanRange = visitPlanRangeEntities.stream()
                .collect(Collectors.groupingBy(SfaVisitPlanRangeEntity::getVisitPlanCode));
        //需要执行的计划范围
        context.setPlanCodeMapPlanRange(planCodeMapPlanRange);
        return context;
    }

    /**
     * WEB后台人工触发解析拜访计划
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void resolveVisitPlanForWeb(List<String> visitPlanIds) {
        if (org.springframework.util.CollectionUtils.isEmpty(visitPlanIds)) {
            throw new BusinessException("请选择需要解析的拜访计划");
        }
        String now = LocalDate.now().format(CrmDateUtils.yyyyMMdd);
        //能够解析的未来日期
        String future = LocalDate.now().plusDays(SfaVisitPlanResolver.futureDays - 1).format(CrmDateUtils.yyyyMMdd);
        //查询勾选的计划
        List<SfaVisitPlanEntity> visitPlanEntities = this.sfaVisitPlanMapper.selectList(Wrappers.lambdaQuery(SfaVisitPlanEntity.class)
                .in(SfaVisitPlanEntity::getId, visitPlanIds)
                .eq(SfaVisitPlanEntity::getEnableStatus, CrmEnableStatusEnum.ENABLE.getCode())
                .eq(SfaVisitPlanEntity::getDelFlag, CrmDelFlagEnum.NORMAL.getCode())
        );
        if (visitPlanEntities.size() != visitPlanIds.size()) {
            throw new BusinessException("选择了不存在或已停用的拜访计划");
        }
        //查询勾选计划的职位列表
        List<String> visitPosCode = visitPlanEntities.stream().map(SfaVisitPlanEntity::getVisitPosCode).collect(Collectors.toList());
        //根据职位查询从现在到能够解析的未来日期范围内覆盖的所有拜访计划
        visitPlanEntities = this.sfaVisitPlanMapper.findListForJob(visitPosCode, now, future);

        if (visitPlanEntities.size() == 0) {
            return;
        }
        //初始化拜访计划解析器上下文
        SfaVisitPlanExecuteContext context = this.initVisitPlanResolverContext(visitPlanEntities);
        if (!context.isDataReady()) {
            throw new BusinessException(context.getDataReadyMsg());
        }
        //解析
        this.doResolveVisitPlan(context);
    }

    /**
     * 定时任务解析拜访计划
     */
    @Override
    @Transactional
    public void resolveVisitPlanForJob() {
        log.warn("#########################拜访计划解析开始#########################");

        String now = LocalDate.now().format(CrmDateUtils.yyyyMMdd);
        String future = LocalDate.now().plusDays(SfaVisitPlanResolver.futureDays - 1).format(CrmDateUtils.yyyyMMdd);

        List<SfaVisitPlanEntity> visitPlanEntities = this.sfaVisitPlanMapper.findListForJob(null, now, future);

        //初始化拜访计划解析器上下文
        SfaVisitPlanExecuteContext context = this.initVisitPlanResolverContext(visitPlanEntities);
        if (!context.isDataReady()) {
            throw new BusinessException(context.getDataReadyMsg());
        }
        //解析
        this.doResolveVisitPlan(context);
        log.warn("#########################拜访计划解析结束#########################");
    }

    /**
     * 解析拜访计划
     *
     * @param context
     */
    private void doResolveVisitPlan(SfaVisitPlanExecuteContext context) {
        Map<String, List<SfaVisitPlanEntity>> userPosMapPlan = context.getUserPosMapPlan();
        //需要执行的拜访计划
        for (Map.Entry<String, List<SfaVisitPlanEntity>> entry : userPosMapPlan.entrySet()) {
            List<SfaVisitPlanEntity> sfaVisitPlanEntities = entry.getValue();
            List<Map<String, SfaVisitPlanExecuteContext.FuturePlanInfo>> planCodeMapFuturePlanInfoOfPlans = Lists.newArrayList();
            for (SfaVisitPlanEntity plan : sfaVisitPlanEntities) {
                context.setCurrentPlanEntity(plan);
                //将拜访计划解析成计划明细
                this.resolveVisitPlanInfo(context);
                Map<String, SfaVisitPlanExecuteContext.FuturePlanInfo> planCodeMapFuturePlanInfo = context.getPlanCodeMapFuturePlanInfo();
                if (null != planCodeMapFuturePlanInfo) {
                    planCodeMapFuturePlanInfoOfPlans.add(planCodeMapFuturePlanInfo);
                }
                context.clean();
            }

            String userAndPosition = entry.getKey();
            List<SfaVisitPlanInfoEntity> planList = Lists.newArrayList();

            //解析出来的计划明细 <redisHash,<redisHashKey, SfaVisitPlanInfoEntity>>
            Map<String, Map<String, SfaVisitPlanInfoEntity>> daysPlanInfo = this.mergeFuturePlanInfo(userAndPosition, planCodeMapFuturePlanInfoOfPlans, planList);
            //注入客户数据到计划明细
            this.injectClientData(planList);
            //保存到redis和ES
            this.saveVisitPlanInfoToRedisAndEs(daysPlanInfo);
        }
    }

    /**
     * 注入客户数据
     *
     * @param planList
     */
    private void injectClientData(List<SfaVisitPlanInfoEntity> planList) {
        List<String> customerCodes = planList.stream().filter(v -> SfaVisitEnum.ClientType.DEALER.getVal().equals(v.getClientType())).map(SfaVisitPlanInfoEntity::getClientCode).collect(Collectors.toList());
        List<String> terminalCodes = planList.stream().filter(v -> SfaVisitEnum.ClientType.TERMINAL.getVal().equals(v.getClientType())).map(SfaVisitPlanInfoEntity::getClientCode).collect(Collectors.toList());
        Map<String, SfaClientData> clientDataMap = SfaClientHelper.loadClientDataMap(customerCodes, terminalCodes);
        for (SfaVisitPlanInfoEntity sfaVisitPlanInfoEntity : planList) {
            SfaClientData clientData = clientDataMap.get(sfaVisitPlanInfoEntity.getClientType() + sfaVisitPlanInfoEntity.getClientCode());
            SfaVisitPlanInfoEntity.copyClientData(sfaVisitPlanInfoEntity, clientData);
        }
    }

    /**
     * <redisHash,<redisHashKey, SfaVisitPlanInfoEntity>>
     *
     * @param daysPlanInfo
     */
    @Override
    public void saveVisitPlanInfoToRedisAndEs(Map<String, Map<String, SfaVisitPlanInfoEntity>> daysPlanInfo) {
        //计划刷新计算
        final Map<String, Map<String, List<SfaVisitPlanInfoEntity>>> needDelete = this.visitPlanInfoHelper.flushCounting(daysPlanInfo);
        daysPlanInfo.forEach((redisHash, objectObjectMap) -> {
//            SfaVisitPlanInfoEntity.VisitRedisHash visitRedisHash = new SfaVisitPlanInfoEntity.VisitRedisHash(redisHash);
            Collection<SfaVisitPlanInfoEntity> planInfoEntityList = objectObjectMap.values();
            try {
                Map<String, List<SfaVisitPlanInfoEntity>> needDeletePlanInfos = needDelete.get(redisHash);
                if (null == needDeletePlanInfos) {
                    needDeletePlanInfos = Maps.newHashMap();
                }
                this.saveToES(planInfoEntityList, needDeletePlanInfos);
                objectObjectMap = planInfoEntityList.stream().collect(Collectors.toMap(SfaVisitPlanInfoEntity::getRedisHashKey, v -> v, (r, r2) -> r2));
                this.visitPlanInfoHelper.saveToRedis(redisHash, objectObjectMap, needDeletePlanInfos.keySet());
            } catch (Exception e) {
                log.error("保存拜访/协访计划:保存数据失败", e);
                throw e;
            }
        });
    }


//    public static class DelAndSaveStore{
//
//    }

    /**
     * 保存到es
     *
     * @param planInfoEntityList
     */
    protected void saveToES(Collection<SfaVisitPlanInfoEntity> planInfoEntityList, final Map<String, List<SfaVisitPlanInfoEntity>> needDeletePlanInfos) {
        List<SfaVisitPlanInfoEntity> valueTemp = Lists.newArrayList();
        for (List<SfaVisitPlanInfoEntity> value : needDeletePlanInfos.values()) {
            valueTemp.addAll(value);
        }
        this.sfaVisitPlanInfoServiceEsImpl.delete(valueTemp);
        valueTemp.forEach(x -> {
            this.iSfaVisitPlanInfoService.remove(Wrappers.lambdaQuery(SfaVisitPlanInfoEntity.class)
                    .eq(SfaVisitPlanInfoEntity::getVisitUserName, x.getVisitUserName())
                    .eq(SfaVisitPlanInfoEntity::getVisitPosCode, x.getVisitPosCode())
                    .eq(SfaVisitPlanInfoEntity::getVisitPlanCode, x.getVisitPlanCode())
                    .eq(SfaVisitPlanInfoEntity::getClientCode, x.getClientCode())
                    .eq(SfaVisitPlanInfoEntity::getVisitDate, x.getVisitDate())
            );
        });
        if (org.springframework.util.CollectionUtils.isEmpty(planInfoEntityList)) {
            log.warn("拜访计划明细同步ES消息>>> 没有可同步的数据！");
            return;
        }
        this.sfaVisitPlanInfoServiceEsImpl.saveAll(planInfoEntityList);
        this.iSfaVisitPlanInfoService.saveOrUpdateBatch(planInfoEntityList);
//        return this.sfaVisitPlanInfoServiceEsImpl.findByVisitPosCodeAndVisitDateAndVisitBigType(visitRedisHash.getVisitPosCode(), visitRedisHash.getVisitDate(), visitRedisHash.getVisitBigType());

    }


    /**
     * <redisHash,<redisHashKey, SfaVisitPlanInfoEntity>>
     * 合并未来的拜访计划
     *
     * @param planCodeMapFuturePlanInfo
     */
    private Map<String, Map<String, SfaVisitPlanInfoEntity>> mergeFuturePlanInfo(String userAndPso
            , List<Map<String, SfaVisitPlanExecuteContext.FuturePlanInfo>> planCodeMapFuturePlanInfo, List<SfaVisitPlanInfoEntity> planList) {
        LocalDate visitDate = LocalDate.now().plusDays(-1);

        //该用户未来的拜访计划明细 <redisHash,<redisHashKey, SfaVisitPlanInfoEntity>>
        Map<String, Map<String, SfaVisitPlanInfoEntity>> daysPlan = Maps.newHashMap();
        for (int i = 0; i < SfaVisitPlanResolver.futureDays; i++) {
            //加一天
            visitDate = visitDate.plusDays(1);
            // 当前拜访日期 visitDate 对应的拜访明细（合并后）
            List<SfaVisitPlanInfoEntity> total = Lists.newArrayList();
            Set<String> hasClientCodes = Sets.newHashSet();
            //遍历合并各种计划（线路组、网点、频率）的明细
            for (Map<String, SfaVisitPlanExecuteContext.FuturePlanInfo> stringFuturePlanInfoMap : planCodeMapFuturePlanInfo) {
                for (Map.Entry<String, SfaVisitPlanExecuteContext.FuturePlanInfo> entry : stringFuturePlanInfoMap.entrySet()) {
                    SfaVisitPlanExecuteContext.FuturePlanInfo futurePlanInfo = entry.getValue();
                    //合并
                    this.doMerge(total, futurePlanInfo.getPlan(visitDate), hasClientCodes);
                }
            }

            planList.addAll(total);
            //该用户未来某一天的拜访计划明细
            Map<String, SfaVisitPlanInfoEntity> dayPlan = total.stream().collect(Collectors.toMap(v -> v.getRedisHashKey(), v -> v, (o, o2) -> o2));


            StringJoiner redisHash = new StringJoiner(RedisParam.DELIMITER);
            redisHash.add(RedisParam.SFA_VISIT).add(SfaVisitPlanInfoEntity.TABLE_NAME).add(SfaVisitEnum.VisitBigType.VISIT.getVal())
                    .add(userAndPso).add(visitDate.format(CrmDateUtils.yyyyMMdd));
            daysPlan.put(redisHash.toString(), dayPlan);
        }
        return daysPlan;
    }

    /**
     * 合并
     *
     * @param total
     * @param newData
     * @param hasClientCodes
     */
    private void doMerge(List<SfaVisitPlanInfoEntity> total, List<SfaVisitPlanInfoEntity> newData, Set<String> hasClientCodes) {
        for (SfaVisitPlanInfoEntity planInfoEntity : newData) {
            String clientCode = planInfoEntity.getClientCode();
            if (hasClientCodes.contains(clientCode)) {
                continue;
            }
            //设置默认状态为在线拜访
            planInfoEntity.setLineStatus(SfaVisitEnum.VISIT_OFF_LINE.online.getVal());
            total.add(planInfoEntity);
            hasClientCodes.add(clientCode);
        }
    }


    /**
     * 将拜访计划解析成计划明细
     *
     * @param context
     */
    private void resolveVisitPlanInfo(SfaVisitPlanExecuteContext context) {
        SfaVisitPlanEntity sfaVisitPlanEntity = context.getCurrentPlanEntity();

        SfaVisitPlanResolver planResolver = this.sfaVisitPlanResolverMap.get(sfaVisitPlanEntity.getRouteType() + SfaVisitPlanResolver.beanNameSuffix);
        if (null == planResolver) {
            throw new BusinessException("拜访计划解析执行失败，未找到计划维度类型[" + sfaVisitPlanEntity.getRouteType() + "]的解析实现!");
        }
        planResolver.resolve(context);
    }


}
