package com.biz.crm.worksign.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.activiti.act.TaActBaseFeign;
import com.biz.crm.activiti.mobile.ActivitiMobileFeign;
import com.biz.crm.aop.CrmAPIDiscard;
import com.biz.crm.base.BusinessException;
import com.biz.crm.common.PageResult;
import com.biz.crm.eunm.ActivitiEnum;
import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.eunm.sfa.SfaCommonEnum;
import com.biz.crm.eunm.sfa.SfaWorkSignEnum;
import com.biz.crm.eunm.sfa.WorkSignEnum;
import com.biz.crm.nebular.activiti.act.req.StartProcessReqVo;
import com.biz.crm.nebular.activiti.common.ProcessCommonVo;
import com.biz.crm.nebular.activiti.start.req.CancelProcessReqVO;
import com.biz.crm.nebular.activiti.task.resp.TaskRspVO;
import com.biz.crm.nebular.mdm.org.resp.MdmOrgRespVo;
import com.biz.crm.nebular.mdm.position.resp.MdmPositionUserOrgRespVo;
import com.biz.crm.nebular.sfa.worksign.req.SfaAuditFinishReqVo;
import com.biz.crm.nebular.sfa.worksign.req.SfaExceptionReportListReqVo;
import com.biz.crm.nebular.sfa.worksign.req.SfaExceptionReportReqVo;
import com.biz.crm.nebular.sfa.worksign.resp.*;
import com.biz.crm.util.*;
import com.biz.crm.worksign.mapper.SfaExceptionReportMapper;
import com.biz.crm.worksign.mapper.SfaWorkSignRecordMapper;
import com.biz.crm.worksign.model.SfaExceptionReportEntity;
import com.biz.crm.worksign.model.SfaSignApplyAttachmentEntity;
import com.biz.crm.worksign.model.SfaWorkSignRecordEntity;
import com.biz.crm.worksign.model.SfaWorkSignRuleInfoEntity;
import com.biz.crm.worksign.service.*;
import com.google.common.collect.Lists;
import jodd.util.ArraysUtil;
import jodd.util.StringUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 异常报备 接口实现
 *
 * @author ren.gang
 * @date 2020-11-23 15:00:38
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name="sfaExceptionReportServiceExpandImpl")
public class SfaExceptionReportServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<SfaExceptionReportMapper, SfaExceptionReportEntity>
        implements ISfaExceptionReportService {

    @Resource
    private SfaExceptionReportMapper mapper;
    @Resource
    private SfaWorkSignRecordMapper sfaWorkSignRecordMapper;
    @Autowired
    private TaActBaseFeign taActBaseFeign;
    @Autowired
    private ActivitiMobileFeign mobileFeign;
    @Autowired
    private ISfaSignFormsService formsService;
    @Autowired
    private ISfaSignApplyAttachmentService attachmentService;
    @Resource
    private ISfaWorkSignRecordService sfaWorkSignRecordService;
    @Resource
    private ISfaWorkSignRuleInfoService sfaWorkSignRuleInfoService;

    /**
     * 获取某个时间端中
     * 存在的补打卡类型
     *
     * @param startTime 开始时间
     * @param endTime   结束时间
     * @return
     */
    @Override
    public List<SfaAddSignTypeRespVo> getAddSignType(String startTime, String endTime) {
        UserRedis user = UserUtils.getUser();
        List<SfaWorkSignRecordEntity> result = sfaWorkSignRecordService.list(Wrappers.lambdaQuery(SfaWorkSignRecordEntity.class)
                .ne(SfaWorkSignRecordEntity::getWorkSignStatus, SfaWorkSignEnum.WorkSignStatus.OK.getVal())
                .between(SfaWorkSignRecordEntity::getRuleDate, startTime, endTime)
                .eq(SfaWorkSignRecordEntity::getWsUserName, user.getUsername()));
        if(CollectionUtil.listEmpty(result)){
            throw new BusinessException("当前时间无异常打卡信息");
        }
        result.forEach(data->{
            //若是自由签到就不返回时间信息
            if(SfaWorkSignEnum.WorkSignRuleType.FREE_TIME.getVal().equals(data.getRuleType())){
                data.setSfaSignTime("");
            }
        });
        List<SfaAddSignTypeRespVo> typeRespVos = CrmBeanUtil.copyList(result, SfaAddSignTypeRespVo.class);
        //当一天存在有两条以上的打卡异常记录就添加全天打卡类型
        if(typeRespVos.size()>=2){
            typeRespVos.add(new SfaAddSignTypeRespVo().setWorkSignType(SfaWorkSignEnum.WorkSignType.CLOCK_ALL.getVal()));
        }
        return typeRespVos;
    }

    /**
     * 列表
     * @param reqVo
     * @return
     */
    @Override
    public PageResult<SfaExceptionReportRespVo> findList(SfaExceptionReportListReqVo reqVo){
        Page<SfaExceptionReportRespVo> page = new Page<>(reqVo.getPageNum(), reqVo.getPageSize());
        if(StringUtils.isNotEmpty(reqVo.getApplicationDateStart())){
            reqVo.setApplicationDateStart(reqVo.getApplicationDateStart() + " 00:00:00");
        }
        if(StringUtils.isNotEmpty(reqVo.getApplicationDateEnd())){
            reqVo.setApplicationDateEnd(reqVo.getApplicationDateEnd() + " 23:59:59");
        }
        List<SfaExceptionReportRespVo> list = mapper.findList(page, reqVo);
        if(list != null && list.size() > 0) {
            list.forEach(vo -> {
                vo.setBpmStatusName(SfaCommonEnum.dataBpmStatus.getDesc(vo.getBpmStatus()));
            });
        }
        return PageResult.<SfaExceptionReportRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    /**
     * 查询
     * @param id
     * @return SfaExceptionReportRespVo
     */
    @Override
    public SfaExceptionReportRespVo query(String id){
//        SfaExceptionReportListReqVo reqVo = new SfaExceptionReportListReqVo();
//        reqVo.setId(id);
//        PageResult<SfaExceptionReportRespVo> result = findList(reqVo);
//        if(result == null || result.getCount() != 1) {
//            throw new BusinessException("id错误");
//        }
//        return result.getData().get(0);
        if(StringUtils.isEmpty(id)){
            throw new BusinessException("请传入ID");
        }
        SfaExceptionReportEntity exceptionReportEntity = this.getOne(Wrappers.lambdaQuery(SfaExceptionReportEntity.class).eq(SfaExceptionReportEntity::getId, id));
        if(exceptionReportEntity == null){
            throw new BusinessException("该ID未查询到相关补打卡信息");
        }
        SfaExceptionReportRespVo reportRespVo = CrmBeanUtil.copy(exceptionReportEntity, SfaExceptionReportRespVo.class);
        reportRespVo.setBpmStatusName(SfaCommonEnum.dataBpmStatus.getDesc(reportRespVo.getBpmStatus()));
        List<SfaAddSignTypeRespVo> signTypeRespVos = new ArrayList<>();
        List<String> strings = Arrays.asList(exceptionReportEntity.getExceptionRecordIds().split(","));
        strings.forEach(data->{
            SfaWorkSignRecordEntity recordEntity = sfaWorkSignRecordService.getOne(Wrappers.lambdaQuery(SfaWorkSignRecordEntity.class).eq(SfaWorkSignRecordEntity::getId, data));
            if(recordEntity==null){
                return;
            }
            SfaAddSignTypeRespVo typeRespVo = CrmBeanUtil.copy(recordEntity, SfaAddSignTypeRespVo.class);
            //自由签到时间规则则设置签到时间为空
            if(SfaWorkSignEnum.WorkSignRuleType.FREE_TIME.getVal().equals(recordEntity.getRuleType())){
                typeRespVo.setSfaSignTime("");
            }
            typeRespVo.setWorkSignTypeName(DictUtil.dictValue("AddSignType",typeRespVo.getWorkSignType()));
            signTypeRespVos.add(typeRespVo);
        });
        reportRespVo.setAddSignTypeRespVos(signTypeRespVos);
        SfaSignApplyAttachmentEntity signApplyAttachmentEntityList = attachmentService.getOne(Wrappers.lambdaQuery(SfaSignApplyAttachmentEntity.class)
                .eq(SfaSignApplyAttachmentEntity::getApplyType, WorkSignEnum.signApplyType.EXCEPTION.getValue())
                .eq(SfaSignApplyAttachmentEntity::getSourceId, reportRespVo.getId()));
        if(signApplyAttachmentEntityList != null ){
            reportRespVo.setAttachmentListJson(signApplyAttachmentEntityList.getAttachmentListJson());
        }
        return reportRespVo;
    }

    /**
     * 根据审核任务ID查询
     * @param auditTaskId
     * @return
     */
    @Override
    public SfaExceptionReportRespVo queryByAuditTaskId(String auditTaskId) {
        SfaExceptionReportListReqVo reqVo = new SfaExceptionReportListReqVo();
        reqVo.setAuditTaskIdList(Arrays.asList(auditTaskId));
        PageResult<SfaExceptionReportRespVo> result = findList(reqVo);
        if(result == null || result.getCount() != 1) {
            throw new BusinessException("审核任务ID错误");
        }
        return result.getData().get(0);
    }


    /**
     * 异常报备申请接口
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public Result apply(SfaExceptionReportReqVo reqVo){
        SfaExceptionReportEntity entity = CrmBeanUtil.copy(reqVo,SfaExceptionReportEntity.class);
        // 判断编辑
        if(StringUtils.isNotEmpty(reqVo.getId())) {
            SfaExceptionReportEntity old = mapper.selectById(reqVo.getId());
            if(old == null) {
                return Result.error("主键id错误");
            }

            // 驳回状态直接创建新的申请
            if(SfaCommonEnum.dataBpmStatus.REJECT.getValue().equals(old.getBpmStatus())) {
                entity.setId(null);
            // 追回的使用原来的申请继续提交
            } else if(SfaCommonEnum.dataBpmStatus.ROLLBACK.getValue().equals(old.getBpmStatus())) {
                entity.setId(old.getId());
            // 非编辑状态判断
            } else {
                return Result.error("该申请不支持编辑");
            }
        }
        // 默认为当前人员申请
        if(StringUtils.isEmpty(entity.getUserName())) {
            entity.setUserName(UserUtils.getUser().getUsername());
        }
        // 查询用户主数组
        MdmPositionUserOrgRespVo mdmPositionUserOrgRespVo = formsService.getUserOrgInfo(entity.getUserName());
        entity.setRealName(mdmPositionUserOrgRespVo.getFullName());
        entity.setPosCode(mdmPositionUserOrgRespVo.getPositionCode());
        entity.setPosName(mdmPositionUserOrgRespVo.getPositionName());
        String orgCode = mdmPositionUserOrgRespVo.getOrgCode();
        //主职位组织
        entity.setOrgCode(orgCode);
        entity.setOrgName(mdmPositionUserOrgRespVo.getOrgName());
        //组织的上级组织
        MdmOrgRespVo mdmOrgRespVo = OrgUtil.getOrgByCode(orgCode);
        if(null != mdmOrgRespVo){
            entity.setParentOrgCode(mdmOrgRespVo.getParentCode());
            entity.setParentOrgName(mdmOrgRespVo.getParentName());
        }
        // 申请时间
        entity.setApplicationDate(CrmDateUtils.yyyyMMddHHmmss.format(LocalDateTime.now()));
        // 验证开始时间和结束时间
        SfaSignUtils.verifyStartAndEndDate(entity.getBeginTime(), entity.getEndTime());
        LocalDate localDate = LocalDate.now();
        // 报备时间只能是当天之前
        if(!LocalDate.parse(entity.getEndTime()).isBefore(localDate)) {
            return Result.error("只能报备当天之前的异常签到");
        }
        // 8日之前可修改上月的数据，8日之后，不可修改上月数据
        if(localDate.getDayOfMonth() > 8) { // 8号之后只能报备本月
            if(LocalDate.parse(entity.getBeginTime()).isBefore(LocalDate.of(localDate.getYear(), localDate.getMonth(), 1))) {
                return Result.error("只能报备当月异常签到");
            }
        } else { // 8号之前可以设置上月
            LocalDate ult = LocalDate.now().minusMonths(1); // 上月
            if(LocalDate.parse(entity.getBeginTime()).isBefore(LocalDate.of(ult.getYear(), ult.getMonth(), 1))) {
                return Result.error("只能报备上月异常签到");
            }
        }
//      // 校验申请时间冲突
//        Integer count = mapper.findRepeatByDates(SfaCommonEnum.dataBpmStatus.REJECT.getValue(), UserUtils.getUser().getUsername(),
//                entity.getBeginTime(), entity.getEndTime(), entity.getId());
//        if(count != null && count > 0) {
//            return Result.error("该时间段内已有申请");
//        }
//        //校验是否包含异常考勤
//        List<SfaWorkSignRecordRespVo> esRecordList = sfaWorkSignRecordMapper.findExceptionSign(entity.getUserName(), entity.getBeginTime(), entity.getEndTime());
//        if(esRecordList == null || esRecordList.size() == 0) {
//            return Result.error("时间段内未存在异常信息");
//        }
        // 校验申请时间冲突+添加异常报备（补打卡）所关联的打卡记录ID 类型
        checkExceptionReport(reqVo,entity);

        // 校验申请人信息
        if(StringUtils.isEmpty(entity.getUserName()) || StringUtil.isEmpty(entity.getRealName())
                || StringUtil.isEmpty(entity.getPosCode())) {
            return Result.error("申请人员信息必须包含以下信息：人员账号、人员姓名、岗位编码，请核对");
        }
        // 设置状态
        entity.setBpmStatus(SfaCommonEnum.dataBpmStatus.COMMIT.getValue());
        this.saveOrUpdate(entity);
        entity.setAuditTaskId(entity.getId());

        //发送审核信息到审核系统
        StartProcessReqVo startProcessReqVO =
                SfaActivitiUtils.createStartProcessReqVO(
                        entity.getUserName(),
                        entity.getPosCode(),
                        entity.getAuditTaskId(),
                        ActivitiEnum.FormTypeEnum.EXCEPTION,
                        entity.getBeginTime(),
                        entity.getEndTime(),
                        entity.getOrgCode()
                );
        String processNo = ActivityUtils.startProcess(startProcessReqVO);
        entity.setProcessNo(processNo);
        this.saveOrUpdate(entity);
        // 保存附件
        attachmentService.save(WorkSignEnum.signApplyType.EXCEPTION.getValue(), entity.getId(), reqVo.getAttachmentList());

        return Result.ok();
    }

    /**
     * 异常报备-追回
     * @param id
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public Result rollback(String id) {
        SfaExceptionReportEntity entity = mapper.selectById(id);
        if(entity == null) {
            return Result.error("数据不存在");
        }
        if(!SfaCommonEnum.dataBpmStatus.COMMIT.getValue().equals(entity.getBpmStatus())) {
            return Result.error("当前申请不能追回");
        }

        CancelProcessReqVO reqVO = new CancelProcessReqVO(entity.getAuditTaskId()
                , UserUtils.getUser().getUsername()
                , ActivitiEnum.FormTypeEnum.EXCEPTION.getCostType()
                , ActivitiEnum.getFormType());
        Result result = mobileFeign.cancelProcess(reqVO);
        if(!result.isSuccess()) {
            return result;
        }

        entity.setBpmStatus(SfaCommonEnum.dataBpmStatus.ROLLBACK.getValue());
        this.saveOrUpdate(entity);
        return Result.ok();
    }


    /**
     * 提交审核信息
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public Result auditCommit(String id) {
        SfaExceptionReportEntity entity = mapper.selectById(id);
        if(entity == null) {
            return Result.error("主键id错误");
        }
        // 验证状态
        if(!entity.getBpmStatus().equals(SfaCommonEnum.dataBpmStatus.COMMIT.getValue()) &&
                !entity.getBpmStatus().equals(SfaCommonEnum.dataBpmStatus.APPROVAL.getValue())) {
            return Result.error("该申请审批状态错误");
        }
        // 已审核过修改状态
        if(!SfaCommonEnum.dataBpmStatus.APPROVAL.getValue().equals(entity.getBpmStatus())) {
            entity.setBpmStatus(SfaCommonEnum.dataBpmStatus.APPROVAL.getValue());
            saveOrUpdate(entity);
        }

        return Result.ok();
    }

    /**
     * 根据任务流返回信息查询审核任务列表
     * @param taskRspVOPageResult
     * @param reqVo
     * @return
     */
    @Override
    public PageResult<SfaAuditListExceptionRespVo> findAuditList(PageResult<TaskRspVO> taskRspVOPageResult, SfaExceptionReportListReqVo reqVo) {
        if(taskRspVOPageResult != null && taskRspVOPageResult.getCount() > 0) {
            // 获取审核任务id集合
            Map<String, TaskRspVO> taskMap = taskRspVOPageResult.getData().stream().collect(Collectors.toMap(TaskRspVO::getFormNo, vo -> vo, (key1, key2) -> key2));
            reqVo.setAuditTaskIdList(new ArrayList<>(taskMap.keySet()));
            // 根据id和其他查询条件去查询业务列表数据
            PageResult<SfaExceptionReportRespVo> pageResult = findList(reqVo);
            if(pageResult.getCount() > 0) {
                // 数据排序
                List data = pageResult.getData().stream().sorted((x, y) -> {
                    return CrmDateUtils.parseyyyyMMddHHmmss(y.getApplicationDate())
                            .compareTo(CrmDateUtils.parseyyyyMMddHHmmss(x.getApplicationDate()));
                }).collect(Collectors.toList());
                pageResult.setData(data);
                // 封装返回数据
                List<SfaAuditListExceptionRespVo> list = new ArrayList<>();
                pageResult.getData().forEach(vo -> {
                    list.add(new SfaAuditListExceptionRespVo(vo, taskMap.get(vo.getAuditTaskId())));
                });
                return PageResult.<SfaAuditListExceptionRespVo>builder()
                        .data(list)
                        .count(pageResult.getCount())
                        .build();
            }
        }
        return PageResult.<SfaAuditListExceptionRespVo>builder()
                .data(Lists.newArrayList())
                .count(0L)
                .build();
    }

    /**
     * 审核完成回调
     * @param reqVo
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void auditFinish(SfaAuditFinishReqVo reqVo) {
        // 查询请假信息
        LambdaQueryWrapper<SfaExceptionReportEntity> wrapper = new LambdaQueryWrapper<SfaExceptionReportEntity>()
                .eq(SfaExceptionReportEntity::getAuditTaskId,reqVo.getFormId());
        SfaExceptionReportEntity entity = mapper.selectOne(wrapper);
        // 处理审核成功
        if(ActivitiEnum.AuditResultType.PASS.getVal().equals(reqVo.getResult())) {
            entity.setBpmStatus(SfaCommonEnum.dataBpmStatus.PASS.getValue());
            // 修改考勤状态为 正常.1、获取规则列表、2执行更新
            String recordIds = entity.getExceptionRecordIds();
            if(null == recordIds){
                recordIds = "";
            }
            List<String> ids = Lists.newArrayList(recordIds.split(","));
//            List<String> ruleInfoIds = sfaWorkSignRecordMapper.getRuleInfoIds(entity.getUserName(), entity.getBeginTime(), entity.getEndTime());
            if(CollectionUtils.isNotEmpty(ids)) {
                List<SfaWorkSignRecordEntity> entities = sfaWorkSignRecordService.lambdaQuery().in(SfaWorkSignRecordEntity :: getId, ids).list();
                for (SfaWorkSignRecordEntity sfaWorkSignRecordEntity : entities) {
                    String remarks = sfaWorkSignRecordEntity.getRemarks();
                    StringJoiner joiner = new StringJoiner(";");
                    if(null != remarks){
                        joiner.add(remarks);
                    }
                    joiner.add("异常报备修改状态");
                    sfaWorkSignRecordService.lambdaUpdate().eq(SfaWorkSignRecordEntity :: getId, sfaWorkSignRecordEntity.getId())
                            .set(SfaWorkSignRecordEntity :: getWorkSignStatus, SfaWorkSignEnum.WorkSignStatus.OK.getVal())
                            .set(SfaWorkSignRecordEntity :: getWorkSignTime, sfaWorkSignRecordEntity.getSfaSignTime())
                            .set(SfaWorkSignRecordEntity :: getRemarks, joiner.toString()).update();
                }

            }
        }
        else if (ActivitiEnum.AuditResultType.REJECT.getVal().equals(reqVo.getResult())){ // 审核驳回
            entity.setBpmStatus(SfaCommonEnum.dataBpmStatus.REJECT.getValue());
        } else if (ActivitiEnum.AuditResultType.RECOVER.getVal().equals(reqVo.getResult())){ // 审核追回
            entity.setBpmStatus(SfaCommonEnum.dataBpmStatus.ROLLBACK.getValue());
        } else {
            throw  new BusinessException("审批失败");
        }

        saveOrUpdate(entity);
    }

    /**
     * 校验申请时间冲突+添加异常报备（补打卡）所关联的打卡记录ID 类型
     * @method checkExceptionReport
     * @date: 2021/2/24 14:59
     * @author: YuanZiJian
     * @param entity
     * @return void
     */
    private void checkExceptionReport (SfaExceptionReportReqVo reqVo,SfaExceptionReportEntity entity){
        List<String> signRecordIdList = reqVo.getWorkSignRecordIdList();
        //校验异常报备（补打卡）所关联的打卡记录ID
        if(CollectionUtil.listEmpty(signRecordIdList)){
            throw new BusinessException("未选择补打卡类型");
        }
        //查询对应的异常报备记录
        List<SfaExceptionReportEntity> entityList = this.list(Wrappers.lambdaQuery(SfaExceptionReportEntity.class)
                .eq(SfaExceptionReportEntity::getBeginTime, entity.getBeginTime())
                .eq(SfaExceptionReportEntity::getEndTime, entity.getEndTime())
                .eq(SfaExceptionReportEntity::getUserName, entity.getUserName())
                .eq(SfaExceptionReportEntity::getPosCode,entity.getPosCode())
                .eq(SfaExceptionReportEntity::getOrgCode,entity.getOrgCode())
                .ne(SfaExceptionReportEntity::getBpmStatus,SfaCommonEnum.dataBpmStatus.REJECT.getValue())
                .ne(SfaExceptionReportEntity::getBpmStatus,SfaCommonEnum.dataBpmStatus.ROLLBACK.getValue()));
        if(CollectionUtil.listNotEmpty(entityList)){
            entityList.forEach(data->{
                List<String> strings = Arrays.asList(data.getExceptionRecordIds().split(","));
                strings.stream().forEach(item->{
                    //查询的异常报备打卡记录ID中与新增的记录中的打卡记录ID是否有相同的Id
                    if(signRecordIdList.contains(item)){
                        throw new BusinessException("存在相关的补打卡审批");
                    }
                });

            });
        }
        //前端传得list中可能会带Null值_去掉NULL
        signRecordIdList.removeAll(Collections.singleton(null));
        //添加异常报备（补打卡）所关联的打卡记录ID 类型
        String exceptionRecordIds = reqVo.getWorkSignRecordIdList().stream().collect(Collectors.joining(","));
        entity.setExceptionRecordIds(exceptionRecordIds);
    }
}
