package com.biz.crm.worksign.service.impl;

import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.activiti.act.TaActBaseFeign;
import com.biz.crm.activiti.mobile.ActivitiMobileFeign;
import com.biz.crm.base.BusinessException;
import com.biz.crm.calculatesalary.service.ISfaCalculateSalaryDateService;
import com.biz.crm.common.CrmIdsReqVo;
import com.biz.crm.common.PageResult;
import com.biz.crm.common.param.ParameterParam;
import com.biz.crm.eunm.ActivitiEnum;
import com.biz.crm.eunm.CrmDelFlagEnum;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.eunm.sfa.SfaCommonEnum;
import com.biz.crm.eunm.sfa.SfaWorkSignEnum;
import com.biz.crm.eunm.sfa.WorkSignEnum;
import com.biz.crm.nebular.activiti.act.req.StartProcessReqVo;
import com.biz.crm.nebular.activiti.common.ProcessCommonVo;
import com.biz.crm.nebular.activiti.start.req.CancelProcessReqVO;
import com.biz.crm.nebular.activiti.task.resp.TaskRspVO;
import com.biz.crm.nebular.mdm.org.resp.MdmOrgRespVo;
import com.biz.crm.nebular.mdm.position.resp.MdmPositionUserOrgRespVo;
import com.biz.crm.nebular.sfa.worksign.req.*;
import com.biz.crm.nebular.sfa.worksign.resp.SfaAuditListLeaveRespVo;
import com.biz.crm.nebular.sfa.worksign.resp.SfaLeaveCancelRespVo;
import com.biz.crm.nebular.sfa.worksign.resp.SfaLeaveRespVo;
import com.biz.crm.nebular.sfa.worksignrule.resp.SfaWorkSignRuleRespVo;
import com.biz.crm.util.*;
import com.biz.crm.worksign.mapper.SfaLeaveCancelMapper;
import com.biz.crm.worksign.mapper.SfaLeaveMapper;
import com.biz.crm.worksign.model.SfaLeaveCancelEntity;
import com.biz.crm.worksign.model.SfaLeaveEntity;
import com.biz.crm.worksign.model.SfaSignApplyAttachmentEntity;
import com.biz.crm.worksign.model.SfaWorkOvertimeEntity;
import com.biz.crm.worksign.service.*;
import com.biz.crm.worksignrule.service.impl.SfaWorkSignRuleServiceImpl;
import com.google.common.collect.Lists;
import jodd.util.StringUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 请假申请 接口实现
 *
 * @author liuhongming
 * @date 2020-10-17 15:00:38
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name="SfaLeaveServiceExpandImpl")
public class SfaLeaveServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<SfaLeaveMapper, SfaLeaveEntity> implements ISfaLeaveService {

    private static final String LEAVE_TYPE_MUST_ATTACHMENT = "leave_type_must_attachment";

    @Resource
    private SfaLeaveCancelMapper sfaLeaveCancelMapper;
    @Resource
    private SfaLeaveMapper sfaLeaveMapper;
    @Autowired
    private ISfaLeaveCancelService leaveCancelService;
    @Autowired
    private TaActBaseFeign taActBaseFeign;
    @Autowired
    private ActivitiMobileFeign mobileFeign;
    @Autowired
    private ISfaSignFormsService formsService;
    @Autowired
    private ISfaWorkOvertimeService overtimeService;
    @Autowired
    private SfaWorkSignRuleServiceImpl signRuleService;
    @Autowired
    private ISfaSignApplyAttachmentService attachmentService;
    @Autowired
    private ISfaCalculateSalaryDateService dateService;

    /**
     * 列表
     * @param reqVo
     * @return
     */
    @Override
    public PageResult<SfaLeaveRespVo> findList(SfaLeaveListReqVo reqVo){
        Page<SfaLeaveRespVo> page = new Page<>(reqVo.getPageNum(), reqVo.getPageSize());
        if(StringUtils.isNotEmpty(reqVo.getApplicationDateStart())){
            reqVo.setApplicationDateStart(reqVo.getApplicationDateStart() + " 00:00:00");
        }
        if(StringUtils.isNotEmpty(reqVo.getApplicationDateEnd())){
            reqVo.setApplicationDateEnd(reqVo.getApplicationDateEnd() + " 23:59:59");
        }
        List<SfaLeaveRespVo> list = sfaLeaveMapper.findList(page, reqVo);
        if(list != null) {
            list.forEach(vo -> {
                setLeaveCancelInfo(vo);
                vo.setBpmStatusName(SfaCommonEnum.dataBpmStatus.getDesc(vo.getBpmStatus()));
                vo.setLeaveTypeName(SfaCommonEnum.leaveTypeEnum.getDesc(vo.getLeaveType()));
            });
        }
        return PageResult.<SfaLeaveRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    /**
     * 设置请假相关的销假信息
     */
    @Override
    public void setLeaveCancelInfo(SfaLeaveRespVo vo) {
        List<SfaLeaveCancelRespVo> list = leaveCancelService.findList(new SfaLeaveCancelReqVo(vo.getId()));
        vo.setLeaveCancelRespVoList(list);
        vo.setOperationCancel(YesNoEnum.yesNoEnum.N.getValue());
        // 设置销假按钮。1、当前登录人员，2审核通过状态，3当前时间未超过请假结束时间
        UserRedis userRedis = UserUtils.getUser();
        if(userRedis.getUsername().equals(vo.getUserName())) {
            if(SfaCommonEnum.dataBpmStatus.PASS.getValue().equals(vo.getBpmStatus())) {
                if(LocalDate.parse(vo.getEndTime()).compareTo(LocalDate.now()) >= 0) {
                    // 1、获取比较开始时间
                    LocalDate startDate = LocalDate.now();
                    if(LocalDate.parse(vo.getBeginTime()).isAfter(startDate)) {
                        startDate = LocalDate.parse(vo.getBeginTime());
                    }
                    // 2、开始执行比较
                    do {
                        // 非工作日时执行下一天
                        if(StringUtils.isNotEmpty(vo.getNonWorkDateListJson()) && vo.getNonWorkDateListJson().indexOf(startDate.toString()) > -1) {
                            startDate = startDate.plusDays(1);
                            continue;
                        }
                        if(list != null && list.size() > 0 && list.get(0) != null) {
                            // 判断比较日期是否有上午销假信息
                            try {
                                leaveCancelService.verifyDateRepeat(list, startDate.toString(), startDate.toString(),
                                        Arrays.asList(new SfaApplyTimeInfoReqVo(startDate.toString(), SfaCommonEnum.dataTimeType.FORENOON.getValue())));
                                vo.setOperationCancel(YesNoEnum.yesNoEnum.Y.getValue());
                                break;
                            } catch (Exception e) { // 上午已有销假信息
                                // 判断下午是否有销假信息
                                try {
                                    leaveCancelService.verifyDateRepeat(list, startDate.toString(), startDate.toString(),
                                            Arrays.asList(new SfaApplyTimeInfoReqVo(startDate.toString(), SfaCommonEnum.dataTimeType.AFTERNOON.getValue())));
                                    if("0.0".equals(vo.getLeaveDuration())){
                                        vo.setOperationCancel(YesNoEnum.yesNoEnum.N.getValue());
                                    }else {
                                        vo.setOperationCancel(YesNoEnum.yesNoEnum.Y.getValue());
                                    }
                                    break;
                                } catch (Exception e1) {
                                    log.debug("当日已有销假信息");
                                }
                            }
                            startDate = startDate.plusDays(1);
                            continue;
                        } else {
                            vo.setOperationCancel(YesNoEnum.yesNoEnum.Y.getValue());
                            break;
                        }
                    } while (startDate.compareTo(LocalDate.parse(vo.getEndTime())) < 1);
                }
            }
        }
    }

    /**
     * 查询
     * @param reqVo
     * @return sfaLeaveRespVo
     */
    @Override
    public SfaLeaveRespVo query(SfaLeaveReqVo reqVo){
        return null;
    }

    /**
     * 根据审核任务id查询
     * @param auditTaskId
     * @return
     */
    @Override
    public SfaLeaveRespVo queryByAuditTaskId(String auditTaskId) {
        SfaLeaveListReqVo reqVo = new SfaLeaveListReqVo();
        //在销假表中查询是否有销假信息
        SfaLeaveCancelEntity leaveCancelEntity = leaveCancelService.getOne(Wrappers.lambdaQuery(SfaLeaveCancelEntity.class).eq(SfaLeaveCancelEntity::getAuditTaskId, auditTaskId));
        if(leaveCancelEntity!=null){
            //设置销假信息对应的请假Id
            reqVo.setId(leaveCancelEntity.getLeaveId());
        }else {
            //设置请假的对应流程号
            reqVo.setAuditTaskIdList(Arrays.asList(auditTaskId));
        }
        PageResult<SfaLeaveRespVo> result = findList(reqVo);
        if(result == null || result.getCount() != 1) {
            throw new BusinessException("审核任务ID错误");
        }
        return result.getData().get(0);
    }

    /**
     * 新增
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void save(SfaLeaveReqVo reqVo){
        SfaLeaveEntity entity = CrmBeanUtil.copy(reqVo,SfaLeaveEntity.class);
        this.save(entity);
    }

    /**
     * 更新
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(SfaLeaveReqVo reqVo){
        SfaLeaveEntity entity = this.getById(reqVo.getId());
        this.updateById(entity);
    }

    /**
     * 删除
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteBatch(CrmIdsReqVo reqVo){
        List<SfaLeaveEntity> sfaLeaveEntities = sfaLeaveMapper.selectBatchIds(reqVo.getIds());
        if(CollectionUtils.isNotEmpty(sfaLeaveEntities)){
            sfaLeaveEntities.forEach(o -> {
                 o.setDelFlag(CrmDelFlagEnum.DELETE.getCode());
            });
        }
        this.updateBatchById(sfaLeaveEntities);
    }

    /**
     * 启用
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void enableBatch(CrmIdsReqVo reqVo){
        //设置状态为启用
        List<SfaLeaveEntity> sfaLeaveEntities = sfaLeaveMapper.selectBatchIds(reqVo.getIds());
        if(CollectionUtils.isNotEmpty(sfaLeaveEntities)){
            sfaLeaveEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
            });
        }
        this.updateBatchById(sfaLeaveEntities);
    }

    /**
     * 禁用
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void disableBatch(CrmIdsReqVo reqVo){
        //设置状态为禁用
        List<SfaLeaveEntity> sfaLeaveEntities = sfaLeaveMapper.selectBatchIds(reqVo.getIds());
        if(CollectionUtils.isNotEmpty(sfaLeaveEntities)){
                sfaLeaveEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.DISABLE.getCode());
            });
        }
        this.updateBatchById(sfaLeaveEntities);
    }

    /**
     * 请假申请接口
     * @param sfaLeaveReqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public Result goApplyLeave(SfaLeaveReqVo sfaLeaveReqVo){

        // 获取申请时长+验证信息
        SfaLeaveEntity entity = countApplyDays(sfaLeaveReqVo);

        // 判断编辑
        if(StringUtils.isNotEmpty(sfaLeaveReqVo.getId())) {
            SfaLeaveEntity old = sfaLeaveMapper.selectById(sfaLeaveReqVo.getId());
            if(old == null) {
                return Result.error("主键id错误");
            }
            // 驳回状态直接创建新的申请
            if(SfaCommonEnum.dataBpmStatus.REJECT.getValue().equals(old.getBpmStatus())) {
                entity.setId(null);
            // 追回的使用原来的申请继续提交
            } else if(SfaCommonEnum.dataBpmStatus.ROLLBACK.getValue().equals(old.getBpmStatus())) {
                entity.setId(old.getId());
            // 非编辑状态判断
            } else {
                return Result.error("该申请不支持编辑");
            }
        }

        // 校验请假时间冲突
        List<SfaLeaveRespVo> repeatList =
                sfaLeaveMapper.findRepeatByDates(entity.getUserName(), entity.getBeginTime(), entity.getEndTime(), entity.getId());
        if(repeatList != null && repeatList.size() > 0) {
            for(SfaLeaveRespVo vo : repeatList) {
                //根据请假的Id获取是否有对应的销假的通过审批信息
                List<SfaLeaveCancelEntity> sfaLeaveCancelEntities = sfaLeaveCancelMapper.selectList(Wrappers.lambdaQuery(SfaLeaveCancelEntity.class)
                        .eq(SfaLeaveCancelEntity::getLeaveId, vo.getId())
                        .eq(SfaLeaveCancelEntity::getBpmStatus, "3"));
                //只比较有请假时长的
                if(!"0".equals(vo.getLeaveDuration())){
                    if(CollectionUtil.listNotEmpty(sfaLeaveCancelEntities)){
                        //如果有销假的信息，则进入销假比对方法(verifyDateRepeatCancel) 如下
                        if (!verifyDateRepeatCancel(entity,sfaLeaveCancelEntities,sfaLeaveReqVo)){
                            //新增时间不在销假通过的时间端中则，可以认为该新增时间与销假无关联，使用原来的时间校验方法（SfaSignUtils.verifyDateRepeat）
                            SfaSignUtils.verifyDateRepeat(entity.getBeginTime(), entity.getEndTime(), vo.getBeginTime(), vo.getEndTime()
                                    , sfaLeaveReqVo.getTimeInfoList(), JSON.parseArray(vo.getTimeInfoListJson(), SfaApplyTimeInfoReqVo.class));
                        }
                    }else {
                        SfaSignUtils.verifyDateRepeat(entity.getBeginTime(), entity.getEndTime(), vo.getBeginTime(), vo.getEndTime()
                                , sfaLeaveReqVo.getTimeInfoList(), JSON.parseArray(vo.getTimeInfoListJson(), SfaApplyTimeInfoReqVo.class));
                    }
                }
            }
        }
        entity.setTimeInfoListJson(JSON.toJSONString(sfaLeaveReqVo.getTimeInfoList()));

        // 通过数据字典配置获取那些类型请假必须上传附件(数据字典配置项)
        String mustTypeByAttachment = "";
        Map<String, String> map = DictUtil.getDictValueMapsByCodes(LEAVE_TYPE_MUST_ATTACHMENT);
        if (map != null) {
            for (String key : map.keySet()) {
                mustTypeByAttachment += "," + key;
            }
        }
        if(mustTypeByAttachment.indexOf(entity.getLeaveType()) > -1) {
            if(sfaLeaveReqVo.getAttachmentList() == null || sfaLeaveReqVo.getAttachmentList().size() == 0) {
                return Result.error("请上传附件");
            }
        }
        // 申请时间
        entity.setApplicationDate(CrmDateUtils.yyyyMMddHHmmss.format(LocalDateTime.now()));
        // 设置审核状态和考勤统计相关信息
        entity.setBpmStatus(SfaCommonEnum.dataBpmStatus.COMMIT.getValue());
        this.saveOrUpdate(entity);
        //发送审核信息到审核系统
        entity.setAuditTaskId(entity.getId());
        StartProcessReqVo startProcessReqVO =
                SfaActivitiUtils.createStartProcessReqVO(
                        entity.getUserName(), entity.getPosCode(), entity.getAuditTaskId(),
                        ActivitiEnum.FormTypeEnum.LEAVE,
                        entity.getBeginTime(),
                        entity.getEndTime(),
                        entity.getOrgCode());
        // 业务参数｛请假天数、是否年假｝
        String isAnnual = YesNoEnum.yesNoEnum.NO.getValue();
        if(SfaCommonEnum.leaveTypeEnum.ANNUAL_LEAVE.getVal().equals(entity.getLeaveType())) {
            isAnnual = YesNoEnum.yesNoEnum.YES.getValue();
        }
        startProcessReqVO.getVariables().put("isAnnual", isAnnual); // 是否年假
        startProcessReqVO.getVariables().put("days", new BigDecimal(entity.getLeaveDuration()));// 请假天数
        String processNo = ActivityUtils.startProcess(startProcessReqVO);
        entity.setProcessNo(processNo);
        this.saveOrUpdate(entity);
        attachmentService.save(WorkSignEnum.signApplyType.LEAVE.getValue(), entity.getId()
                , sfaLeaveReqVo.getAttachmentList());
        return Result.ok();
    }

    /**
     * 计算请假时长
     * @param reqVo
     * @return
     */
    @Override
    public SfaLeaveEntity countApplyDays(SfaLeaveReqVo reqVo) {
        SfaLeaveEntity entity = CrmBeanUtil.copy(reqVo,SfaLeaveEntity.class);

        // 请假类型校验
        if (StringUtil.isEmpty(entity.getLeaveType())){
            throw new BusinessException("请假类型必传");
        }

        //设置创建人信息，当前登录人信息
        UserRedis userRedis = UserUtils.getUser();
        if (null==userRedis|| StringUtil.isEmpty(userRedis.getUsername())){
            throw new BusinessException(CommonConstant.NO_LOGIN_401, "请重新登录");
        }

        // 默认为当前人员申请
        if(StringUtils.isEmpty(entity.getUserName())) {
            entity.setUserName(userRedis.getUsername());
        }
        // 查询用户主数组
        MdmPositionUserOrgRespVo mdmPositionUserOrgRespVo = formsService.getUserOrgInfo(entity.getUserName());

        entity.setRealName(mdmPositionUserOrgRespVo.getFullName());

        entity.setPosCode(mdmPositionUserOrgRespVo.getPositionCode());
        entity.setPosName(mdmPositionUserOrgRespVo.getPositionName());
        String orgCode = mdmPositionUserOrgRespVo.getOrgCode();
        //主职位组织
        entity.setOrgCode(orgCode);
        entity.setOrgName(mdmPositionUserOrgRespVo.getOrgName());
        //组织的上级组织
        MdmOrgRespVo mdmOrgRespVo = OrgUtil.getOrgByCode(orgCode);
        if(null != mdmOrgRespVo){
            entity.setParentOrgCode(mdmOrgRespVo.getParentCode());
            entity.setParentOrgName(mdmOrgRespVo.getParentName());
        }

        // 校验申请人信息
        if(StringUtils.isEmpty(entity.getUserName()) || StringUtil.isEmpty(entity.getRealName())
                || StringUtil.isEmpty(entity.getPosCode()) || StringUtil.isEmpty(entity.getOrgCode())) {
            throw new BusinessException("申请人员信息必须包含以下信息：人员账号、人员姓名、岗位编码、组织编码，请核对");
        }

        // 校验并获取全部时间的申请时长
        BigDecimal countApplyDays = SfaSignUtils.countApplyDays(entity.getBeginTime(), entity.getEndTime()
                , reqVo.getTimeInfoList());

        // 请假-调休或年假，触发校验规则
        if(SfaCommonEnum.leaveTypeEnum.ANNUAL_LEAVE.getVal().equals(entity.getLeaveType())
                || SfaCommonEnum.leaveTypeEnum.DAYS_OFF.getVal().equals(entity.getLeaveType())) {
            // 获取考勤规则
            SfaWorkSignRuleRespVo ruleRespVo = signRuleService.getSignRuleByOrgCode(entity.getOrgCode());

            // 验证开始时间、结束时间 为工作日
            if(!YesNoEnum.yesNoEnum.ZERO.getValue().equals(
                    SfaSignUtils.isWorkDay(ruleRespVo, entity.getBeginTime(), dateService.findHolidayByDate(entity.getBeginTime()))
            )) {
                throw new BusinessException("开始时间请选择工作日");
            }
            if(!YesNoEnum.yesNoEnum.ZERO.getValue().equals(
                    SfaSignUtils.isWorkDay(ruleRespVo, entity.getEndTime(), dateService.findHolidayByDate(entity.getEndTime()))
            )) {
                throw new BusinessException("结束时间请选择工作日");
            }

            // 扣除扣除休息日
            List<SfaApplyTimeInfoReqVo> nonWorkDateList = new ArrayList<>();
            List<SfaApplyTimeInfoReqVo> timeInfoList = SfaSignUtils.fillTimeInfoAndCheck(reqVo.getTimeInfoList(), entity.getBeginTime(), entity.getEndTime());
            for (int r = 0; r < timeInfoList.size(); r++) {
                if (r == 0 || r == timeInfoList.size() - 1) { // 开始和结束已经验证过了
                    continue;
                }
                // 当天为非工作日
                if (!YesNoEnum.yesNoEnum.ZERO.getValue().equals(
                        SfaSignUtils.isWorkDay(ruleRespVo, timeInfoList.get(r).getTimeStr(), dateService.findHolidayByDate(timeInfoList.get(r).getTimeStr()))
                )) {
                    countApplyDays = countApplyDays.subtract(new BigDecimal(1));
                    nonWorkDateList.add(timeInfoList.get(r));
                }
            }
            // 保存非工作日信息
            if(nonWorkDateList.size() > 0) {
                entity.setNonWorkDateListJson(JSON.toJSONString(nonWorkDateList));
            }
        }

        // 设置返回信息
        entity.setLeaveDuration(countApplyDays.toString());

        return entity;
    }

    /**
     * 请假申请-追回
     * @param id
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public Result rollbackLeave(String id) {
        SfaLeaveEntity entity = sfaLeaveMapper.selectById(id);
        if(entity == null) {
            return Result.error("数据不存在");
        }
        if(!SfaCommonEnum.dataBpmStatus.COMMIT.getValue().equals(entity.getBpmStatus())) {
            return Result.error("当前申请不能追回");
        }

        // 调用工作流
        CancelProcessReqVO reqVO = new CancelProcessReqVO(entity.getAuditTaskId()
                , UserUtils.getUser().getUsername()
                , ActivitiEnum.FormTypeEnum.LEAVE.getCostType()
                , ActivitiEnum.FormTypeEnum.LEAVE.getFormType());
        Result result = mobileFeign.cancelProcess(reqVO);
        if(!result.isSuccess()) {
            return result;
        }

        entity.setBpmStatus(SfaCommonEnum.dataBpmStatus.ROLLBACK.getValue());
        this.saveOrUpdate(entity);

        return Result.ok();
    }

    /**
     * 追加附件信息
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public Result addAttachment(SfaLeaveReqVo reqVo) {
        SfaLeaveEntity entity = sfaLeaveMapper.selectById(reqVo.getId());
        if(entity == null) {
            return Result.error("信息不存在");
        }
        // 只有审批中的才能追加附件
        if(!SfaCommonEnum.dataBpmStatus.COMMIT.getValue().equals(entity.getBpmStatus()) &&
                !SfaCommonEnum.dataBpmStatus.APPROVAL.getValue().equals(entity.getBpmStatus())) {
            return Result.error("当前状态不能追加附件");
        }

        List<SfaApplyAttachmentReqVo> attachmentList = new ArrayList<>();
        // 获取附件信息
        SfaSignApplyAttachmentEntity attachmentEntity =
                attachmentService.getEntity(WorkSignEnum.signApplyType.LEAVE.getValue(), entity.getId());
        if(attachmentEntity != null && StringUtils.isNotEmpty(attachmentEntity.getAttachmentListJson())) {
            attachmentList = JSON.parseArray(attachmentEntity.getAttachmentListJson(), SfaApplyAttachmentReqVo.class);
        }
        attachmentList.addAll(reqVo.getAttachmentList());

        attachmentService.save(WorkSignEnum.signApplyType.LEAVE.getValue(), entity.getId()
                , attachmentList);
        return Result.ok();
    }

    /**
     * 销假
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public Result cancelLeave(SfaAuditCancelReqVo reqVo) {
        SfaLeaveEntity entity = sfaLeaveMapper.selectById(reqVo.getId());
        if(entity == null) {
            return Result.error("信息不存在");
        }
        if(!SfaCommonEnum.dataBpmStatus.PASS.getValue().equals(entity.getBpmStatus())) {
            return Result.error("当前申请不允许销假");
        }

        if(reqVo.getTimeInfoReqVoList() == null || reqVo.getTimeInfoReqVoList().size() == 0) {
            return Result.error("日期明细列表不能为空");
        }
        // 销假日期明细全列表
        List<SfaApplyTimeInfoReqVo> cancelTimeInfoList = reqVo.getTimeInfoReqVoList();
        String startDate = cancelTimeInfoList.get(0).getTimeStr();
        String endDate = cancelTimeInfoList.get(cancelTimeInfoList.size()-1).getTimeStr();
        //获取请假的日期明细，与销假的比对
        List<SfaApplyTimeInfoReqVo> sfaApplyTimeInfoReqVos = JsonPropertyUtil.toArray(entity.getTimeInfoListJson(), SfaApplyTimeInfoReqVo.class);
        if(!checkDateLeave(cancelTimeInfoList,sfaApplyTimeInfoReqVos)){
            throw new BusinessException("销假时间请选择在请假时间之内");
        }
        // 验证销假时间是否正确
        if(!(LocalDate.parse(entity.getBeginTime()).compareTo(LocalDate.parse(startDate)) <= 0
                && LocalDate.parse(entity.getEndTime()).compareTo(LocalDate.parse(endDate)) >= 0)) {
            return Result.error("销假时间请选择在请假时间之内");
        }
        boolean isCancel = true;
        try {
            // 申请时间如果没有和请假时间重合，表示销假时间在请假时间之外
            SfaSignUtils.verifyDateRepeat(
                    startDate,
                    endDate,
                    entity.getBeginTime(),
                    entity.getEndTime(),
                    reqVo.getTimeInfoReqVoList(),
                    JSON.parseArray(entity.getTimeInfoListJson(), SfaApplyTimeInfoReqVo.class)
            );
        } catch (Exception e) {
            isCancel = false;
        }
        if(isCancel) {
            return Result.error("销假时间请选择在请假时间之内");
        }

        // 销假信息，计算销假天数。并减去非工作日
        BigDecimal cancelDays = SfaSignUtils.countApplyDays(startDate, endDate, cancelTimeInfoList);
        if(StringUtils.isNotEmpty(entity.getNonWorkDateListJson())) {
            // 销假开始日期不能是非工作日
            if(entity.getNonWorkDateListJson().indexOf(startDate) > -1) {
                throw new BusinessException("销假开始不能为非工作日");
            }
            if(entity.getNonWorkDateListJson().indexOf(endDate) > -1) {
                throw new BusinessException("销假结束不能为非工作日");
            }
            cancelTimeInfoList = SfaSignUtils.fillTimeInfoAndCheck(cancelTimeInfoList, startDate, endDate);
            for (SfaApplyTimeInfoReqVo vo : cancelTimeInfoList) {
                if (entity.getNonWorkDateListJson().indexOf(vo.getTimeStr()) > -1) {
                    cancelDays = cancelDays.subtract(new BigDecimal(1));
                }
            }
        }

        // 默认走审批流程
        boolean isAudit = true;
        try {
            if (YesNoEnum.yesNoEnum.NO.getValue().equals(ParamUtil.getParameterValue(ParameterParam.SFA_LEAVE_CANCEL_AUDIT))) {
                isAudit = false;
            }
        } catch (Exception e){}

        // 保存销假信息
        SfaLeaveCancelEntity leaveCancelEntity = leaveCancelService.save(new SfaLeaveCancelReqVo(entity.getId(), startDate, endDate, cancelDays.toString(), reqVo.getTimeInfoReqVoList()), isAudit);

        if(!isAudit) {
            rollBackDeduction(leaveCancelEntity);
        }

        return Result.ok();
    }

    /**
     * 根据任务流返回信息查询审核任务列表
     * @param taskRspVOPageResult
     * @param reqVo
     * @return
     */
    @Override
    public PageResult<SfaAuditListLeaveRespVo> findAuditList(PageResult<TaskRspVO> taskRspVOPageResult
            , SfaLeaveListReqVo reqVo) {
        if(taskRspVOPageResult != null && taskRspVOPageResult.getCount() > 0) {
            // 获取审核任务id集合
            Map<String, TaskRspVO> taskMap = taskRspVOPageResult.getData().stream().collect(Collectors.toMap(TaskRspVO::getFormNo, vo -> vo, (key1, key2) -> key2));
            reqVo.setAuditTaskIdList(new ArrayList<>(taskMap.keySet()));

            // 根据id和其他查询条件去查询业务列表数据
            PageResult<SfaLeaveRespVo> pageResult = findList(reqVo);
            if(pageResult.getCount() > 0) {
                // 数据排序
                List data = pageResult.getData().stream().sorted((x, y) -> {
                    return CrmDateUtils.parseyyyyMMddHHmmss(y.getApplicationDate())
                            .compareTo(CrmDateUtils.parseyyyyMMddHHmmss(x.getApplicationDate()));
                }).collect(Collectors.toList());
                pageResult.setData(data);
                // 封装返回数据
                List<SfaAuditListLeaveRespVo> list = new ArrayList<>();
                pageResult.getData().forEach(vo -> {
                    list.add(new SfaAuditListLeaveRespVo(vo, taskMap.get(vo.getAuditTaskId())));
                });
                return PageResult.<SfaAuditListLeaveRespVo>builder()
                        .data(list)
                        .count(pageResult.getCount())
                        .build();
            }
        }
        return PageResult.<SfaAuditListLeaveRespVo>builder()
                .data(Lists.newArrayList())
                .count(0L)
                .build();
    }

    /**
     * 提交审核信息
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public Result auditCommit(String id) {
        SfaLeaveEntity entity = sfaLeaveMapper.selectById(id);
        if(entity == null) {
            return Result.error("主键id错误");
        }
        // 验证状态
        if(!entity.getBpmStatus().equals(SfaCommonEnum.dataBpmStatus.COMMIT.getValue()) &&
                !entity.getBpmStatus().equals(SfaCommonEnum.dataBpmStatus.APPROVAL.getValue())) {
            return Result.error("该申请审批状态错误");
        }
        // 已审核过修改状态
        if(!SfaCommonEnum.dataBpmStatus.APPROVAL.getValue().equals(entity.getBpmStatus())) {
            entity.setBpmStatus(SfaCommonEnum.dataBpmStatus.APPROVAL.getValue());
            saveOrUpdate(entity);
        }

        return Result.ok();
    }

    /**
     * 审核完成回调
     * @param reqVo
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void auditFinish(SfaAuditFinishReqVo reqVo) {
        // 查询请假信息
        LambdaQueryWrapper<SfaLeaveEntity> wrapper = new LambdaQueryWrapper<SfaLeaveEntity>()
                .eq(SfaLeaveEntity::getAuditTaskId,reqVo.getFormId());
        SfaLeaveEntity entity = sfaLeaveMapper.selectOne(wrapper);
        // 处理审核通过
        if(ActivitiEnum.AuditResultType.PASS.getVal().equals(reqVo.getResult())) {
            entity.setBpmStatus(SfaCommonEnum.dataBpmStatus.PASS.getValue());
            entity.setPassStatusDate(CrmDateUtils.yyyyMMddHHmmss.format(LocalDateTime.now()));
            // 请假类型为调休时，根据调休规则核算调休信息
            if(SfaCommonEnum.leaveTypeEnum.DAYS_OFF.getVal().equals(entity.getLeaveType())) {
                // 获取当前调休天数
                BigDecimal lastDays = new BigDecimal(entity.getLeaveDuration());
                // 获取有效期内可以抵扣的工作日调整列表
                List<SfaWorkOvertimeEntity> overtimeList = overtimeService.findNotUseDaysList(entity.getUserName());
                if(overtimeList != null && overtimeList.size() > 0) {
                    /**
                     * 1、计算 抵扣天数 = 剩余天数-可抵扣天数, 剩余天数 = 剩余天数-抵扣天数
                     * 1.1、正数： 抵扣天数为可抵扣天数
                     * 1.2、负数或0： 抵扣天数为剩余天数
                     * 2、添加抵扣信息
                     * 3、剩余天数 <= 0 时，结束循环
                     */
                    for(SfaWorkOvertimeEntity overtimeEntity : overtimeList) {
                        BigDecimal days = lastDays.subtract(overtimeEntity.getLastDays());
                        if(days.compareTo(BigDecimal.ZERO) > 0) {
                            days = overtimeEntity.getLastDays();
                        } else {
                            days = lastDays;
                        }
                        lastDays = lastDays.subtract(overtimeEntity.getLastDays());
                        // 保存抵扣信息
                        overtimeEntity.setUseDeductionIds(SfaSignUtils.addDeductionIds(overtimeEntity.getUseDeductionIds(), entity.getId(), days));
                        overtimeEntity.setUseDays(overtimeEntity.getUseDays().add(days));
                        overtimeEntity.setLastDays(overtimeEntity.getLastDays().subtract(days));
                        overtimeService.saveOrUpdate(overtimeEntity);
                        entity.setUseDeductionIds(SfaSignUtils.addDeductionIds(entity.getUseDeductionIds(), overtimeEntity.getId(), days));

                        if(lastDays.compareTo(BigDecimal.ZERO) <= 0) {
                            break;
                        }
                    }
                }
                // 设置抵扣最终信息
                entity.setLastDays(lastDays);
                entity.setUseDays(new BigDecimal(entity.getLeaveDuration()).subtract(lastDays));
            }
        } else if (ActivitiEnum.AuditResultType.REJECT.getVal().equals(reqVo.getResult())){ // 审核驳回
            entity.setBpmStatus(SfaCommonEnum.dataBpmStatus.REJECT.getValue());
        } else if (ActivitiEnum.AuditResultType.RECOVER.getVal().equals(reqVo.getResult())){ // 审核追回
            entity.setBpmStatus(SfaCommonEnum.dataBpmStatus.ROLLBACK.getValue());
        } else {
            throw  new BusinessException("审批失败");
        }

        // 保存信息
        saveOrUpdate(entity);
    }

    /**
     * 销假审批完成
     * @param reqVo
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void auditCancelFinish(SfaAuditFinishReqVo reqVo) {
        // 保存审批结果
        SfaLeaveCancelEntity leaveCancelEntity = leaveCancelService.auditCancelFinish(reqVo);

        // 处理审核通过-回滚抵扣
        if(ActivitiEnum.AuditResultType.PASS.getVal().equals(reqVo.getResult())) {
            rollBackDeduction(leaveCancelEntity);
        }
    }

    /**
     * 执行抵扣回滚
     * @param leaveCancelEntity
     */
    private void rollBackDeduction(SfaLeaveCancelEntity leaveCancelEntity) {
        SfaLeaveEntity entity = this.getById(leaveCancelEntity.getLeaveId());
        if(entity == null) {
            throw new BusinessException("请假信息不存在");
        }
        BigDecimal cancelDays = new BigDecimal(leaveCancelEntity.getCancelDays());
        // 更新请假时长，和销假总时长
        entity.setLeaveDuration(new BigDecimal(entity.getLeaveDuration()).subtract(cancelDays).toString());
        if(StringUtils.isNotEmpty(entity.getCancelDays())) {
            entity.setCancelDays(new BigDecimal(entity.getCancelDays()).add(cancelDays).toString());
        } else{
            entity.setCancelDays(cancelDays.toString());
        }

        /**
         * 1、计算中的请假和销假
         * 2、剩余天数减去销假天数
         * 2.1、正数或0：更新剩余抵扣天数为 (2)的值
         * 2.2、负数：回滚抵扣信息-3
         * 3、遍历抵扣信息，。回滚数+抵扣数
         * 3.1、负数或0：同时回滚抵扣数、并更新回滚数
         * 3.2、正数：同时更新回滚数，并更新回滚数为0
         * 3.3 回滚数=0时，结束循环
         */
        // 销假触发抵扣回滚。
        if(SfaCommonEnum.leaveTypeEnum.DAYS_OFF.getVal().equals(entity.getLeaveType())) {
            // 如果剩余抵扣天数 >= 销假天数，只需要更新剩余抵扣天数
            BigDecimal backNum = entity.getLastDays().subtract(cancelDays);
            if(backNum.compareTo(BigDecimal.ZERO) != -1) {
                entity.setLastDays(backNum);
            } else { // 回滚抵扣
                entity.setLastDays(BigDecimal.ZERO);
                entity.setUseDays(entity.getUseDays().subtract(cancelDays));
                // 当前已抵扣的工作日调整列表
                List<Map> useDeductionIdsList = SfaSignUtils.decodeDeductionIds(entity.getUseDeductionIds());
                for(int r = useDeductionIdsList.size(); r > 0; r --) {
                    String id = SfaSignUtils.getDeductionId(useDeductionIdsList.get(r-1)); // 工作日调整id
                    BigDecimal days = SfaSignUtils.getDeductionDays(useDeductionIdsList.get(r-1)); // 抵扣天数
                    // 修改抵扣天数
                    BigDecimal updateDays = backNum.add(days);
                    backNum = backNum.add(days);
                    if(updateDays.compareTo(BigDecimal.ZERO) == 1) {
                        updateDays = days.subtract(updateDays);
                        backNum = BigDecimal.ZERO;
                    } else {
                        updateDays = days;
                    }
                    // 回滚工作日调整
                    SfaWorkOvertimeEntity overtimeEntity = overtimeService.getById(id);
                    overtimeEntity.setUseDeductionIds(SfaSignUtils.backDeductionIds(overtimeEntity.getUseDeductionIds(), entity.getId(), updateDays));
                    overtimeEntity.setUseDays(overtimeEntity.getUseDays().subtract(updateDays));
                    overtimeEntity.setLastDays(overtimeEntity.getLastDays().add(updateDays));
                    overtimeService.saveOrUpdate(overtimeEntity);

                    // 请假信息更新
                    entity.setUseDeductionIds(SfaSignUtils.backDeductionIds(entity.getUseDeductionIds(), id, updateDays));

                    if(backNum.compareTo(BigDecimal.ZERO) == 0){
                        break;
                    }
                }
            }
        }
        // 保存信息
        this.saveOrUpdate(entity);
    }

    /**
     * 获取数据字典请假有效期配置
     * @return
     */
    @Override
    public int getLeaveIndate() {
        // 设置有效期
        Map<String, String> map = DictUtil.getDictValueMapsByCodes(SfaWorkSignEnum.SignRuleIndateKey.SIGN_RULE_INDATE.getVal());
        if(map != null && map.get(SfaWorkSignEnum.SignRuleIndateKey.LEAVE_INDATE.getVal()) != null) {
            try {
                return Integer.parseInt(map.get(SfaWorkSignEnum.SignRuleIndateKey.LEAVE_INDATE.getVal()));
            }catch (Exception e) {
                throw new BusinessException("数据字典"+SfaWorkSignEnum.SignRuleIndateKey.SIGN_RULE_INDATE.getVal()+"配置错误");
            }
        }
        return 0;
    }

    /**
     * 获取有效期内可以抵扣的请假整列表
     * @return
     */
    @Override
    public List<SfaLeaveEntity> findNotUseDaysList(String userName) {
        return sfaLeaveMapper.findNotUseDays(userName, getLeaveIndate());
    }

    /**
     * 保证销假一定在请假日期范围内
     * 因为是销假请假是连续的时间段
     * 1.保证请假的第一天包含销假的第一天。
     * 2.保证请假最后一天包含销假的最后一天即可。
     *
     * @method checkDateLeave
     * @date: 2021/2/2 11:03
     * @author: YuanZiJian
     * @param cancelTimeInfoList    销假明细
     * @param sfaApplyTimeInfoReqVos 请假明细
     * @return void
     */
    private boolean checkDateLeave(List<SfaApplyTimeInfoReqVo> cancelTimeInfoList,List<SfaApplyTimeInfoReqVo> sfaApplyTimeInfoReqVos){
        Boolean flag = false;
        SfaApplyTimeInfoReqVo sfaApplyTimeInfoFirstDay = sfaApplyTimeInfoReqVos.get(0);
        SfaApplyTimeInfoReqVo sfaApplyTimeInfoReqVoEndDay = sfaApplyTimeInfoReqVos.get(sfaApplyTimeInfoReqVos.size() - 1);
        SfaApplyTimeInfoReqVo cancelTimeInfoReqVoFirstDay = cancelTimeInfoList.get(0);
        SfaApplyTimeInfoReqVo cancelTimeInfoReqVoEndDay = cancelTimeInfoList.get(cancelTimeInfoList.size() - 1);
        if("1".equals(sfaApplyTimeInfoFirstDay.getTimeType()) ||sfaApplyTimeInfoFirstDay.getTimeType().equals(cancelTimeInfoReqVoFirstDay.getTimeType())){
                if("1".equals(sfaApplyTimeInfoReqVoEndDay.getTimeType()) ||sfaApplyTimeInfoReqVoEndDay.getTimeType().equals(cancelTimeInfoReqVoEndDay.getTimeType())){
                    flag = true;
                }
        }
        return flag;
    }

    /**
     * 校验新增时间段被包含在销假的时间端
     * @method verifyDateRepeatCancel
     * @date: 2021/2/2 15:08
     * @author: YuanZiJian
     * @param entity 新增的销假信息
     * @param sfaLeaveCancelEntities 销假信息
     * @return void
     */
    private Boolean verifyDateRepeatCancel(SfaLeaveEntity entity , List<SfaLeaveCancelEntity> sfaLeaveCancelEntities,SfaLeaveReqVo sfaLeaveReqVo){
        //获取新增的请假开始和结束时间
        boolean flag = false;
        String newBeginTime = entity.getBeginTime();
        String newEndTime = entity.getEndTime();
        //获取新增的日期类型(1全天，2上午，3下午)
        List<SfaApplyTimeInfoReqVo> timeInfoList = sfaLeaveReqVo.getTimeInfoList();
        String beginType = timeInfoList.get(0).getTimeType();
        String endType = timeInfoList.get(timeInfoList.size()-1).getTimeType();
        //去遍历销假信息的集合，比对销假和新增的开始结束时间
        for (SfaLeaveCancelEntity data: sfaLeaveCancelEntities) {
            //获取销假的日期类型(1全天，2上午，3下午)
            List<SfaApplyTimeInfoReqVo> timeInfoCancel = JsonPropertyUtil.toArray(data.getTimeInfoListJson(), SfaApplyTimeInfoReqVo.class);
            String beginCancelType = timeInfoCancel.get(0).getTimeType();
            String endCancelType = timeInfoCancel.get(timeInfoCancel.size()-1).getTimeType();
            //只有当新增的时间段被销假的时间段包含时，才会通过
            int compareToBegin = LocalDate.parse(newBeginTime).compareTo(LocalDate.parse(data.getBeginTime()));
            int compareToEnd = LocalDate.parse(newEndTime).compareTo(LocalDate.parse(data.getEndTime()));
            //当新增的时间段被销假的时间段包含时（新增的时间段与销假的时间段不相等）
            if(compareToBegin>0&&compareToEnd<0){
                flag = true;
            }
            /*
             * 新增的时间段与销假的时间段不完全相等时
             * 情况一: 新增起始时间等于销假起始时间
             * 情况二: 新增结束时间等于销假结束时间
             */
            //情况一: 新增起始时间等于销假起始时间
            else  if (compareToBegin==0&&compareToEnd<=0){
                if(checkType(beginType,beginCancelType)){
                    flag=true;
                }
            }
            //情况二: 新增结束时间等于销假结束时间
            else  if (compareToBegin>=0&&compareToEnd==0){
                if(checkType(endType,endCancelType)){
                    flag=true;
                }
            }
        }
        return flag;
    }


    /**
     * 校验同一天
     * 新增和销假的全天，上午，下午的包含关系（全天包含上午下午）
     * @method checkType
     * @date: 2021/2/2 16:41
     * @author: YuanZiJian
     * @param saveType 新增的日期类型
     * @param chancelType 销假的日期类型
     * @return boolean
     */
    private boolean checkType(String saveType ,String chancelType){
            //type相等时间段必定相等
            if(saveType.equals(chancelType)){
                return true;
            }
            //销假的type为1（全天）则 新增的type可以
            if (chancelType.equals("1")){
                return true;
            }
            //以上条件不满足则返回false
            return false;
    }

}
