package com.biz.crm.worksign.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.activiti.act.TaActBaseFeign;
import com.biz.crm.activiti.mobile.ActivitiMobileFeign;
import com.biz.crm.base.BusinessException;
import com.biz.crm.common.PageResult;
import com.biz.crm.eunm.ActivitiEnum;
import com.biz.crm.eunm.sfa.SfaCommonEnum;
import com.biz.crm.nebular.activiti.act.req.StartProcessReqVo;
import com.biz.crm.nebular.activiti.common.ProcessCommonVo;
import com.biz.crm.nebular.activiti.start.req.CancelProcessReqVO;
import com.biz.crm.nebular.activiti.task.resp.TaskRspVO;
import com.biz.crm.nebular.mdm.position.resp.MdmPositionUserOrgRespVo;
import com.biz.crm.nebular.sfa.worksign.req.*;
import com.biz.crm.nebular.sfa.worksign.resp.SfaAuditListTravelRespVo;
import com.biz.crm.nebular.sfa.worksign.resp.SfaTravelRespVo;
import com.biz.crm.nebular.sfa.worksign.resp.SfaTravelUserRespVo;
import com.biz.crm.util.*;
import com.biz.crm.worksign.mapper.SfaTravelMapper;
import com.biz.crm.worksign.model.SfaTravelEntity;
import com.biz.crm.worksign.service.ISfaSignFormsService;
import com.biz.crm.worksign.service.ISfaTravelService;
import com.biz.crm.worksign.service.ISfaTravelUserService;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 出差申请 接口实现
 *
 * @author liuhongming
 * @date 2020-10-17 15:00:38
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name="sfaTravelServiceExpandImpl")
public class SfaTravelServiceImpl <M extends BaseMapper<T>,T> extends ServiceImpl<SfaTravelMapper, SfaTravelEntity> implements ISfaTravelService {

    @Resource
    private SfaTravelMapper sfaTravelMapper;
    @Autowired
    private ISfaTravelUserService sfaTravelUserService;
    @Autowired
    private TaActBaseFeign taActBaseFeign;
    @Autowired
    private ActivitiMobileFeign mobileFeign;
    @Autowired
    private ISfaSignFormsService formsService;

    /**
     * 列表
     * @param reqVo
     * @return
     */
    @Override
    public PageResult<SfaTravelRespVo> findList(SfaTravelReqVo reqVo){
        Page<SfaTravelRespVo> page = new Page<>(reqVo.getPageNum(), reqVo.getPageSize());
        if(StringUtils.isNotEmpty(reqVo.getApplicationDateStart())){
            reqVo.setApplicationDateStart(reqVo.getApplicationDateStart() + " 00:00:00");
        }
        if(StringUtils.isNotEmpty(reqVo.getApplicationDateEnd())){
            reqVo.setApplicationDateEnd(reqVo.getApplicationDateEnd() + " 23:59:59");
        }
        List<SfaTravelRespVo> list = sfaTravelMapper.findList(page, reqVo);
        if(list != null && list.size() > 0 && list.get(0) != null) {
            list.forEach(vo -> {
                List<SfaTravelUserRespVo> userRespVos = sfaTravelUserService.findList(vo.getId());
                vo.setSfaTravelUserRespVos(userRespVos);
                if(CollectionUtil.listNotEmpty(userRespVos)) {
                    vo.setTravelUserNames(userRespVos.stream().map(SfaTravelUserRespVo::getRealName).collect(Collectors.joining(",")));
                }
                vo.setBpmStatusName(SfaCommonEnum.dataBpmStatus.getDesc(vo.getBpmStatus()));
            });
        }
        return PageResult.<SfaTravelRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    @Override
    public List<SfaTravelRespVo> findDataList(SfaTravelReqVo reqVo){
        List<SfaTravelRespVo> list = sfaTravelMapper.findDataList(reqVo);
        return list;
    }

    /**
     * 查询
     * @param id
     * @return sfaTravelRespVo
     */
    @Override
    public SfaTravelRespVo query(String id){
        SfaTravelReqVo reqVo = new SfaTravelReqVo();
        reqVo.setId(id);
        PageResult<SfaTravelRespVo>  pageResult = this.findList(reqVo);
        if(pageResult == null || pageResult.getCount() != 1) {
            throw new BusinessException("出差信息ID错误");
        }
        return pageResult.getData().get(0);
    }

    /**
     * 出差申请接口
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public Result goApplyTravel(SfaTravelApplyReqVo reqVo){
        // 数据校验
        checkInfo(reqVo);
        //获取当前登录人信息
        UserRedis userRedis = UserUtils.getUser();
        if(userRedis == null) {
            throw new BusinessException("请登录");
        }
        SfaTravelEntity entity = CrmBeanUtil.copy(reqVo,SfaTravelEntity.class);
        // 再次提交设置
        if(StringUtils.isNotEmpty(entity.getId())) {
            SfaTravelEntity old = sfaTravelMapper.selectOne(Wrappers.lambdaQuery(SfaTravelEntity.class).eq(SfaTravelEntity::getId, entity.getId()));
            if(old == null) {
                throw new BusinessException("出差信息id错误");
            }
            // 驳回状态直接创建新的申请
            if(SfaCommonEnum.dataBpmStatus.REJECT.getValue().equals(old.getBpmStatus())) {
                entity.setId(null);
                // 追回也直接创建新的申请继续提交
            } else if(SfaCommonEnum.dataBpmStatus.ROLLBACK.getValue().equals(old.getBpmStatus())) {
                entity.setId(null);
                // 非编辑状态判断
            } else {
                return Result.error("该申请不支持编辑");
            }
            entity.setApplyUserName(old.getApplyUserName());
        } else {
            // 申请人信息设置
            entity.setApplyUserName(userRedis.getUsername());
        }
        // 查询用户主数组
        MdmPositionUserOrgRespVo mdmPositionUserOrgRespVo = formsService.getUserOrgInfo(entity.getApplyUserName());
        entity.setApplyRealName(mdmPositionUserOrgRespVo.getFullName());
        entity.setApplyOrgCode(mdmPositionUserOrgRespVo.getOrgCode());
        entity.setApplyOrgName(mdmPositionUserOrgRespVo.getOrgName());
        entity.setApplyPosCode(mdmPositionUserOrgRespVo.getPositionCode());
        entity.setApplyPosName(mdmPositionUserOrgRespVo.getPositionName());
        entity.setApplyParentOrgCode(mdmPositionUserOrgRespVo.getParentOrgCode());
        entity.setApplyParentOrgName(mdmPositionUserOrgRespVo.getParentOrgName());

        // 申请时间
        entity.setApplicationDate(CrmDateUtils.yyyyMMddHHmmss.format(LocalDateTime.now()));
        // 设置审核状态和考勤统计相关信息
        entity.setBpmStatus(SfaCommonEnum.dataBpmStatus.COMMIT.getValue());
        // 保存出差信息
        this.saveOrUpdate(entity);
        // 保存出差人员
        sfaTravelUserService.saveTravelUser(reqVo.getSfaTravelUserReqVos(), entity);
        //发送审核信息到审核系统
        entity.setAuditTaskId(entity.getId());
        StartProcessReqVo startProcessReqVO =
                SfaActivitiUtils.createStartProcessReqVO(
                        entity.getApplyUserName(), entity.getApplyPosCode(), entity.getAuditTaskId(),
                        ActivitiEnum.FormTypeEnum.TRAVEL,
                        entity.getBeginTime(),
                        entity.getEndTime(),
                        entity.getApplyOrgCode());

        String processNo = ActivityUtils.startProcess(startProcessReqVO);
        entity.setProcessNo(processNo);
        // 保存出差信息
        this.saveOrUpdate(entity);
        return Result.ok();
    }

    /**
     * 根据任务流返回信息查询审核任务列表
     * @param taskRspVOPageResult
     * @param reqVo
     * @return
     */
    @Override
    public PageResult<SfaAuditListTravelRespVo> findAuditList(PageResult<TaskRspVO> taskRspVOPageResult, SfaTravelReqVo reqVo) {
        if(taskRspVOPageResult != null && taskRspVOPageResult.getCount() > 0) {
            // 获取审核任务id集合
            Map<String, TaskRspVO> taskMap = taskRspVOPageResult.getData().stream().collect(Collectors.toMap(TaskRspVO::getFormNo, vo -> vo, (key1, key2) -> key2));
            reqVo.setAuditTaskIdList(new ArrayList<>(taskMap.keySet()));
            // 根据id和其他查询条件去查询业务列表数据
            PageResult<SfaTravelRespVo> pageResult = findList(reqVo);
            if(pageResult.getCount() > 0) {
                // 数据排序
                List data = pageResult.getData().stream().sorted((x, y) -> {
                    return CrmDateUtils.parseyyyyMMddHHmmss(y.getApplicationDate())
                            .compareTo(CrmDateUtils.parseyyyyMMddHHmmss(x.getApplicationDate()));
                }).collect(Collectors.toList());
                pageResult.setData(data);
                // 封装返回数据
                List<SfaAuditListTravelRespVo> list = new ArrayList<>();
                pageResult.getData().forEach(vo -> {
                    list.add(new SfaAuditListTravelRespVo(vo, taskMap.get(vo.getAuditTaskId())));
                });
                return PageResult.<SfaAuditListTravelRespVo>builder()
                        .data(list)
                        .count(pageResult.getCount())
                        .build();
            }
        }
        return PageResult.<SfaAuditListTravelRespVo>builder()
                .data(Lists.newArrayList())
                .count(0L)
                .build();
    }

    /**
     * 追回
     * @param id
     */
    @Override
    public Result rollback(String id) {
        SfaTravelEntity entity = sfaTravelMapper.selectById(id);
        if(entity == null) {
            return Result.error("出差申请主键id错误");
        }
        if(!SfaCommonEnum.dataBpmStatus.COMMIT.getValue().equals(entity.getBpmStatus())) {
            return Result.error("当前申请不能追回");
        }

        // 调用工作流
        CancelProcessReqVO reqVO = new CancelProcessReqVO(entity.getAuditTaskId()
                , UserUtils.getUser().getUsername()
                , ActivitiEnum.FormTypeEnum.TRAVEL.getCostType()
                , ActivitiEnum.FormTypeEnum.TRAVEL.getFormType());
        Result result = mobileFeign.cancelProcess(reqVO);
        if(!result.isSuccess()) {
            return result;
        }

        entity.setBpmStatus(SfaCommonEnum.dataBpmStatus.ROLLBACK.getValue());
        this.saveOrUpdate(entity);

        return Result.ok();
    }

    /**
     * 提交审核信息
     * @param id
     * @return
     */
    @Override
    public Result auditCommit(String id) {
        SfaTravelEntity entity = sfaTravelMapper.selectById(id);
        if(entity == null) {
            return Result.error("出差申请主键id错误");
        }
        // 验证状态
        if(!entity.getBpmStatus().equals(SfaCommonEnum.dataBpmStatus.COMMIT.getValue()) &&
                !entity.getBpmStatus().equals(SfaCommonEnum.dataBpmStatus.APPROVAL.getValue())) {
            return Result.error("该申请审批状态错误");
        }
        // 已审核过修改状态
        if(!SfaCommonEnum.dataBpmStatus.APPROVAL.getValue().equals(entity.getBpmStatus())) {
            entity.setBpmStatus(SfaCommonEnum.dataBpmStatus.APPROVAL.getValue());
            saveOrUpdate(entity);
        }

        return Result.ok();
    }

    /**
     * 审核完成回调
     * @param reqVo
     */
    @Override
    public void auditFinish(SfaAuditFinishReqVo reqVo) {
        SfaTravelEntity entity = this.lambdaQuery().eq(SfaTravelEntity::getAuditTaskId, reqVo.getFormId()).one();
        // 处理审核成功
        if(ActivitiEnum.AuditResultType.PASS.getVal().equals(reqVo.getResult())) {
            entity.setBpmStatus(SfaCommonEnum.dataBpmStatus.PASS.getValue());
            entity.setPassStatusDate(CrmDateUtils.yyyyMMddHHmmss.format(LocalDateTime.now()));
        } else if (ActivitiEnum.AuditResultType.REJECT.getVal().equals(reqVo.getResult())){ // 审核驳回
            entity.setBpmStatus(SfaCommonEnum.dataBpmStatus.REJECT.getValue());
        } else if (ActivitiEnum.AuditResultType.RECOVER.getVal().equals(reqVo.getResult())){ // 审核追回
            entity.setBpmStatus(SfaCommonEnum.dataBpmStatus.ROLLBACK.getValue());
        } else {
            throw new BusinessException("审批失败");
        }
        saveOrUpdate(entity);
    }

    /**
     * 根据审核任务ID查询
     * @param auditTaskId
     * @return
     */
    @Override
    public SfaTravelRespVo queryByAuditTaskId(String auditTaskId) {
        SfaTravelReqVo reqVo = new SfaTravelReqVo();
        reqVo.setAuditTaskIdList(Arrays.asList(auditTaskId));
        PageResult<SfaTravelRespVo> result = findList(reqVo);
        if(result == null || result.getCount() != 1) {
            throw new BusinessException("审核任务ID错误");
        }
        return result.getData().get(0);
    }


//    /**
//     * 删除
//     * @param reqVo
//     * @return
//     */
//    @Override
//    @Transactional(rollbackFor = Exception.class)
//    public void deleteBatch(SfaTravelReqVo reqVo){
//        List<SfaTravelEntity> sfaTravelEntities = sfaTravelMapper.selectBatchIds(reqVo.getIds());
//        if(CollectionUtils.isNotEmpty(sfaTravelEntities)){
//            sfaTravelEntities.forEach(o -> {
//                 o.setDelFlag(CrmDelFlagEnum.DELETE.getCode());
//            });
//        }
//        this.updateBatchById(sfaTravelEntities);
//    }
//
//    /**
//     * 启用
//     * @param reqVo
//     * @return
//     */
//    @Override
//    @Transactional(rollbackFor = Exception.class)
//    public void enableBatch(SfaTravelReqVo reqVo){
//        //设置状态为启用
//        List<SfaTravelEntity> sfaTravelEntities = sfaTravelMapper.selectBatchIds(reqVo.getIds());
//        if(CollectionUtils.isNotEmpty(sfaTravelEntities)){
//            sfaTravelEntities.forEach(o -> {
//                o.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
//            });
//        }
//        this.updateBatchById(sfaTravelEntities);
//    }
//
//    /**
//     * 禁用
//     * @param reqVo
//     * @return
//     */
//    @Override
//    @Transactional(rollbackFor = Exception.class)
//    public void disableBatch(SfaTravelReqVo reqVo){
//        //设置状态为禁用
//        List<SfaTravelEntity> sfaTravelEntities = sfaTravelMapper.selectBatchIds(reqVo.getIds());
//        if(CollectionUtils.isNotEmpty(sfaTravelEntities)){
//                sfaTravelEntities.forEach(o -> {
//                o.setEnableStatus(CrmEnableStatusEnum.DISABLE.getCode());
//            });
//        }
//        this.updateBatchById(sfaTravelEntities);
//    }



    /**
     * 信息检查
     * @param reqVo
     */
    private void checkInfo(SfaTravelApplyReqVo reqVo) {
        AssertUtils.isNotEmpty(reqVo.getSfaTravelUserReqVos(), "出差人员不能为空");
        for(int r = 0; r < reqVo.getSfaTravelUserReqVos().size(); r ++) {
            SfaTravelUserReqVo vo  = reqVo.getSfaTravelUserReqVos().get(r);
            AssertUtils.isNotEmpty(vo.getUserName(), "第"+(r+1)+"组出差人员:人员帐号不能为空");
            AssertUtils.isNotEmpty(vo.getRealName(), "第"+(r+1)+"组出差人员:人员名称不能为空");
        }
        AssertDateTimeUtils.checkDateAndNull(reqVo.getBeginTime(), "请选择出差开始时间", "出差开始时间错误");
        AssertDateTimeUtils.checkDateAndNull(reqVo.getEndTime(), "请选择出差结束时间", "出差结束时间错误");
        AssertDateTimeUtils.notAfterDate(reqVo.getBeginTime(), reqVo.getEndTime(), "开始时间不能在结束时间之后");
        AssertUtils.isNotEmpty(reqVo.getAddress(), "请填写出差地点");
        AssertUtils.isNotEmpty(reqVo.getTravelObjective(), "请填写出差目的");

    }
}
