package com.biz.crm.worksignrule.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.aop.CrmLog;
import com.biz.crm.common.PageResult;
import com.biz.crm.eunm.CrmDelFlagEnum;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.eunm.sfa.SfaWorkSignEnum;
import com.biz.crm.nebular.sfa.worksignrule.req.SfaWorkSignPlaceReqVo;
import com.biz.crm.nebular.sfa.worksignrule.resp.SfaWorkSignPlaceRespVo;
import com.biz.crm.util.AssertUtils;
import com.biz.crm.util.CrmBeanUtil;
import com.biz.crm.util.StringUtils;
import com.biz.crm.worksignrule.mapper.SfaWorkSignPlaceMapper;
import com.biz.crm.worksignrule.model.SfaWorkSignPlaceEntity;
import com.biz.crm.worksignrule.service.ISfaWorkSignPlaceService;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 考勤规则-打卡地点 接口实现
 *
 * @author liuhongming
 * @date 2020-09-16 15:06:20
 */
@Slf4j
@Service
@Transactional
@ConditionalOnMissingBean(name="SfaWorkSignPlaceServiceExpandImpl")
public class SfaWorkSignPlaceServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<SfaWorkSignPlaceMapper, SfaWorkSignPlaceEntity> implements ISfaWorkSignPlaceService {

    @Resource
    private SfaWorkSignPlaceMapper sfaWorkSignPlaceMapper;

    /**
     * 列表
     * @param reqVo
     * @return
     */
    @Override
    @CrmLog
    public PageResult<SfaWorkSignPlaceRespVo> findList(SfaWorkSignPlaceReqVo reqVo){
        Page<SfaWorkSignPlaceRespVo> page = new Page<>(reqVo.getPageNum(), reqVo.getPageSize());
        List<SfaWorkSignPlaceRespVo> list = sfaWorkSignPlaceMapper.findList(page, reqVo);
        return PageResult.<SfaWorkSignPlaceRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }
    /**
     * 根据规则编码查询 Mapping 列表
     * @param ruleCodes
     * @return
     */
    @Override
    public Map<String, List<SfaWorkSignPlaceEntity>> selectMappingByRuleCodes(Set<String> ruleCodes) {
        if(org.springframework.util.CollectionUtils.isEmpty(ruleCodes)){
            return Maps.newHashMap();
        }
        return sfaWorkSignPlaceMapper.selectList(Wrappers.lambdaQuery(SfaWorkSignPlaceEntity.class)
                .in(SfaWorkSignPlaceEntity :: getRuleCode, ruleCodes)).stream().collect(Collectors.groupingBy(SfaWorkSignPlaceEntity :: getRuleCode));
    }
    /**
     * 查询
     * @param reqVo
     * @return sfaWorkSignPlaceRespVo
     */
    @Override
    @CrmLog
    public SfaWorkSignPlaceRespVo query(SfaWorkSignPlaceReqVo reqVo){
        List<SfaWorkSignPlaceRespVo> list = this.findList(reqVo).getData();
        if(CollectionUtils.isEmpty(list)){
            return new SfaWorkSignPlaceRespVo();
        }
        SfaWorkSignPlaceRespVo sfaWorkSignPersonnelRespVo = list.get(0);
        return sfaWorkSignPersonnelRespVo;
    }

    /**
     * 新增
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @CrmLog
    public void save(SfaWorkSignPlaceReqVo reqVo){
        SfaWorkSignPlaceEntity entity = CrmBeanUtil.copy(reqVo,SfaWorkSignPlaceEntity.class);
        this.save(entity);
    }

    /**
     * 更新
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @CrmLog
    public void update(SfaWorkSignPlaceReqVo reqVo){
        SfaWorkSignPlaceEntity entity = this.getById(reqVo.getId());
        this.updateById(entity);
    }

    /**
     * 删除
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @CrmLog
    public void deleteBatch(SfaWorkSignPlaceReqVo reqVo){
        List<SfaWorkSignPlaceEntity> sfaWorkSignPlaceEntities = sfaWorkSignPlaceMapper.selectBatchIds(reqVo.getIds());
        if(CollectionUtils.isNotEmpty(sfaWorkSignPlaceEntities)){
            sfaWorkSignPlaceEntities.forEach(o -> {
                 o.setDelFlag(CrmDelFlagEnum.DELETE.getCode());
            });
        }
        this.updateBatchById(sfaWorkSignPlaceEntities);
    }

    /**
     * 启用
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @CrmLog
    public void enableBatch(SfaWorkSignPlaceReqVo reqVo){
        //设置状态为启用
        List<SfaWorkSignPlaceEntity> sfaWorkSignPlaceEntities = sfaWorkSignPlaceMapper.selectBatchIds(reqVo.getIds());
        if(CollectionUtils.isNotEmpty(sfaWorkSignPlaceEntities)){
            sfaWorkSignPlaceEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
            });
        }
        this.updateBatchById(sfaWorkSignPlaceEntities);
    }

    /**
     * 禁用
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @CrmLog
    public void disableBatch(SfaWorkSignPlaceReqVo reqVo){
        //设置状态为禁用
        List<SfaWorkSignPlaceEntity> sfaWorkSignPlaceEntities = sfaWorkSignPlaceMapper.selectBatchIds(reqVo.getIds());
        if(CollectionUtils.isNotEmpty(sfaWorkSignPlaceEntities)){
                sfaWorkSignPlaceEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.DISABLE.getCode());
            });
        }
        this.updateBatchById(sfaWorkSignPlaceEntities);
    }

    /**
     * 新增,编辑,校验
     * @param placeReqVo
     */
    @Override
    @CrmLog
    public void saveCheck(SfaWorkSignPlaceReqVo placeReqVo){
        if(StringUtils.isEmpty(placeReqVo.getPlaceSignType())) { //默认上下班打卡
            placeReqVo.setPlaceSignType(SfaWorkSignEnum.WorkSignType.CLOCK_ALL.getVal());
        }
        AssertUtils.isNotEmpty(placeReqVo.getWspNo(), "打卡地点，序号不能为空");
        AssertUtils.isNotEmpty(placeReqVo.getPlaceName(), "序号"+placeReqVo.getWspNo()+"打卡地点，打卡地点名称不能为空");
        AssertUtils.isNotEmpty(placeReqVo.getPlaceLngin(),"序号"+placeReqVo.getWspNo()+"打卡地点，地点经度不能为空");
        AssertUtils.isNotEmpty(placeReqVo.getPlaceLatin(),"序号"+placeReqVo.getWspNo()+"打卡地点，地点纬度不能为空");
        AssertUtils.isNotEmpty(placeReqVo.getPlaceSignType(), "序号"+placeReqVo.getWspNo()+"打卡地点，打卡类型不能为空");
        AssertUtils.isNotEmpty(SfaWorkSignEnum.WorkSignType.getDesc(placeReqVo.getPlaceSignType()), "序号"+placeReqVo.getWspNo()+"打卡地点，打卡类型错误");
        AssertUtils.isNotEmpty(placeReqVo.getPlaceRange(),"序号"+placeReqVo.getWspNo()+"打卡地点，打卡范围不能为空");
    }

}
