package com.biz.crm.worksignrule.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.aop.CrmLog;
import com.biz.crm.base.ApiResultUtil;
import com.biz.crm.base.BusinessException;
import com.biz.crm.cache.util.SfaWorkSignRuleInfoUtil;
import com.biz.crm.cache.util.SfaWorkSignRuleUtil;
import com.biz.crm.calculatesalary.service.ISfaCalculateSalaryDateService;
import com.biz.crm.common.PageResult;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.eunm.sfa.SfaWorkSignEnum;
import com.biz.crm.eunm.sfa.WorkSignEnum;
import com.biz.crm.mdm.org.MdmOrgFeign;
import com.biz.crm.nebular.mdm.org.resp.MdmOrgRespVo;
import com.biz.crm.nebular.mdm.org.resp.MdmOrgTreeRespVo;
import com.biz.crm.nebular.mdm.org.resp.MdmOrgWithPositionRespVo;
import com.biz.crm.nebular.mdm.position.resp.MdmPositionRespVo;
import com.biz.crm.nebular.mdm.position.resp.MdmPositionUserOrgRespVo;
import com.biz.crm.nebular.sfa.worksignrule.SfaWorkSignRuleVo;
import com.biz.crm.nebular.sfa.worksignrule.req.*;
import com.biz.crm.nebular.sfa.worksignrule.resp.*;
import com.biz.crm.util.*;
import com.biz.crm.worksign.model.SfaWorkSignRecordEntity;
import com.biz.crm.worksign.model.SfaWorkSignRuleInfoEntity;
import com.biz.crm.worksign.service.ISfaWorkSignRecordService;
import com.biz.crm.worksign.service.ISfaWorkSignRuleInfoService;
import com.biz.crm.worksignrule.mapper.*;
import com.biz.crm.worksignrule.model.*;
import com.biz.crm.worksignrule.service.*;
import com.biz.crm.worksignrule.vo.ExecuteWorkSignRuleVo;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import lombok.Synchronized;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.time.LocalDate;
import java.time.LocalTime;
import java.util.*;
import java.util.concurrent.CountDownLatch;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

/**
 * 考勤规则;考勤规则接口实现
 *
 * @author liuhongming
 * @date 2020-09-16 14:50:03
 */
@Slf4j
@Service
@Transactional
@ConditionalOnMissingBean(name="SfaWorkSignRuleServiceExpandImpl")
public class SfaWorkSignRuleServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<SfaWorkSignRuleMapper, SfaWorkSignRuleEntity> implements ISfaWorkSignRuleService {

    @Resource
    private ISfaWorkSignPersonnelService iSfaWorkSignPersonnelService;
    @Resource
    private ISfaWorkSignPlaceService iSfaWorkSignPlaceService;
    @Resource
    private ISfaWorkSignTimeService iSfaWorkSignTimeService;
    @Resource
    private ISfaWorkSignSpecialService iSfaWorkSignSpecialService;
    @Resource
    private ISfaWorkSignRuleInfoService iSfaWorkSignRuleInfoService;
    @Resource
    private ISfaWorkSignRecordService iSfaWorkSignRecordService;
    @Resource
    private SfaWorkSignRuleMapper sfaWorkSignRuleMapper;
    @Resource
    private SfaWorkSignPersonnelMapper personnelMapper;
    @Resource
    private SfaWorkSignPlaceMapper placeMapper;
    @Resource
    private SfaWorkSignSpecialMapper specialMapper;
    @Resource
    private SfaWorkSignTimeMapper signTimeMapper;
    @Resource
    private MdmOrgFeign mdmOrgFeign;
    @Resource
    private ISfaCalculateSalaryDateService dateService;
    @Resource
    private SfaWorkSignRuleUtil sfaWorkSignRuleUtil;
    @Resource
    private SfaWorkSignRuleInfoUtil sfaWorkSignRuleInfoUtil;

    @Resource
    protected SignRuleResolveHelper signRuleResolveHelper;

    /**
     * 列表
     * @return
     */
    @Override
    @CrmLog
    public PageResult<SfaWorkSignRuleRespVo> findList(SfaWorkSignRuleVo sfaWorkSignRuleVo){
        Page<SfaWorkSignRuleVo> page = new Page<>(sfaWorkSignRuleVo.getPageNum(), sfaWorkSignRuleVo.getPageSize());
        List<SfaWorkSignRuleRespVo> list = sfaWorkSignRuleMapper.findList(page, sfaWorkSignRuleVo);
        return PageResult.<SfaWorkSignRuleRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    /**
     * 根据id查询详情
     * @param id
     * @return
     */
    @Override
    public SfaWorkSignRuleRespVo findDetailsById(String id) {
        if(StringUtils.isBlank(id)) {
            return null;
        }
        SfaWorkSignRuleEntity entity = this.sfaWorkSignRuleMapper.selectById(id);
        if(entity == null) {
            return null;
        }
        return sfaWorkSignRuleUtil.getObj(entity.getRuleCode());
    }

    /**
     * 查询
     * @param reqVo
     * @return sfaWorkSignRuleRespVo
     */
    @Override
    @CrmLog
    public SfaWorkSignRuleRespVo query(SfaWorkSignRuleReqVo reqVo){

        List<SfaWorkSignRuleRespVo> list = this.findList(reqVo).getData();
        if(CollectionUtils.isEmpty(list)){
            return new SfaWorkSignRuleRespVo();
        }
        SfaWorkSignRuleRespVo sfaWorkSignRuleRespVo = list.get(0);
        /**考勤规则-打卡人员*/
        SfaWorkSignPersonnelReqVo sfaWorkSignPersonnelReqVo = new SfaWorkSignPersonnelReqVo();
        sfaWorkSignPersonnelReqVo.setRuleCode(sfaWorkSignRuleRespVo.getRuleCode());
        sfaWorkSignPersonnelReqVo.setPageSize(-1);
        List<SfaWorkSignPersonnelEntity> personnelEntities = iSfaWorkSignPersonnelService.list(Wrappers.lambdaQuery(SfaWorkSignPersonnelEntity.class)
                .eq(SfaWorkSignPersonnelEntity::getRuleCode, sfaWorkSignRuleRespVo.getRuleCode()));
        List<SfaWorkSignPersonnelRespVo> sfaWorkSignPersonnelRespVos = CrmBeanUtil.copyList(personnelEntities, SfaWorkSignPersonnelRespVo.class);
        if (CollectionUtil.listNotEmpty(sfaWorkSignPersonnelRespVos)){
//            //打卡人员包含
//            List<SfaWorkSignPersonnelRespVo> sfaWorkSignPersonnelRespVo = new ArrayList<>();
//            //打卡人员非包含
//            List<SfaWorkSignPersonnelRespVo> personnelRespNonInclusionVos = new ArrayList<>();
//            for (SfaWorkSignPersonnelRespVo personnelRespVo:sfaWorkSignPersonnelRespVos){
////                if (SfaCommonEnum.rangeTypeEnum.One.getVal().equals(personnelRespVo.getRangeType())){
////                    personnelRespContainVos.add(personnelRespVo);
////                }else{
////                    personnelRespNonInclusionVos.add(personnelRespVo);
////                }
//            }
            //区分组织和职位级别 通过codeType
            List<SfaWorkSignPersonnelRespVo> org = sfaWorkSignPersonnelRespVos.stream().filter(data -> WorkSignEnum.codeType.ORG.getVal().equals(data.getCodeType())).collect(Collectors.toList());
            List<SfaWorkSignPersonnelRespVo> posLevel = sfaWorkSignPersonnelRespVos.stream().filter(data -> WorkSignEnum.codeType.POS_LEVEL.getVal().equals(data.getCodeType())).collect(Collectors.toList());
            sfaWorkSignRuleRespVo.setSfaWorkSignPersonnelRespVo(org);
            sfaWorkSignRuleRespVo.setSfaWorkSignPersonnelPosRespVo(posLevel);
        }
        /**考勤规则-打卡地点*/
        SfaWorkSignPlaceReqVo sfaWorkSignPlaceReqVo = new SfaWorkSignPlaceReqVo();
        sfaWorkSignPlaceReqVo.setRuleCode(sfaWorkSignRuleRespVo.getRuleCode());
        sfaWorkSignPlaceReqVo.setPageSize(-1);
        List<SfaWorkSignPlaceRespVo> sfaWorkSignPlaceRespVos =
                iSfaWorkSignPlaceService.findList(sfaWorkSignPlaceReqVo).getData();
        sfaWorkSignRuleRespVo.setSfaWorkSignPlaceRespVos(sfaWorkSignPlaceRespVos);
        /**考勤规则-打卡时间*/
        SfaWorkSignTimeReqVo sfaWorkSignTimeReqVo = new SfaWorkSignTimeReqVo();
        sfaWorkSignTimeReqVo.setRuleCode(sfaWorkSignRuleRespVo.getRuleCode());
        sfaWorkSignTimeReqVo.setPageSize(-1);
        List<SfaWorkSignTimeRespVo> sfaWorkSignTimeRespVos =
                iSfaWorkSignTimeService.findList(sfaWorkSignTimeReqVo).getData();
        sfaWorkSignRuleRespVo.setSfaWorkSignTimeRespVos(sfaWorkSignTimeRespVos);
        /**考勤规则-特殊日期*/
        SfaWorkSignSpecialReqVo sfaWorkSignSpecialReqVo = new SfaWorkSignSpecialReqVo();
        sfaWorkSignSpecialReqVo.setRuleCode(sfaWorkSignRuleRespVo.getRuleCode());
        List<SfaWorkSignSpecialRespVo> sfaWorkSignSpecialRespVos =
                iSfaWorkSignSpecialService.findNonPageList(sfaWorkSignSpecialReqVo);
        if (CollectionUtil.listNotEmpty(sfaWorkSignSpecialRespVos)){
            //必须打卡的日期
            List<SfaWorkSignSpecialRespVo> sfaWorkSignSpecialMustRespVos = new ArrayList<>();
            //不用打卡的日期
            List<SfaWorkSignSpecialRespVo> sfaWorkSignSpecialNotRespVos = new ArrayList<>();
            for (SfaWorkSignSpecialRespVo specialRespVos:sfaWorkSignSpecialRespVos){
                if(SfaWorkSignEnum.workSignSpecialType.One.getVal().equals(specialRespVos.getWssType())){
                    sfaWorkSignSpecialMustRespVos.add(specialRespVos);
                }else{
                    sfaWorkSignSpecialNotRespVos.add(specialRespVos);
                }
            }
            sfaWorkSignRuleRespVo.setSfaWorkSignSpecialMustRespVos(sfaWorkSignSpecialMustRespVos);
            sfaWorkSignRuleRespVo.setSfaWorkSignSpecialNotRespVos(sfaWorkSignSpecialNotRespVos);
        }
        return sfaWorkSignRuleRespVo;
    }

    /**
     * 新增
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @CrmLog
    @Synchronized
    public void save(SfaWorkSignRuleReqVo reqVo){
        this.saveCheck(reqVo);
        SfaWorkSignRuleEntity entity = CrmBeanUtil.copy(reqVo,SfaWorkSignRuleEntity.class);
        //TODO 规则编码
        String code = CodeUtil.getCodeDefault();
        entity.setRuleCode(code);
        this.save(entity);
        this.saveData(reqVo,entity);
    }

    /**
     * 保存业务明细数据
     * @param reqVo
     * @param entity
     */
    public void saveData(SfaWorkSignRuleReqVo reqVo,SfaWorkSignRuleEntity entity){
        //打卡人员-组织和职位的
        if (!org.springframework.util.CollectionUtils.isEmpty(reqVo.getSfaWorkSignPersonnelReqVo())){
            //存在职位级别就将职位级别List添加到组织的List中统一操作
            if(CollectionUtil.listNotEmpty(reqVo.getSfaWorkSignPersonnelPosReqVo())){
                reqVo.getSfaWorkSignPersonnelReqVo().addAll(reqVo.getSfaWorkSignPersonnelPosReqVo());
            }
            List<SfaWorkSignPersonnelEntity> personnelContainList = reqVo.getSfaWorkSignPersonnelReqVo().stream().map(o -> {
                SfaWorkSignPersonnelEntity personnelEntity = new SfaWorkSignPersonnelEntity();
                CrmBeanUtil.copyProperties(o, personnelEntity);
                personnelEntity.setId(null);
                personnelEntity.setRuleCode(entity.getRuleCode());
                return personnelEntity;
            }).collect(Collectors.toList());
            iSfaWorkSignPersonnelService.saveOrUpdateBatch(personnelContainList, 200);
        }
        //打卡地点
        if (CollectionUtil.listNotEmptyNotSizeZero(reqVo.getSfaWorkSignPlaceReqVos())){
            List<SfaWorkSignPlaceEntity> signPlaceEntityList = reqVo.getSfaWorkSignPlaceReqVos().stream().map(o -> {
                SfaWorkSignPlaceEntity signPlaceEntity = new SfaWorkSignPlaceEntity();
                CrmBeanUtil.copyProperties(o, signPlaceEntity);
                signPlaceEntity.setRuleCode(entity.getRuleCode());
                return signPlaceEntity;
            }).collect(Collectors.toList());
            iSfaWorkSignPlaceService.saveOrUpdateBatch(signPlaceEntityList);
        }
        //打卡时间
        if (CollectionUtil.listNotEmptyNotSizeZero(reqVo.getSfaWorkSignTimeReqVos())){
            List<SfaWorkSignTimeReqVo> sfaWorkSignTimeReqVos = reqVo.getSfaWorkSignTimeReqVos();
            List<SfaWorkSignTimeEntity> signTimeEntityList = Lists.newArrayList();
            Integer i = 1;
            for(SfaWorkSignTimeReqVo temp : sfaWorkSignTimeReqVos){
                SfaWorkSignTimeEntity signTimeEntity = CrmBeanUtil.copy(temp, SfaWorkSignTimeEntity.class);
                signTimeEntity.setRuleCode(entity.getRuleCode());
                signTimeEntity.setWstNo(i.toString());
                signTimeEntityList.add(signTimeEntity);
                i++;
            }
            iSfaWorkSignTimeService.saveOrUpdateBatch(signTimeEntityList);
        }
        //特殊日期--必须打卡的日期
        if (CollectionUtil.listNotEmptyNotSizeZero(reqVo.getSfaWorkSignSpecialMustReqVos())){
            List<SfaWorkSignSpecialEntity> signSpecialEntityList = reqVo.getSfaWorkSignSpecialMustReqVos().stream().map(o -> {
                SfaWorkSignSpecialEntity signSpecialEntity = new SfaWorkSignSpecialEntity();
                CrmBeanUtil.copyProperties(o, signSpecialEntity);
                signSpecialEntity.setWssType(SfaWorkSignEnum.workSignSpecialType.One.getVal());
                signSpecialEntity.setRuleCode(entity.getRuleCode());
                return signSpecialEntity;
            }).collect(Collectors.toList());
            iSfaWorkSignSpecialService.saveOrUpdateBatch(signSpecialEntityList);
        }
        //特殊日期--不用打卡的日期
        if (CollectionUtil.listNotEmptyNotSizeZero(reqVo.getSfaWorkSignSpecialNotReqVos())){
            List<SfaWorkSignSpecialEntity> specialEntityList = reqVo.getSfaWorkSignSpecialNotReqVos().stream().map(o -> {
                SfaWorkSignSpecialEntity specialEntity = new SfaWorkSignSpecialEntity();
                CrmBeanUtil.copyProperties(o, specialEntity);
                specialEntity.setWssType(SfaWorkSignEnum.workSignSpecialType.Two.getVal());
                specialEntity.setRuleCode(entity.getRuleCode());
                return specialEntity;
            }).collect(Collectors.toList());
            iSfaWorkSignSpecialService.saveOrUpdateBatch(specialEntityList);
        }
    }


    /**
     * 更新
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @CrmLog
    public void update(SfaWorkSignRuleReqVo reqVo){
        if(StringUtils.isEmpty(reqVo.getId())){
            throw new BusinessException("数据主键不能为空");
        }
        SfaWorkSignRuleEntity entity = this.getById(reqVo.getId());
        if(Objects.isNull(entity)){
            throw new BusinessException("考勤规则数据不存在");
        }
        reqVo.setRuleCode(entity.getRuleCode());
        this.saveCheck(reqVo);
        SfaWorkSignRuleEntity updateEntity = CrmBeanUtil.copy(reqVo, SfaWorkSignRuleEntity.class);
        updateEntity.setRuleCode(null);
        this.updateById(updateEntity);
        //业务数据
        reqVo.setRuleCode(entity.getRuleCode());
        // 删除历史数据
        this.deleteBatchDate(reqVo);
        // 保存其他级联数据
        this.saveData(reqVo,entity);
        //删除缓存
        sfaWorkSignRuleUtil.deleteKey(entity.getRuleCode());
    }

    public void deleteBatchDate(SfaWorkSignRuleReqVo reqVo){
        if(StringUtils.isNotEmpty(reqVo.getRuleCode())||CollectionUtil.listNotEmptyNotSizeZero(reqVo.getRuleCodes())){
            //打卡人员
            SfaWorkSignPersonnelReqVo signPersonnelReqVo = new SfaWorkSignPersonnelReqVo();
            signPersonnelReqVo.setRuleCode(reqVo.getRuleCode());
            signPersonnelReqVo.setRuleCodes(reqVo.getRuleCodes());
            personnelMapper.deleteProductsByParams(signPersonnelReqVo);
            //打卡地点
            SfaWorkSignPlaceReqVo signPlaceReqVo = new SfaWorkSignPlaceReqVo();
            signPlaceReqVo.setRuleCode(reqVo.getRuleCode());
            signPlaceReqVo.setRuleCodes(reqVo.getRuleCodes());
            placeMapper.deleteProductsByParams(signPlaceReqVo);
            //打卡时间
            SfaWorkSignTimeReqVo signTimeReqVo = new SfaWorkSignTimeReqVo();
            signTimeReqVo.setRuleCode(reqVo.getRuleCode());
            signTimeReqVo.setRuleCodes(reqVo.getRuleCodes());
            signTimeMapper.deleteProductsByParams(signTimeReqVo);
            //特殊日期
            SfaWorkSignSpecialReqVo signSpecialReqVo = new SfaWorkSignSpecialReqVo();
            signSpecialReqVo.setRuleCode(reqVo.getRuleCode());
            signSpecialReqVo.setRuleCodes(reqVo.getRuleCodes());
            specialMapper.deleteProductsByParams(signSpecialReqVo);
        }
    }

    /**
     * 删除
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @CrmLog
    public void deleteBatch(SfaWorkSignRuleReqVo reqVo){
        if(StringUtils.isEmpty(reqVo.getId())&&CollectionUtil.listEmpty(reqVo.getIds())){
            throw new BusinessException("数据主键不能为空");
        }
        reqVo.setPageSize(-1);
        List<SfaWorkSignRuleRespVo> list = this.findList(reqVo).getData();
        if(CollectionUtil.listEmpty(list)){
            throw new BusinessException("考勤规则数据不存在");
        }
        if(list.size() != reqVo.getIds().size()) {
            throw new BusinessException("主键ID错误，请检查");
        }
        // 检查是否有考勤打卡数据生成
        List<String> result = sfaWorkSignRuleMapper.findUseRuleCodeByIds(reqVo.getIds());
        if(result != null && result.size() > 0) {
            throw new BusinessException("规则："+ result.get(0) + " 已生成考勤数据");
        }

        sfaWorkSignRuleMapper.deleteProductsByParams(reqVo);
        List<String> ruleCodes = list.stream().map(SfaWorkSignRuleRespVo:: getRuleCode).collect(Collectors.toList());
        reqVo.setRuleCodes(ruleCodes);
        this.deleteBatchDate(reqVo);
        //删除缓存
        ruleCodes.forEach(code -> sfaWorkSignRuleUtil.deleteKey(code));
    }

    /**
     * 启用
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @CrmLog
    public void enableBatch(SfaWorkSignRuleReqVo reqVo){
        if(StringUtils.isEmpty(reqVo.getId())&&CollectionUtil.listEmpty(reqVo.getIds())){
            throw new BusinessException("请选择将启用信息");
        }
        if (StringUtils.isNotEmpty(reqVo.getId())){
            List<String> ids = new ArrayList<>();
            if (CollectionUtil.listNotEmptyNotSizeZero(reqVo.getIds())){
                ids = reqVo.getIds();
            }
            ids.add(reqVo.getId());
            reqVo.setIds(ids);
        }
        //设置状态为启用
        List<SfaWorkSignRuleEntity> sfaWorkSignRuleEntities = sfaWorkSignRuleMapper.selectBatchIds(reqVo.getIds());
        if(CollectionUtils.isNotEmpty(sfaWorkSignRuleEntities)){
            if(sfaWorkSignRuleEntities.size() != reqVo.getIds().size()) {
                throw new BusinessException("主键ID错误，请检查");
            }

            sfaWorkSignRuleEntities.forEach(o -> {
                // 检查组织冲突。1、查询当前规则中的组织，2、验证是否冲突
//                List<SfaWorkSignPersonnelRespVo> list = iSfaWorkSignPersonnelService.findListJoinRule(null, null, o.getRuleCode(), null);
                List<SfaWorkSignPersonnelEntity> personnelEntities = iSfaWorkSignPersonnelService.list(Wrappers.lambdaQuery(SfaWorkSignPersonnelEntity.class).eq(SfaWorkSignPersonnelEntity::getRuleCode, o.getRuleCode()));
                if(CollectionUtil.listEmpty(personnelEntities)) {
                    throw new BusinessException("该规则未配置打卡组织:"+o.getRuleCode());
                }
                List<SfaWorkSignPersonnelReqVo> signPersonnelReqVos = CrmBeanUtil.copyList(personnelEntities, SfaWorkSignPersonnelReqVo.class);
                List<SfaWorkSignPersonnelReqVo> orgReqVos = signPersonnelReqVos.stream().filter(data -> WorkSignEnum.codeType.ORG.getVal().equals(data.getCodeType())).collect(Collectors.toList());
                List<SfaWorkSignPersonnelReqVo> posReqVos = signPersonnelReqVos.stream().filter(data -> WorkSignEnum.codeType.POS_LEVEL.getVal().equals(data.getCodeType())).collect(Collectors.toList());
                SfaWorkSignRuleReqVo sfaWorkSignRuleReqVo = CrmBeanUtil.copy(o, SfaWorkSignRuleReqVo.class);
                sfaWorkSignRuleReqVo.setSfaWorkSignPersonnelReqVo(orgReqVos);
                sfaWorkSignRuleReqVo.setSfaWorkSignPersonnelPosReqVo(posReqVos);
                checkSignRange(sfaWorkSignRuleReqVo.getSfaWorkSignPersonnelReqVo(),sfaWorkSignRuleReqVo);
//                list = iSfaWorkSignPersonnelService.findListJoinRule(list.stream().map(SfaWorkSignPersonnelRespVo::getWspCode).collect(Collectors.toList())
//                        , o.getRuleCode(), null, CrmEnableStatusEnum.ENABLE.getCode());
//                if(CollectionUtil.listNotEmpty(list)) {
//                    throw new BusinessException("组织重复添加：编码" + list.get(0).getWspCode() + "-规则" + list.get(0).getRuleCode());
//                }

                o.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
            });
        }
        this.updateBatchById(sfaWorkSignRuleEntities);
        //删除缓存
        if(CollectionUtil.listNotEmpty(sfaWorkSignRuleEntities)) {
            sfaWorkSignRuleEntities.forEach(entity -> sfaWorkSignRuleUtil.deleteKey(entity.getRuleCode()));
        }
    }

    /**
     * 禁用
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @CrmLog
    public void disableBatch(SfaWorkSignRuleReqVo reqVo){
        if(StringUtils.isEmpty(reqVo.getId())&&CollectionUtil.listEmpty(reqVo.getIds())){
            throw new BusinessException("数据主键不能为空");
        }
        if (StringUtils.isNotEmpty(reqVo.getId())){
            List<String> ids = new ArrayList<>();
            if (CollectionUtil.listNotEmptyNotSizeZero(reqVo.getIds())){
                ids = reqVo.getIds();
            }
            ids.add(reqVo.getId());
            reqVo.setIds(ids);
        }
        //设置状态为禁用
        List<SfaWorkSignRuleEntity> sfaWorkSignRuleEntities = sfaWorkSignRuleMapper.selectBatchIds(reqVo.getIds());
        if(sfaWorkSignRuleEntities.size() != reqVo.getIds().size()) {
            throw new BusinessException("主键ID错误，请检查");
        }
        if(CollectionUtils.isNotEmpty(sfaWorkSignRuleEntities)){
            sfaWorkSignRuleEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.DISABLE.getCode());
            });
        }
        this.updateBatchById(sfaWorkSignRuleEntities);
        //这些规则被禁用后需要删除当天已经生成的对应规则的打卡明细和规则明细
        List<String> ruleCodes = sfaWorkSignRuleEntities.stream().map(SfaWorkSignRuleEntity::getRuleCode).collect(Collectors.toList());
        String nowDate = LocalDate.now().format(CrmDateUtils.yyyyMMdd);
        iSfaWorkSignRecordService.remove(Wrappers.lambdaQuery(SfaWorkSignRecordEntity.class)
                .eq(SfaWorkSignRecordEntity::getRuleDate,nowDate)
                .in(SfaWorkSignRecordEntity::getRuleCode,ruleCodes));
        iSfaWorkSignRuleInfoService.remove(Wrappers.lambdaQuery(SfaWorkSignRuleInfoEntity.class)
                .eq(SfaWorkSignRuleInfoEntity::getRuleDate,nowDate)
                .in(SfaWorkSignRuleInfoEntity::getRuleCode,ruleCodes));
        //删除缓存
        if(CollectionUtil.listNotEmpty(sfaWorkSignRuleEntities)) {
            sfaWorkSignRuleEntities.forEach(entity -> this.sfaWorkSignRuleUtil.deleteKey(entity.getRuleCode()));
        }
    }

    /**
     * 设置默认值
     * @param reqVo
     */
    private void defCheck(SfaWorkSignRuleReqVo reqVo){
        // 规则类型
        if(StringUtils.isBlank(reqVo.getRuleType())){
            reqVo.setRuleType(SfaWorkSignEnum.WorkSignRuleType.STATIC.getVal());
        }
        //电子围栏
        if(StringUtils.isBlank(reqVo.getElectronFence())){
            reqVo.setElectronFence(SfaWorkSignEnum.ElectronFenceEnum.NONE.getVal());
        }
        //规则生效日期
        if(StringUtils.isBlank(reqVo.getRuleEffective())){
            reqVo.setRuleEffective(SfaWorkSignEnum.RuleEffective.TOMORROW.getVal());
        }
        //节假日
        if (!YesNoEnum.yesNoEnum.YES.getValue().equals(reqVo.getHolidayWhether())){
            reqVo.setHolidayWhether(YesNoEnum.yesNoEnum.NO.getValue());
        }
        //拍照打卡
        if (!YesNoEnum.yesNoEnum.YES.getValue().equals(reqVo.getWsrPhotograph())){
            reqVo.setWsrPhotograph(YesNoEnum.yesNoEnum.NO.getValue());
        }
        //拍照打卡
        if (!YesNoEnum.yesNoEnum.YES.getValue().equals(reqVo.getWsrPhotograph())){
            reqVo.setWsrPhotograph(YesNoEnum.yesNoEnum.NO.getValue());
        }
        //非工作日是否对打卡进行时间范围规则限制（Y/N 默认N）
        if (!YesNoEnum.yesNoEnum.YES.getValue().equals(reqVo.getNonWorkingDaySignAstrict())) {
            reqVo.setNonWorkingDaySignAstrict(YesNoEnum.yesNoEnum.NO.getValue());
        }
    }
    /**
     * 新增,编辑,校验
     * @param reqVo
     */
    public void saveCheck(SfaWorkSignRuleReqVo reqVo){
        //有默认填充值的字段
        this.defCheck(reqVo);
        //校验数据不为空
        AssertUtils.isNotEmpty(reqVo.getRuleName(),"请输入规则名称");
        AssertUtils.isNotEmpty(reqVo.getRuleType(), "请选择规则类型");
        //校验打卡地点
        this.checkPlace(reqVo.getElectronFence(), reqVo.getSfaWorkSignPlaceReqVos());
        //校验特殊日期配置
        this.checkSpecialDate(reqVo.getSfaWorkSignSpecialMustReqVos());
        this.checkSpecialDate(reqVo.getSfaWorkSignSpecialNotReqVos());
        //校验打卡范围(组织)配置
        this.checkSignRange(reqVo.getSfaWorkSignPersonnelReqVo(), reqVo);
        // 校验打卡时间
        this.checkSignTime(reqVo);

        //工作日
        AssertUtils.isNotEmpty(reqVo.getWorkingDay(),"请选择工作日");
        //打卡必填验证
        AssertUtils.isNotEmpty(reqVo.getGooffWorkSignAstrict(),"请选择下班打卡限制");
        this.checkDataExist(reqVo);



        //默认为打卡人员设置codeType
        reqVo.getSfaWorkSignPersonnelReqVo().forEach(data->data.setCodeType(WorkSignEnum.codeType.ORG.getVal()));
        if(CollectionUtil.listNotEmpty(reqVo.getSfaWorkSignPersonnelPosReqVo())){
            reqVo.getSfaWorkSignPersonnelPosReqVo().forEach(data->data.setCodeType(WorkSignEnum.codeType.POS_LEVEL.getVal()));
        }
    }

    /**
     * 校验打卡地点
     * @param electronFence 电子围栏(NONE:无电子围栏/OUT_SIGN_EX:允许范围外打卡，地点记录为异常/OUT_SIGN_OK:允许范围外打卡，地点记录为正常/NO_OUT_SIGN:不允许范围外打卡)
     * @param placeReqVos 打卡地点
     */
    private void checkPlace(String electronFence, List<SfaWorkSignPlaceReqVo> placeReqVos) {
        // 校验打卡地点数据
        if (CollectionUtil.listNotEmpty(placeReqVos)){
            int p = 1;
            for (SfaWorkSignPlaceReqVo placeReqVo : placeReqVos){
                AssertUtils.isNotEmpty(placeReqVo.getWspNo(), "第" + p + "条打卡地点，序号不能为空");
                iSfaWorkSignPlaceService.saveCheck(placeReqVo);
                p++;
            }
        }
        // 设置了电子围栏信息时
        if(!SfaWorkSignEnum.ElectronFenceEnum.NONE.getVal().equals(electronFence)) {
            // 地点不能为空
            AssertUtils.isNotEmpty(placeReqVos, "请添加打卡地点");
            // 地点必须包含上下班打卡(上下班打卡||上班签到+下班签退)
            boolean nonClockAll = false, nonClockIn = false, nonClockOut = false;
            for(SfaWorkSignPlaceReqVo vo : placeReqVos) {
                if(SfaWorkSignEnum.WorkSignType.CLOCK_ALL.getVal().equals(vo.getPlaceSignType())) {
                    nonClockAll = true;
                } else if(SfaWorkSignEnum.WorkSignType.CLOCK_IN.getVal().equals(vo.getPlaceSignType())) {
                    nonClockIn = true;
                } else if(SfaWorkSignEnum.WorkSignType.CLOCK_OUT.getVal().equals(vo.getPlaceSignType())) {
                    nonClockOut = true;
                }
            }
            if(!(nonClockAll || (nonClockIn && nonClockOut))) {
                throw new BusinessException("打卡地点必须包含上下班打卡！");
            }
        }
    }

    /**
     * 校验打卡时间
     * @param reqVo
     */
    private void checkSignTime(SfaWorkSignRuleReqVo reqVo) {
        // 固定上下班时间
        if(SfaWorkSignEnum.WorkSignRuleType.STATIC.getVal().equals(reqVo.getRuleType())){
            AssertUtils.isNotEmpty(reqVo.getSfaWorkSignTimeReqVos(), "请选择打卡时间");
        } else if(SfaWorkSignEnum.WorkSignRuleType.FREE_TIME.getVal().equals(reqVo.getRuleType())) {
            // 自由上下班打卡时，默认值设置
            SfaWorkSignTimeReqVo vo = new SfaWorkSignTimeReqVo();
            vo.setGotoTime(CrmDateUtils.TIME_STR_00);
            vo.setGooffTime(CrmDateUtils.TIME_STR_235959);
            vo.setGotoStartTime(CrmDateUtils.TIME_STR_00);
            vo.setGotoEndTime(CrmDateUtils.TIME_STR_235959);
            vo.setGooffStartTime(CrmDateUtils.TIME_STR_00);
            vo.setGooffEndTime(CrmDateUtils.TIME_STR_235959);
            List<SfaWorkSignTimeReqVo> sfaWorkSignTimeReqVos = reqVo.getSfaWorkSignTimeReqVos();
            if(!org.springframework.util.CollectionUtils.isEmpty(sfaWorkSignTimeReqVos)){
                vo.setId(sfaWorkSignTimeReqVos.get(0).getId());
            }
            reqVo.setSfaWorkSignTimeReqVos(Arrays.asList(vo));
        }

        List<SfaWorkSignTimeReqVo> sfaWorkSignTimeReqVos = reqVo.getSfaWorkSignTimeReqVos();
        AssertUtils.isNotNull(sfaWorkSignTimeReqVos, "请配置打卡时间");
        // 校验多段时间的，开始和结束不能重合
        String endTime = null;
        int i = 1;
        for (SfaWorkSignTimeReqVo timeReqVo : sfaWorkSignTimeReqVos) {
            iSfaWorkSignTimeService.saveCheck(timeReqVo);
            if(StringUtils.isEmpty(endTime)) {
                endTime = timeReqVo.getGooffEndTime();
            } else {
                if(LocalTime.parse(endTime).compareTo(LocalTime.parse(timeReqVo.getGotoStartTime())) > 0) {
                    throw new BusinessException("<第"+(i-1)+">段打卡下班结束时间不能早于<第"+i+">段打卡上班开始时间");
                } else {
                    endTime = timeReqVo.getGooffEndTime();
                }
            }
            i ++;
        }
    }

    /**
     * 校验打卡范围配置
     * @param sfaWorkSignPersonnelReqVos
     */
    private void checkSignRange(List<SfaWorkSignPersonnelReqVo> sfaWorkSignPersonnelReqVos, SfaWorkSignRuleReqVo reqVo){
        AssertUtils.isNotEmpty(sfaWorkSignPersonnelReqVos, "请添加打卡组织");
        List<String> wspCodes = new ArrayList<>();
        for(SfaWorkSignPersonnelReqVo vo : sfaWorkSignPersonnelReqVos){
            AssertUtils.isNotEmpty(vo.getWspCode(), "组织编码不能为空");
            AssertUtils.isNotEmpty(vo.getWspName(), "组织名称不能为空");
            AssertUtils.isNotEmpty(vo.getCodeType(), "编码类型不能为空");
//            AssertUtils.isNotEmpty(vo.getCurrentAble(), "请指定组织是否当前组织");
            vo.setCurrentAble(YesNoEnum.yesNoEnum.NO.getValue()); // 丰谷默认为当前及下级组织
            wspCodes.add(vo.getWspCode());
        }
        List<SfaWorkSignPersonnelReqVo> personnelPosReqVo = reqVo.getSfaWorkSignPersonnelPosReqVo();
        //表的该字段不能为空，故添加
        personnelPosReqVo.forEach(data->data.setCurrentAble(YesNoEnum.yesNoEnum.NO.getValue()));
        //组织是否在其他规则中被选择
        List<String> stringsOrg = reqVo.getSfaWorkSignPersonnelReqVo().stream().map(SfaWorkSignPersonnelReqVo::getWspCode).collect(Collectors.toList());
        List<SfaWorkSignPersonnelEntity> listOrg = Lists.newArrayList();
        List<List<String>> partition = Lists.partition(stringsOrg, 999);
        for (List<String> strings : partition) {
            listOrg.addAll(iSfaWorkSignPersonnelService.list(Wrappers.lambdaQuery(SfaWorkSignPersonnelEntity.class)
                    .select(SfaWorkSignPersonnelEntity :: getId, SfaWorkSignPersonnelEntity :: getRuleCode)
                    .in(SfaWorkSignPersonnelEntity::getWspCode, strings)));
        }

        if(CollectionUtil.listEmpty(listOrg)){
            //为空说明传入的组织没有被规则覆盖，可以直接跳过下面关于组织下的职位校验
            return;
        }
        Set<String> stringsRuleCode = listOrg.stream().map(SfaWorkSignPersonnelEntity::getRuleCode).collect(Collectors.toSet());
        //筛出没有被禁用的规则
        LambdaQueryWrapper<SfaWorkSignRuleEntity> queryWrapper = Wrappers.lambdaQuery(SfaWorkSignRuleEntity.class);
        //若是更新则同时去掉被更新规则
        if(com.biz.crm.util.StringUtils.isNotEmpty(reqVo.getId())){
            queryWrapper.ne(SfaWorkSignRuleEntity::getId,reqVo.getId());
        }
        List<SfaWorkSignRuleEntity> ruleEntities = this.list(queryWrapper.eq(SfaWorkSignRuleEntity::getEnableStatus, CrmEnableStatusEnum.ENABLE.getCode())
                .in(SfaWorkSignRuleEntity::getRuleCode, stringsRuleCode));
        if(CollectionUtil.listEmpty(ruleEntities)){
            //筛出没有被禁用的规则后查询为空、说明传入的组织没有被启用的规则覆盖，可以直接跳过下面关于组织下的职位校验
            return;
        }
        Set<String> stringsRuleCodeNew = ruleEntities.stream().map(SfaWorkSignRuleEntity::getRuleCode).collect(Collectors.toSet());
        //查询相同规则编码中被覆盖的职位级别
        if(CollectionUtil.listNotEmpty(reqVo.getSfaWorkSignPersonnelPosReqVo())){
            List<String> stringsPos = reqVo.getSfaWorkSignPersonnelPosReqVo().stream().map(SfaWorkSignPersonnelReqVo::getWspCode).collect(Collectors.toList());
            List<SfaWorkSignPersonnelEntity> personnelEntities = Lists.newArrayList();
            List<List<String>> partitionPos = Lists.partition(stringsPos, 999);
            for (List<String> strings : partitionPos) {
                personnelEntities.addAll(iSfaWorkSignPersonnelService.list(Wrappers.lambdaQuery(SfaWorkSignPersonnelEntity.class)
                        .select(SfaWorkSignPersonnelEntity :: getId, SfaWorkSignPersonnelEntity :: getRuleCode)
                        .in(SfaWorkSignPersonnelEntity::getWspCode, strings)
                        .eq(SfaWorkSignPersonnelEntity::getCodeType, WorkSignEnum.codeType.POS_LEVEL.getVal())
                        .in(SfaWorkSignPersonnelEntity::getRuleCode, stringsRuleCodeNew)));
            }
            if(CollectionUtil.listNotEmpty(personnelEntities)){
                throw new BusinessException("职位重复添加:规则冲突—— 规则编码："+personnelEntities.get(0).getRuleCode());
            }
        } else {
            // 组织是否重复添加
            Set<String> ruleCodeCollect = ruleEntities.stream().map(SfaWorkSignRuleEntity::getRuleCode).collect(Collectors.toSet());
            List<SfaWorkSignPersonnelEntity> personnelEntities = iSfaWorkSignPersonnelService.list(Wrappers.lambdaQuery(SfaWorkSignPersonnelEntity.class)
                    .in(SfaWorkSignPersonnelEntity::getRuleCode, ruleCodeCollect)
                    .eq(SfaWorkSignPersonnelEntity::getCodeType, WorkSignEnum.codeType.POS_LEVEL.getVal()));
            Set<String> posLevelString = personnelEntities.stream().map(SfaWorkSignPersonnelEntity::getRuleCode).collect(Collectors.toSet());
            List<String> collect = ruleCodeCollect.stream().filter(data -> !posLevelString.contains(data)).collect(Collectors.toList());
            //2021-12-16 修改（允许同个组织配置多个，但时间不能冲突）
            List<SfaWorkSignRuleEntity> lists = ruleEntities.stream().filter(data -> collect.contains(data.getRuleCode())).collect(Collectors.toList());
            lists.stream().forEach(data ->{
                if(data.getWorkingDay().equals(reqVo.getWorkingDay())) {
                    throw new BusinessException("组织冲突的规则编码：" + data.getRuleCode() +"已经存在");
                }
            });
        }
    }
    /**
     * 校验特殊日期配置
     * @param sfaWorkSignSpecialReqVos
     */
    private void checkSpecialDate(List<SfaWorkSignSpecialReqVo> sfaWorkSignSpecialReqVos){
        if(org.springframework.util.CollectionUtils.isEmpty(sfaWorkSignSpecialReqVos)){
            return;
        }
        for(SfaWorkSignSpecialReqVo sfaWorkSignSpecialReqVo : sfaWorkSignSpecialReqVos){
            if(StringUtils.isBlank(sfaWorkSignSpecialReqVo.getWssDate())){
                throw new BusinessException("请选择特殊日期");
            }
            try {
                LocalDate.parse(sfaWorkSignSpecialReqVo.getWssDate());
            }catch (Exception e){
                throw new BusinessException("非法的特殊日期格式，可用的格式：[yyyy-MM-dd]", e);
            }
        }
    }

    /**
     * 校验数据是否已经存在
     * @param reqVo
     * @return
     */
    public void checkDataExist(SfaWorkSignRuleReqVo reqVo ) {
        LambdaQueryWrapper<SfaWorkSignRuleEntity> wrapper = new LambdaQueryWrapper<SfaWorkSignRuleEntity>().eq(SfaWorkSignRuleEntity::getRuleName,reqVo.getRuleName());
        if(StringUtils.isNotEmpty(reqVo.getId())){
            wrapper.ne(SfaWorkSignRuleEntity::getId,reqVo.getId());
        }
        SfaWorkSignRuleEntity entity = sfaWorkSignRuleMapper.selectOne(wrapper);
        if(null != entity){
            throw new BusinessException("已存在考勤规则[" + reqVo.getRuleName() + "]");
        }
    }


    /**
     * 执行考勤规则，生成指定日期的明细
     *
     * @param reqVo
     */
    @Override
    @Transactional
    public void executeWorkSignRule(SfaWorkSignExecuteReqVo reqVo) {
        // 查询全部已生效的考勤规则
        LambdaQueryWrapper<SfaWorkSignRuleEntity> wrappers = Wrappers.lambdaQuery(SfaWorkSignRuleEntity.class);
        wrappers = wrappers.eq(SfaWorkSignRuleEntity :: getEnableStatus, CrmEnableStatusEnum.ENABLE.getCode())
                .eq(SfaWorkSignRuleEntity :: getDelFlag, CrmEnableStatusEnum.ENABLE.getCode());
        List<SfaWorkSignRuleEntity> rules = sfaWorkSignRuleMapper.selectList(wrappers);
        if(org.springframework.util.CollectionUtils.isEmpty(rules)){
            log.warn("执行考勤规则:无可执行的规则");
            throw new BusinessException("无可执行的规则");
        }
        this.resolveSignRule(reqVo, rules);
    }

    /**
     * 解析规则
     * @param reqVoTemp
     * @param rules
     */
    protected void resolveSignRule(final SfaWorkSignExecuteReqVo reqVoTemp, final List<SfaWorkSignRuleEntity> rules){

        ExecuteSignRuleContext contextOfBase = new ExecuteSignRuleContext(rules);
        LocalDate localDate = this.buildBaseContext(contextOfBase, reqVoTemp);
        List<Thread> resolveThread = Lists.newArrayList();
        List<ExecuteSignRuleContext> contextOfThread = Lists.newArrayList();
        CountDownLatch countDownLatch = new CountDownLatch(SfaWorkSignRuleEntity.RULE_RESOLVE_DAYS);
        for(int i = 0; i < SfaWorkSignRuleEntity.RULE_RESOLVE_DAYS; i ++){
            localDate = localDate.plusDays(1);
            final String executeDate = localDate.format(CrmDateUtils.yyyyMMdd);
            SfaWorkSignExecuteReqVo reqVo = CrmBeanUtil.copy(reqVoTemp, SfaWorkSignExecuteReqVo.class);
            reqVo.setExecuteDate(executeDate);
            //拷贝上下文（只拷贝必要前置数据）
            ExecuteSignRuleContext context = contextOfBase.cloneContext();
            context.setSignDate(executeDate);
            contextOfThread.add(context);
            resolveThread.add(new Thread(() -> {
                try {
                    log.warn("###############################" + Thread.currentThread().getName() + "：开始解析规则 ###############################");
                    UserUtils.setToken(reqVo.getToken());
                    //解析
                    this.resolveInfoAndRecord(context);
                    log.warn("###############################" + Thread.currentThread().getName() + "：结束解析规则 ###############################");
                }catch (Exception e){
                    context.setStatus(false);
                    log.error(Thread.currentThread().getName() + "：解析失败！", e);
                    throw e;
                }finally{
                    countDownLatch.countDown();
                }
            }, "解析[" + executeDate + "]"));
        }
        resolveThread.forEach(thread -> thread.start());

        this.saveSignRuleInfo(contextOfBase, contextOfThread, countDownLatch);

    }

    protected LocalDate buildBaseContext(ExecuteSignRuleContext contextOfBase, final SfaWorkSignExecuteReqVo reqVoTemp){
        String token = UserUtils.getToken();
        if(StringUtils.isBlank(token)){
            UserUtils.doTokenForNull();
            token = UserUtils.getToken();
        }
        reqVoTemp.setToken(token);
        String executeDateTemp = reqVoTemp.getExecuteDate();
        if(StringUtils.isBlank(executeDateTemp)){
            executeDateTemp = LocalDate.now().format(CrmDateUtils.yyyyMMdd);
        }
        CrmDateUtils.checkTimeFormatThrows(executeDateTemp, CrmDateUtils.yyyyMMdd, "创建考勤明细解析线程失败，非法的日期格式！");
        LocalDate localDate = LocalDate.parse(executeDateTemp, CrmDateUtils.yyyyMMdd);
        localDate = localDate.plusDays(-1);//先减一天

        contextOfBase.setSignDate(executeDateTemp);

        //加载所需要的前置数据到上下文对象
        this.loadContextData(contextOfBase);
        return localDate;
    }

    protected void saveSignRuleInfo(ExecuteSignRuleContext contextOfBase, List<ExecuteSignRuleContext> contextOfThread, CountDownLatch countDownLatch){
        try {
            countDownLatch.await();
            List<SfaWorkSignRuleInfoEntity> ruleInfoEntities = Lists.newArrayList();
            List<SfaWorkSignRecordEntity> recordEntities = Lists.newArrayList();
            contextOfThread.forEach(v -> {
                if(v.getStatus()){
                    ruleInfoEntities.addAll(v.getRuleInfoEntities());
                    recordEntities.addAll(v.getRecordEntities());
                }

            });
            contextOfBase.setRuleInfoEntities(ruleInfoEntities);
            contextOfBase.setRecordEntities(recordEntities);
            // 保存信息
            signRuleResolveHelper.saveSignRuleInfo(contextOfBase.getSignDate(), contextOfBase);
        } catch (InterruptedException e) {
            log.error("保存失败！", e);
            throw new BusinessException("保存失败!", e);
        }
    }

    @Override
    @Transactional
    public void executeWorkSignRule(ExecuteWorkSignRuleVo executeWorkSignRuleVo){
        // 查询全部已生效的考勤规则
        List<SfaWorkSignRuleEntity> sfaWorkSignRuleEntities =  sfaWorkSignRuleUtil.getAllRule();
        if(CollectionUtil.listEmpty(sfaWorkSignRuleEntities)){
            return;
        }
        for(String userName: executeWorkSignRuleVo.getUserNames()){
            executeWorkSignRule(sfaWorkSignRuleEntities,userName, executeWorkSignRuleVo.getExecuteDate());
        }
    }

    /**
     * 按用户维度执行
     * @param sfaWorkSignRuleEntities
     * @param userName
     * @param executeDate
     */
    @Transactional
    public void executeWorkSignRule(List<SfaWorkSignRuleEntity> sfaWorkSignRuleEntities, String userName, String executeDate){

         //检查用户+日期 在redis是否存在数据,如果已存在则不再生成
        SfaWorkSignRuleInfoEntity oldEntity = sfaWorkSignRuleInfoUtil.getObj(userName, executeDate);
        if(oldEntity != null) {
            return;
        }
        //查询用户所有职位
        List<MdmPositionRespVo> mdmPositionRespVos =  PositionUtil.getAllPositionByUsername(userName);
        for(SfaWorkSignRuleEntity rule : sfaWorkSignRuleEntities){
            //todo 检查是否设置职位级别==>sfaWorkSignRuleUtil.checkPosition
            //判断是否存在职位级别的考勤范围，如果存在则先校验职位级别是否满足，如果不存在默认校验通过
            boolean executeFlag = sfaWorkSignRuleUtil.checkPositionLevel(rule);
//            //todo 判断当前用户所有职位对应 职位级别是否存在规则中 sfaWorkSignRuleUtil.checkPositionLevel
//            boolean levelInRule = sfaWorkSignRuleUtil.checkPositionLevel(rule, mdmPositionRespVos);
//            //todo 判断当前用户所有职位对应组织是否存在规则中  sfaWorkSignRuleUtil.checkPositionOrg
//            boolean OrgInRule = sfaWorkSignRuleUtil.checkPositionOrg(rule, mdmPositionRespVos);
            //如果需要校验职级，则校验职级，默认校验通过（选择性校验）
            executeFlag = executeFlag ? sfaWorkSignRuleUtil.checkPositionLevel(rule, mdmPositionRespVos)
                    : true;
            //如果职级校验通过，则校验组织是否通过，默认校验失败（必须校验）
            executeFlag = executeFlag ? sfaWorkSignRuleUtil.checkPositionOrg(rule, mdmPositionRespVos) : false;
            //todo 检查今天是否不用打卡3个判断  sfaWorkSignRuleUtil.checkMust
            //如果组织校验通过，则校验打卡3个判断，默认失败（必须校验）
            executeFlag = executeFlag ? sfaWorkSignRuleUtil.checkMust(rule, executeDate) : false;
            if(!executeFlag) {
                continue;
            }
            //保存数据 到数据库
            SfaWorkSignRuleInfoEntity infoEntity = this.buildAndSaveSignRuleInfo(rule, userName, executeDate);
            //同步到redis缓存
            sfaWorkSignRuleInfoUtil.setObj(infoEntity, sfaWorkSignRuleInfoUtil.getKey(userName, executeDate));
        }
    }





    /**
     * 根据组织编码获取考勤信息
     * @param orgCode
     * @return
     */
    @Override
    public SfaWorkSignRuleRespVo getSignRuleByOrgCode(String orgCode) {
        //查询出所有已启用的考勤规则
        List<SfaWorkSignRuleEntity> tempRules = sfaWorkSignRuleMapper.selectList(Wrappers
                .lambdaQuery(SfaWorkSignRuleEntity.class)
                .eq(SfaWorkSignRuleEntity :: getEnableStatus, CrmEnableStatusEnum.ENABLE.getCode()));
        if(tempRules == null || tempRules.size() == 0) {
            throw new BusinessException("当前无可执行的考勤规则");
        }
        // 根据规则编码查询出所有定义的组织编码
        Map<String, List<SfaWorkSignPersonnelEntity>> personnelEntityMap = iSfaWorkSignPersonnelService.selectMappingByRuleCodes(tempRules.stream().map(SfaWorkSignRuleEntity :: getRuleCode).collect(Collectors.toSet()));
        // 根据组织树，设置组织所对应的考勤规则
        Map<String, ExecuteSignRuleContext.OrgToRule> orgToRuleMap = getOrgToRuleMap(personnelEntityMap);
        ExecuteSignRuleContext.OrgToRule orgToRule = orgToRuleMap.get(orgCode);
        if(orgToRule == null) {
            throw new BusinessException("当前用户组织未配置考勤规则");
        }
        SfaWorkSignRuleReqVo sfaWorkSignRuleReqVo = new SfaWorkSignRuleReqVo();
        sfaWorkSignRuleReqVo.setRuleCode(orgToRule.getRuleCode());
        SfaWorkSignRuleRespVo ruleRespVo = query(sfaWorkSignRuleReqVo);
        if(ruleRespVo == null) {
            throw new BusinessException("当前用户组织未配置考勤规则");
        }

        return ruleRespVo;
    }

    /**
     * 根据考勤规则构建并保存打卡信息
     * @param rule
     * @param userName
     * @param executeDate
     */
    @Override
    @Transactional
    public SfaWorkSignRuleInfoEntity buildAndSaveSignRuleInfo(SfaWorkSignRuleEntity rule, String userName, String executeDate) {
        //如果参数为空，则不打卡，或者抛异常 TODO 待定
        if(rule == null || StringUtils.isBlank(rule.getRuleCode()) || StringUtils.isBlank(userName)) {
            return null;
        }
//        TODO ValidateUtils.isTrue(rule != null && StringUtils.isNotBlank(rule.getRuleCode()) && StringUtils.isNotBlank(userName), "保存打卡信息时，参数不能为空");
        Set<String> ruleCodes = Sets.newHashSet(rule.getRuleCode());
        SfaWorkSignRuleInfoEntity oldEntity = this.sfaWorkSignRuleInfoUtil.getObj(userName, executeDate);
        //如果已存在规则明细记录，则不再创建
        if(oldEntity != null) {
            return oldEntity;
        }
        SfaWorkSignRuleInfoEntity entity = new SfaWorkSignRuleInfoEntity();
        entity.setUserName(userName);
        entity.setRuleCode(rule.getRuleCode());
        entity.setRuleName(rule.getRuleName());
        entity.setRuleType(rule.getRuleType());
        entity.setRuleDate(executeDate);
        ValidateUtils.isTrue(Pattern.compile("^\\d{4}\\-\\d{2}\\-\\d{2}$").matcher(executeDate).matches(), "考勤日期格式不正确");
        entity.setRuleYear(executeDate.substring(0, 4));
        entity.setRuleMonth(executeDate.substring(5, 7));
        entity.setRuleYearMonth(executeDate.substring(0, 7));
        entity.setNonWorkingDaySignAstrict(rule.getNonWorkingDaySignAstrict());
        entity.setElectronFence(rule.getElectronFence());
        entity.setWsrPhotograph(rule.getWsrPhotograph());
        boolean holiday = dateService.isHoliday(executeDate);
        entity.setSignMust(holiday ? CommonConstant.GLOBAL.YesOrNo.N.getItemCode() : CommonConstant.GLOBAL.YesOrNo.Y.getItemCode());
        //如果为工作日，打卡或不打卡类型为工作日打卡，否则为特殊日期打卡
        if(com.biz.crm.util.StringUtils.isNotEmpty(rule.getWorkingDay()) && rule.getWorkingDay().contains(DateUtil.getWeek(executeDate))) {
            entity.setSignOrNonType(WorkSignEnum.signOrNonType.WORKDAY_SIGN.getValue());
        } else {
            entity.setSignOrNonType(WorkSignEnum.signOrNonType.SPECIAL_DAY_SIGN.getValue());
        }
        //保存打卡规则明细
        this.iSfaWorkSignRuleInfoService.save(entity);
        List<SfaWorkSignRecordEntity> recordEntities = this.packageRecords(entity, rule);
        //保存打卡记录
        this.iSfaWorkSignRecordService.saveBatch(recordEntities);
        entity.setSfaWorkSignRecords(recordEntities);
        return entity;
    }

    /**
     * 打包打卡记录信息
     * @param entity
     * @param rule
     * @return
     */
    private List<SfaWorkSignRecordEntity> packageRecords(SfaWorkSignRuleInfoEntity entity, SfaWorkSignRuleEntity rule) {
        if(entity == null) {
            return Lists.newArrayList();
        }
        List<SfaWorkSignTimeEntity> signTimeEntities = this.iSfaWorkSignTimeService.findByRuleCode(entity.getRuleCode());
        List<SfaWorkSignRecordEntity> recordEntities = Lists.newArrayList();
        for(SfaWorkSignTimeEntity time : signTimeEntities) {
            //上班打卡
            SfaWorkSignRecordEntity start = new SfaWorkSignRecordEntity();
            start.setWsRuleInfoId(entity.getId());
            start.setWorkSignType(SfaWorkSignEnum.WorkSignType.CLOCK_IN.getVal());
            start.setWorkSignDesc(SfaWorkSignEnum.WorkSignType.CLOCK_IN.getDesc());
            start.setSfaSignTimeId(time.getId());
            start.setSfaSignTimeBegin(time.getGotoStartTime());
            start.setSfaSignTimeEnd(time.getGotoEndTime());
            start.setWorkSignStatus(SfaWorkSignEnum.WorkSignStatus.NONE.getVal());
            start.setWsUserName(entity.getUserName());
            recordEntities.add(start);
            //下班打卡
            SfaWorkSignRecordEntity end = new SfaWorkSignRecordEntity();
            end.setWsRuleInfoId(entity.getId());
            end.setWorkSignType(SfaWorkSignEnum.WorkSignType.CLOCK_OUT.getVal());
            end.setWorkSignDesc(SfaWorkSignEnum.WorkSignType.CLOCK_OUT.getDesc());
            end.setSfaSignTimeId(time.getId());
            end.setSfaSignTimeBegin(time.getGooffStartTime());
            end.setSfaSignTimeEnd(time.getGooffEndTime());
            end.setWorkSignStatus(SfaWorkSignEnum.WorkSignStatus.NONE.getVal());
            end.setWsUserName(entity.getUserName());
            recordEntities.add(end);
        }
        return recordEntities;
    }

    /**
     * 组装用户的规则明细和考勤上下班记录
     * @param context 上下文信息
     * @param entity 规则
     * @param mdmOrgWithPositionRespVo 人员列表
     * @param holiday 是否节假日
     */
    protected void buildRuleInfoAndRecord(ExecuteSignRuleContext context, SfaWorkSignRuleEntity entity,
                                        MdmOrgWithPositionRespVo mdmOrgWithPositionRespVo, boolean holiday,Map<String, List<SfaWorkSignPersonnelEntity>> posLevelListMap){
        if(null == mdmOrgWithPositionRespVo){
            return;
        }
        List<MdmPositionUserOrgRespVo> positionRespVos = mdmOrgWithPositionRespVo.getPositionList();
        if(CollectionUtil.listEmpty(positionRespVos)){
            return;
        }
        String orgCode = mdmOrgWithPositionRespVo.getOrgCode();
        context.setOrgCodeNow(orgCode);
        context.setOrgNameNow(mdmOrgWithPositionRespVo.getOrgName());
        //获取组织
        MdmOrgRespVo parentOrg = OrgUtil.getOrgByCode(orgCode);
        if(null != parentOrg){
            //上级组织
            context.setParentOrgCodeNow(parentOrg.getParentCode());
            context.setParentOrgNameNow(parentOrg.getParentName());
        }
        for(MdmPositionUserOrgRespVo positionRespVo : positionRespVos){
            //检查该用户是否已经被其他规则计算过
            if(!context.addUser(positionRespVo.getUserName())){
                log.warn("考勤规则任务计算：忽略已被其他规则计算过或无法识别的用户 user={}", positionRespVo.getUserName());
                continue;
            }
            Map<String, ExecuteSignRuleContext.OrgToRule> levelToRuleMap = context.getOrgAndPosLevelToRuleMap();
            if(CollectionUtil.mapNotEmpty(levelToRuleMap)){
                ExecuteSignRuleContext.OrgToRule rule = levelToRuleMap.get(positionRespVo.getOrgCode() + positionRespVo.getPositionLevelCode());
                if(rule != null){
                    String ruleCode = rule.getRuleCode();
                    entity = context.getRuleEntityMap().get(ruleCode);
                }
            }
            List<SfaWorkSignTimeEntity> sfaWorkSignTimeEntities = context.getTimeMapping().get(entity.getRuleCode());
            //获取对应规则该用户是否被职位级别覆盖
            List<SfaWorkSignPersonnelEntity> workSignPersonnelEntities = posLevelListMap.get(entity.getRuleCode());
            List<String> stringsPos = null ;
            if(CollectionUtil.listNotEmpty(workSignPersonnelEntities)){
                stringsPos = workSignPersonnelEntities.stream().map(SfaWorkSignPersonnelEntity::getWspCode).collect(Collectors.toList());
            }
            //如果存在职位级别
            if(stringsPos != null ){
                //检查该用户是否被职位级别覆盖
                if(!stringsPos.contains(positionRespVo.getPositionLevelCode())){
                    //未被覆盖就跳过
                    continue;
                }
            }
            //组装用户考勤规则明细
            SfaWorkSignRuleInfoEntity infoEntity = SfaWorkSignRuleInfoEntity.buildRuleInfoEntity(context
                    , entity, positionRespVo);
            // 设置是否工作日
            setSignMust(infoEntity, context.getSignDate(), context.getSpecialMapping().get(entity.getRuleCode()),
                    holiday, entity.getWorkingDay());
            context.addRuleInfo(infoEntity);
            for(SfaWorkSignTimeEntity sfaWorkSignTimeEntity : sfaWorkSignTimeEntities){
                //组装用户考勤记录
                context.addRecord(SfaWorkSignRecordEntity.buildClockInOut(infoEntity, sfaWorkSignTimeEntity));
            }

        }
    }

    /**
     * 设置是否必须打卡和类型
     * @param infoEntity 打卡规则详情
     * @param signDate 打卡日期
     * @param specialEntityList 特殊日期
     * @param holiday 是否节假日
     * @param workingDay 工作日
     */
    private void setSignMust(SfaWorkSignRuleInfoEntity infoEntity, String signDate,List<SfaWorkSignSpecialEntity> specialEntityList, boolean holiday, String workingDay) {
        // 节假日
        if(holiday) {
            infoEntity.setSignMust(YesNoEnum.yesNoEnum.NO.getValue());
            infoEntity.setSignOrNonType(WorkSignEnum.signOrNonType.HOLIDAY_NO_SIGN.getValue());
            return;
        }
        // 判断特殊日期必须打卡
        if(specialEntityList != null && specialEntityList.size() > 0) {
            for(SfaWorkSignSpecialEntity specialEntity : specialEntityList) {
                if(signDate.equals(specialEntity.getWssDate())) {
                    if(YesNoEnum.yesNoEnum.ONE.getValue().equals(specialEntity.getWssType())) {
                        infoEntity.setSignMust(YesNoEnum.yesNoEnum.YES.getValue());
                        infoEntity.setSignOrNonType(WorkSignEnum.signOrNonType.SPECIAL_DAY_SIGN.getValue());
                        return;
                    } else {
                        infoEntity.setSignOrNonType(WorkSignEnum.signOrNonType.SPECIAL_DAY_NO_SIGN.getValue());
                        break;
                    }
                }
            }
        }

        // 工作日判断
        int weekNo = LocalDate.now().getDayOfWeek().getValue();
        if(workingDay.indexOf(weekNo+"") > -1) {
            infoEntity.setSignMust(YesNoEnum.yesNoEnum.YES.getValue());
            if(StringUtils.isEmpty(infoEntity.getSignOrNonType())) {
                infoEntity.setSignOrNonType(WorkSignEnum.signOrNonType.WORKDAY_SIGN.getValue());
            }
        } else {
            infoEntity.setSignMust(YesNoEnum.yesNoEnum.NO.getValue());
            if(StringUtils.isEmpty(infoEntity.getSignOrNonType())) {
                infoEntity.setSignOrNonType(WorkSignEnum.signOrNonType.WORK_DAY_NO_SIGN.getValue());
            }
        }
    }

    /**
     * 加载所需要的前置数据到上下文对象
     * @param context
     */
    private void loadContextData(ExecuteSignRuleContext context){
        Set<String> ruleCodes = context.getRuleCodes();

        log.info("规则考勤地点配置");
        context.setPlaceMapping(iSfaWorkSignPlaceService.selectMappingByRuleCodes(ruleCodes));
        log.info("规则考勤时间配置");
        context.setTimeMapping(iSfaWorkSignTimeService.selectMappingByRuleCodes(ruleCodes));
        log.info("规则考勤特殊时间配置");
        context.setSpecialMapping(iSfaWorkSignSpecialService.selectMappingByRuleCodes(ruleCodes));
        log.info("规则考勤组织配置");
        context.setPersonnelMapping(iSfaWorkSignPersonnelService.selectMappingByRuleCodes(ruleCodes));
        log.info("根据组织树，设置规则对应组织及下级对应的规则信息");
        Map<String, ExecuteSignRuleContext.OrgToRule> orgToRuleMap = getOrgToRuleMap(context.getPersonnelMapping());
        context.setOrgToRuleMap(orgToRuleMap);
        log.info("规则考勤职位级别");
        Map<String, List<SfaWorkSignPersonnelEntity>> posLevelListMap = iSfaWorkSignPersonnelService.list(Wrappers.lambdaQuery(SfaWorkSignPersonnelEntity.class)
                    .in(SfaWorkSignPersonnelEntity::getRuleCode, ruleCodes)
                    .eq(SfaWorkSignPersonnelEntity::getCodeType, WorkSignEnum.codeType.POS_LEVEL.getVal()))
                .stream().collect(Collectors.groupingBy(SfaWorkSignPersonnelEntity::getRuleCode));
        context.setPosLevelListMap(posLevelListMap);
        log.info("设置规则对应组织和职位级别的规则信息");
        if(CollectionUtil.mapNotEmpty(posLevelListMap)){
            Map<String, ExecuteSignRuleContext.OrgToRule> orgAndPosLevelToRuleMap = new HashMap<>();
            orgAndPosLevelToRuleMap = this.getOrgAndPosLevelToRuleMap(context.getPersonnelMapping(),posLevelListMap);
            context.setOrgAndPosLevelToRuleMap(orgAndPosLevelToRuleMap);
        }
        Map<String, MdmOrgWithPositionRespVo> orgPositionMappingCurrentY = new HashMap<>();
        //分组获取职位和用户信息每组10个
        List<String> map = new ArrayList<>(context.getOrgToRuleMap().keySet());
        List<List<String>> orgCodes = CollectionUtil.groupListByQuantity(map,10);
        for(List<String> codes : orgCodes){
            log.info("循环查询组织下的职位和用户信息-当前组织==>"+codes);
            List<MdmOrgWithPositionRespVo> mdmOrgWithPositionRespVos =   ApiResultUtil.objResult(mdmOrgFeign.findOrgWithSinglePositionList(codes),true);
            Map<String, MdmOrgWithPositionRespVo> tempMap = mdmOrgWithPositionRespVos.stream().collect(Collectors.toMap(MdmOrgWithPositionRespVo:: getOrgCode, v -> {
                //筛选主职位
                v.setPositionList(filterPrimary(v.getPositionList()));
                return v;
            }, (v1, v2) -> v1));
            orgPositionMappingCurrentY.putAll(tempMap);
        }
        context.setOrgPositionMappingCurrentY(orgPositionMappingCurrentY);
    }

    /**
     * 解析
     * @param context
     */
    protected void resolveInfoAndRecord(ExecuteSignRuleContext context){

        log.info("判断是否法定节假日");
        boolean holiday = dateService.isHoliday(context.getSignDate());
        log.info("遍历组织对应规则信息，生成考勤记录和详情");
        for(String orgCode : context.getOrgToRuleMap().keySet()) {
            ExecuteSignRuleContext.OrgToRule orgToRule = context.getOrgToRuleMap().get(orgCode);
            // 获取当前组织的人员信息
            MdmOrgWithPositionRespVo positionRespVos = context.getOrgPositionMappingCurrentY().get(orgCode);
            this.buildRuleInfoAndRecord(context, context.getRuleEntityMap().get(orgToRule.getRuleCode()),  positionRespVos, holiday, context.getPosLevelListMap());
        }
    }

    /**
     * 设置组织对应规则信息，在组织树种找到最近的规则
     * @param personnelMapping
     */
    public Map<String, ExecuteSignRuleContext.OrgToRule> getOrgToRuleMap(Map<String, List<SfaWorkSignPersonnelEntity>> personnelMapping) {
        // 获取组织树列表
        List<MdmOrgTreeRespVo> treeList = ApiResultUtil.objResult(mdmOrgFeign.getOrgTree(), true);
        // 遍历组织树，标记组织对应规则编码，层级。如果遍历到的层级高于新规则的层级，则更新层级和规则信息
        Map<String, ExecuteSignRuleContext.OrgToRule> orgToRuleMap = new HashMap<>();
        personnelMapping.forEach((k,v) -> {
            v.forEach(vo -> {
                findOrg(treeList, vo, k, orgToRuleMap);
            });
        });
        return orgToRuleMap;
    }

    /**
     * 设置组织加职位级别为KEY对应的组织及下级对应的规则信息
     * @param personnelMapping
     */
    public Map<String, ExecuteSignRuleContext.OrgToRule> getOrgAndPosLevelToRuleMap(Map<String, List<SfaWorkSignPersonnelEntity>> personnelMapping,Map<String, List<SfaWorkSignPersonnelEntity>> posLevelListMap) {
        Map<String, ExecuteSignRuleContext.OrgToRule> orgAndPosLevelToRuleMap = new HashMap<>();
        personnelMapping.forEach((k,v)->{
            v.forEach(data->{
                List<SfaWorkSignPersonnelEntity> sfaWorkSignPersonnelEntities = posLevelListMap.get(k);
                if(CollectionUtil.listNotEmpty(sfaWorkSignPersonnelEntities)){
                    sfaWorkSignPersonnelEntities.forEach(item->{
                        orgAndPosLevelToRuleMap.put(data.getWspCode()+item.getWspCode(),new ExecuteSignRuleContext.OrgToRule(0, item.getRuleCode()));
                    });
                }
            });
        });
        return orgAndPosLevelToRuleMap;
    }

    /**
     * 查找第一层级组织编码
     * @param treeList 组织树
     * @param personnelEntity 考勤组织实体
     * @param ruleCode 当前规则编码
     * @param orgToRuleMap 组织信息对应规则-键值对
     */
    private void findOrg(List<MdmOrgTreeRespVo> treeList, SfaWorkSignPersonnelEntity personnelEntity, String ruleCode, Map<String, ExecuteSignRuleContext.OrgToRule> orgToRuleMap) {
        for(MdmOrgTreeRespVo vo :treeList) {
            if (vo.getOrgCode().equals(personnelEntity.getWspCode())) {
                orgToRuleMap.put(personnelEntity.getWspCode(), new ExecuteSignRuleContext.OrgToRule(0, ruleCode));
                if(CollectionUtils.isNotEmpty(vo.getChildren())) {
                    setOrgToRuleMap(vo.getChildren(), 1, ruleCode, orgToRuleMap);
                }
                break;
            }
            // 如果没有找到的话，继续向下遍历
            if(CollectionUtils.isNotEmpty(vo.getChildren())) {
                findOrg(vo.getChildren(), personnelEntity, ruleCode, orgToRuleMap);
            }
        }
    }

    /**
     * 设置规则编码及层次
     * 如果当前组织编码已经在其他规则中级别更高的话就不用设置 。0级最高
     * @param treeList
     * @param leave
     * @param ruleCode
     */
    private void setOrgToRuleMap(List<MdmOrgTreeRespVo> treeList, Integer leave, String ruleCode, Map<String, ExecuteSignRuleContext.OrgToRule> orgToRuleMap) {
        for(MdmOrgTreeRespVo vo : treeList) {
            ExecuteSignRuleContext.OrgToRule orgToRule = orgToRuleMap.get(vo.getOrgCode());
            if(orgToRule != null && orgToRule.getLeave() < leave) {
                continue;
            }
            orgToRuleMap.put(vo.getOrgCode(), new ExecuteSignRuleContext.OrgToRule(leave, ruleCode));
            if(CollectionUtils.isNotEmpty(vo.getChildren())) {
                setOrgToRuleMap(vo.getChildren(), leave+1, ruleCode, orgToRuleMap);
            }
        }
    }

    /**
     * 筛选主职位
     * @param positionRespVos
     * @return
     */
    private List<MdmPositionUserOrgRespVo> filterPrimary( List<MdmPositionUserOrgRespVo> positionRespVos){
        if(null == positionRespVos){
            return Lists.newArrayList();
        }
        // 筛选主职位
        return positionRespVos.stream().filter(v -> { return YesNoEnum.yesNoEnum.ONE.getValue().equals(v.getPrimaryFlag());}).collect(Collectors.toList());
    }

}
