package com.biz.crm.worksignrule.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.aop.CrmLog;
import com.biz.crm.common.PageResult;
import com.biz.crm.eunm.CrmDelFlagEnum;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.nebular.sfa.worksignrule.req.SfaWorkSignTimeReqVo;
import com.biz.crm.nebular.sfa.worksignrule.resp.SfaWorkSignTimeRespVo;
import com.biz.crm.util.AssertDateTimeUtils;
import com.biz.crm.util.CollectionUtil;
import com.biz.crm.util.CrmBeanUtil;
import com.biz.crm.util.StringUtils;
import com.biz.crm.worksignrule.mapper.SfaWorkSignTimeMapper;
import com.biz.crm.worksignrule.model.SfaWorkSignTimeEntity;
import com.biz.crm.worksignrule.service.ISfaWorkSignTimeService;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 考勤规则-打卡时间;接口实现
 *
 * @author liuhongming
 * @date 2020-09-16 15:06:20
 */
@Slf4j
@Service
@Transactional
@ConditionalOnMissingBean(name="SfaWorkSignTimeServiceExpandImpl")
public class SfaWorkSignTimeServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<SfaWorkSignTimeMapper, SfaWorkSignTimeEntity> implements ISfaWorkSignTimeService {

    @Resource
    private SfaWorkSignTimeMapper sfaWorkSignTimeMapper;

    /**
     * 列表
     * @param reqVo
     * @return
     */
    @Override
    @CrmLog
    public PageResult<SfaWorkSignTimeRespVo> findList(SfaWorkSignTimeReqVo reqVo){
        Page<SfaWorkSignTimeRespVo> page = new Page<>(reqVo.getPageNum(), reqVo.getPageSize());
        List<SfaWorkSignTimeRespVo> list = sfaWorkSignTimeMapper.findList(page, reqVo);
        return PageResult.<SfaWorkSignTimeRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }
    /**
     * 根据规则编码查询 Mapping 列表
     * @param ruleCodes
     * @return
     */
    @Override
    public Map<String, List<SfaWorkSignTimeEntity>> selectMappingByRuleCodes(Set<String> ruleCodes) {
        if(org.springframework.util.CollectionUtils.isEmpty(ruleCodes)){
            return Maps.newHashMap();
        }
        return sfaWorkSignTimeMapper.selectList(Wrappers.lambdaQuery(SfaWorkSignTimeEntity.class)
                .in(SfaWorkSignTimeEntity :: getRuleCode, ruleCodes)).stream().collect(Collectors.groupingBy(SfaWorkSignTimeEntity :: getRuleCode));
    }
    /**
     * 查询
     * @param reqVo
     * @return sfaWorkSignTimeRespVo
     */
    @Override
    @CrmLog
    public SfaWorkSignTimeRespVo query(SfaWorkSignTimeReqVo reqVo){
        List<SfaWorkSignTimeRespVo> list = this.findList(reqVo).getData();
        if(CollectionUtils.isEmpty(list)){
            return new SfaWorkSignTimeRespVo();
        }
        SfaWorkSignTimeRespVo sfaWorkSignTimeRespVo = list.get(0);
        return sfaWorkSignTimeRespVo;
    }

    /**
     * 新增
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @CrmLog
    public void save(SfaWorkSignTimeReqVo reqVo){
        SfaWorkSignTimeEntity entity = CrmBeanUtil.copy(reqVo,SfaWorkSignTimeEntity.class);
        this.save(entity);
    }

    /**
     * 更新
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @CrmLog
    public void update(SfaWorkSignTimeReqVo reqVo){
        SfaWorkSignTimeEntity entity = this.getById(reqVo.getId());
        this.updateById(entity);
    }

    /**
     * 删除
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @CrmLog
    public void deleteBatch(SfaWorkSignTimeReqVo reqVo){
        List<SfaWorkSignTimeEntity> sfaWorkSignTimeEntities = sfaWorkSignTimeMapper.selectBatchIds(reqVo.getIds());
        if(CollectionUtils.isNotEmpty(sfaWorkSignTimeEntities)){
            sfaWorkSignTimeEntities.forEach(o -> {
                 o.setDelFlag(CrmDelFlagEnum.DELETE.getCode());
            });
        }
        this.updateBatchById(sfaWorkSignTimeEntities);
    }

    /**
     * 启用
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @CrmLog
    public void enableBatch(SfaWorkSignTimeReqVo reqVo){
        //设置状态为启用
        List<SfaWorkSignTimeEntity> sfaWorkSignTimeEntities = sfaWorkSignTimeMapper.selectBatchIds(reqVo.getIds());
        if(CollectionUtils.isNotEmpty(sfaWorkSignTimeEntities)){
            sfaWorkSignTimeEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
            });
        }
        this.updateBatchById(sfaWorkSignTimeEntities);
    }

    /**
     * 禁用
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @CrmLog
    public void disableBatch(SfaWorkSignTimeReqVo reqVo){
        //设置状态为禁用
        List<SfaWorkSignTimeEntity> sfaWorkSignTimeEntities = sfaWorkSignTimeMapper.selectBatchIds(reqVo.getIds());
        if(CollectionUtils.isNotEmpty(sfaWorkSignTimeEntities)){
                sfaWorkSignTimeEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.DISABLE.getCode());
            });
        }
        this.updateBatchById(sfaWorkSignTimeEntities);
    }

    /**
     * 新增,编辑,校验
     * @param timeReqVo
     */
    @Override
    @CrmLog
    public void saveCheck(SfaWorkSignTimeReqVo timeReqVo){
        // 数据校验
        AssertDateTimeUtils.checkTime(timeReqVo.getGotoTime(), "上班打卡时间错误");
        AssertDateTimeUtils.checkTime(timeReqVo.getGooffTime(), "下班打卡时间错误");
        AssertDateTimeUtils.checkTime(timeReqVo.getGotoStartTime(), "打卡时间限制-可打上班卡开始时间错误");
        AssertDateTimeUtils.checkTime(timeReqVo.getGotoEndTime(), "打卡时间限制-可打上班卡结束时间错误");
        AssertDateTimeUtils.checkTime(timeReqVo.getGooffStartTime(), "打卡时间限制-可打下班打卡开始时间错误");
        AssertDateTimeUtils.checkTime(timeReqVo.getGooffEndTime(), "打卡时间限制-可打下班卡结束时间错误");

        // 规则校验
//        AssertDateTimeUtils.notAfterTime(timeReqVo.getGotoTime(),timeReqVo.getGooffTime(), "上班时间不能大于下班时间");
        AssertDateTimeUtils.notAfterTime(timeReqVo.getGotoStartTime(),timeReqVo.getGotoEndTime(), "上班打卡开始时间不能大于上班打卡结束时间");
//        AssertDateTimeUtils.notAfter(timeReqVo.getGotoEndTime(),timeReqVo.getGooffStartTime(), "上班打卡结束时间不能大于下班打卡开始时间");
        AssertDateTimeUtils.notAfterTime(timeReqVo.getGooffStartTime(),timeReqVo.getGooffEndTime(), "下班打卡开始时间不能大于下班打卡结束时间");
    }

    /**
     * 根据规则编码查询列表
     * @param ruleCode
     * @return
     */
    @Override
    public List<SfaWorkSignTimeEntity> findByRuleCode(String ruleCode) {
        if(StringUtils.isEmpty(ruleCode)) {
            return Lists.newArrayList();
        }
        List<SfaWorkSignTimeEntity> entities = this.sfaWorkSignTimeMapper.selectList(Wrappers.<SfaWorkSignTimeEntity>lambdaQuery()
        .eq(SfaWorkSignTimeEntity::getRuleCode, ruleCode));
        if(CollectionUtil.listEmpty(entities)) {
            return Lists.newArrayList();
        }
        return entities;
    }
}
