package com.biz.crm.activity.service.impl;

import cn.hutool.json.JSONUtil;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.activity.mapper.*;
import com.biz.crm.activity.model.*;
import com.biz.crm.activity.req.ActivityExecutionResolveContext;
import com.biz.crm.activity.req.ActivityExecutionResolveData;
import com.biz.crm.activity.service.*;
import com.biz.crm.base.BusinessException;
import com.biz.crm.base.SfaClientData;
import com.biz.crm.base.SfaClientHelper;
import com.biz.crm.common.PageResult;
import com.biz.crm.config.CrmDictMethod;
import com.biz.crm.eunm.CrmDelFlagEnum;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.eunm.sfa.SfaActivityEnum;
import com.biz.crm.mq.RocketMQConstant;
import com.biz.crm.mq.RocketMQMessageBody;
import com.biz.crm.mq.RocketMQProducer;
import com.biz.crm.nebular.sfa.activity.req.SfaActivityBgSetmealReqVo;
import com.biz.crm.nebular.sfa.activity.req.SfaActivityExecutionReqVo;
import com.biz.crm.nebular.sfa.activity.resp.*;
import com.biz.crm.nebular.sfa.picture.resp.SfaVisitPictureRespVo;
import com.biz.crm.picture.mapper.SfaVisitPictureMapper;
import com.biz.crm.picture.model.SfaVisitPictureEntity;
import com.biz.crm.picture.service.ISfaVisitPictureService;
import com.biz.crm.util.*;
import com.biz.crm.visitnote.service.component.resolver.SfaVisitPlanResolver;
import com.biz.crm.visitstep.repositories.SfaVisitStepActivityCostExecutionEsDataRepositories;
import com.biz.crm.visitstep.repositories.SfaVisitStepActivityDisplayExecutionEsDataRepositories;
import com.biz.crm.visitstep.service.ISfaVisitStepActivityExecutionService;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.time.LocalDate;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 活动执行;接口实现
 *
 * @author lf
 * @date 2020-12-30 15:24:11
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "SfaActivityExecutionServiceExpandImpl")
@Transactional
public class SfaActivityExecutionServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<SfaActivityExecutionMapper, SfaActivityExecutionEntity> implements ISfaActivityExecutionService, JobHelper.JobExecutor {

    @Resource
    private SfaActivityExecutionMapper sfaActivityExecutionMapper;
    @Resource
    private ISfaVisitStepActivityExecutionService sfaVisitStepActivityExecutionService;
    @Resource
    private SfaActivityRangeMapper sfaActivityRangeMapper;
    @Resource
    private ISfaActivityRangeService sfaActivityRangeService;
    @Resource
    private SfaActivityProductMapper sfaActivityProductMapper;
    @Resource
    private ISfaActivityProductService sfaActivityProductService;
    @Resource
    private SfaActivityBgSetmealMapper sfaActivityBgSetmealMapper;
    @Resource
    private ISfaActivityBgSetmealService sfaActivityBgSetmealService;
    @Resource
    private SfaActivityBgProductMapper sfaActivityBgProductMapper;
    @Resource
    private ISfaActivityBgProductService sfaActivityBgProductService;
    @Resource
    private SfaVisitPictureMapper sfaVisitPictureMapper;
    @Resource
    private ISfaVisitPictureService sfaVisitPictureService;
    @Resource
    private JobHelper jobHelper;
    @Resource
    private RocketMQProducer rocketMQProducer;
    @Resource
    private SfaVisitStepActivityCostExecutionEsDataRepositories activityCostExecutionEsDataRepositories;
    @Resource
    private SfaVisitStepActivityDisplayExecutionEsDataRepositories activityDisplayExecutionEsDataRepositories;

    /**
     * 列表
     *
     * @param reqVo
     * @return
     */
    @Override
    @CrmDictMethod
    public PageResult<SfaActivityExecutionRespVo> findList(SfaActivityExecutionReqVo reqVo) {
        Page<SfaActivityExecutionRespVo> page = new Page<>(reqVo.getPageNum(), reqVo.getPageSize());
        List<SfaActivityExecutionRespVo> list = sfaActivityExecutionMapper.findList(page, reqVo);
        return PageResult.<SfaActivityExecutionRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }


    /**
     * 通过id查询详情
     *
     * @param id
     * @return
     */
    @Override
    @CrmDictMethod
    public SfaActivityExecutionRespVo queryDetailById(String id) {
        AssertUtils.isNotEmpty(id, "数据主键id不能为空");
        SfaActivityExecutionEntity sfaActivityExecutionEntity = this.lambdaQuery().eq(SfaActivityExecutionEntity::getId, id).one();
        if (sfaActivityExecutionEntity == null) {
            throw new BusinessException("未查询到活动执行明细");
        }
        SfaActivityExecutionRespVo sfaActivityExecutionRespVo = CrmBeanUtil.copy(sfaActivityExecutionEntity, SfaActivityExecutionRespVo.class);
        StringBuilder stringBuilder = new StringBuilder();
        String[] activityRequireList = sfaActivityExecutionRespVo.getActivityRequire().split(",");
        for (int i = 0; i < activityRequireList.length; i++) {
            if (stringBuilder != null && stringBuilder.length() > 0) {
                stringBuilder.append(",");
            }
            stringBuilder.append(SfaActivityEnum.activityRequire.GETMAP.get(activityRequireList[i]));
        }
        sfaActivityExecutionRespVo.setActivityRequireDesc(stringBuilder.toString());
        List<SfaActivityRangeEntity> rangeEntityList = sfaActivityRangeService.lambdaQuery()
                .eq(SfaActivityRangeEntity::getActivityExecutionId, sfaActivityExecutionEntity.getId()).list();
        if (CollectionUtil.listNotEmptyNotSizeZero(rangeEntityList)) {
            List<SfaActivityRangeRespVo> rangeRespVos = CrmBeanUtil.copyList(rangeEntityList, SfaActivityRangeRespVo.class);
            Map<String, List<SfaActivityRangeRespVo>> mapRange = rangeRespVos.stream().collect(Collectors.groupingBy(SfaActivityRangeRespVo::getRangeType));
            for (Map.Entry<String, List<SfaActivityRangeRespVo>> map : mapRange.entrySet()) {
                if (map.getKey().equals(SfaActivityEnum.rangeType.CHANNEL.getVal())) {
                    sfaActivityExecutionRespVo.setChannelRangeList(map.getValue());
                }
                if (map.getKey().equals(SfaActivityEnum.rangeType.ORG.getVal())) {
                    sfaActivityExecutionRespVo.setOrgRangeList(map.getValue());
                }
                if (map.getKey().equals(SfaActivityEnum.rangeType.CUSTOMER.getVal())) {
                    sfaActivityExecutionRespVo.setCustomerRangeList(map.getValue());
                }
            }
            sfaActivityExecutionRespVo.setActivityRangeList(rangeRespVos);
        }
        List<SfaActivityProductEntity> productEntityList = sfaActivityProductService.lambdaQuery()
                .eq(SfaActivityProductEntity::getActivityExecutionId, sfaActivityExecutionEntity.getId()).list();
        if (CollectionUtil.listNotEmptyNotSizeZero(productEntityList)) {
            List<SfaActivityProductRespVo> productRespVos = CrmBeanUtil.copyList(productEntityList, SfaActivityProductRespVo.class);
            sfaActivityExecutionRespVo.setActivityProductList(productRespVos);
        }
        List<SfaVisitPictureEntity> pictureEntities = sfaVisitPictureService.lambdaQuery()
                .eq(SfaVisitPictureEntity::getBusinessId, sfaActivityExecutionEntity.getId()).list();
        if (CollectionUtil.listNotEmptyNotSizeZero(pictureEntities)) {
            List<SfaVisitPictureRespVo> pictureRespVos = CrmBeanUtil.copyList(pictureEntities, SfaVisitPictureRespVo.class);
            sfaActivityExecutionRespVo.setVisitPictureList(pictureRespVos);
        }
        List<SfaActivityBgSetmealEntity> setmealEntitieList = sfaActivityBgSetmealService.lambdaQuery()
                .eq(SfaActivityBgSetmealEntity::getActivityExecutionId, sfaActivityExecutionEntity.getId()).list();
        if (CollectionUtil.listNotEmptyNotSizeZero(setmealEntitieList)) {
            List<SfaActivityBgSetmealRespVo> activityBgSetmealList = CrmBeanUtil.copyList(setmealEntitieList, SfaActivityBgSetmealRespVo.class);
            List<SfaActivityBgProductEntity> productBgEntityList = sfaActivityBgProductService.lambdaQuery()
                    .eq(SfaActivityBgProductEntity::getActivityExecutionId, sfaActivityExecutionEntity.getId()).list();
            List<SfaActivityBgProductRespVo> bgProductRespVos = CrmBeanUtil.copyList(productBgEntityList, SfaActivityBgProductRespVo.class);
            Map<String, List<SfaActivityBgProductRespVo>> mapRange = bgProductRespVos.stream().collect(Collectors.groupingBy(SfaActivityBgProductRespVo::getSetmealId));
            for (SfaActivityBgSetmealRespVo activityBgSetmeal : activityBgSetmealList) {
                if (mapRange.containsKey(activityBgSetmeal.getId())) {
                    List<SfaActivityBgProductRespVo> productRespVoList = mapRange.get(activityBgSetmeal.getId());
                    Map<String, List<SfaActivityBgProductRespVo>> mapProductRange = productRespVoList.stream().collect(Collectors.groupingBy(SfaActivityBgProductRespVo::getItmeType));
                    if (mapProductRange.containsKey(SfaActivityEnum.bpItemType.bp.getVal())) {
                        activityBgSetmeal.setBpBgProductList(mapProductRange.get(SfaActivityEnum.bpItemType.bp.getVal()));
                    }
                    if (mapProductRange.containsKey(SfaActivityEnum.bpItemType.zp.getVal())) {
                        activityBgSetmeal.setZpBgProductList(mapProductRange.get(SfaActivityEnum.bpItemType.zp.getVal()));
                    }
                }
            }
            sfaActivityExecutionRespVo.setActivityBgSetmealList(activityBgSetmealList);
        }
        return sfaActivityExecutionRespVo;
    }

    /**
     * 查询
     *
     * @param reqVo
     * @return sfaActivityExecutionRespVo
     */
    @Override
    public SfaActivityExecutionRespVo query(SfaActivityExecutionReqVo reqVo) {
        return null;
    }

    /**
     * 新增
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void save(SfaActivityExecutionReqVo reqVo) {
        //参数校验
        this.checkParam(reqVo);
        SfaActivityExecutionEntity entity = CrmBeanUtil.copy(reqVo, SfaActivityExecutionEntity.class);
        entity.setActivityCode(CodeUtil.getCodeDefault());
        entity.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
        entity.setDelFlag(CrmDelFlagEnum.NORMAL.getCode());
        log.info("entity[{}]", JSONUtil.toJsonStr(entity));
        this.save(entity);
        String activityExecutionId = entity.getId();
        reqVo.getActivityRangeList().forEach(data -> {
            data.setActivityExecutionId(activityExecutionId);
        });
        List<SfaActivityRangeEntity> activityRangeEntityList = reqVo.getActivityRangeList().stream().map(v -> {
            SfaActivityRangeEntity rangeEntity = CrmBeanUtil.copy(v, SfaActivityRangeEntity.class);
            rangeEntity.setId(null);
            return rangeEntity;
        }).collect(Collectors.toList());
        sfaActivityRangeService.saveBatch(activityRangeEntityList);
        if (CollectionUtil.listNotEmptyNotSizeZero(reqVo.getVisitPictureList())) {
            List<SfaVisitPictureEntity> sfaVisitPictureEntities = CrmBeanUtil.copyList(reqVo.getVisitPictureList(), SfaVisitPictureEntity.class);
            sfaVisitPictureEntities.forEach(data -> {
                data.setBusinessId(activityExecutionId);
            });
            sfaVisitPictureService.saveBatch(sfaVisitPictureEntities);
        }
        if (CollectionUtil.listNotEmptyNotSizeZero(reqVo.getActivityProductList())) {
            List<SfaActivityProductEntity> activityProductEntities = CrmBeanUtil.copyList(reqVo.getActivityProductList(), SfaActivityProductEntity.class);
            activityProductEntities.forEach(data -> {
                data.setActivityExecutionId(activityExecutionId);
            });
            sfaActivityProductService.saveBatch(activityProductEntities);
        }
        if (CollectionUtil.listNotEmptyNotSizeZero(reqVo.getActivityBgSetmealList())) {
            for (SfaActivityBgSetmealReqVo setmealReqVo : reqVo.getActivityBgSetmealList()) {
                SfaActivityBgSetmealEntity setmealEntity = CrmBeanUtil.copy(setmealReqVo, SfaActivityBgSetmealEntity.class);
                setmealEntity.setActivityExecutionId(activityExecutionId);
                sfaActivityBgSetmealService.save(setmealEntity);
                if (CollectionUtil.listNotEmptyNotSizeZero(setmealReqVo.getActivityBgProductList())) {
                    List<SfaActivityBgProductEntity> activityBgProductEntities = CrmBeanUtil.copyList(setmealReqVo.getActivityBgProductList(), SfaActivityBgProductEntity.class);
                    activityBgProductEntities.forEach(data -> {
                        data.setActivityExecutionId(activityExecutionId);
                        data.setSetmealId(setmealEntity.getId());
                    });
                    sfaActivityBgProductService.saveBatch(activityBgProductEntities);
                }
            }
        }
    }


    /**
     * 校验参数信息
     *
     * @param reqVo
     */
    private void checkParam(SfaActivityExecutionReqVo reqVo) {
        UserRedis user = UserUtils.getUser();
        AssertUtils.isNotEmpty(reqVo.getActivityType(), "活动类型为空");
        if (!SfaActivityEnum.activityType.GETMAP.containsKey(reqVo.getActivityType())) {
            throw new BusinessException("活动类型错误:" + reqVo.getActivityType());
        }
        AssertUtils.isNotEmpty(reqVo.getActivityStartTime(), "活动执行开始时间为空");
        AssertUtils.isNotEmpty(reqVo.getActivityEndTime(), "活动执行结束时间为空");
        AssertUtils.isNotEmpty(reqVo.getActivityRequire(), "活动要求为空为空");
        AssertUtils.isNotEmpty(reqVo.getActivityName(), "活动名称为空");
        AssertUtils.isNotEmpty(reqVo.getActivityRangeList(), "活动范围为空");
        LocalDate activityStartTime = null;
        LocalDate activityEndTime = null;
        LocalDate now = LocalDate.now();
        try {
            activityStartTime = LocalDate.parse(reqVo.getActivityStartTime());
            activityEndTime = LocalDate.parse(reqVo.getActivityEndTime());
        } catch (Exception e) {
            throw new BusinessException("活动执行时间错误");
        }
        if (!activityStartTime.isBefore(activityEndTime)) {
            throw new BusinessException("活动执行开始时间不能大于活动执行结束时间");
        }
        if (now.isAfter(activityEndTime)) {
            throw new BusinessException("活动执行结束时间不可小于当前时间");
        }
        if (now.isBefore(activityStartTime)) {
            reqVo.setActivityStatus(SfaActivityEnum.activityStatus.NOT_START.getVal());
        } else {
            reqVo.setActivityStatus(SfaActivityEnum.activityStatus.ALREADY_START.getVal());
        }
        // 移除activityRangeList链表中null元素
        reqVo.getActivityRangeList().removeAll(Collections.singleton(null));
        reqVo.getActivityRangeList().forEach(data -> {
            AssertUtils.isNotEmpty(data.getRangeType(), "活动范围类型为空");
            if (!SfaActivityEnum.rangeType.GETMAP.containsKey(data.getRangeType())) {
                throw new BusinessException("活动范围类型错误");
            }
            AssertUtils.isNotEmpty(data.getCode(), "编码为空");
            AssertUtils.isNotEmpty(data.getName(), "名称为空");
        });
        //判断为费用活动
        if (SfaActivityEnum.activityType.COST_ACTIVITY.getVal().equals(reqVo.getActivityType())) {
            AssertUtils.isNotEmpty(reqVo.getSalesVolume(), "活动预估销售额为空");
            AssertUtils.isNotEmpty(reqVo.getApplyAmount(), "活动费用申请金额为空");
        } else {
            if (reqVo.getActivityFrequency() == null) {
                throw new BusinessException("活动执行频率不能为空");
            }
//            AssertUtils.isNotEmpty(reqVo.getVisitPictureList(),"活动示例图片为空");
//            AssertUtils.isNotEmpty(reqVo.getActivityProductList(),"活动陈列要求产品为空");
//            reqVo.getActivityProductList().forEach(data->{
//                AssertUtils.isNotEmpty(data.getProductCode(),"产品编码为空");
//                AssertUtils.isNotEmpty(data.getProductName(),"产品名称为空");
//                AssertUtils.isNotEmpty(data.getDisplayNum(),"最低陈列面数为空");
//            });
        }
        reqVo.setUserName(user.getUsername());
        reqVo.setRealName(user.getRealname());
        reqVo.setPosCode(user.getPoscode());
        reqVo.setPosName(user.getPosname());
        reqVo.setOrgCode(user.getOrgcode());
        reqVo.setOrgName(user.getOrgname());
    }

    /**
     * 更新
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(SfaActivityExecutionReqVo reqVo) {
        //校验参数信息
        this.checkParam(reqVo);
        SfaActivityExecutionEntity entity = this.getById(reqVo.getId());
        String activityType = entity.getActivityType();
        String id = entity.getId();
        entity = CrmBeanUtil.copy(reqVo, SfaActivityExecutionEntity.class);
        entity.setId(id);
        entity.setActChanged(YesNoEnum.yesNoEnum.YES.getValue());
        entity.setActivityType(activityType);
        this.updateById(entity);
        LambdaQueryWrapper<SfaActivityProductEntity> productQueryWrapper = new LambdaQueryWrapper<SfaActivityProductEntity>()
                .eq(SfaActivityProductEntity::getActivityExecutionId, id);
        LambdaQueryWrapper<SfaActivityRangeEntity> rangeQueryWrapper = new LambdaQueryWrapper<SfaActivityRangeEntity>()
                .eq(SfaActivityRangeEntity::getActivityExecutionId, id);
        LambdaQueryWrapper<SfaVisitPictureEntity> picQueryWrapper = new LambdaQueryWrapper<SfaVisitPictureEntity>()
                .eq(SfaVisitPictureEntity::getBusinessId, id);
        LambdaQueryWrapper<SfaActivityBgSetmealEntity> bgSetmealQueryWrapper = new LambdaQueryWrapper<SfaActivityBgSetmealEntity>()
                .eq(SfaActivityBgSetmealEntity::getActivityExecutionId, id);
        LambdaQueryWrapper<SfaActivityBgProductEntity> bgProductQueryWrapper = new LambdaQueryWrapper<SfaActivityBgProductEntity>()
                .eq(SfaActivityBgProductEntity::getActivityExecutionId, id);
        sfaActivityProductMapper.delete(productQueryWrapper);
        sfaActivityRangeMapper.delete(rangeQueryWrapper);
        sfaVisitPictureMapper.delete(picQueryWrapper);
        sfaActivityBgSetmealMapper.delete(bgSetmealQueryWrapper);
        sfaActivityBgProductMapper.delete(bgProductQueryWrapper);

        reqVo.getActivityRangeList().forEach(data -> {
            data.setActivityExecutionId(id);
        });
        List<SfaActivityRangeEntity> activityRangeEntityList = CrmBeanUtil.copyList(reqVo.getActivityRangeList(), SfaActivityRangeEntity.class);
        if(CollectionUtils.isNotEmpty(activityRangeEntityList)){
            for (SfaActivityRangeEntity item : activityRangeEntityList) {
                item.setId(null);
            }
        }
        sfaActivityRangeService.saveBatch(activityRangeEntityList);
        if (CollectionUtil.listNotEmptyNotSizeZero(reqVo.getVisitPictureList())) {
            List<SfaVisitPictureEntity> sfaVisitPictureEntities = CrmBeanUtil.copyList(reqVo.getVisitPictureList(), SfaVisitPictureEntity.class);
            sfaVisitPictureEntities.forEach(data -> {
                data.setBusinessId(id);
            });
            sfaVisitPictureService.saveBatch(sfaVisitPictureEntities);
        }
        if (CollectionUtil.listNotEmptyNotSizeZero(reqVo.getActivityProductList())) {
            List<SfaActivityProductEntity> activityProductEntities = CrmBeanUtil.copyList(reqVo.getActivityProductList(), SfaActivityProductEntity.class);
            activityProductEntities.forEach(data -> {
                data.setActivityExecutionId(id);
            });
            sfaActivityProductService.saveBatch(activityProductEntities);
        }
        if (CollectionUtil.listNotEmptyNotSizeZero(reqVo.getActivityBgSetmealList())) {
            for (SfaActivityBgSetmealReqVo setmealReqVo : reqVo.getActivityBgSetmealList()) {
                SfaActivityBgSetmealEntity setmealEntity = CrmBeanUtil.copy(setmealReqVo, SfaActivityBgSetmealEntity.class);
                setmealEntity.setActivityExecutionId(id);
                sfaActivityBgSetmealService.save(setmealEntity);
                if (CollectionUtil.listNotEmptyNotSizeZero(setmealReqVo.getActivityBgProductList())) {
                    List<SfaActivityBgProductEntity> activityBgProductEntities = CrmBeanUtil.copyList(setmealReqVo.getActivityBgProductList(), SfaActivityBgProductEntity.class);
                    activityBgProductEntities.forEach(data -> {
                        data.setActivityExecutionId(id);
                        data.setSetmealId(setmealEntity.getId());
                    });
                    sfaActivityBgProductService.saveBatch(activityBgProductEntities);
                }
            }
        }
    }

    /**
     * 删除
     *
     * @param ids
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteBatch(List<String> ids) {
        if (!CollectionUtil.listNotEmptyNotSizeZero(ids)) {
            throw new BusinessException("数据主键不能为空");
        }
        List<SfaActivityExecutionEntity> sfaActivityExecutionEntities = sfaActivityExecutionMapper.selectBatchIds(ids);
        if (CollectionUtils.isNotEmpty(sfaActivityExecutionEntities)) {
            sfaActivityExecutionEntities.forEach(o -> {
                o.setDelFlag(CrmDelFlagEnum.DELETE.getCode());
            });
        }
        this.updateBatchById(sfaActivityExecutionEntities);
        activityCostExecutionEsDataRepositories.deleteByActivityExecutionIdIn(ids);
        activityDisplayExecutionEsDataRepositories.deleteByActivityExecutionIdIn(ids);
    }

    /**
     * 启用
     *
     * @param ids
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void enableBatch(List<String> ids) {
        if (!CollectionUtil.listNotEmptyNotSizeZero(ids)) {
            throw new BusinessException("数据主键不能为空");
        }
        //设置状态为启用
        List<SfaActivityExecutionEntity> sfaActivityExecutionEntities = sfaActivityExecutionMapper.selectBatchIds(ids);
        if (CollectionUtils.isNotEmpty(sfaActivityExecutionEntities)) {
            sfaActivityExecutionEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
            });
        }
        this.updateBatchById(sfaActivityExecutionEntities);
    }

    /**
     * 禁用
     *
     * @param ids
     * @return
     */
    @Override
    @Transactional
    public void disableBatch(List<String> ids) {
        if (!CollectionUtil.listNotEmptyNotSizeZero(ids)) {
            throw new BusinessException("数据主键不能为空");
        }
        //设置状态为禁用
        List<SfaActivityExecutionEntity> sfaActivityExecutionEntities = sfaActivityExecutionMapper.selectBatchIds(ids);
        if (CollectionUtils.isNotEmpty(sfaActivityExecutionEntities)) {
            sfaActivityExecutionEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.DISABLE.getCode());
            });
        }
        this.updateBatchById(sfaActivityExecutionEntities);
    }

    @Override
    @Transactional
    public void executeJob(Object jobParam) {
        this.resolveSfaVisitStepActivityExecution((Boolean) jobParam);
    }

    @Override
    @Transactional
    public void resolveSfaVisitStepActivityExecutionJob(Boolean compensation) {
        JobHelper.JobContext context = new JobHelper.JobContext();
        context.setJob(JobHelper.Job.RESOLVE_SFA_VISIT_STEP_ACTIVITY_EXECUTION_JOB);
        context.setExecutorBeanClass(SfaActivityExecutionServiceImpl.class);
        context.setJobParam(compensation);
        this.jobHelper.executeJobSaveContext(context);
    }

    @Override
    @Transactional
    public void resolveSfaVisitStepActivityExecution(Boolean compensation) {
        log.info("resolveSfaVisitStepActivityExecution");
        LocalDate localDateNow = LocalDate.now();
        String now = localDateNow.format(CrmDateUtils.yyyyMMdd);

        List<SfaActivityExecutionEntity> activityList = this.lambdaQuery()
                .eq(SfaActivityExecutionEntity::getDelFlag,CrmDelFlagEnum.NORMAL.getCode())
                .eq(SfaActivityExecutionEntity::getEnableStatus, CrmEnableStatusEnum.ENABLE.getCode())
                .le(SfaActivityExecutionEntity::getActivityStartTime, now)
                .ge(SfaActivityExecutionEntity::getActivityEndTime, now).list();
        log.info("activityList[{}]",activityList);
        for (SfaActivityExecutionEntity sfaActivityExecutionEntity : activityList) {
            //已经解析过的天数
//            long resolvedDays = this.getResolvedDays(sfaActivityExecutionEntity.getLastResolveEndDate(), localDateNow, compensation);
            long resolvedDays;
//            if(!YesNoEnum.yesNoEnum.NO.getValue().equals(sfaActivityExecutionEntity.getActChanged())){
            //活动有变更，清除该活动上次解析的明细数据
            this.sfaVisitStepActivityExecutionService.cleanExecutionData(sfaActivityExecutionEntity);
            //变更之后直接从当前日期开始解析
            resolvedDays = 0;
//            }

            //TODO 暂时全部解析所有活动
            //如果活动没有变更，并且已经解析过的天数满足期望解析的天数，那么就无需再次解析该活动
//            if(YesNoEnum.yesNoEnum.NO.getValue().equals(sfaActivityExecutionEntity.getActChanged())
//                    && resolvedDays >= SfaVisitPlanResolver.futureDays){
//                log.warn("活动执行明细解析:该活动没有变更，且[已经解析过的天数]满足期望解析的天数! resolvedDays={},futureDays={},activityCode={}",
//                        resolvedDays, SfaVisitPlanResolver.futureDays, sfaActivityExecutionEntity.getActivityCode());
//                continue;
//            }
            log.warn("活动执行明细解析:开始解析活动 resolvedDays={},futureDays={},activityCode={}",
                    resolvedDays, SfaVisitPlanResolver.futureDays, sfaActivityExecutionEntity.getActivityCode());
            this.doResolveSfaVisitStepActivityExecution(sfaActivityExecutionEntity, now, resolvedDays, localDateNow);

        }
    }

    private void doResolveSfaVisitStepActivityExecution(SfaActivityExecutionEntity sfaActivityExecutionEntity, String now, long resolvedDays, LocalDate localDateNow) {
        //更新该活动的【上次解析活动明细的结束日期】
        this.lambdaUpdate().eq(SfaActivityExecutionEntity::getId, sfaActivityExecutionEntity.getId())
                .set(SfaActivityExecutionEntity::getLastResolveEndDate, localDateNow.plusDays(SfaVisitPlanResolver.futureDays - 1).format(CrmDateUtils.yyyyMMdd))
                .set(SfaActivityExecutionEntity::getActChanged, YesNoEnum.yesNoEnum.NO.getValue()).update();
        ActivityExecutionResolveContext context = new ActivityExecutionResolveContext();
        for (; resolvedDays < SfaVisitPlanResolver.futureDays; resolvedDays++) {
            if (SfaActivityEnum.activityType.COST_ACTIVITY.getVal().equals(sfaActivityExecutionEntity.getActivityType())
                    && context.costActResolved(sfaActivityExecutionEntity.getActivityCode())) {
                break;
            }
            LocalDate resolveLocalDate = localDateNow.plusDays(resolvedDays);
            String resolveLocalDateStr = resolveLocalDate.format(CrmDateUtils.yyyyMMdd);
            if (resolveLocalDate.compareTo(LocalDate.parse(sfaActivityExecutionEntity.getActivityEndTime())) > 0) {
                log.warn("活动执行明细解析:已超出活动结束日期，结束该活动的解析! resolveDate={},activityEndTime={},activityCode={}",
                        resolveLocalDateStr, sfaActivityExecutionEntity.getActivityEndTime(), sfaActivityExecutionEntity.getActivityCode());
                break;
            }
            if (this.checkFrequency(sfaActivityExecutionEntity, resolveLocalDate)) {
                ActivityExecutionResolveContext.DayContext dayContext = new ActivityExecutionResolveContext.DayContext();
                ActivityExecutionResolveData resolveData = new ActivityExecutionResolveData();
                resolveData.setResolveDate(resolveLocalDateStr);
                resolveData.setActivityExecutionEntity(sfaActivityExecutionEntity);
                dayContext.setResolveData(resolveData);
                context.setDayContext(dayContext);
                this.resolveClientActList(context);
            } else {
                log.warn("活动执行明细解析:当前日期不满足活动执行频率，忽略该日期的解析! resolveDate={},activityFrequency={},activityCode={}",
                        resolveLocalDateStr, sfaActivityExecutionEntity.getActivityFrequency(), sfaActivityExecutionEntity.getActivityCode());
            }
        }
    }

    /**
     * 频率检查，根据给定活动的频率配置，检查给定日期是否需要执行给定活动
     *
     * @param sfaActivityExecutionEntity
     * @param resolveDate
     * @return
     */
    private boolean checkFrequency(SfaActivityExecutionEntity sfaActivityExecutionEntity, LocalDate resolveDate) {
        //陈列活动
        if (SfaActivityEnum.activityType.DISPLAY_ACTIVITY.getVal().equals(sfaActivityExecutionEntity.getActivityType())) {
            if (org.apache.commons.lang3.StringUtils.isBlank(sfaActivityExecutionEntity.getActivityFrequency())) {
                return false;
            }
            //频率
            Integer frequency = Integer.valueOf(sfaActivityExecutionEntity.getActivityFrequency());
            LocalDate startDate = LocalDate.parse(sfaActivityExecutionEntity.getActivityStartTime());
            long days = resolveDate.toEpochDay() - startDate.toEpochDay();
            return days % frequency == 0;
        } else {
            //费用活动只做一次
//            if(StringUtils.isNotBlank(sfaActivityExecutionEntity.getLastResolveEndDate())){
//                return false;
//            }
            return true;
        }

    }

    private void resolveClientActList(ActivityExecutionResolveContext context) {
        ActivityExecutionResolveContext.DayContext dayContext = context.getDayContext();
        final ActivityExecutionResolveData resolveData = dayContext.getResolveData();
        SfaActivityExecutionEntity sfaActivityExecutionEntity = resolveData.getActivityExecutionEntity();
        List<SfaActivityRangeEntity> ranges = this.sfaActivityRangeService.lambdaQuery().eq(SfaActivityRangeEntity::getActivityExecutionId, sfaActivityExecutionEntity.getId()).list();
        Map<String, List<SfaActivityRangeEntity>> rangeTypeMapping = ranges.stream().collect(Collectors.groupingBy(SfaActivityRangeEntity::getRangeType));
        rangeTypeMapping.forEach((rangeType, sfaActivityRangeEntities) -> {
            List<String> codes = sfaActivityRangeEntities.stream().map(SfaActivityRangeEntity::getCode).collect(Collectors.toList());
            //客户范围，直接发送mq
            if (SfaActivityEnum.rangeType.CUSTOMER.getVal().equals(rangeType)) {
                List<SfaClientData> clientDataList = SfaClientHelper.loadClientDataList(null, codes);
                this.sendMq(clientDataList, context);
                return;
            }
            //非客户范围，分页发送mq
            this.sendOrgAndChannelMq(context, rangeType, codes);
        });

    }

    private void sendOrgAndChannelMq(ActivityExecutionResolveContext context, String rangeType, List<String> codes) {
        PageDataAdviser pageDataAdviser = new PageDataAdviser(jobHelper.getJobBizDataPageSize());
        long total = jobHelper.getJobBizDataPageSize();
        while (true) {
            if (pageDataAdviser.hasNext(total)) {
                PageResult<SfaClientData> pageResult;
                if (SfaActivityEnum.rangeType.CHANNEL.getVal().equals(rangeType)) {
                    pageResult = SfaClientHelper.loadTerminalClientDataByChannelCodes(codes, pageDataAdviser.nextPage(), pageDataAdviser.getPageSize());
                } else {
                    pageResult = SfaClientHelper.loadTerminalClientDataByOrgCodes(codes, pageDataAdviser.nextPage(), pageDataAdviser.getPageSize());
                }
                List<SfaClientData> clientDataList = pageResult.getData();
                this.sendMq(clientDataList, context);
                total = pageResult.getCount();
            } else {
                break;
            }
        }
    }

    /**
     * 发送mq
     *
     * @param clientDataList
     * @param context
     */
    private void sendMq(List<SfaClientData> clientDataList, ActivityExecutionResolveContext context) {
//        ActivityExecutionResolveContext.DayContext dayContext = context.getDayContext();
//        final ActivityExecutionResolveData resolveData = dayContext.getResolveData();
//        RocketMQMessageBody mqMessageBody = new RocketMQMessageBody();
//        for (SfaClientData clientData : clientDataList) {
//            if(dayContext.needSend(clientData.getClientCode())){
//                resolveData.setClientData(clientData);
//                mqMessageBody.setMsgBody(JsonPropertyUtil.toJsonString(resolveData));
//                mqMessageBody.setTag(RocketMQConstant.CRM_MQ_TAG.SFA_ACTIVITY_EXECUTION_RESOLVE);
//                this.rocketMQProducer.convertAndSend(mqMessageBody);
//            }
//        }
        //进行分组集合传递，避免数据过多，导致发送mq失败
        ActivityExecutionResolveContext.DayContext dayContext = context.getDayContext();
        List<ActivityExecutionResolveData> resolveDataList = new ArrayList<>();
        final ActivityExecutionResolveData resolveData = dayContext.getResolveData();
        clientDataList.forEach(data -> {
            if (dayContext.needSend(data.getClientCode())) {
                ActivityExecutionResolveData temp = new ActivityExecutionResolveData();
                temp.setClientData(data);
                temp.setActivityExecutionEntity(resolveData.getActivityExecutionEntity());
                temp.setResolveDate(resolveData.getResolveDate());
                resolveDataList.add(temp);
            }
        });
        RocketMQMessageBody mqMessageBody = new RocketMQMessageBody();
        List<List<ActivityExecutionResolveData>> list = Lists.partition(resolveDataList, 100);
        for (List<ActivityExecutionResolveData> data : list) {
            mqMessageBody.setMsgBody(JsonPropertyUtil.toJsonStringNotEmptyVal(data));
            mqMessageBody.setTag(RocketMQConstant.CRM_MQ_TAG.SFA_ACTIVITY_EXECUTION_RESOLVE);
            this.rocketMQProducer.convertAndSend(mqMessageBody);
        }
    }

    /**
     * 当返回值为0时，则从当前日期开始解析，
     * 小于0时，并且 compensation 为true，则为[补偿解析]
     * 大于0时，则跳跃该天数，开始解析
     *
     * @param lastResolveEndDate
     * @param localDateNow
     * @param compensation
     * @return
     */
    private long getResolvedDays(String lastResolveEndDate, LocalDate localDateNow, boolean compensation) {
        LocalDate localDateLastResolveEnd;
        if (StringUtils.isBlank(lastResolveEndDate)) {
            localDateLastResolveEnd = LocalDate.now();
        } else {
            localDateLastResolveEnd = LocalDate.parse(lastResolveEndDate, CrmDateUtils.yyyyMMdd);
        }
        //当前时间 与 上次解析时间 的差值
        long days = localDateNow.toEpochDay() - localDateLastResolveEnd.toEpochDay();

        /**
         * 当 days = 0时
         */
        if (days > 0) {//上次解析时间 小于 当前时间
            if (compensation) {
                // 当前时间 与 上次解析时间 的差值 的差值（这时，他们差值内的日期可能从来没有被解析过，所以叫做补偿解析）
                // 比如：上次解析时间为 2021-04-10, 当前时间为2021-04-15。
                // 2021-04-15减去2021-04-10 差值为5天， 需要补偿解析的日期为 11、12、12、14号
                days = -days;
            } else {
                //不用补偿
                return 0;
            }
        } else if (days < 0) {//上次解析时间 大于 当前时间
            //大于0时，则跳跃该天数，开始解析
            days = Math.abs(days) + 1;
        }
        return days;
    }


}
