package com.biz.crm.ai.service.impl;

import cn.hutool.core.date.LocalDateTimeUtil;
import cn.hutool.core.util.NumberUtil;
import cn.hutool.json.JSONUtil;
import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.activity.model.SfaActivityProductEntity;
import com.biz.crm.activity.service.ISfaActivityProductService;
import com.biz.crm.ai.mapper.SfaAiTaskMapper;
import com.biz.crm.ai.model.SfaAiProductEntity;
import com.biz.crm.ai.model.SfaAiTaskEntity;
import com.biz.crm.ai.model.SfaAiTaskExtEntity;
import com.biz.crm.ai.service.SfaAiProductService;
import com.biz.crm.ai.service.SfaAiTaskExtService;
import com.biz.crm.ai.service.SfaAiTaskService;
import com.biz.crm.ai.util.AiTaskStatusEnums;
import com.biz.crm.ai.util.AiUtil;
import com.biz.crm.ai.vo.AiCallBackDto;
import com.biz.crm.ai.vo.recognition.result.GoodInfo;
import com.biz.crm.ai.vo.recognition.result.RecognitionResponse;
import com.biz.crm.ai.vo.recognition.result.RecognitionResultVo;
import com.biz.crm.ai.vo.recognition.task.ImgInfo;
import com.biz.crm.ai.vo.recognition.task.ImgLibInfo;
import com.biz.crm.ai.vo.recognition.task.Recognition;
import com.biz.crm.ai.vo.recognition.task.RecognitionTaskDto;
import com.biz.crm.ai.vo.recognition.task.RecognitionTaskVo;
import com.biz.crm.base.CrmAttachment;
import com.biz.crm.moblie.controller.visit.req.step.ActivityStepExecuteData;
import com.biz.crm.moblie.controller.visit.req.step.ActivityStepExecuteData.ActivityRequireReqVo;
import com.biz.crm.visitstep.service.ISfaVisitStepActivityExecutionService;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import java.time.LocalDate;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.stream.Collectors;
import javax.annotation.Resource;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

/**
 * @author sunx
 * @date 2021/9/26
 */
@Service
@Slf4j
public class SfaAiTaskServiceImpl<M extends BaseMapper<T>, T> extends
    ServiceImpl<SfaAiTaskMapper, SfaAiTaskEntity> implements
    SfaAiTaskService {

  @Value("${sfa.ai.appkey:}")
  private String appKey;
  @Value("${sfa.ai.enterpriseid:}")
  private String enterpriseId;
  @Value("${sfa.ai.appsecret:}")
  private String appSecret;
  @Value("${sfa.ai.callback.recognition-url:}")
  private String recognitionUrl;
  @Value("${sfa.ai.callback.repeat-url:}")
  private String repeatUrl;

  private static final String LABELS = "ALL";

  @Resource
  private SfaAiTaskExtService sfaAiTaskExtService;

  @Resource
  private ISfaActivityProductService sfaActivityProductService;

  @Resource
  private SfaAiProductService sfaAiProductService;

  @Resource
  private ISfaVisitStepActivityExecutionService sfaVisitStepActivityExecutionService;

  /**
   * 开始执行陈列识别
   *
   * @param stepId                   执行任务id
   * @param stepCode                 stepCode
   * @param executionId              活动id
   * @param clientCode               客户编码
   * @param activityRequireReqVoList 活动图片信息
   */
  @Override
  @Transactional
  public void startRecognitionTask(String stepId, String stepCode, String executionId,
      String clientCode,
      List<ActivityStepExecuteData.ActivityRequireReqVo> activityRequireReqVoList) {
    log.info("\r\n*startRecognitionTask*[{}],[{}],[{}],[{}],[{}]**\r\n", stepId, stepCode,
        executionId,
        clientCode,
        JSONUtil.toJsonStr(activityRequireReqVoList));
    if (CollectionUtils.isEmpty(activityRequireReqVoList)) {
      return;
    }
    List<String> imgList = Lists.newArrayList();
    for (ActivityRequireReqVo item : activityRequireReqVoList) {
      // 2 陈列照
      if (!"2".equals(item.getActivityRequire())) {
        continue;
      }
      if (CollectionUtils.isNotEmpty(item.getPictureList())) {
        for (CrmAttachment sub : item.getPictureList()) {
          if (StringUtils.isNotBlank(sub.getUrl())) {
            imgList.add(sub.getUrl());
          }
        }
      }
    }
    if (CollectionUtils.isEmpty(imgList)) {
      return;
    }
    log.info("enterpriseId={},appKey={},appSecret={}", enterpriseId, appKey, appSecret);
    AiUtil aiUtil = AiUtil.aiBuilder().setEnterpriseId(enterpriseId).setAppKey(appKey)
        .setAppSecret(appSecret).bulid();
    RecognitionTaskVo recognitionTaskVo = aiUtil
        .submitRecognitionTask(getRecognitionTaskDto(clientCode, imgList));

    SfaAiTaskEntity sfaAiTaskEntity = new SfaAiTaskEntity();
    sfaAiTaskEntity.setStepId(stepId);
    sfaAiTaskEntity.setStepCode(stepCode);
    sfaAiTaskEntity.setId(stepId);
    sfaAiTaskEntity.setExecutionId(executionId);
    sfaAiTaskEntity.setClientCode(clientCode);
    sfaAiTaskEntity.setTaskStatus(AiTaskStatusEnums.RECOGNITION_STATUS_2.getValue());
    sfaAiTaskEntity.setRecognitionTaskId("");
    sfaAiTaskEntity.setRepeatTaskId("");
    sfaAiTaskEntity.setErrMsg("");
    if (recognitionTaskVo != null) {
      sfaAiTaskEntity.setRecognitionTaskId(recognitionTaskVo.getTaskId());
      sfaAiTaskEntity.setTaskStatus(AiTaskStatusEnums.RECOGNITION_STATUS_1.getValue());
    }
    this.saveOrUpdate(sfaAiTaskEntity);
    SfaAiTaskExtEntity sfaAiTaskExtEntity = new SfaAiTaskExtEntity();
    sfaAiTaskExtEntity.setId(stepId);
    sfaAiTaskExtEntity.setParentId(stepId);
    sfaAiTaskExtEntity.setReqParamsJson(JSONUtil.toJsonStr(activityRequireReqVoList));
    sfaAiTaskExtEntity.setReqTime(LocalDateTimeUtil.format(LocalDate.now(), "yyyy-MM-dd HH:mm:ss"));
    sfaAiTaskExtEntity.setRecognitionJson("");
    sfaAiTaskExtEntity.setRecognitionTime("");
    sfaAiTaskExtEntity.setRepeatJson("");
    sfaAiTaskExtEntity.setRepeatTime("");
    sfaAiTaskExtService.saveOrUpdate(sfaAiTaskExtEntity);
  }

  /**
   * 开始执行陈列识别new
   *
   * @param stepId                   执行任务id
   * @param stepCode                 stepCode
   * @param executionId              活动id
   * @param clientCode               客户编码
   * @param activityRequireReqVoList 活动图片信息
   */
  @Override
  public void startRecognitionTaskNew(String stepId, String stepCode, String executionId,
      String clientCode, List<ActivityRequireReqVo> activityRequireReqVoList) {
    log.info("\r\n*startRecognitionTaskNew*[{}],[{}],[{}],[{}],[{}]**\r\n", stepId, stepCode,
        executionId,
        clientCode,
        JSONUtil.toJsonStr(activityRequireReqVoList));
    if (CollectionUtils.isEmpty(activityRequireReqVoList)) {
      return;
    }
    List<String> imgList = Lists.newArrayList();
    for (ActivityRequireReqVo item : activityRequireReqVoList) {
      // 2 陈列照
      if (!"2".equals(item.getActivityRequire())) {
        continue;
      }
      if (CollectionUtils.isNotEmpty(item.getPictureList())) {
        for (CrmAttachment sub : item.getPictureList()) {
          if (StringUtils.isNotBlank(sub.getUrl())) {
            imgList.add(sub.getUrl());
          }
        }
      }
    }
    if (CollectionUtils.isEmpty(imgList)) {
      return;
    }
    log.info("enterpriseId={},appKey={},appSecret={}", enterpriseId, appKey, appSecret);
    AiUtil aiUtil = AiUtil.aiBuilder().setEnterpriseId(enterpriseId).setAppKey(appKey)
        .setAppSecret(appSecret).bulid();

    RecognitionTaskDto recognitionTaskDto = getRecognitionTaskDto(clientCode, imgList);
    RecognitionResultVo recognitionResultVo = aiUtil
        .execRecognitionTask(recognitionTaskDto);

    SfaAiTaskEntity sfaAiTaskEntity = new SfaAiTaskEntity();
    sfaAiTaskEntity.setStepId(stepId);
    sfaAiTaskEntity.setStepCode(stepCode);
    sfaAiTaskEntity.setId(stepId);
    sfaAiTaskEntity.setExecutionId(executionId);
    sfaAiTaskEntity.setClientCode(clientCode);
    sfaAiTaskEntity.setTaskStatus(AiTaskStatusEnums.RECOGNITION_STATUS_2.getValue());
    sfaAiTaskEntity.setRecognitionTaskId("");
    sfaAiTaskEntity.setRepeatTaskId("");
    sfaAiTaskEntity.setErrMsg("");

    SfaAiTaskExtEntity sfaAiTaskExtEntity = new SfaAiTaskExtEntity();
    sfaAiTaskExtEntity.setId(stepId);
    sfaAiTaskExtEntity.setParentId(stepId);
    sfaAiTaskExtEntity.setReqParamsJson(JSONUtil.toJsonStr(recognitionTaskDto));
    sfaAiTaskExtEntity.setReqTime(LocalDateTimeUtil.format(LocalDate.now(), "yyyy-MM-dd HH:mm:ss"));
    sfaAiTaskExtEntity.setRecognitionJson("");
    sfaAiTaskExtEntity.setRecognitionTime("");
    sfaAiTaskExtEntity.setRepeatJson("");
    sfaAiTaskExtEntity.setRepeatTime("");

    if (recognitionResultVo == null || CollectionUtils
        .isEmpty(recognitionResultVo.getRecognitionResponseList())) {
      sfaAiTaskEntity.setTaskStatus(AiTaskStatusEnums.FAIL.getValue());
    } else {
      // 陈列任务检查
      displayTaskCheck(sfaAiTaskEntity, sfaAiTaskExtEntity,
          getAiRecognitionGoodsMap(recognitionResultVo));
      sfaAiTaskExtEntity.setRecognitionJson(JSONUtil.toJsonStr(recognitionResultVo));
    }
    this.saveOrUpdate(sfaAiTaskEntity);
    sfaAiTaskExtService.saveOrUpdate(sfaAiTaskExtEntity);
    if (sfaAiTaskEntity.getTaskStatus().equals(AiTaskStatusEnums.FAIL.getValue())) {
      sfaAiTaskEntity.setErrMsg("陈列要求未达标");
    }
    if (sfaAiTaskEntity.getTaskStatus().equals(AiTaskStatusEnums.FAIL.getValue()) ||
        sfaAiTaskEntity.getTaskStatus().equals(AiTaskStatusEnums.SUCCESS.getValue())) {
      // TODO 更改识别任务状态
      sfaVisitStepActivityExecutionService
          .finishDisplayVisitStep(sfaAiTaskEntity.getStepId(), sfaAiTaskEntity.getStepCode(),
              sfaAiTaskEntity.getTaskStatus(), sfaAiTaskEntity.getErrMsg());
    }
  }

  /**
   * 陈列识别回调
   *
   * @param dto
   */
  @Override
  @Transactional
  public void recognitionCallBack(AiCallBackDto dto) {
    log.info("\r\n*recognitionCallBack[{}]*\n", JSONUtil.toJsonStr(dto));
    SfaAiTaskEntity sfaAiTaskEntity = findByRecognitionTaskId(dto.getTaskId());
    if (sfaAiTaskEntity == null) {
      log.error("识别任务不存在[{}]", dto);
      return;
    }
    SfaAiTaskExtEntity sfaAiTaskExtEntity = findExtEntityById(sfaAiTaskEntity.getId());
    if (sfaAiTaskExtEntity == null) {
      log.error("识别任务不存在ext[{}]", dto);
      return;
    }
    if (sfaAiTaskEntity.getTaskStatus().equals(AiTaskStatusEnums.FAIL.getValue())
        || sfaAiTaskEntity.getTaskStatus().equals(AiTaskStatusEnums.SUCCESS.getValue())) {
      log.info("**recognitionCallBack**[{}],[{}]", JSONUtil.toJsonStr(dto),
          sfaAiTaskEntity.getTaskStatus());
      return;
    }
    // 执行识别结果查询
    log.info("enterpriseId={},appKey={},appSecret={}", enterpriseId, appKey, appSecret);
    AiUtil aiUtil = AiUtil.aiBuilder().setEnterpriseId(enterpriseId).setAppKey(appKey)
        .setAppSecret(appSecret).bulid();
    RecognitionResultVo recognitionResultVo = aiUtil.recognitionResult(dto.getTaskId());
    sfaAiTaskExtEntity
        .setRecognitionTime(LocalDateTimeUtil.format(LocalDate.now(), "yyyy-MM-dd HH:mm:ss"));
    if (recognitionResultVo == null || CollectionUtils
        .isEmpty(recognitionResultVo.getRecognitionResponseList())) {
      sfaAiTaskEntity.setTaskStatus(AiTaskStatusEnums.FAIL.getValue());
    } else {
      // 陈列任务检查
      displayTaskCheck(sfaAiTaskEntity, sfaAiTaskExtEntity,
          getAiRecognitionGoodsMap(recognitionResultVo));
      sfaAiTaskExtEntity.setRecognitionJson(JSONUtil.toJsonStr(recognitionResultVo));
    }
    this.saveOrUpdate(sfaAiTaskEntity);
    sfaAiTaskExtService.saveOrUpdate(sfaAiTaskExtEntity);
    if (sfaAiTaskEntity.getTaskStatus().equals(AiTaskStatusEnums.FAIL.getValue())) {
      sfaAiTaskEntity.setErrMsg("陈列要求未达标");
    }
    if (sfaAiTaskEntity.getTaskStatus().equals(AiTaskStatusEnums.FAIL.getValue()) ||
        sfaAiTaskEntity.getTaskStatus().equals(AiTaskStatusEnums.SUCCESS.getValue())) {
      // TODO 更改识别任务状态
      sfaVisitStepActivityExecutionService
          .finishDisplayVisitStep(sfaAiTaskEntity.getStepId(), sfaAiTaskEntity.getStepCode(),
              sfaAiTaskEntity.getTaskStatus(), sfaAiTaskEntity.getErrMsg());
    }
  }


  /**
   * 陈列任务检查
   */
  @Transactional
  public void displayTaskCheck(SfaAiTaskEntity sfaAiTaskEntity,
      SfaAiTaskExtEntity sfaAiTaskExtEntity, Map<String, Integer> goodsInfoMap) {
    Map<String, String> mapProductDisplayNum = Maps.newHashMap();
    // 获取活动商品及其陈列面数
    List<SfaActivityProductEntity> activityProductEntities = sfaActivityProductService
        .lambdaQuery()
        .eq(SfaActivityProductEntity::getActivityExecutionId, sfaAiTaskEntity.getExecutionId())
        .select(SfaActivityProductEntity::getProductCode, SfaActivityProductEntity::getDisplayNum)
        .list();
    if (CollectionUtils.isNotEmpty(activityProductEntities)) {
      mapProductDisplayNum = activityProductEntities.stream().filter(
          a -> StringUtils.isNotBlank(a.getProductCode()) && NumberUtil
              .isNumber(a.getDisplayNum()))
          .collect(Collectors.toMap(SfaActivityProductEntity::getProductCode,
              SfaActivityProductEntity::getDisplayNum, (a, b) -> a));
    }

    // 验证任务采集数据是否达标
    if (!validateTask(mapProductDisplayNum, goodsInfoMap)) {
      sfaAiTaskEntity.setTaskStatus(AiTaskStatusEnums.FAIL.getValue());
      sfaAiTaskEntity.setErrMsg("陈列要求未达标");
    } else {
      // TODO 无后续操作  直接设置状态为完成
//      sfaAiTaskEntity.setTaskStatus(AiTaskStatusEnums.RECOGNITION_STATUS_3.getValue());
      sfaAiTaskEntity.setTaskStatus(AiTaskStatusEnums.SUCCESS.getValue());
      // 执行查重任务提交
//      RecognitionTaskVo recognitionTaskVo = aiUtil.submitRecognitionTask();
    }
  }

  /**
   * 查重任务回调
   *
   * @param dto
   */
  @Override
  @Transactional
  public void repeatCallBack(AiCallBackDto dto) {

  }

  /**
   * 测试后续逻辑
   *
   * @param taskId
   * @param status
   */
  @Override
  public void test(String taskId, String status) {
    sfaVisitStepActivityExecutionService.deleteAll();
//    log.info("\r\n*test[{}]*\n", taskId);
//    SfaAiTaskEntity sfaAiTaskEntity = findByRecognitionTaskId(taskId);
//    if (sfaAiTaskEntity == null) {
//      log.error("识别任务不存在[{}]", taskId);
//      return;
//    }
//    SfaAiTaskExtEntity sfaAiTaskExtEntity = findExtEntityById(sfaAiTaskEntity.getId());
//    if (sfaAiTaskExtEntity == null) {
//      log.error("识别任务不存在ext[{}]", taskId);
//      return;
//    }
//    // 执行识别结果查询
//    sfaAiTaskEntity.setTaskStatus(status);
//    this.saveOrUpdate(sfaAiTaskEntity);
//    sfaAiTaskExtService.saveOrUpdate(sfaAiTaskExtEntity);
//
//    if (sfaAiTaskEntity.getTaskStatus().equals(AiTaskStatusEnums.FAIL.getValue()) ||
//        sfaAiTaskEntity.getTaskStatus().equals(AiTaskStatusEnums.SUCCESS.getValue())) {
//      // TODO 更改识别任务状态
//      sfaVisitStepActivityExecutionService
//          .finishDisplayVisitStep(sfaAiTaskEntity.getStepId(), sfaAiTaskEntity.getStepCode(),
//              sfaAiTaskEntity.getTaskStatus(), sfaAiTaskEntity.getErrMsg());
//    }
  }

  /**
   * 构建任务提交参数信息
   *
   * @param clientCode
   * @param imgList
   * @return
   */
  private RecognitionTaskDto getRecognitionTaskDto(String clientCode,
      List<String> imgList) {
    RecognitionTaskDto dto = new RecognitionTaskDto();
    dto.setCallbackUrl(recognitionUrl);
    dto.setImgUrls(imgList);
    Recognition recognition = new Recognition();
    ImgInfo imgInfo = new ImgInfo();
    imgInfo.setEnterpriseId(enterpriseId);
    imgInfo.setShotTime(System.currentTimeMillis() + "");
    imgInfo.setStoreId(clientCode);
    recognition.setImgInfo(imgInfo);
    ImgLibInfo imgLibInfo = new ImgLibInfo();
    imgLibInfo.setEnable(true);
    imgLibInfo.setLabels(Lists.newArrayList(LABELS));
    recognition.setImgLibInfo(imgLibInfo);
    dto.setRecognition(recognition);
    return dto;
  }

  /**
   * 获取ai识别商品对应的陈列面数据信息
   *
   * @param recognitionResultVo
   * @return
   */
  private Map<String, Integer> getAiRecognitionGoodsMap(RecognitionResultVo recognitionResultVo) {
    Map<String, Integer> map = Maps.newHashMap();
    Map<String, Integer> re = Maps.newHashMap();
    for (RecognitionResponse recognitionResponse : recognitionResultVo
        .getRecognitionResponseList()) {
      for (GoodInfo goodInfo : recognitionResponse.getGoodsInfo()) {
        map.put(goodInfo.getCode(), map.getOrDefault(goodInfo.getCode(), 0) + 1);
      }
    }
    // 映射crm商品编码信息
    if (map != null && !map.isEmpty()) {
      List<SfaAiProductEntity> mapEntities = sfaAiProductService.lambdaQuery()
          .in(SfaAiProductEntity::getCode, map.keySet())
          .select(SfaAiProductEntity::getCode, SfaAiProductEntity::getProductCode).list();
      if (CollectionUtils.isNotEmpty(mapEntities)) {
        Map<String, String> map1 = mapEntities.stream().collect(Collectors
            .toMap(SfaAiProductEntity::getProductCode, SfaAiProductEntity::getCode));
        for (Entry<String, String> item : map1.entrySet()) {
          re.put(item.getKey(), map.get(item.getValue()));
        }
      }
    }
    return re;
  }

  /**
   * 验证任务采集数据是否达标
   *
   * @param mapProductDisplayNum
   * @param goodsInfoMap
   * @return
   */
  private boolean validateTask(Map<String, String> mapProductDisplayNum,
      Map<String, Integer> goodsInfoMap) {
    if (mapProductDisplayNum == null || goodsInfoMap == null || mapProductDisplayNum.isEmpty()
        || goodsInfoMap.isEmpty()) {
      return false;
    }
    for (Entry<String, String> item : mapProductDisplayNum.entrySet()) {
      // 判断活动配置商品的陈列面数量最低限制
      if (goodsInfoMap.getOrDefault(item.getKey(), 0).compareTo(Integer.valueOf(item.getValue()))
          < 0) {
        return false;
      }
    }
    return true;
  }

  private SfaAiTaskEntity findByRecognitionTaskId(String taskId) {
    List<SfaAiTaskEntity> list = lambdaQuery()
        .eq(SfaAiTaskEntity::getRecognitionTaskId, taskId).list();
    if (CollectionUtils.isNotEmpty(list)) {
      return list.get(0);
    }
    return null;
  }

  private SfaAiTaskEntity findByRepeatTaskId(String taskId) {
    List<SfaAiTaskEntity> list = lambdaQuery()
        .eq(SfaAiTaskEntity::getRepeatTaskId, taskId).list();
    if (CollectionUtils.isNotEmpty(list)) {
      return list.get(0);
    }
    return null;
  }

  private SfaAiTaskExtEntity findExtEntityById(String id) {
    return sfaAiTaskExtService.getById(id);
  }
}
