package com.biz.crm.calculatesalary.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.calculatesalary.mapper.SfaCalculateSalaryDateMapper;
import com.biz.crm.calculatesalary.model.SfaCalculateSalaryDateEntity;
import com.biz.crm.calculatesalary.service.ISfaCalculateSalaryDateService;
import com.biz.crm.common.PageResult;
import com.biz.crm.eunm.CrmDelFlagEnum;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.nebular.sfa.calculateSalary.req.SfaCalculateSalaryDateReqVo;
import com.biz.crm.nebular.sfa.calculateSalary.resp.SfaCalculateSalaryDateRespVo;
import com.biz.crm.util.CrmBeanUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.time.LocalDate;
import java.util.List;

/**
 * 计算薪资日期接口实现
 *
 * @author lf
 * @date 2020-12-28 15:00:09
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name="SfaCalculateSalaryDateServiceExpandImpl")
public class SfaCalculateSalaryDateServiceImpl<M extends BaseMapper<T>,T> extends ServiceImpl<SfaCalculateSalaryDateMapper, SfaCalculateSalaryDateEntity> implements ISfaCalculateSalaryDateService {

    @Resource
    private SfaCalculateSalaryDateMapper sfaCalculateSalaryDateMapper;

    /**
     * 列表
     * @param reqVo
     * @return
     */
    @Override
    public PageResult<SfaCalculateSalaryDateRespVo> findList(SfaCalculateSalaryDateReqVo reqVo){
        Page<SfaCalculateSalaryDateRespVo> page = new Page<>(reqVo.getPageNum(), reqVo.getPageSize());
        List<SfaCalculateSalaryDateRespVo> list = sfaCalculateSalaryDateMapper.findList(page, reqVo);
        return PageResult.<SfaCalculateSalaryDateRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    /**
     * 查询
     * @param reqVo
     * @return sfaCalculateSalaryDateRespVo
     */
    @Override
    public SfaCalculateSalaryDateRespVo query(SfaCalculateSalaryDateReqVo reqVo){
        return null;
    }

    /**
     * 新增
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void save(SfaCalculateSalaryDateReqVo reqVo){
        SfaCalculateSalaryDateEntity entity = CrmBeanUtil.copy(reqVo,SfaCalculateSalaryDateEntity.class);
        this.save(entity);
    }

    /**
     * 更新
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void update(SfaCalculateSalaryDateReqVo reqVo){
        SfaCalculateSalaryDateEntity entity = this.getById(reqVo.getId());
        this.updateById(entity);
    }

    /**
     * 删除
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteBatch(SfaCalculateSalaryDateReqVo reqVo){
        List<SfaCalculateSalaryDateEntity> sfaCalculateSalaryDateEntities = sfaCalculateSalaryDateMapper.selectBatchIds(reqVo.getIds());
        if(CollectionUtils.isNotEmpty(sfaCalculateSalaryDateEntities)){
            sfaCalculateSalaryDateEntities.forEach(o -> {
                 o.setDelFlag(CrmDelFlagEnum.DELETE.getCode());
            });
        }
        this.updateBatchById(sfaCalculateSalaryDateEntities);
    }

    /**
     * 启用
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void enableBatch(SfaCalculateSalaryDateReqVo reqVo){
        //设置状态为启用
        List<SfaCalculateSalaryDateEntity> sfaCalculateSalaryDateEntities = sfaCalculateSalaryDateMapper.selectBatchIds(reqVo.getIds());
        if(CollectionUtils.isNotEmpty(sfaCalculateSalaryDateEntities)){
            sfaCalculateSalaryDateEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
            });
        }
        this.updateBatchById(sfaCalculateSalaryDateEntities);
    }

    /**
     * 禁用
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void disableBatch(SfaCalculateSalaryDateReqVo reqVo){
        //设置状态为禁用
        List<SfaCalculateSalaryDateEntity> sfaCalculateSalaryDateEntities = sfaCalculateSalaryDateMapper.selectBatchIds(reqVo.getIds());
        if(CollectionUtils.isNotEmpty(sfaCalculateSalaryDateEntities)){
                sfaCalculateSalaryDateEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.DISABLE.getCode());
            });
        }
        this.updateBatchById(sfaCalculateSalaryDateEntities);
    }

    /**
     * 获取指定年的节假日列表
     * @param year
     * @return
     */
    @Override
    public List<SfaCalculateSalaryDateRespVo> findHoliday(String year) {
        SfaCalculateSalaryDateReqVo reqVo = new SfaCalculateSalaryDateReqVo();
        reqVo.setYear(year);
        reqVo.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
        List<SfaCalculateSalaryDateRespVo> list = sfaCalculateSalaryDateMapper.findList(null, reqVo);
        return list;
    }

    /**
     * 获取指定日期的年度节假日列表
     * @param date
     * @return
     */
    @Override
    public List<SfaCalculateSalaryDateRespVo> findHolidayByDate(String date) {
        return findHoliday(LocalDate.parse(date).getYear() + "");
    }

    /**
     * 判断指定日期是否节假日(不判断是否有年度信息)
     * @param date
     * @return
     */
    @Override
    @Transactional(readOnly = true)
    public boolean isHoliday(String date) {
        LambdaQueryWrapper<SfaCalculateSalaryDateEntity> wrapper = new LambdaQueryWrapper<SfaCalculateSalaryDateEntity>()
                .eq(SfaCalculateSalaryDateEntity::getEnableStatus, CrmEnableStatusEnum.ENABLE.getCode())
                .eq(SfaCalculateSalaryDateEntity::getDay, date);
        SfaCalculateSalaryDateEntity entity = sfaCalculateSalaryDateMapper.selectOne(wrapper);
        if(entity != null) {
            return true;
        }
        return false;
    }

    /**
     * 是否计算薪资日期
     * @param date
     * @return
     */
    @Override
    public boolean isCalculateSalary(String date) {
        LambdaQueryWrapper<SfaCalculateSalaryDateEntity> wrapper = new LambdaQueryWrapper<SfaCalculateSalaryDateEntity>()
                .eq(SfaCalculateSalaryDateEntity::getEnableStatus, CrmEnableStatusEnum.ENABLE.getCode())
                .eq(SfaCalculateSalaryDateEntity::getDay, date);
        SfaCalculateSalaryDateEntity entity = sfaCalculateSalaryDateMapper.selectOne(wrapper);
        if(entity != null && YesNoEnum.yesNoEnum.ONE.getValue().equals(entity.getIsCalculationSalary())) {
            return true;
        }
        return false;
    }
}
