package com.biz.crm.checkin.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.biz.crm.base.ApiResultUtil;
import com.biz.crm.base.BusinessException;
import com.biz.crm.checkin.model.SfaCheckInDateEntity;
import com.biz.crm.checkin.model.SfaCheckInGroupEntity;
import com.biz.crm.checkin.model.SfaCheckInSignRecordPictureEntity;
import com.biz.crm.checkin.service.*;
import com.biz.crm.eunm.CrmDelFlagEnum;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.eunm.sfa.SfaWorkSignEnum;
import com.biz.crm.eunm.sfa.VisitStepFromEnum;
import com.biz.crm.mdm.position.MdmPositionFeign;
import com.biz.crm.nebular.mdm.PageVo;
import com.biz.crm.nebular.mdm.position.req.MdmPositionUserOrgReqVo;
import com.biz.crm.nebular.mdm.position.resp.MdmPositionUserOrgRespVo;
import com.biz.crm.nebular.sfa.checkin.req.SfaCheckInGroupReqVo;
import com.biz.crm.nebular.sfa.checkin.req.SfaCheckInSignRecordReqVo;
import com.biz.crm.nebular.sfa.checkin.resp.*;
import com.biz.crm.nebular.sfa.worksign.req.SfaSignHistoryListReqVo;
import com.biz.crm.util.*;
import com.biz.crm.worksign.service.ISfaSignFormsService;
import io.jsonwebtoken.lang.Collections;
import io.swagger.annotations.ApiModelProperty;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.common.PageResult;
import com.biz.crm.checkin.model.SfaCheckInSignRecordEntity;
import com.biz.crm.checkin.mapper.SfaCheckInSignRecordMapper;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

import lombok.extern.slf4j.Slf4j;
import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;

/**
 * 签到组-签到记录接口实现
 *
 * @author fily
 * @date 2021-01-05 13:12:38
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name="SfaCheckInSignRecordServiceExpandImpl")
public class SfaCheckInSignRecordServiceImpl<M extends BaseMapper<T>,T> extends ServiceImpl<SfaCheckInSignRecordMapper, SfaCheckInSignRecordEntity> implements ISfaCheckInSignRecordService {

    @Resource
    private SfaCheckInSignRecordMapper sfaCheckInSignRecordMapper;
    @Autowired
    private ISfaCheckInSignRecordPictureService iSfaCheckInSignRecordPictureService;
    @Autowired
    private ISfaCheckInGroupService iSfaCheckInGroupService;
    @Autowired
    private ISfaSignFormsService formsService;
    @Autowired
    private ISfaCheckInSignGroupReportService iSfaCheckInSignGroupReportService;
    @Autowired
    private MdmPositionFeign mdmPositionFeign;
    @Resource
    private ISfaCheckInDateService iSfaCheckInDateService;

    /**
     * 列表
     * @param reqVo
     * @return
     */
    @Override
    public PageResult<SfaCheckInSignRecordRespVo> findList(SfaSignHistoryListReqVo reqVo){
        UserRedis userRedis = UserUtils.getUser();
        Page<SfaCheckInSignRecordRespVo> page = new Page<>(reqVo.getPageNum(), reqVo.getPageSize());

        List<SfaCheckInSignRecordRespVo> list = sfaCheckInSignRecordMapper.findSignHistory(page, reqVo);
//        if(list != null && list.size() > 0 && list.get(0) != null) {
//            list.forEach(vo -> {
//                vo.setPictureRespVoList(iSfaCheckInSignRecordPictureService.findList(vo.getId()));
//            });
//        }
        return PageResult.<SfaCheckInSignRecordRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    /**
     * 查询
     * @param id
     * @return sfaCheckInSignRecordRespVo
     */
    @Override
    public SfaCheckInSignRecordRespVo query(String id){
        AssertUtils.isNotEmpty(id,"传入ID参数为空");
        SfaCheckInSignRecordEntity entity = sfaCheckInSignRecordMapper.selectOne(
                Wrappers.lambdaQuery(SfaCheckInSignRecordEntity.class).eq(SfaCheckInSignRecordEntity::getId, id)
        );
        if(entity == null) {
            throw new BusinessException("签到记录id错误");
        }

        SfaCheckInSignRecordRespVo respVo = CrmBeanUtil.copy(entity, SfaCheckInSignRecordRespVo.class);
        respVo.setPictureRespVoList(iSfaCheckInSignRecordPictureService.findList(entity.getId()));
        return respVo;
    }

    /**
     * 我的签到组列表
     * @return
     */
    @Override
    public PageResult<SfaCheckInGroupRespVo> myCheckInGroup(PageVo pageVo) {
        UserRedis user =  UserUtils.getUser();
        if(user == null) {
            throw new BusinessException("请登录");
        }
        // 查询条件
        SfaCheckInGroupReqVo reqVo = new SfaCheckInGroupReqVo();
        // 分页信息
        reqVo.setPageNum(pageVo.getPageNum());
        reqVo.setPageSize(pageVo.getPageSize());
        reqVo.setEnableStatus(CrmDelFlagEnum.NORMAL.getCode());
        // 1、当前日期，2人员或组织
        reqVo.setCheckInDate(LocalDate.now().toString());
        reqVo.setUserOrOrgByUser(user.getUsername());
        reqVo.setUserOrOrgByOrg(user.getOrgcode());
        return iSfaCheckInGroupService.myCheckInGroup(reqVo);
    }

    /**
     * 自由签到
     * @param reqVo
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void checkInGroupSign(SfaCheckInSignRecordReqVo reqVo) {
        LocalDateTime localDateTime = LocalDateTime.now();
        String signDate = localDateTime.format(CrmDateUtils.yyyyMMdd);
        // 打卡人员信息
        UserRedis user = UserUtils.getUser();

        SfaCheckInSignRecordEntity entityOld= new SfaCheckInSignRecordEntity();
        //如果有签到组编码 说明是选择了签到组的自由签到
        if(StringUtils.isNotEmpty(reqVo.getGroupCode())){
            AssertUtils.isNotEmpty(reqVo.getCheckInTypeId(),"请选择签到类型");
            //查询同样的签到组中、同样的签到时间、在同一天中是否有签到
            entityOld = this.lambdaQuery().eq(SfaCheckInSignRecordEntity :: getGroupCode, reqVo.getGroupCode() )
                    .eq(SfaCheckInSignRecordEntity :: getCheckInTypeId, reqVo.getCheckInTypeId() )
                    .eq(SfaCheckInSignRecordEntity :: getSignDate, signDate)
                    .eq(SfaCheckInSignRecordEntity::getUserName,user.getUsername()).one();
            if(null == entityOld){
                //若没有签过到、就新建一个签到组签到实体
                entityOld = CrmBeanUtil.copy(reqVo, SfaCheckInSignRecordEntity.class);
            }else {
                //否则已经签到过——（同一天只能对每一种签到类型签到一次）将提示：“已存在该类型的签到！”
                throw new BusinessException("今日已存在该类型的自由签到！");
            }
        }
        CrmBeanUtil.copyPropertiesIgnoreEmpty(reqVo, entityOld);
        SfaCheckInSignRecordEntity entity = entityOld;

        if(user == null){
            throw new BusinessException("请登录");
        }
        if(!Collections.isEmpty(reqVo.getPictureReqVos())) {
            reqVo.getPictureReqVos().forEach(vo -> {
                if(StringUtils.isEmpty(vo.getPicUrl()) && StringUtils.isEmpty(vo.getPicPath())) {
                    throw new BusinessException("照片信息不完整");
                }
            });
        }
        entity.setUserName(user.getUsername());
        entity.setRealName(user.getRealname());
        entity.setOrgName(user.getOrgname());
        entity.setOrgCode(user.getOrgcode());
        entity.setPosName(user.getPosname());
        entity.setPosCode(user.getPoscode());
        // 设置上级组织信息
        MdmPositionUserOrgRespVo mdm = formsService.getUserOrgInfo(entity.getUserName());
        entity.setParentOrgCode(mdm.getParentOrgCode());
        entity.setParentOrgName(mdm.getParentOrgName());
        // 签到时间和日期
        entity.setSignDate(signDate);
        entity.setSignTime(localDateTime.format(CrmDateUtils.HHmmss));
        entity.setSignPlace(reqVo.getWorkSignPlace());
        // 指定签到组并选择签到类型
        if(StringUtils.isNotEmpty(entity.getGroupCode())) {
            /**
             * 校验
              */
            AssertUtils.isNotEmpty(entity.getCheckInTypeId(), "打卡类型不能为空");
            SfaCheckInGroupRespVo groupRespVo = iSfaCheckInGroupService.getByGroupCode(entity.getGroupCode());
            if(groupRespVo == null) {
                throw new BusinessException("签到组编码错误");
            }
            // 打卡日期
            // 判断时间类型
            if(VisitStepFromEnum.timeType.DIY_TIME.getValue().equals(groupRespVo.getTimeType())){
                List<SfaCheckInDateRespVo> dateRespVos = groupRespVo.getDateRespVos().stream().filter(vo -> {
                    //校验是否在签到规则时间段中
                    LocalDate dateStart = LocalDate.parse(vo.getStartTime());
                    LocalDate dateEnd = LocalDate.parse(vo.getEndTime());
                    LocalDate dateSignDate = LocalDate.parse(entity.getSignDate());
                    int compareToStart = dateStart.compareTo(dateSignDate);
                    int compareToEnd = dateEnd.compareTo(dateSignDate);
                    if(compareToStart<=0&&compareToEnd>=0){
                        return true;
                    }
                    return false;
                }).collect(Collectors.toList());
                if(dateRespVos == null || dateRespVos.size() == 0 || dateRespVos.get(0) == null) {
                    throw new BusinessException("当前日期不能打卡");
                }
                SfaCheckInDateRespVo dateRespVo = dateRespVos.get(0);
                entity.setCheckInDateId(dateRespVo.getId());
            }else {
                //长期有效的就无对应的时间明细——存null
                entity.setCheckInDateId(null);
            }
            // 打卡类型
            List<SfaCheckInTypeRespVo> typeRespVos = groupRespVo.getTypeRespVos().stream().filter(vo -> vo.getId().equals(entity.getCheckInTypeId())).collect(Collectors.toList());
            if(typeRespVos == null || typeRespVos.size() == 0 || typeRespVos.get(0) == null) {
                throw new BusinessException("打卡类型id不存在");
            }
            SfaCheckInTypeRespVo typeRespVo = typeRespVos.get(0);
            // 打卡照片
            if(YesNoEnum.yesNoEnum.YES.getValue().equals(typeRespVo.getPhotograph())) {
                AssertUtils.isNotEmpty(reqVo.getPictureReqVos(), "打卡照片不能为空");
            }
            // 打卡时间
            if(YesNoEnum.yesNoEnum.YES.getValue().equals(typeRespVo.getAppointTime())) {
                AssertDateTimeUtils.checkBetweenTime(entity.getSignTime(), typeRespVo.getBeginTime(), typeRespVo.getEndTime(), "当前时间不能打卡");
            }
            // 重复打卡
            SfaCheckInSignRecordEntity old = sfaCheckInSignRecordMapper.selectOne(Wrappers.lambdaQuery(SfaCheckInSignRecordEntity.class)
                    .eq(SfaCheckInSignRecordEntity::getGroupCode, entity.getGroupCode())
                    .eq(SfaCheckInSignRecordEntity::getCheckInDateId, entity.getCheckInDateId())
                    .eq(SfaCheckInSignRecordEntity::getCheckInTypeId, entity.getCheckInTypeId())
                    .eq(SfaCheckInSignRecordEntity::getUserName, entity.getUserName())
            );
            if(old != null) {
                throw new BusinessException("当前打卡类型已经签到，请勿重复打卡！");
            }
            // 电子围栏 打卡地址 校验
            YesNoEnum.yesNoEnum yesNoEnum = this.checkSignPlace(groupRespVo, entity);
            /**
             * 装载签到组数据
              */
            entity.setGroupName(groupRespVo.getGroupName());
//            entity.setCheckInDateId(dateRespVo.getId());
            entity.setCheckInTypeName(typeRespVo.getTypeName());
            // 保存签到信息
            this.saveOrUpdate(entity);
            // 更新统计打卡信息
            iSfaCheckInSignGroupReportService.refreshReportByUserName(entity, yesNoEnum);
        }
        // 保存签到信息
        this.saveOrUpdate(entity);
        // 保存签到图片
        if(!Collections.isEmpty(reqVo.getPictureReqVos())) {
            reqVo.getPictureReqVos().forEach(vo -> {
                SfaCheckInSignRecordPictureEntity pictureEntity = CrmBeanUtil.copy(vo, SfaCheckInSignRecordPictureEntity.class);
                pictureEntity.setSignRecordId(entity.getId());
                iSfaCheckInSignRecordPictureService.save(pictureEntity);
            });
        }
    }

    /**
     *  电子围栏 打卡地址 校验
     * @param groupRespVo
     * @param entity
     */
    private YesNoEnum.yesNoEnum checkSignPlace(SfaCheckInGroupRespVo groupRespVo, SfaCheckInSignRecordEntity entity){
        String electronFenceType = groupRespVo.getElectronFenceType();
        //无电子围栏
        if(SfaWorkSignEnum.ElectronFenceEnum.NONE.getVal().equals(electronFenceType)){
            return YesNoEnum.yesNoEnum.NO;
        }

        //是否超出电子围栏范围
        boolean outOfRange = true;
        List<SfaCheckInPlaceRespVo> placeRespVos = groupRespVo.getPlaceRespVos();
        if(CollectionUtil.listNotEmpty(placeRespVos)){
            for (SfaCheckInPlaceRespVo placeRespVo : placeRespVos) {
                //规定距离
                double distance = Double.parseDouble(placeRespVo.getSignPlaceScope());
                //两地的经纬度
                double latinOne = Double.parseDouble(placeRespVo.getPlaceLatin());
                double lnginOne = Double.parseDouble(placeRespVo.getPlaceLngin());

                double lnginTwo = Double.parseDouble(entity.getLngIn());
                double latinTwo = Double.parseDouble(placeRespVo.getPlaceLatin());
                //两相距距离
                double distanceTrue = DistanceUtil.getDistance(latinOne, lnginOne, latinTwo, lnginTwo);
                //只要有一个在规定距离内，则视为正常
                if(distance > distanceTrue){
                    outOfRange = false;
                    break;
                }
            }
        }
        //超出范围
        if(outOfRange){
            if(SfaWorkSignEnum.ElectronFenceEnum.NO_OUT_SIGN.getVal().equals(electronFenceType)){
                throw  new  BusinessException("离规定打卡地点过远");
            }
            if(SfaWorkSignEnum.ElectronFenceEnum.OUT_SIGN_EX.getVal().equals(electronFenceType)){
                return YesNoEnum.yesNoEnum.YES;
            }
            return YesNoEnum.yesNoEnum.NO;
        }
        return YesNoEnum.yesNoEnum.NO;

    }
}
