package com.biz.crm.mqconsumer;

import com.biz.crm.eunm.sfa.SfaVisitEnum;
import com.biz.crm.moblie.controller.visit.req.GetVisitListReq;
import com.biz.crm.mq.AbstractRocketMQConsumer;
import com.biz.crm.mq.RocketMQConstant;
import com.biz.crm.mq.RocketMQMessageBody;
import com.biz.crm.nebular.mdm.position.resp.MdmPositionPageRespVo;
import com.biz.crm.service.RedisService;
import com.biz.crm.util.CollectionUtil;
import com.biz.crm.util.CrmBeanUtil;
import com.biz.crm.util.CrmDateUtils;
import com.biz.crm.util.JsonPropertyUtil;
import com.biz.crm.visitinfo.model.SfaVisitPlanInfoEntity;
import com.biz.crm.visitinfo.model.SfaVisitPlanInfoRedisData;
import com.biz.crm.visitinfo.req.PlanInfoSettlementReq;
import com.biz.crm.visitinfo.service.ISfaVisitPlanInfoService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.rocketmq.spring.annotation.ConsumeMode;
import org.apache.rocketmq.spring.annotation.RocketMQMessageListener;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.time.LocalDate;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;


/**
 * 拜访计划结算
 *
 * @author: luoqi
 * @Date: 2021-4-13 16:57
 * @version: V1.0
 * @Description:
 */
@Slf4j
@Component
@RocketMQMessageListener(topic = "${rocketmq.topic}" + "${rocketmq.environment-variable}",
        /**
         * tag
         * 可用 || 监听多个tag： "tag1 || tag2 || tag3"
         * 请把tag定义在 RocketMQConstant.CRM_MQ_TAG 内统一维护
        */
        selectorExpression = RocketMQConstant.CRM_MQ_TAG.VISIT_PLAN_INFO_SETTLEMENT,
        /**
         * 相同分组下 consumer 可自动负载均衡
         * 请把consumerGroup定义在 RocketMQConstant.CONSUMER_GROUP 内统一维护
        */
        consumerGroup = RocketMQConstant.CONSUMER_GROUP.VISIT_PLAN_INFO_SETTLEMENT + "${rocketmq.environment-variable}",
        /**
         * 默认集群消费， 可以设置 ConsumeMode.ORDERLY 使用广播消费
         * 也可使用集群模式模拟广播模式：
         * 启动多个不同 consumerGroup 的consumer实例
        */
        consumeMode = ConsumeMode.CONCURRENTLY)
public class VisitPlanInfoSettlementConsumer extends AbstractRocketMQConsumer {

    @Resource
    private RedisService redisService;
    @Resource
    private ISfaVisitPlanInfoService sfaVisitPlanInfoService;


    //获取消息
    @Override
    protected Object handleMessage(RocketMQMessageBody message) {
        String msgBody = message.getMsgBody();
        PlanInfoSettlementReq req = JsonPropertyUtil.toObject(msgBody, PlanInfoSettlementReq.class);
        List<PlanInfoSettlementReq.VisitUserPosition> users = req.getVisitUserPositionList();
        if (CollectionUtil.listNotEmptyNotSizeZero(users)) {
            users.forEach(x -> {
                PlanInfoSettlementReq settlementReq = new PlanInfoSettlementReq();
                settlementReq.setVisitDate(req.getVisitDate());
                settlementReq.setVisitPositionCode(x.getVisitPositionCode());
                settlementReq.setVisitUserName(x.getVisitUserName());
                this.planInfoSettlement(settlementReq);
            });
        }
        return "结算完成";
    }

    /**
     * 结算给定日期的拜访计划明细
     *
     * @param req
     */
    protected void planInfoSettlement(PlanInfoSettlementReq req) {

        List<SfaVisitPlanInfoEntity> planInfoEntities = this.loadUserVisitPlan(req);

        for (SfaVisitPlanInfoEntity planInfoEntity : planInfoEntities) {
            SfaVisitPlanInfoRedisData sfaVisitPlanInfoRedisData = null;
            //是否有执行缓存数据（拜访中才有执行数据的缓存）
            if (this.hasRedisData(planInfoEntity)) {
                //拜访计划明细执行
                sfaVisitPlanInfoRedisData = (SfaVisitPlanInfoRedisData) this.redisService
                        .hmget(SfaVisitPlanInfoRedisData.getInstance().redisHash(planInfoEntity.getVisitDate(), planInfoEntity.getVisitUserName(), planInfoEntity.getVisitPosCode(), planInfoEntity.getVisitBigType()).toString()
                                , planInfoEntity.getRedisHashKey());
            }
            //持久化接口
            this.sfaVisitPlanInfoService.dataDurability(planInfoEntity, sfaVisitPlanInfoRedisData);

            //更新缓存
            this.redisService.hmset(planInfoEntity.redisHash(planInfoEntity.getVisitDate(), planInfoEntity.getVisitUserName(), planInfoEntity.getVisitPosCode(), planInfoEntity.getVisitBigType()).toString()
                    , planInfoEntity.buildRedisDataForWrite(), SfaVisitPlanInfoEntity.CACHE_TIME);
            if (null != sfaVisitPlanInfoRedisData) {
                //计划明细执行数据
                this.redisService.hmset(sfaVisitPlanInfoRedisData.redisHash(planInfoEntity.getVisitDate(), planInfoEntity.getVisitUserName(), planInfoEntity.getVisitPosCode(), planInfoEntity.getVisitBigType()).toString()
                        , sfaVisitPlanInfoRedisData.buildRedisDataForWrite(), SfaVisitPlanInfoEntity.CACHE_TIME);
            }
        }

    }

    /**
     * 是否有执行缓存数据（拜访中才有执行数据的缓存）
     *
     * @param planInfoEntity
     * @return
     */
    protected boolean hasRedisData(SfaVisitPlanInfoEntity planInfoEntity) {
        if (SfaVisitEnum.VisitBigType.HELP_VISIT.getVal().equals(planInfoEntity.getVisitBigType())) {
            if (SfaVisitEnum.HelpVisitStatus.NO_COMMIT.getVal().equals(planInfoEntity.getVisitStatus())) {
                return true;
            }
        } else {
            if (SfaVisitEnum.visitStatus.V5.getVal().equals(planInfoEntity.getVisitStatus())) {
                return true;
            }
        }
        return false;
    }

    /**
     * 加载用户指定日期的拜访计划
     *
     * @param settlementReq
     * @return
     */
    protected List<SfaVisitPlanInfoEntity> loadUserVisitPlan(PlanInfoSettlementReq settlementReq) {
        GetVisitListReq req = CrmBeanUtil.copy(settlementReq, GetVisitListReq.class);
        //拜访
        req.setVisitBigType(SfaVisitEnum.VisitBigType.VISIT.getVal());
        List<SfaVisitPlanInfoEntity> planInfoTemps = this.sfaVisitPlanInfoService.doGetVisitInfoList(req);
        //协访
        req.setVisitBigType(SfaVisitEnum.VisitBigType.HELP_VISIT.getVal());
        planInfoTemps.addAll(this.sfaVisitPlanInfoService.doGetVisitInfoList(req));
        //陌拜
        req.setVisitBigType(SfaVisitEnum.VisitBigType.UNFAMILIAR_VISIT.getVal());
        planInfoTemps.addAll(this.sfaVisitPlanInfoService.doGetVisitInfoList(req));
        //筛选待拜访和拜访中的计划,并修改成对应状态
        List<SfaVisitPlanInfoEntity> planInfoEntities = planInfoTemps.stream().filter(v -> {
            LocalDate date = LocalDate.parse(v.getVisitDate(), CrmDateUtils.yyyyMMdd);
            v.setVisitDateOfYearMonth(date.format(CrmDateUtils.yyyyMM));
            v.setVisitDateOfYear(date.format(CrmDateUtils.yyyy));
            if (SfaVisitEnum.VisitBigType.HELP_VISIT.getVal().equals(v.getVisitBigType())) {
                if (SfaVisitEnum.HelpVisitStatus.READY_HELP_DEFENSE.getVal().equals(v.getVisitStatus())) {
                    v.setVisitStatus(SfaVisitEnum.HelpVisitStatus.ALREADY_OVERDUE.getVal());
                    v.setVisitStatusName(SfaVisitEnum.HelpVisitStatus.ALREADY_OVERDUE.getDesc());
                    return true;
                }
                if (SfaVisitEnum.HelpVisitStatus.IN_HELP_DEFENSE.getVal().equals(v.getVisitStatus())) {
                    v.setVisitStatus(SfaVisitEnum.HelpVisitStatus.NO_COMMIT.getVal());
                    v.setVisitStatusName(SfaVisitEnum.HelpVisitStatus.NO_COMMIT.getDesc());
                    v.setRemarks("该用户在执行业务时未主动提交执行数据（任务/后台批量结算）");
                    return true;
                }
            } else {
                if (SfaVisitEnum.visitStatus.V1.getVal().equals(v.getVisitStatus())) {
                    v.setVisitStatus(SfaVisitEnum.visitStatus.V6.getVal());
                    v.setVisitStatusName(SfaVisitEnum.visitStatus.V6.getDesc());
                    return true;
                }
                if (SfaVisitEnum.visitStatus.V2.getVal().equals(v.getVisitStatus())) {
                    v.setVisitStatus(SfaVisitEnum.visitStatus.V5.getVal());
                    v.setVisitStatusName(SfaVisitEnum.visitStatus.V5.getDesc());
                    v.setRemarks("该用户在执行业务时未主动提交执行数据（任务/后台批量结算）");
                    return true;
                }
            }

            return false;
        }).collect(Collectors.toList());
        return planInfoEntities;
    }
}
