package com.biz.crm.notice.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.base.BusinessException;
import com.biz.crm.common.PageResult;
import com.biz.crm.eunm.CrmDelFlagEnum;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.eunm.sfa.SfaCodeEnum;
import com.biz.crm.mdm.org.MdmOrgFeign;
import com.biz.crm.nebular.mdm.org.req.MdmOrgReqVo;
import com.biz.crm.nebular.sfa.notice.req.*;
import com.biz.crm.nebular.sfa.notice.resp.SfaNoticeFileRespVo;
import com.biz.crm.nebular.sfa.notice.resp.SfaNoticeReadRespVo;
import com.biz.crm.nebular.sfa.notice.resp.SfaNoticeReceivingRespVo;
import com.biz.crm.nebular.sfa.notice.resp.SfaNoticeRespVo;
import com.biz.crm.notice.mapper.SfaNoticeFileMapper;
import com.biz.crm.notice.mapper.SfaNoticeMapper;
import com.biz.crm.notice.mapper.SfaNoticeReadMapper;
import com.biz.crm.notice.mapper.SfaNoticeReceivingMapper;
import com.biz.crm.notice.model.SfaNoticeEntity;
import com.biz.crm.notice.model.SfaNoticeFileEntity;
import com.biz.crm.notice.model.SfaNoticeReceivingEntity;
import com.biz.crm.notice.service.ISfaNoticeFileService;
import com.biz.crm.notice.service.ISfaNoticeReadService;
import com.biz.crm.notice.service.ISfaNoticeReceivingService;
import com.biz.crm.notice.service.ISfaNoticeService;
import com.biz.crm.util.*;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 公告管理 接口实现
 *
 * @author liuhongming
 * @date 2020-09-17 17:14:42
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name = "SfaNoticeServiceExpandImpl")
@Transactional(rollbackFor = Exception.class)
public class SfaNoticeServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<SfaNoticeMapper, SfaNoticeEntity> implements ISfaNoticeService {

    @Resource
    private SfaNoticeMapper sfaNoticeMapper;
    @Resource
    private SfaNoticeFileMapper noticeFileMapper;
    @Resource
    private SfaNoticeReceivingMapper noticeReceivingMapper;

    @Resource
    private SfaNoticeReadMapper sfaNoticeReadMapper;

    @Autowired
    private ISfaNoticeFileService iSfaNoticeFileService;
    @Autowired
    private ISfaNoticeReceivingService iSfaNoticeReceivingService;
    @Autowired
    private ISfaNoticeReadService iSfaNoticeReadService;

    @Resource
    private MdmOrgFeign mdmOrgFeign;


    /**
     * 列表
     *
     * @param reqVo
     * @return
     */
    @Override
    public PageResult<SfaNoticeRespVo> findList(SfaNoticeReqVo reqVo) {
        Page<SfaNoticeRespVo> page = new Page<>(reqVo.getPageNum(), reqVo.getPageSize());
        List<SfaNoticeRespVo> list = sfaNoticeMapper.findList(page, reqVo);
        if (CollectionUtil.listNotEmptyNotSizeZero(list)) {
            List<String> noticeCodes = list.stream().map(SfaNoticeRespVo::getNoticeCode).collect(Collectors.toList());
            SfaNoticeReadReqVo readReqVo = new SfaNoticeReadReqVo();
            readReqVo.setNoticeCodes(noticeCodes);
            readReqVo.setPageSize(-1);
            List<SfaNoticeReadRespVo> readRespVoList = iSfaNoticeReadService.findList(readReqVo).getData();
            Map<String, List<SfaNoticeReadRespVo>> readRespMap = readRespVoList.stream()
                    .collect(Collectors.groupingBy(SfaNoticeReadRespVo::getNoticeCode));
            for (SfaNoticeRespVo noticeVo : list) {
                noticeVo.setNoticeOrg(noticeVo.getCreateOrgName());
                noticeVo.setNoticeDate(noticeVo.getCreateDate());
                noticeVo.setEnableStatusName(CrmEnableStatusEnum.getDesc(noticeVo.getEnableStatus()));
                List<SfaNoticeReadRespVo> noticeReadRespVos = readRespMap.get(noticeVo.getNoticeCode());
                if (noticeReadRespVos != null && noticeReadRespVos.size() > 0) {
                    //todo 访问人数是否进行去重处理
                    noticeVo.setVisitNum(noticeReadRespVos.size());
                    Set<String> readusers = noticeReadRespVos.stream().map(SfaNoticeReadRespVo::getUserName).collect(Collectors.toSet());
                    noticeVo.setReadUserNum(readusers.size());
                } else {
                    noticeVo.setVisitNum(0);
                    noticeVo.setReadUserNum(0);
                }
            }
        }
        return PageResult.<SfaNoticeRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    /**
     * 小程序公告列表
     *
     * @param reqVo
     * @return
     */
    @Override
    public PageResult<SfaNoticeRespVo> findListByMiniPrg(SfaNoticeMiniReqVo reqVo) {
        UserRedis user = UserUtils.getUser();
        Page<SfaNoticeRespVo> page = new Page<>(reqVo.getPageNum(), reqVo.getPageSize());
        //账号
        reqVo.setUserName(user.getUsername());
        //当前组织code
        reqVo.setThisCode(user.getOrgcode());
        //职位
        reqVo.setThisPost(user.getPoscode());
        List<SfaNoticeRespVo> miniPrgList = this.sfaNoticeMapper.findListMiniPrg(page, reqVo);
        List<String> notices = new ArrayList<>();
        for (SfaNoticeRespVo sfaNoticeRespVo : miniPrgList) {
            notices.add(sfaNoticeRespVo.getNoticeCode());
        }
        if (!CollectionUtil.listEmpty(notices)) {
            Map map = this.sfaNoticeReadMapper.findIsItRight(notices, reqVo.getUserName(), reqVo.getThisPost(),user.getOrgcode());
            if (!map.isEmpty()) {
                for (SfaNoticeRespVo sfaNoticeRespVo : miniPrgList) {
                    if (map.containsKey(sfaNoticeRespVo.getNoticeCode())) {
                        //是否已读
                        sfaNoticeRespVo.setHaveRead("1");
                        continue;
                    }
                    sfaNoticeRespVo.setHaveRead("0");
                }
            }
        }
        return PageResult.<SfaNoticeRespVo>builder()
                .data(miniPrgList)
                .count(page.getTotal())
                .build();
    }

    /**
     * 小程序公告最新
     *
     * @return
     */
    @Override
    public SfaNoticeRespVo queryMiniPrgNew() {
        SfaNoticeMiniReqVo vo = new SfaNoticeMiniReqVo();
        vo.setPageNum(0);
        vo.setPageSize(1);
        List<SfaNoticeRespVo> data = this.findListByMiniPrg(vo).getData();
        if (CollectionUtil.listEmpty(data)) {
            return new SfaNoticeRespVo();
        }
        return data.get(0);
    }


    /**
     * 小程序详情
     *
     * @param id
     * @return
     */
    @Override
    public SfaNoticeRespVo queryMiniPrg(String id) {
        AssertUtils.isNotEmpty(id, "ID不能为空");
        SfaNoticeRespVo sfaNoticeRespVo = sfaNoticeMapper.queyMimiPrg(id);
        String noticeCode = sfaNoticeRespVo.getNoticeCode();
        List<SfaNoticeFileRespVo> miniList = noticeFileMapper.findMiniList(noticeCode);
        if (!CollectionUtil.listEmpty(miniList)) {
            //截取文件类型后缀
            miniList.forEach(m -> m.setFileSuffix
                    (StringUtils.isEmpty(m.getFileName()) ? "" : m.getFileName().substring(m.getFileName().lastIndexOf(".") + 1)));
        }
        sfaNoticeRespVo.setNoticeFileRespVos(miniList);
        return sfaNoticeRespVo;
    }

    /**
     * 上一页下一页
     *
     * @param id
     * @param upBelow
     * @return
     */
    @Override
    public SfaNoticeRespVo queryMiniPrgByUpBelow(String id, String upBelow) {
        AssertUtils.isNotEmpty(upBelow, "翻页类型不能为空");
        UserRedis user = UserUtils.getUser();
        MdmOrgReqVo mdmOrgReqVo = new MdmOrgReqVo();
        mdmOrgReqVo.setOrgCode(user.getOrgcode());

        String noticeId = sfaNoticeMapper.queryMinPrgUpOrBelow(id, upBelow, user.getOrgcode());
        if (StringUtils.isEmpty(noticeId)) {
            if ("0".equals(upBelow)) {
                AssertUtils.isNotEmpty(noticeId, "已经是最后一条了");
            } else {
                AssertUtils.isNotEmpty(noticeId, "已经是第一条了");
            }
        }
        SfaNoticeRespVo sfaNoticeRespVo = this.queryMiniPrg(noticeId);
        List<SfaNoticeFileRespVo> miniList = noticeFileMapper.findMiniList(sfaNoticeRespVo.getNoticeCode());
        sfaNoticeRespVo.setNoticeFileRespVos(miniList);
        return sfaNoticeRespVo;
    }

    /**
     * 查询
     *
     * @param reqVo
     * @return sfaNoticeRespVo
     */
    @Override
    public SfaNoticeRespVo query(SfaNoticeReqVo reqVo) {
        if (null == reqVo || (StringUtils.isEmpty(reqVo.getId())) && CollectionUtil.listEmpty(reqVo.getIds())) {
            return new SfaNoticeRespVo();
        }
        List<SfaNoticeRespVo> list = this.findList(reqVo).getData();
        if (CollectionUtils.isEmpty(list)) {
            return new SfaNoticeRespVo();
        }
        SfaNoticeRespVo sfaNoticeRespVo = list.get(0);
        //查询公告附件
        SfaNoticeFileReqVo fileReqVo = new SfaNoticeFileReqVo();
        fileReqVo.setNoticeCode(sfaNoticeRespVo.getNoticeCode());
        fileReqVo.setPageSize(-1);
        List<SfaNoticeFileRespVo> fileRespVos =
                iSfaNoticeFileService.findList(fileReqVo).getData();
        if (CollectionUtil.listNotEmptyNotSizeZero(fileRespVos)) {
            sfaNoticeRespVo.setNoticeFileRespVos(fileRespVos);
        }
        //查询公告发布范围
        SfaNoticeReceivingReqVo receivingReqVo = new SfaNoticeReceivingReqVo();
        receivingReqVo.setNoticeCode(sfaNoticeRespVo.getNoticeCode());
        receivingReqVo.setPageSize(-1);
        List<SfaNoticeReceivingRespVo> receivingRespVos = iSfaNoticeReceivingService.findList(receivingReqVo).getData();
        sfaNoticeRespVo.setNoticeReceivingRespVos(receivingRespVos);
        return sfaNoticeRespVo;
    }

    /**
     * 新增,编辑,校验
     *
     * @param reqVo
     */
    public void saveCheck(SfaNoticeReqVo reqVo) {
        AssertUtils.isNotEmpty(reqVo.getNoticeTitle(), "请输入公告标题");
        AssertUtils.isNotEmpty(reqVo.getNoticeType(), "请选择公告类型");
        AssertUtils.isNotEmpty(reqVo.getBeginDate(), "有效开始时间不能为空");
        AssertUtils.isNotEmpty(reqVo.getNoticeContent(), "请输入公告内容");
        AssertUtils.isNotEmpty(reqVo.getNoticeReceivingReqVos(), "请选择公告范围");
        LocalDateTime beginDate = null;
        try {
            beginDate = LocalDateTime.parse(reqVo.getBeginDate(),CrmDateUtils.yyyyMMddHHmmss);
        } catch (Exception e) {
            throw new BusinessException("有效开始时间格式错误");
        }
        AssertUtils.isNotEmpty(reqVo.getEndDate(), "有效结束时间不能为空");
        LocalDateTime endDate = null;
        try {
            endDate = LocalDateTime.parse(reqVo.getEndDate(),CrmDateUtils.yyyyMMddHHmmss);
        } catch (Exception e) {
            throw new BusinessException("有效结束时间格式错误");
        }
        if (!(beginDate.isBefore(endDate))) {
            throw new BusinessException("有效开始时间不能大于有效结束时间");
        }
        AssertUtils.isNotEmpty(reqVo.getNoticeContent(), "公告内容不能为空");
    }

    /**
     * 新增
     *
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void save(SfaNoticeReqVo reqVo) {
        this.saveCheck(reqVo);
        SfaNoticeEntity entity = CrmBeanUtil.copy(reqVo, SfaNoticeEntity.class);
        //公告编码
        String code = CodeUtil.generateCode(SfaCodeEnum.NoticeCode.SFA_NOTICE_CODE.getVal());
        entity.setNoticeCode(code);
        entity.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
        entity.setDelFlag(CrmDelFlagEnum.NORMAL.getCode());
        this.save(entity);
        this.saveData(reqVo, entity);
    }

    /**
     * 保存业务明细数据
     *
     * @param reqVo
     * @param entity
     */
    public void saveData(SfaNoticeReqVo reqVo, SfaNoticeEntity entity) {
        //公告发布范围
        if (CollectionUtil.listNotEmptyNotSizeZero(reqVo.getNoticeReceivingReqVos())) {
            List<SfaNoticeReceivingEntity> receivingEntityList = reqVo.getNoticeReceivingReqVos().stream().map(o -> {
                SfaNoticeReceivingEntity receivingEntity = new SfaNoticeReceivingEntity();
                CrmBeanUtil.copyProperties(o, receivingEntity);
                receivingEntity.setId(null);
                receivingEntity.setNoticeCode(entity.getNoticeCode());
                return receivingEntity;
            }).collect(Collectors.toList());
            iSfaNoticeReceivingService.saveBatch(receivingEntityList);
        }
        //公告附件
        if (CollectionUtil.listNotEmptyNotSizeZero(reqVo.getNoticeFileReqVos())) {
            List<SfaNoticeFileEntity> fileEntityList = reqVo.getNoticeFileReqVos().stream().map(o -> {
                SfaNoticeFileEntity fileEntity = new SfaNoticeFileEntity();
                CrmBeanUtil.copyProperties(o, fileEntity);
                fileEntity.setId(null);
                fileEntity.setNoticeCode(entity.getNoticeCode());
                return fileEntity;
            }).collect(Collectors.toList());
            iSfaNoticeFileService.saveBatch(fileEntityList);
        }
    }

    /**
     * 更新
     *
     * @param reqVo
     * @return
     */
    @Override
    public void update(SfaNoticeReqVo reqVo) {
        if (StringUtils.isEmpty(reqVo.getId())) {
            throw new BusinessException("数据主键不能为空");
        }
        SfaNoticeEntity entity = this.getById(reqVo.getId());
        if (Objects.isNull(entity)) {
            throw new BusinessException("公告数据不存在");
        }
        this.saveCheck(reqVo);
        CrmBeanUtil.copyProperties(reqVo, entity);
        this.updateById(entity);
        this.deleteBatchDate(reqVo);
        this.saveData(reqVo, entity);
    }

    /**
     * 删除
     *
     * @param ids
     * @return
     */
    @Override
    public void deleteBatch(List<String> ids) {
        if (!CollectionUtil.listNotEmptyNotSizeZero(ids)) {
            throw new BusinessException("数据主键不能为空");
        }
        SfaNoticeReqVo reqVo = new SfaNoticeReqVo() {{
            this.setIds(ids);
        }};
        reqVo.setPageSize(-1);
        List<SfaNoticeRespVo> list = this.findList(reqVo).getData();
        if (CollectionUtils.isEmpty(list)) {
            throw new BusinessException("公告数据不存在");
        }
        sfaNoticeMapper.deleteProductsByParams(reqVo);
        List<String> noticeCodes = list.stream().map(SfaNoticeRespVo::getNoticeCode).collect(Collectors.toList());
        reqVo.setNoticeCodes(noticeCodes);
        this.deleteBatchDate(reqVo);
    }


    public void deleteBatchDate(SfaNoticeReqVo reqVo) {
        if (StringUtils.isEmpty(reqVo.getNoticeCode()) && CollectionUtil.listEmpty(reqVo.getNoticeCodes())) {
            throw new BusinessException("数据主键不能为空");
        }
        SfaNoticeFileReqVo sfaNoticeFileReqVo = new SfaNoticeFileReqVo();
        sfaNoticeFileReqVo.setNoticeCode(reqVo.getNoticeCode());
        sfaNoticeFileReqVo.setNoticeCodes(reqVo.getNoticeCodes());
        noticeFileMapper.deleteProductsByParams(sfaNoticeFileReqVo);

        SfaNoticeReceivingReqVo sfaNoticeReceivingReqVo = new SfaNoticeReceivingReqVo();
        sfaNoticeReceivingReqVo.setNoticeCode(reqVo.getNoticeCode());
        sfaNoticeReceivingReqVo.setNoticeCodes(reqVo.getNoticeCodes());
        noticeReceivingMapper.deleteProductsByParams(sfaNoticeReceivingReqVo);
    }

    /**
     * 启用
     *
     * @param ids
     * @return
     */
    @Override
    public void enableBatch(List<String> ids) {
        if (!CollectionUtil.listNotEmptyNotSizeZero(ids)) {
            throw new BusinessException("数据主键不能为空");
        }
        //设置状态为启用
        List<SfaNoticeEntity> sfaNoticeEntities = sfaNoticeMapper.selectBatchIds(ids);
        if (CollectionUtils.isNotEmpty(sfaNoticeEntities)) {
            sfaNoticeEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
            });
        }
        this.updateBatchById(sfaNoticeEntities);
    }

    /**
     * 禁用
     *
     * @param ids
     * @return
     */
    @Override
    public void disableBatch(List<String> ids) {
        SfaNoticeReqVo reqVo = new SfaNoticeReqVo();
        if (!CollectionUtil.listNotEmptyNotSizeZero(ids)) {
            throw new BusinessException("数据主键不能为空");
        }
        //设置状态为禁用
        List<SfaNoticeEntity> sfaNoticeEntities = sfaNoticeMapper.selectBatchIds(ids);
        if (CollectionUtils.isNotEmpty(sfaNoticeEntities)) {
            sfaNoticeEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.DISABLE.getCode());
            });
        }
        this.updateBatchById(sfaNoticeEntities);
    }
}
