package com.biz.crm.visitstepbase;

import com.biz.crm.annotation.Klock;
import com.biz.crm.asexecution.model.SfaAsTreatyEntity;
import com.biz.crm.asexecution.service.ISfaAsTreatyService;
import com.biz.crm.base.BusinessException;
import com.biz.crm.base.SfaClientData;
import com.biz.crm.base.SfaClientHelper;
import com.biz.crm.common.param.RedisParam;
import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.eunm.sfa.SfaCodeEnum;
import com.biz.crm.moblie.controller.visit.component.AbstractVisitStepListener;
import com.biz.crm.moblie.controller.visit.component.VisitStepListener;
import com.biz.crm.nebular.activiti.vo.AttachmentVo;
import com.biz.crm.nebular.sfa.SfaTpmActDisplayVo;
import com.biz.crm.nebular.sfa.tpmact.actcollect.SfaTpmActCollectVo;
import com.biz.crm.nebular.sfa.tpmact.displaytreaty.DisplayTreatyVo;
import com.biz.crm.nebular.sfa.tpmact.resp.SfaTpmActDetailRespVo;
import com.biz.crm.service.RedisService;
import com.biz.crm.tpmact.mapper.SfaTpmActDetailMapper;
import com.biz.crm.tpmact.model.SfaTpmActBaseEntity;
import com.biz.crm.tpmact.model.SfaTpmActDetailEntity;
import com.biz.crm.tpmact.service.ISfaTpmActDetailService;
import com.biz.crm.util.*;
import com.biz.crm.visitinfo.model.SfaVisitPlanInfoEntity;
import com.biz.crm.visitinfo.service.ISfaVisitPlanInfoService;
import com.biz.crm.visitstep.model.SfaVisitStepOrderEntity;
import com.biz.crm.visitstep.resp.SfaVisitStepFromRespVo;
import com.biz.crm.visitstep.service.ISfaVisitStepFromService;
import com.biz.crm.visitstep.service.ISfaVisitStepOrderService;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;
import org.redisson.Redisson;
import org.redisson.api.RLock;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.time.LocalDate;
import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * @author lf
 * @describe
 * @project crm
 * @package com.biz.crm.visitstepbase
 * @date 2021/6/15 11:12
 */
@Component
@ConditionalOnMissingBean(name = "visitBaseComponentExpandImpl")
public class VisitBaseComponent {

    public final static String SFA_TPM_ACT_AGAIN_SET = "SFA_TPM_ACT_AGAIN_SET";

    @Resource
    private RedisService redisService;
    @Resource
    protected ISfaVisitStepFromService sfaVisitStepFromService;
    @Resource
    private ISfaVisitPlanInfoService sfaVisitPlanInfoService;
    @Resource
    private ISfaTpmActDetailService sfaTpmActDetailService;
    @Resource
    private ISfaVisitStepOrderService sfaVisitStepOrderService;
    @Resource
    private ISfaAsTreatyService sfaAsTreatyService;
    @Resource
    private SfaTpmActDetailMapper sfaTpmActDetailMapper;

    /**
     * 加载计划明细数据
     *
     * @param redisHashKey
     */
    public SfaVisitPlanInfoEntity loadAndCheckSfaVisitPlanInfoEntity(String redisHashKey, String visitBigType) {
        String nowDate = LocalDate.now().format(CrmDateUtils.yyyyMMdd);
        SfaVisitPlanInfoEntity sfaVisitPlanInfoEntity = (SfaVisitPlanInfoEntity) this.redisService
                .hmget(SfaVisitPlanInfoEntity.getInstance().redisHashCurrent(nowDate, visitBigType).toString(), redisHashKey);
        if (null == sfaVisitPlanInfoEntity) {
            throw new BusinessException("不存在的拜访计划，或该计划不在今日计划内！");
        }
        UserRedis userRedis = UserUtils.getUser();
        if (!org.apache.commons.lang3.StringUtils.equals(sfaVisitPlanInfoEntity.getVisitPosCode(), userRedis.getPoscode())) {
            throw new BusinessException("你没有权限访问该数据，请重新登陆或切换职位再尝试！");
        }
        return sfaVisitPlanInfoEntity;
    }


    /**
     * 表单数据加载
     *
     * @param formId
     */
    public SfaVisitStepFromRespVo getFormData(String formId, String stepCode) {
        SfaVisitStepFromRespVo fromRespVo = this.sfaVisitStepFromService.queryById(formId);
        if (null == fromRespVo) {
            throw new BusinessException("未查询到表单[" + formId + "]配置数据");
        }
        if (!stepCode.equals(fromRespVo.getStepCode())) {
            throw new BusinessException("表单ID传输错误,该步骤属于TPM");
        }
        return fromRespVo;
    }


    /**
     * 更新步骤状态
     *
     * @author: luoqi
     * @Date: 2021-3-5 22:29
     * @version: V1.0
     * @Description:
     */
    public void updateStepStatus(String redisHashKey, String stepCode, String visitBigType) {
        this.sfaVisitPlanInfoService.updateStepStatus(redisHashKey, stepCode, visitBigType);
    }


    /**
     * sfa-tpm活动金额校验-用于协议校验
     *
     * @param actDetailCode
     * @param totalPrice
     */
    public void checkTpmActTreatyAmount(String actDetailCode, BigDecimal totalPrice) {
        SfaTpmActDetailRespVo respVo = sfaTpmActDetailMapper.findActDetailByActDetailCode(actDetailCode);
        if (YesNoEnum.yesNoEnum.Y.getValue().equals(respVo.getIsControlActFee())) {
            List<SfaAsTreatyEntity> treatyEntityList = sfaAsTreatyService.lambdaQuery()
                    .eq(SfaAsTreatyEntity::getActivityCode, actDetailCode).list();
            BigDecimal usedAmount = BigDecimal.ZERO;
            if (CollectionUtil.listNotEmptyNotSizeZero(treatyEntityList)) {
                usedAmount = treatyEntityList.stream().map(SfaAsTreatyEntity::getTotalPrice).reduce(BigDecimal.ZERO, BigDecimal::add);
            }
            SfaTpmActDetailEntity actDetailEntity = sfaTpmActDetailService.lambdaQuery()
                    .eq(SfaTpmActDetailEntity::getActDetailCode, actDetailCode)
                    .select(SfaTpmActDetailEntity::getActDetailCode,
                            SfaTpmActDetailEntity::getApplyAmount).one();
            BigDecimal surplusAmount = actDetailEntity.getApplyAmount().subtract(usedAmount);
            if (totalPrice.compareTo(surplusAmount) == 1) {
                throw new BusinessException("当前终端活动金额超额,剩余可用终端活动金额:" + surplusAmount);
            }
        }
    }


    /**
     * sfa-tpm活动金额校验-用于分销订单金额校验
     *
     * @param actDetailCode
     * @param totalPrice
     */
    public void checkTpmActDistributionOrderAmount(String actDetailCode, BigDecimal totalPrice, String terminalCode) {
        SfaTpmActDisplayVo displayVo = getTpmActTreatyRedis(actDetailCode, terminalCode);
        if (displayVo.getIsControlActFee().equals(YesNoEnum.yesNoEnum.Y.getValue())) {
            if (totalPrice.compareTo(displayVo.getSurplusAmount()) == 1) {
                throw new BusinessException("当前终端活动金额超额,剩余可用终端活动金额:" + displayVo.getSurplusAmount());
            }
        }
    }

    /**
     * 获取当前终端所属活动是否签署协议
     *
     * @param actDetailCode
     * @param terminalCode
     */
    public SfaTpmActDisplayVo getTpmActTreatyRedis(String actDetailCode, String terminalCode) {
        SfaTpmActDisplayVo displayVo = (SfaTpmActDisplayVo) redisService.hmget(SfaTpmActDisplayVo.buildRedisHashKey(actDetailCode),
                actDetailCode + RedisParam.DELIMITER + terminalCode);
        if (null == displayVo) {
            displayVo = buildSfaTpmActDisplayVo(actDetailCode, terminalCode);
            redisService.hmset(displayVo.buildRedisHashKey(actDetailCode), displayVo.buildRedisMap(terminalCode), SfaTpmActDisplayVo.REDIS_EXPIRE_TIME);
        }
        return displayVo;
    }


    /**
     * 订单结算过后进行tpm活动金额重新设置
     *
     * @param actDetailCode
     * @param terminalCode
     */
    @Klock(keys = {SFA_TPM_ACT_AGAIN_SET, "#actDetailCode"}, waitTime = 5, leaseTime = 8)
    public void saveOrderSetTpmActAmount(String actDetailCode, String terminalCode) {
        SfaTpmActDisplayVo displayVo = buildSfaTpmActDisplayVo(actDetailCode, terminalCode);
        redisService.hmset(displayVo.buildRedisHashKey(actDetailCode), displayVo.buildRedisMap(terminalCode), SfaTpmActDisplayVo.REDIS_EXPIRE_TIME);
    }

    /**
     * 进行数据分装
     *
     * @param actDetailCode
     * @param terminalCode
     * @return
     */
    protected SfaTpmActDisplayVo buildSfaTpmActDisplayVo(String actDetailCode, String terminalCode) {
        SfaTpmActDisplayVo displayVo = new SfaTpmActDisplayVo();
        //查询当前活动是否有签署协议
        SfaTpmActDetailRespVo respVo = sfaTpmActDetailMapper.findActDetailByActDetailCode(actDetailCode);
        if (respVo.getIsControlActFee().equals(YesNoEnum.yesNoEnum.Y.getValue())) {
            BigDecimal totalPrice = BigDecimal.ZERO;
            BigDecimal surplusAmount = BigDecimal.ZERO;
            //判断当前tpm活动是否签署协议
            if (YesNoEnum.yesNoEnum.Y.getValue().equals(respVo.getIsSignDisplayAgreement())) {
                SfaAsTreatyEntity treatyEntity = sfaAsTreatyService.lambdaQuery()
                        .eq(SfaAsTreatyEntity::getTerminalCode, terminalCode)
                        .eq(SfaAsTreatyEntity::getActivityCode, actDetailCode).one();
                if (null == treatyEntity) {
                    throw new BusinessException("该终端未进行协议签署,不可进行订单采集/活动数据采集");
                }
                BigDecimal orderPrice = sfaVisitStepOrderService.lambdaQuery()
                        .eq(SfaVisitStepOrderEntity::getActDetailCode, actDetailCode)
                        .eq(SfaVisitStepOrderEntity::getClientCode, terminalCode)
                        .select(SfaVisitStepOrderEntity::getTotalPrice).list()
                        .stream().map(SfaVisitStepOrderEntity::getTotalPrice).reduce(BigDecimal.ZERO, BigDecimal::add);
                totalPrice = treatyEntity.getTotalPrice();
                surplusAmount = totalPrice.subtract(orderPrice);
            } else {
                //没有签署协议，则进行整个活动的金额计算
                SfaTpmActDetailEntity actDetailEntity = sfaTpmActDetailService.lambdaQuery()
                        .eq(SfaTpmActDetailEntity::getActDetailCode, actDetailCode)
                        .select(SfaTpmActDetailEntity::getActDetailCode,
                                SfaTpmActDetailEntity::getApplyAmount).one();
                totalPrice = actDetailEntity.getApplyAmount();
                BigDecimal orderPrice = sfaVisitStepOrderService.lambdaQuery()
                        .eq(SfaVisitStepOrderEntity::getActDetailCode, actDetailCode)
                        .select(SfaVisitStepOrderEntity::getTotalPrice).list()
                        .stream().map(SfaVisitStepOrderEntity::getTotalPrice).reduce(BigDecimal.ZERO, BigDecimal::add);
                surplusAmount = totalPrice.subtract(orderPrice);
            }
            displayVo.setTotalPrice(totalPrice);
            displayVo.setSurplusAmount(surplusAmount);
        }
        displayVo.setIsControlActFee(respVo.getIsControlActFee());
        displayVo.setIsSignDisplayAgreement(respVo.getIsSignDisplayAgreement());
        return displayVo;
    }


    /**
     * tpm活动调用监听器
     *
     * @param committedData
     */
    public void tpmDoListener(SfaTpmActBaseEntity committedData) {
        if (null == committedData) {
            return;
        }
        SfaClientData clientData = SfaClientHelper.loadClientDataNotRequired(committedData.getClientType(), committedData.getClientCode());
        if (null != clientData) {
            committedData.setClientSubclass(clientData.getClientSubclass());
            committedData.setClientSubclassName(clientData.getClientSubclassName());
        }
        String stepCode = SfaCodeEnum.VisitStepCode.SFA_STEP_CODE_TPM.getVal();
        committedData.setStepCode(stepCode);
        List<VisitStepListener.VisitStepListenerCommittedData> committedDataList = Lists.newArrayList(committedData);
        List<AbstractVisitStepListener> listeners = AbstractVisitStepListener.getListeners(stepCode);
        for (AbstractVisitStepListener listener : listeners) {
            if (null == listener) {
                continue;
            }
            VisitStepListener.VisitStepListenerCommittedEvent committedEvent = new VisitStepListener.VisitStepListenerCommittedEvent();
            committedEvent.setEntities(committedDataList);
            listener.committed(committedEvent);
        }
    }


    /**
     * 短信验证
     *
     * @param vo
     */
    public void textCheck(DisplayTreatyVo vo) {
        AssertUtils.isNotEmpty(vo.getStorePhone(), "店主手机号为空");
        AssertUtils.isNotEmpty(vo.getVerificationCode(), "验证码为空");
    }

    /**
     * 活动数据采集图片验证
     *
     * @param collectFormList
     */
    public void checkActTpmCollectPicList(List<SfaTpmActCollectVo.CollectForm> collectFormList) {
        collectFormList.forEach(pic -> {
            this.checkPics(pic.getAttachmentList(), 2, 5, pic.getExampleName());
        });
    }


    /**
     * 校验照片
     *
     * @param sfaVisitPics 照片
     * @param min          最少限制
     * @param max          最多限制
     * @param bizTag       错误提示
     */
    private void checkPics(List<AttachmentVo> sfaVisitPics, int min, int max, String bizTag) {
        min = min < 0 ? 0 : min;
        if (max < min) {
            max = min;
        }
        bizTag = null == bizTag ? "" : bizTag;
        if (null == sfaVisitPics) {
            sfaVisitPics = Lists.newArrayList();
        }
        if (sfaVisitPics.size() < min) {
            throw new BusinessException(bizTag + "图片数量不能少于[" + min + "]张");
        }
        if (sfaVisitPics.size() > max) {
            throw new BusinessException(bizTag + "图片数量不能超过[" + max + "]张");
        }
        for (AttachmentVo sfaVisitPictureReqVo : sfaVisitPics) {
            if (org.apache.commons.lang3.StringUtils.isBlank(sfaVisitPictureReqVo.getUrlPathPrefix())) {
                throw new BusinessException(bizTag + "照片域名为空");
            }
            if (StringUtils.isBlank(sfaVisitPictureReqVo.getUrlPathPrefix())) {
                throw new BusinessException(bizTag + "照片相对路径为空");
            }
        }
    }
}
