package com.biz.crm.worksign.service.impl;

import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.activiti.act.TaActBaseFeign;
import com.biz.crm.base.BusinessException;
import com.biz.crm.common.PageResult;
import com.biz.crm.eunm.ActivitiEnum;
import com.biz.crm.eunm.sfa.SfaCommonEnum;
import com.biz.crm.nebular.activiti.act.req.StartProcessReqVo;
import com.biz.crm.nebular.activiti.task.resp.TaskRspVO;
import com.biz.crm.nebular.sfa.worksign.req.SfaApplyTimeInfoReqVo;
import com.biz.crm.nebular.sfa.worksign.req.SfaAuditFinishReqVo;
import com.biz.crm.nebular.sfa.worksign.req.SfaAuditListLeaveCancelReqVo;
import com.biz.crm.nebular.sfa.worksign.req.SfaLeaveCancelReqVo;
import com.biz.crm.nebular.sfa.worksign.resp.*;
import com.biz.crm.util.*;
import com.biz.crm.worksign.mapper.SfaLeaveCancelMapper;
import com.biz.crm.worksign.model.SfaLeaveCancelEntity;
import com.biz.crm.worksign.service.ISfaLeaveCancelService;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @author ren.gang
 * @ClassName SfaLeaveCancelServiceImpl.java
 * @Description TODO
 * @createTime 2020年12月09日 22:45:00
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name="sfaLeaveCancelServiceExpandImpl")
public class SfaLeaveCancelServiceImpl extends ServiceImpl<SfaLeaveCancelMapper, SfaLeaveCancelEntity> implements ISfaLeaveCancelService {

    @Resource
    private SfaLeaveCancelMapper mapper;
    @Autowired
    private TaActBaseFeign taActBaseFeign;

    /**
     * 列表
     * @param reqVo
     * @return
     */
    @Override
    public List<SfaLeaveCancelRespVo> findList(SfaLeaveCancelReqVo reqVo) {
        List<SfaLeaveCancelRespVo> list = mapper.findList(reqVo);
        list.forEach(vo -> {
            vo.setTimeInfoList(JSON.parseArray(vo.getTimeInfoListJson(), SfaApplyTimeInfoReqVo.class));
            vo.setBpmStatusName(SfaCommonEnum.dataBpmStatus.getDesc(vo.getBpmStatus()));
        });
        return list;
    }
    /**
     * 新增
     * @param reqVo
     * @return
     */
    @Override
    public SfaLeaveCancelEntity save(SfaLeaveCancelReqVo reqVo, boolean isAudit) {
        UserRedis userRedis = UserUtils.getUser();
        if(userRedis == null) {
            throw new BusinessException("请登录");
        }
        // 校验是否冲突
        List<SfaLeaveCancelRespVo> list = findList(new SfaLeaveCancelReqVo(reqVo.getLeaveId()));
        verifyDateRepeat(list, reqVo.getBeginTime(), reqVo.getEndTime(), reqVo.getTimeInfoList());

        SfaLeaveCancelEntity entity = CrmBeanUtil.copy(reqVo, SfaLeaveCancelEntity.class);
        entity.setTimeInfoListJson(JSON.toJSONString(reqVo.getTimeInfoList()));
        entity.setApplicationDate(CrmDateUtils.yyyyMMddHHmmss.format(LocalDateTime.now()));

        if(isAudit) {
            //发送审核信息到审核系统
            entity.setBpmStatus(SfaCommonEnum.dataBpmStatus.COMMIT.getValue());
            this.saveOrUpdate(entity);
            entity.setAuditTaskId(entity.getId());
            StartProcessReqVo startProcessReqVO =
                    SfaActivitiUtils.createStartProcessReqVO(
                            userRedis.getUsername(), userRedis.getPoscode(), entity.getAuditTaskId(),
                            ActivitiEnum.FormTypeEnum.LEAVECANCEL,
                            entity.getBeginTime(),
                            entity.getEndTime(),
                            userRedis.getOrgcode());
            // 业务参数｛｝
            String processNo = ActivityUtils.startProcess(startProcessReqVO);
        } else {
            entity.setBpmStatus(SfaCommonEnum.dataBpmStatus.PASS.getValue());
        }

        this.saveOrUpdate(entity);

        return entity;
    }

    /**
     * 校验申请时间冲突
     * @param list 销假列表
     * @param beginTime 开始时间
     * @param endTime 结束时间
     * @param timeInfoList 时间明细
     * @param list 销假列表
     */
    @Override
    public void verifyDateRepeat(List<SfaLeaveCancelRespVo> list, String beginTime, String endTime, List<SfaApplyTimeInfoReqVo> timeInfoList) {
        if(list != null && list.size() > 0) {
            for(SfaLeaveCancelRespVo vo : list) {
                if(SfaCommonEnum.dataBpmStatus.REJECT.getValue().equals(vo.getBpmStatus())) {
                    continue;
                }
                SfaSignUtils.verifyDateRepeat(
                        beginTime,
                        endTime,
                        vo.getBeginTime(),
                        vo.getEndTime(),
                        timeInfoList,
                        vo.getTimeInfoList());
            }
        }
    }

    /**
     * 销假详情列表
     * @param reqVo
     * @return
     */
    @Override
    public PageResult<SfaLeaveCancelInfoRespVo> findInfoList(SfaAuditListLeaveCancelReqVo reqVo) {
        Page<SfaLeaveCancelInfoRespVo> page = new Page<>(reqVo.getPageNum(), reqVo.getPageSize());
        List<SfaLeaveCancelInfoRespVo> list = mapper.findInfoList(page, reqVo);
        if(list != null) {
            list.forEach(vo -> {
                vo.setBpmStatusName(SfaCommonEnum.dataBpmStatus.getDesc(vo.getBpmStatus()));
                vo.setLeaveTypeName(SfaCommonEnum.leaveTypeEnum.getDesc(vo.getLeaveType()));
            });
        }
        return PageResult.<SfaLeaveCancelInfoRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }


    /**
     * 提交审核信息
     * @param id
     * @return
     */
    @Override
    public Result auditCommit(String id) {
        SfaLeaveCancelEntity entity = mapper.selectById(id);
        AssertUtils.isNotNull(entity, "销假信息id错误");
        // 验证状态
        if(!entity.getBpmStatus().equals(SfaCommonEnum.dataBpmStatus.COMMIT.getValue()) &&
                !entity.getBpmStatus().equals(SfaCommonEnum.dataBpmStatus.APPROVAL.getValue())) {
            return Result.error("该申请审批状态错误");
        }
        // 已审核过修改状态
        if(!SfaCommonEnum.dataBpmStatus.APPROVAL.getValue().equals(entity.getBpmStatus())) {
            entity.setBpmStatus(SfaCommonEnum.dataBpmStatus.APPROVAL.getValue());
            saveOrUpdate(entity);
        }

        return Result.ok();
    }

    /**
     * 审批完成状态变更
     * @param reqVo
     */
    @Override
    public SfaLeaveCancelEntity auditCancelFinish(SfaAuditFinishReqVo reqVo) {
        // 查询请假信息
        LambdaQueryWrapper<SfaLeaveCancelEntity> wrapper = new LambdaQueryWrapper<SfaLeaveCancelEntity>()
                .eq(SfaLeaveCancelEntity::getAuditTaskId,reqVo.getFormId());
        SfaLeaveCancelEntity entity = mapper.selectOne(wrapper);
        if(entity == null) {
            throw new BusinessException("销假信息不存在");
        }
        // 处理审核通过
        if(ActivitiEnum.AuditResultType.PASS.getVal().equals(reqVo.getResult())) {
            entity.setBpmStatus(SfaCommonEnum.dataBpmStatus.PASS.getValue());
            entity.setPassStatusDate(CrmDateUtils.yyyyMMddHHmmss.format(LocalDateTime.now()));
        } else if (ActivitiEnum.AuditResultType.REJECT.getVal().equals(reqVo.getResult())){ // 审核驳回
            entity.setBpmStatus(SfaCommonEnum.dataBpmStatus.REJECT.getValue());
        } else if (ActivitiEnum.AuditResultType.RECOVER.getVal().equals(reqVo.getResult())){ // 审核追回
            entity.setBpmStatus(SfaCommonEnum.dataBpmStatus.ROLLBACK.getValue());
        } else {
            throw  new BusinessException("审批失败");
        }

        // 保存信息
        saveOrUpdate(entity);

        return entity;
    }

    /**
     * 根据任务流返回信息查询审核任务列表
     * @param taskRspVOPageResult
     * @param reqVo
     * @return
     */
    @Override
    public PageResult<SfaAuditListLeaveCancelRespVo> findAuditList(PageResult<TaskRspVO> taskRspVOPageResult, SfaAuditListLeaveCancelReqVo reqVo) {

        if(taskRspVOPageResult != null && taskRspVOPageResult.getCount() > 0) {
            // 获取审核任务id集合
            Map<String, TaskRspVO> taskMap = taskRspVOPageResult.getData().stream().collect(Collectors.toMap(TaskRspVO::getFormNo, vo -> vo, (key1, key2) -> key2));
            reqVo.setAuditTaskIdList(new ArrayList<>(taskMap.keySet()));
            // 根据id和其他查询条件去查询业务列表数据
            PageResult<SfaLeaveCancelInfoRespVo> pageResult = findInfoList(reqVo);
            if(pageResult.getCount() > 0) {
                // 数据排序
                List data = pageResult.getData().stream().sorted((x, y) -> {
                    return CrmDateUtils.parseyyyyMMddHHmmss(y.getApplicationDate())
                            .compareTo(CrmDateUtils.parseyyyyMMddHHmmss(x.getApplicationDate()));
                }).collect(Collectors.toList());
                pageResult.setData(data);
                // 封装返回数据
                List<SfaAuditListLeaveCancelRespVo> list = new ArrayList<>();
                pageResult.getData().forEach(vo -> {
                    list.add(new SfaAuditListLeaveCancelRespVo(vo, taskMap.get(vo.getAuditTaskId())));
                });
                return PageResult.<SfaAuditListLeaveCancelRespVo>builder()
                        .data(list)
                        .count(pageResult.getCount())
                        .build();
            }
        }
        return PageResult.<SfaAuditListLeaveCancelRespVo>builder()
                .data(Lists.newArrayList())
                .count(0L)
                .build();
    }


}
