package com.biz.crm.worksign.service.impl;

import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.activiti.act.TaActBaseFeign;
import com.biz.crm.activiti.mobile.ActivitiMobileFeign;
import com.biz.crm.base.BusinessException;
import com.biz.crm.calculatesalary.service.ISfaCalculateSalaryDateService;
import com.biz.crm.common.PageResult;
import com.biz.crm.eunm.ActivitiEnum;
import com.biz.crm.eunm.YesNoEnum;
import com.biz.crm.eunm.sfa.SfaCommonEnum;
import com.biz.crm.eunm.sfa.SfaWorkSignEnum;
import com.biz.crm.nebular.activiti.act.req.StartProcessReqVo;
import com.biz.crm.nebular.activiti.common.ProcessCommonVo;
import com.biz.crm.nebular.activiti.start.req.CancelProcessReqVO;
import com.biz.crm.nebular.activiti.task.resp.TaskRspVO;
import com.biz.crm.nebular.mdm.org.resp.MdmOrgRespVo;
import com.biz.crm.nebular.mdm.position.resp.MdmPositionUserOrgRespVo;
import com.biz.crm.nebular.sfa.worksign.req.*;
import com.biz.crm.nebular.sfa.worksign.resp.SfaAuditListOvertimeRespVo;
import com.biz.crm.nebular.sfa.worksign.resp.SfaWorkOvertimeRespVo;
import com.biz.crm.nebular.sfa.worksignrule.resp.SfaWorkSignRuleRespVo;
import com.biz.crm.util.*;
import com.biz.crm.worksign.mapper.SfaWorkOvertimeMapper;
import com.biz.crm.worksign.model.SfaLeaveEntity;
import com.biz.crm.worksign.model.SfaWorkOvertimeEntity;
import com.biz.crm.worksign.model.SfaWorkSignRuleInfoEntity;
import com.biz.crm.worksign.service.ISfaLeaveService;
import com.biz.crm.worksign.service.ISfaSignFormsService;
import com.biz.crm.worksign.service.ISfaWorkOvertimeService;
import com.biz.crm.worksign.service.ISfaWorkOvertimeTimeInfoService;
import com.biz.crm.worksignrule.service.impl.SfaWorkSignRuleServiceImpl;
import com.google.common.collect.Lists;
import jodd.util.StringUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 工作日调整业务处理 接口实现
 *
 * @author ren.gang
 * @date 2020-11-20 15:00:38
 */
@Slf4j
@Service
@ConditionalOnMissingBean(name="sfaWorkOvertimeServiceExpandImpl")
public class SfaWorkOvertimeServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<SfaWorkOvertimeMapper, SfaWorkOvertimeEntity>
        implements ISfaWorkOvertimeService {


    @Resource
    private SfaWorkOvertimeMapper mapper;
    @Autowired
    private ISfaWorkOvertimeTimeInfoService overtimeTimeInfoService;
    @Autowired
    private SfaWorkSignRuleServiceImpl signRuleService;
    @Autowired
    private TaActBaseFeign taActBaseFeign;
    @Autowired
    private ActivitiMobileFeign mobileFeign;
    @Autowired
    private ISfaSignFormsService formsService;
    @Autowired
    private ISfaLeaveService leaveService;
    @Autowired
    private ISfaCalculateSalaryDateService dateService;

    /**
     * 列表
     * @param reqVo
     * @return
     */
    @Override
    public PageResult<SfaWorkOvertimeRespVo> findList(SfaWorkOvertimeListReqVo reqVo){
        Page<SfaWorkOvertimeRespVo> page = new Page<>(reqVo.getPageNum(), reqVo.getPageSize());
        if(StringUtils.isNotEmpty(reqVo.getApplicationDateStart())){
            reqVo.setApplicationDateStart(reqVo.getApplicationDateStart() + " 00:00:00");
        }
        if(StringUtils.isNotEmpty(reqVo.getApplicationDateEnd())){
            reqVo.setApplicationDateEnd(reqVo.getApplicationDateEnd() + " 23:59:59");
        }
        List<SfaWorkOvertimeRespVo> list = mapper.findList(page, reqVo);
        if(list != null && list.size() > 0) {
            list.forEach(vo -> {
                vo.setBpmStatusName(SfaCommonEnum.dataBpmStatus.getDesc(vo.getBpmStatus()));
            });
        }
        return PageResult.<SfaWorkOvertimeRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    /**
     * 查询
     * @param id
     * @return SfaWorkOvertimeRespVo
     */
    @Override
    public SfaWorkOvertimeRespVo query(String  id){
        SfaWorkOvertimeListReqVo reqVo = new SfaWorkOvertimeListReqVo();
        reqVo.setId(id);
        PageResult<SfaWorkOvertimeRespVo> result = findList(reqVo);
        if(result == null || result.getCount() != 1) {
            throw new BusinessException("id错误");
        }
        return result.getData().get(0);
    }

    /**
     * 查询单个明细-根据审核任务id
     * @param auditTaskId
     * @return
     */
    @Override
    public SfaWorkOvertimeRespVo queryByAuditTaskId(String auditTaskId) {
        SfaWorkOvertimeListReqVo reqVo = new SfaWorkOvertimeListReqVo();
        reqVo.setAuditTaskIdList(Arrays.asList(auditTaskId));
        PageResult<SfaWorkOvertimeRespVo> result = findList(reqVo);
        if(result == null || result.getCount() != 1) {
            throw new BusinessException("审核任务ID错误");
        }
        return result.getData().get(0);
    }

    /**
     * 当前考勤明细是否有工作日调整
     * @param workSignRuleInfoEntity
     * @return
     */
    @Override
    @Transactional(readOnly = true)
    public boolean hasOvertime(SfaWorkSignRuleInfoEntity workSignRuleInfoEntity){
        // 查询该明细当天的 工作日调整记录
        List<SfaWorkOvertimeEntity> entities = this.lambdaQuery()
                //用户
                .eq(SfaWorkOvertimeEntity :: getUserName, workSignRuleInfoEntity.getUserName())
                //工作日调整记录审批状态
                .in(SfaWorkOvertimeEntity :: getBpmStatus
                        , Lists.newArrayList(SfaCommonEnum.dataBpmStatus.COMMIT.getValue()
                                , SfaCommonEnum.dataBpmStatus.APPROVAL.getValue(), SfaCommonEnum.dataBpmStatus.PASS.getValue()))
                //工作日调整开始时间
                .le(SfaWorkOvertimeEntity :: getBeginTime, workSignRuleInfoEntity.getRuleDate())
                //工作日调整结束时间
                .ge(SfaWorkOvertimeEntity :: getEndTime, workSignRuleInfoEntity.getRuleDate()).list();
        if(org.springframework.util.CollectionUtils.isEmpty(entities)){
            return false;
        }
        return true;
    }

    /**
     * 工作日调整申请接口
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public Result apply(SfaWorkOvertimeReqVo reqVo){
        Result result = new Result<>();
        SfaWorkOvertimeEntity entity = CrmBeanUtil.copy(reqVo,SfaWorkOvertimeEntity.class);

        // 判断编辑
        if(StringUtils.isNotEmpty(reqVo.getId())) {
            SfaWorkOvertimeEntity old = mapper.selectById(reqVo.getId());
            if(old == null) {
                return Result.error("主键id错误");
            }

            // 驳回状态直接创建新的申请
            if(SfaCommonEnum.dataBpmStatus.REJECT.getValue().equals(old.getBpmStatus())) {
                entity.setId(null);
                // 追回的使用原来的申请继续提交
            } else if(SfaCommonEnum.dataBpmStatus.ROLLBACK.getValue().equals(old.getBpmStatus())) {
                entity.setId(old.getId());
                // 非编辑状态判断
            } else {
                return Result.error("该申请不支持编辑");
            }
        }

        //设置创建人信息，当前登录人信息
        UserRedis userRedis = UserUtils.getUser();
        if (null==userRedis|| StringUtil.isEmpty(userRedis.getUsername())){
            result.error401("请重新登录");
            return result;
        }

        // 默认为当前人员申请
        if(StringUtils.isEmpty(entity.getUserName())) {
            entity.setUserName(userRedis.getUsername());
        }
        // 查询用户主数组
        MdmPositionUserOrgRespVo mdmPositionUserOrgRespVo = formsService.getUserOrgInfo(entity.getUserName());
        entity.setRealName(mdmPositionUserOrgRespVo.getFullName());
        entity.setPosCode(mdmPositionUserOrgRespVo.getPositionCode());
        entity.setPosName(mdmPositionUserOrgRespVo.getPositionName());

        String orgCode = mdmPositionUserOrgRespVo.getOrgCode();
        //主职位组织
        entity.setOrgCode(orgCode);
        entity.setOrgName(mdmPositionUserOrgRespVo.getOrgName());
        //组织的上级组织
        MdmOrgRespVo mdmOrgRespVo = OrgUtil.getOrgByCode(orgCode);
        if(null != mdmOrgRespVo){
            entity.setParentOrgCode(mdmOrgRespVo.getParentCode());
            entity.setParentOrgName(mdmOrgRespVo.getParentName());
        }

        // 校验申请人信息
        if(StringUtils.isEmpty(entity.getUserName()) || StringUtil.isEmpty(entity.getRealName())
                || StringUtil.isEmpty(entity.getPosCode()) || StringUtil.isEmpty(entity.getOrgCode())) {
            return Result.error("申请人员信息必须包含以下信息：人员账号、人员姓名、岗位编码、组织编码，请核对");
        }



        // 校验开始、结束时间、日期明细、申请时长
        SfaSignUtils.verifyApply(entity.getBeginTime(), entity.getEndTime()
                , reqVo.getTimeInfoList(), entity.getOvertimeDuration());

        // 获取考勤规则
        SfaWorkSignRuleRespVo ruleRespVo = signRuleService.getSignRuleByOrgCode(entity.getOrgCode());

        // 校验日期是否可以调整，以及申请是否包含节假日
        String overtimeType = ActivitiEnum.SfaOvertimeSmallTypeEnum.NORMAL.getVal();
        for(SfaApplyTimeInfoReqVo timeInfo : SfaSignUtils.fillTimeInfoAndCheck(reqVo.getTimeInfoList(), entity.getBeginTime(), entity.getEndTime())) {
            // 获取工作日类型
            String workType = SfaSignUtils.isWorkDay(ruleRespVo, timeInfo.getTimeStr(), dateService.findHolidayByDate(timeInfo.getTimeStr()));
            if(YesNoEnum.yesNoEnum.ZERO.getValue().equals(workType)) {
                return Result.error("日期不能调整-工作日："+timeInfo.getTimeStr());
            }
            // 判断是否包含计算薪资日期
            if(YesNoEnum.yesNoEnum.THREE.getValue().equals(workType)) {
                overtimeType = ActivitiEnum.SfaOvertimeSmallTypeEnum.LEGAL.getVal();
            }

        };
        entity.setOvertimeType(overtimeType);

        // 校验时间冲突
        List<SfaWorkOvertimeRespVo> repeatList =
                mapper.findRepeatByDates(entity.getUserName(), entity.getBeginTime(), entity.getEndTime(), entity.getId());
        if(repeatList != null && repeatList.size() > 0) {
            for(SfaWorkOvertimeRespVo vo : repeatList) {
                SfaSignUtils.verifyDateRepeat(entity.getBeginTime(), entity.getEndTime(), vo.getBeginTime(), vo.getEndTime()
                        , reqVo.getTimeInfoList(), JSON.parseArray(vo.getTimeInfoListJson(), SfaApplyTimeInfoReqVo.class));
            }
        }
        entity.setTimeInfoListJson(JSON.toJSONString(reqVo.getTimeInfoList()));
        // 申请时间
        entity.setApplicationDate(CrmDateUtils.yyyyMMddHHmmss.format(LocalDateTime.now()));
        // 设置审核状态和考勤统计相关信息
        entity.setBpmStatus(SfaCommonEnum.dataBpmStatus.COMMIT.getValue());
        this.saveOrUpdate(entity);

        //发送审核信息到审核系统
        entity.setAuditTaskId(entity.getId());
        StartProcessReqVo startProcessReqVO =
                SfaActivitiUtils.createStartProcessReqVO(
                        entity.getUserName(),
                        entity.getPosCode(),
                        entity.getAuditTaskId(),
                        ActivitiEnum.FormTypeEnum.OVERTIME,
                        entity.getBeginTime(),
                        entity.getEndTime(),
                        entity.getOrgCode()
                );
        // 业务参数｛是否包含节假日｝
        String isHoliday = YesNoEnum.yesNoEnum.NO.getValue();
        if(ActivitiEnum.SfaOvertimeSmallTypeEnum.LEGAL.getVal().equals(entity.getOvertimeType())) {
            isHoliday = YesNoEnum.yesNoEnum.YES.getValue();
        }
        startProcessReqVO.getVariables().put("isHoliday", isHoliday); // 是否包含节假日
        String processNo = ActivityUtils.startProcess(startProcessReqVO);
        entity.setProcessNo(processNo);
        this.saveOrUpdate(entity);
        return Result.ok();
    }

    /**
     * 工作日调整申请-追回
     * @param id
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public Result rollback(String id) {
        SfaWorkOvertimeEntity entity = mapper.selectById(id);
        if(entity == null) {
            return Result.error("数据不存在");
        }
        if(!SfaCommonEnum.dataBpmStatus.COMMIT.getValue().equals(entity.getBpmStatus())) {
            return Result.error("当前申请不能追回");
        }

        CancelProcessReqVO reqVO = new CancelProcessReqVO(entity.getAuditTaskId()
                , UserUtils.getUser().getUsername()
                , ActivitiEnum.FormTypeEnum.OVERTIME.getCostType()
                , ActivitiEnum.getFormType());
        Result result = mobileFeign.cancelProcess(reqVO);
        if(!result.isSuccess()) {
            return result;
        }

        entity.setBpmStatus(SfaCommonEnum.dataBpmStatus.ROLLBACK.getValue());
        this.saveOrUpdate(entity);
        return Result.ok();
    }

    /**
     * 提交审核信息
     * @param id
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public Result auditCommit(String id) {
        SfaWorkOvertimeEntity entity = mapper.selectById(id);
        if(entity == null) {
            return Result.error("主键id错误");
        }
        // 验证状态
        if(!entity.getBpmStatus().equals(SfaCommonEnum.dataBpmStatus.COMMIT.getValue()) &&
                !entity.getBpmStatus().equals(SfaCommonEnum.dataBpmStatus.APPROVAL.getValue())) {
            return Result.error("该申请审批状态错误");
        }
        // 已审核过修改状态
        if(!SfaCommonEnum.dataBpmStatus.APPROVAL.getValue().equals(entity.getBpmStatus())) {
            entity.setBpmStatus(SfaCommonEnum.dataBpmStatus.APPROVAL.getValue());
            saveOrUpdate(entity);
        }

        return Result.ok();
    }

    /**
     * 根据任务流返回信息查询审核任务列表
     * @param taskRspVOPageResult
     * @param reqVo
     * @return
     */
    @Override
    public PageResult<SfaAuditListOvertimeRespVo> findAuditList(PageResult<TaskRspVO> taskRspVOPageResult, SfaWorkOvertimeListReqVo reqVo) {
        if(taskRspVOPageResult != null && taskRspVOPageResult.getCount() > 0) {
            // 获取审核任务id集合
            Map<String, TaskRspVO> taskMap = taskRspVOPageResult.getData().stream().collect(Collectors.toMap(TaskRspVO::getFormNo, vo -> vo, (key1, key2) -> key2));
            reqVo.setAuditTaskIdList(new ArrayList<>(taskMap.keySet()));
            // 根据id和其他查询条件去查询业务列表数据
            PageResult<SfaWorkOvertimeRespVo> pageResult = findList(reqVo);
            if(pageResult.getCount() > 0) {
                // 数据排序
                List data = pageResult.getData().stream().sorted((x, y) -> {
                    return CrmDateUtils.parseyyyyMMddHHmmss(y.getApplicationDate())
                            .compareTo(CrmDateUtils.parseyyyyMMddHHmmss(x.getApplicationDate()));
                }).collect(Collectors.toList());
                pageResult.setData(data);
                // 封装返回数据
                List<SfaAuditListOvertimeRespVo> list = new ArrayList<>();
                pageResult.getData().forEach(vo -> {
                    list.add(new SfaAuditListOvertimeRespVo(vo, taskMap.get(vo.getAuditTaskId())));
                });
                return PageResult.<SfaAuditListOvertimeRespVo>builder()
                        .data(list)
                        .count(pageResult.getCount())
                        .build();
            }
        }
        return PageResult.<SfaAuditListOvertimeRespVo>builder()
                .data(Lists.newArrayList())
                .count(0L)
                .build();
    }

    /**
     * 审核完成回调
     * @param reqVo
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void auditFinish(SfaAuditFinishReqVo reqVo) {
        // 查询工作日调整信息
        LambdaQueryWrapper<SfaWorkOvertimeEntity> wrapper = new LambdaQueryWrapper<SfaWorkOvertimeEntity>()
                .eq(SfaWorkOvertimeEntity::getAuditTaskId,reqVo.getFormId());
        SfaWorkOvertimeEntity entity = mapper.selectOne(wrapper);
        // 处理审核通过
        if(ActivitiEnum.AuditResultType.PASS.getVal().equals(reqVo.getResult())) {
            // 记录状态，通过时间
            entity.setBpmStatus(SfaCommonEnum.dataBpmStatus.PASS.getValue());
            entity.setPassStatusDate(CrmDateUtils.yyyyMMddHHmmss.format(LocalDateTime.now()));

            // 保存日期明细
            overtimeTimeInfoService.saveTimeInfo(entity);

            // 统计出现计算薪资日期
            BigDecimal holidayDays = new BigDecimal(0);
            List<SfaApplyTimeInfoReqVo> timeInfoList = SfaSignUtils.fillTimeInfoAndCheck(
                    JSON.parseArray(entity.getTimeInfoListJson(), SfaApplyTimeInfoReqVo.class)
                    ,entity.getBeginTime(),entity.getEndTime()
            );
            for(SfaApplyTimeInfoReqVo vo : timeInfoList) {
                if(dateService.isCalculateSalary(vo.getTimeStr())) {
                    if(SfaCommonEnum.dataTimeType.ALL_DAY.getValue().equals( vo.getTimeType())) {
                        holidayDays = holidayDays.add(new BigDecimal(1));
                    } else {
                        holidayDays = holidayDays.add(new BigDecimal(0.5));
                    }
                }
            }


            // 剩余抵扣时长 = 原时长 - 计算薪资时长
            BigDecimal lastDays = new BigDecimal(entity.getOvertimeDuration()).subtract(holidayDays);

            // 获取有效期内可以抵扣的调休列表
            List<SfaLeaveEntity> leaveList = leaveService.findNotUseDaysList(entity.getUserName());
            if(leaveList != null && lastDays.compareTo(BigDecimal.ZERO) > 0) {
                /**
                 * 1、计算 抵扣天数 = 剩余天数-可抵扣天数, 剩余天数 = 剩余天数-抵扣天数
                 * 1.1、正数： 抵扣天数为可抵扣天数
                 * 1.2、负数或0： 抵扣天数为剩余天数
                 * 2、添加抵扣信息
                 * 3、剩余天数 <= 0 时，结束循环
                 */
                for(SfaLeaveEntity leaveEntity : leaveList) {
                    BigDecimal days = lastDays.subtract(leaveEntity.getLastDays());
                    if(days.compareTo(BigDecimal.ZERO) > 0) {
                        days = leaveEntity.getLastDays();
                    } else {
                        days = lastDays;
                    }
                    lastDays = lastDays.subtract(leaveEntity.getLastDays());
                    // 保存抵扣信息
                    leaveEntity.setUseDeductionIds(SfaSignUtils.addDeductionIds(leaveEntity.getUseDeductionIds(), entity.getId(), days));
                    leaveEntity.setUseDays(leaveEntity.getUseDays().add(days));
                    leaveEntity.setLastDays(leaveEntity.getLastDays().subtract(days));
                    leaveService.saveOrUpdate(leaveEntity);
                    entity.setUseDeductionIds(SfaSignUtils.addDeductionIds(entity.getUseDeductionIds(), leaveEntity.getId(), days));

                    if(lastDays.compareTo(BigDecimal.ZERO) <= 0) {
                        break;
                    }
                }
            }
            // 设置抵扣最终信息
            entity.setLastDays(lastDays);
            entity.setUseDays(new BigDecimal(entity.getOvertimeDuration()).subtract(holidayDays).subtract(lastDays));
            entity.setHolidayDays(holidayDays);
        }
        else if (ActivitiEnum.AuditResultType.REJECT.getVal().equals(reqVo.getResult())){ // 审核驳回
            entity.setBpmStatus(SfaCommonEnum.dataBpmStatus.REJECT.getValue());
        } else if (ActivitiEnum.AuditResultType.RECOVER.getVal().equals(reqVo.getResult())){ // 审核追回
            entity.setBpmStatus(SfaCommonEnum.dataBpmStatus.ROLLBACK.getValue());
        } else {
            throw  new BusinessException("审批失败");
        }

        // 保存信息
        saveOrUpdate(entity);
    }

    /**
     * 获取数据字典加班申请有效期配置
     * @return
     */
    @Override
    public int getOvertimeIndate() {
        // 设置有效期
        Map<String, String> map = DictUtil.getDictValueMapsByCodes(SfaWorkSignEnum.SignRuleIndateKey.SIGN_RULE_INDATE.getVal());
        if(map != null && map.get(SfaWorkSignEnum.SignRuleIndateKey.OVERTIME_INDATE.getVal()) != null) {
            try {
                return Integer.parseInt(map.get(SfaWorkSignEnum.SignRuleIndateKey.OVERTIME_INDATE.getVal()));
            }catch (Exception e) {
                throw new BusinessException("数据字典"+SfaWorkSignEnum.SignRuleIndateKey.SIGN_RULE_INDATE.getVal()+"配置错误");
            }
        }
        return 0;
    }

    /**
     * 获取有效期内可以抵扣的工作日调整列表
     * 1、状态为审批通过,2、有效期内，3、剩余天数大于0
     * @return
     */
    @Override
    public List<SfaWorkOvertimeEntity> findNotUseDaysList(String userName) {
        return mapper.findNotUseDays(userName, getOvertimeIndate());
    }


}
