package com.biz.crm.worksignrule.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.biz.crm.aop.CrmLog;
import com.biz.crm.common.PageResult;
import com.biz.crm.eunm.CrmDelFlagEnum;
import com.biz.crm.eunm.CrmEnableStatusEnum;
import com.biz.crm.eunm.sfa.WorkSignEnum;
import com.biz.crm.nebular.sfa.worksignrule.req.SfaWorkSignPersonnelReqVo;
import com.biz.crm.nebular.sfa.worksignrule.resp.SfaWorkSignPersonnelRespVo;
import com.biz.crm.util.CollectionUtil;
import com.biz.crm.util.CrmBeanUtil;
import com.biz.crm.util.StringUtils;
import com.biz.crm.worksignrule.mapper.SfaWorkSignPersonnelMapper;
import com.biz.crm.worksignrule.model.SfaWorkSignPersonnelEntity;
import com.biz.crm.worksignrule.service.ISfaWorkSignPersonnelService;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 考勤规则-打卡人员 接口实现
 *
 * @author liuhongming
 * @date 2020-09-16 15:06:20
 */
@Slf4j
@Service
@Transactional
@ConditionalOnMissingBean(name="SfaWorkSignPersonnelServiceExpandImpl")
public class SfaWorkSignPersonnelServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<SfaWorkSignPersonnelMapper, SfaWorkSignPersonnelEntity> implements ISfaWorkSignPersonnelService {

    @Resource
    private SfaWorkSignPersonnelMapper sfaWorkSignPersonnelMapper;

    /**
     * 列表
     * @param reqVo
     * @return
     */
    @Override
    @CrmLog
    public PageResult<SfaWorkSignPersonnelRespVo> findList(SfaWorkSignPersonnelReqVo reqVo){
        Page<SfaWorkSignPersonnelRespVo> page = new Page<>(reqVo.getPageNum(), reqVo.getPageSize());
        List<SfaWorkSignPersonnelRespVo> list = sfaWorkSignPersonnelMapper.findList(page, reqVo);
        return PageResult.<SfaWorkSignPersonnelRespVo>builder()
                .data(list)
                .count(page.getTotal())
                .build();
    }

    /**
     * 关联规则表查询考勤组织
     * @param wspCodes 组织编码集合
     * @param nonRuleCode 非指定规则码
     * @param ruleCode 指定规则码
     * @param enableStatus 规则状态
     * @return
     */
    @Override
    @CrmLog
    public List<SfaWorkSignPersonnelRespVo> findListJoinRule(List<String> wspCodes, String nonRuleCode, String ruleCode, String enableStatus) {
        return sfaWorkSignPersonnelMapper.findListJoinRule(wspCodes, nonRuleCode, ruleCode, enableStatus);
    }

    /**
     * 根据规则编码查询 Mapping 列表
     * @param ruleCodes
     * @return
     */
    @Override
    public Map<String, List<SfaWorkSignPersonnelEntity>> selectMappingByRuleCodes(Set<String> ruleCodes) {
        if(org.springframework.util.CollectionUtils.isEmpty(ruleCodes)){
            return Maps.newHashMap();
        }
        return sfaWorkSignPersonnelMapper.selectList(Wrappers.lambdaQuery(SfaWorkSignPersonnelEntity.class)
                .in(SfaWorkSignPersonnelEntity :: getRuleCode, ruleCodes).eq(SfaWorkSignPersonnelEntity::getCodeType, WorkSignEnum.codeType.ORG.getVal()))
                .stream().collect(Collectors.groupingBy(SfaWorkSignPersonnelEntity :: getRuleCode));
    }

    /**
     * 查询
     * @param sfaWorkSignPersonnelReqVo
     * @return sfaWorkSignPersonnelRespVo
     */
    @Override
    @CrmLog
    public SfaWorkSignPersonnelRespVo query(SfaWorkSignPersonnelReqVo sfaWorkSignPersonnelReqVo){
        List<SfaWorkSignPersonnelRespVo> list = this.findList(sfaWorkSignPersonnelReqVo).getData();
        if(CollectionUtils.isEmpty(list)){
            return new SfaWorkSignPersonnelRespVo();
        }
        SfaWorkSignPersonnelRespVo sfaWorkSignPersonnelRespVo = list.get(0);
        return sfaWorkSignPersonnelRespVo;
    }

    /**
     * 新增
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @CrmLog
    public void save(SfaWorkSignPersonnelReqVo reqVo){
        SfaWorkSignPersonnelEntity entity = CrmBeanUtil.copy(reqVo,SfaWorkSignPersonnelEntity.class);
        this.save(entity);
    }

    /**
     * 更新
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @CrmLog
    public void update(SfaWorkSignPersonnelReqVo reqVo){
        SfaWorkSignPersonnelEntity entity = this.getById(reqVo.getId());
        this.updateById(entity);
    }

    /**
     * 删除
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @CrmLog
    public void deleteBatch(SfaWorkSignPersonnelReqVo reqVo){
        List<SfaWorkSignPersonnelEntity> sfaWorkSignPersonnelEntities = sfaWorkSignPersonnelMapper.selectBatchIds(reqVo.getIds());
        if(CollectionUtils.isNotEmpty(sfaWorkSignPersonnelEntities)){
            sfaWorkSignPersonnelEntities.forEach(o -> {
                 o.setDelFlag(CrmDelFlagEnum.DELETE.getCode());
            });
        }
        this.updateBatchById(sfaWorkSignPersonnelEntities);
    }

    /**
     * 启用
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @CrmLog
    public void enableBatch(SfaWorkSignPersonnelReqVo reqVo){
        //设置状态为启用
        List<SfaWorkSignPersonnelEntity> sfaWorkSignPersonnelEntities = sfaWorkSignPersonnelMapper.selectBatchIds(reqVo.getIds());
        if(CollectionUtils.isNotEmpty(sfaWorkSignPersonnelEntities)){
            sfaWorkSignPersonnelEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.ENABLE.getCode());
            });
        }
        this.updateBatchById(sfaWorkSignPersonnelEntities);
    }

    /**
     * 禁用
     * @param reqVo
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    @CrmLog
    public void disableBatch(SfaWorkSignPersonnelReqVo reqVo){
        //设置状态为禁用
        List<SfaWorkSignPersonnelEntity> sfaWorkSignPersonnelEntities = sfaWorkSignPersonnelMapper.selectBatchIds(reqVo.getIds());
        if(CollectionUtils.isNotEmpty(sfaWorkSignPersonnelEntities)){
                sfaWorkSignPersonnelEntities.forEach(o -> {
                o.setEnableStatus(CrmEnableStatusEnum.DISABLE.getCode());
            });
        }
        this.updateBatchById(sfaWorkSignPersonnelEntities);
    }

    /**
     * 根据考勤规则编码和考勤范围卫星编码查询列表
     * @param ruleCode
     * @param codeType
     * @return
     */
    @Override
    public List<SfaWorkSignPersonnelEntity> findByRuleCodeAndCodeType(String ruleCode, String codeType) {
        List<SfaWorkSignPersonnelEntity> entities = this.sfaWorkSignPersonnelMapper.selectList(
                Wrappers.<SfaWorkSignPersonnelEntity>lambdaQuery()
                .eq(StringUtils.isNotEmpty(ruleCode), SfaWorkSignPersonnelEntity::getRuleCode, ruleCode)
                .eq(StringUtils.isNotEmpty(codeType), SfaWorkSignPersonnelEntity::getCodeType, codeType)
        );
        if(CollectionUtil.listEmpty(entities)) {
            return Lists.newArrayList();
        }
        return entities;
    }
}
